require.config({"config": {
        "jsbuild":{"Magento_Catalog/js/product/list/columns/final-price.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'uiRegistry',\n    'mageUtils',\n    'uiCollection'\n], function (_, registry, utils, Collection) {\n    'use strict';\n\n    return Collection.extend({\n        defaults: {\n            label: false,\n            headerTmpl: 'ui/grid/columns/text',\n            showMinimalPrice: false,\n            showMaximumPrice: false,\n            useLinkForAsLowAs: false,\n            bodyTmpl: 'Magento_Catalog/product/final_price',\n            priceWrapperCssClasses: '',\n            priceWrapperAttr: {}\n        },\n\n        /**\n         * Get product final price.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} final price html\n         */\n        getPrice: function (row) {\n            return row['price_info']['formatted_prices']['final_price'];\n        },\n\n        /**\n         * UnsanitizedHtml version of getPrice.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} final price html\n         */\n        getPriceUnsanitizedHtml: function (row) {\n            return this.getPrice(row);\n        },\n\n        /**\n         * Get product regular price.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} regular price html\n         */\n        getRegularPrice: function (row) {\n            return row['price_info']['formatted_prices']['regular_price'];\n        },\n\n        /**\n         * UnsanitizedHtml version of getRegularPrice.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} regular price html\n         */\n        getRegularPriceUnsanitizedHtml: function (row) {\n            return this.getRegularPrice(row);\n        },\n\n        /**\n         * Check if product has a price range.\n         *\n         * @param {Object} row\n         * @return {Boolean}\n         */\n        hasPriceRange: function (row) {\n            return row['price_info']['max_regular_price'] !== row['price_info']['min_regular_price'];\n        },\n\n        /**\n         * Check if product has special price.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} special price html\n         */\n        hasSpecialPrice: function (row) {\n            return row['price_info']['regular_price'] > row['price_info']['final_price'];\n        },\n\n        /**\n         * Check if product has minimal price.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} minimal price html\n         */\n        isMinimalPrice: function (row) {\n            return row['price_info']['minimal_price'] < row['price_info']['final_price'];\n        },\n\n        /**\n         * Get product minimal price.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} minimal price html\n         */\n        getMinimalPrice: function (row) {\n            return row['price_info']['formatted_prices']['minimal_price'];\n        },\n\n        /**\n         * UnsanitizedHtml version of getMinimalPrice.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} minimal price html\n         */\n        getMinimalPriceUnsanitizedHtml: function (row) {\n            return this.getMinimalPrice(row);\n        },\n\n        /**\n         * Check if product is salable.\n         *\n         * @param {Object} row\n         * @return {Boolean}\n         */\n        isSalable: function (row) {\n            return row['is_salable'];\n        },\n\n        /**\n         * Get product maximum price.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} maximum price html\n         */\n        getMaxPrice: function (row) {\n            return row['price_info']['formatted_prices']['max_price'];\n        },\n\n        /**\n         * UnsanitizedHtml version of getMaxPrice.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} maximum price html\n         */\n        getMaxPriceUnsanitizedHtml: function (row) {\n            return this.getMaxPrice(row);\n        },\n\n        /**\n         * Get product maximum regular price in case of price range and special price.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} maximum regular price html\n         */\n        getMaxRegularPrice: function (row) {\n            return row['price_info']['formatted_prices']['max_regular_price'];\n        },\n\n        /**\n         * UnsanitizedHtml version of getMaxRegularPrice.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} maximum regular price html\n         */\n        getMaxRegularPriceUnsanitizedHtml: function (row) {\n            return this.getMaxRegularPrice(row);\n        },\n\n        /**\n         * Get product minimal regular price in case of price range and special price.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} minimal regular price html\n         */\n        getMinRegularPrice: function (row) {\n            return row['price_info']['formatted_prices']['min_regular_price'];\n        },\n\n        /**\n         * UnsanitizedHtml version of getMinRegularPrice.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} minimal regular price html\n         */\n        getMinRegularPriceUnsanitizedHtml: function (row) {\n            return this.getMinRegularPrice(row);\n        },\n\n        /**\n         * Get adjustments names and return as string.\n         *\n         * @return {String} adjustments classes\n         */\n        getAdjustmentCssClasses: function () {\n            return _.pluck(this.getAdjustments(), 'index').join(' ');\n        },\n\n        /**\n         * Get product minimal price as number.\n         *\n         * @param {Object} row\n         * @return {Number} minimal price amount\n         */\n        getMinimalPriceAmount: function (row) {\n            return row['price_info']['minimal_price'];\n        },\n\n        /**\n         * UnsanitizedHtml version of getMinimalPriceAmount\n         *\n         * @param {Object} row\n         * @return {Number} minimal price amount\n         */\n        getMinimalPriceAmountUnsanitizedHtml: function (row) {\n            return this.getMinimalPriceAmount(row);\n        },\n\n        /**\n         * Get product minimal regular price as number in case of special price.\n         *\n         * @param {Object} row\n         * @return {Number} minimal regular price amount\n         */\n        getMinimalRegularPriceAmount: function (row) {\n            return row['price_info']['min_regular_price'];\n        },\n\n        /**\n         * Get product maximum price as number.\n         *\n         * @param {Object} row\n         * @return {Number} maximum price amount\n         */\n        getMaximumPriceAmount: function (row) {\n            return row['price_info']['max_price'];\n        },\n\n        /**\n         * Get product maximum regular price as number in case of special price.\n         *\n         * @param {Object} row\n         * @return {Number} maximum regular price amount\n         */\n        getMaximumRegularPriceAmount: function (row) {\n            return row['price_info']['max_regular_price'];\n        },\n\n        /**\n         * Check if minimal regular price exist for product.\n         *\n         * @param {Object} row\n         * @return {Boolean}\n         */\n        showMinRegularPrice: function (row) {\n            return this.getMinimalPriceAmount(row) < this.getMinimalRegularPriceAmount(row);\n        },\n\n        /**\n         * Check if maximum regular price exist for product.\n         *\n         * @param {Object} row\n         * @return {Boolean}\n         */\n        showMaxRegularPrice: function (row) {\n            return this.getMaximumPriceAmount(row) < this.getMaximumRegularPriceAmount(row);\n        },\n\n        /**\n         * Get path to the columns' body template.\n         *\n         * @returns {String}\n         */\n        getBody: function () {\n            return this.bodyTmpl;\n        },\n\n        /**\n         * Get all price adjustments.\n         *\n         * @returns {Object}\n         */\n        getAdjustments: function () {\n            var adjustments = this.elems();\n\n            _.each(adjustments, function (adjustment) {\n                adjustment.setPriceType(this.priceType);\n                adjustment.source = this.source;\n            }, this);\n\n            return adjustments;\n        }\n    });\n});\n","Magento_Catalog/js/product/list/columns/image.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'Magento_Ui/js/grid/columns/column',\n    'Magento_Catalog/js/product/list/column-status-validator'\n], function (_, Element, columnStatusValidator) {\n    'use strict';\n\n    return Element.extend({\n        defaults: {\n            bodyTmpl: 'Magento_Catalog/product/list/columns/image',\n            imageCode: 'default',\n            image: {}\n        },\n\n        /**\n         * Find image by code in scope of images\n         *\n         * @param {Object} images\n         * @returns {*|T}\n         */\n        getImage: function (images) {\n            return _.filter(images, function (image) {\n                return this.imageCode === image.code;\n            }, this).pop();\n        },\n\n        /**\n         * Get image path.\n         *\n         * @param {Object} row\n         * @return {String}\n         */\n        getImageUrl: function (row) {\n            return this.getImage(row.images).url;\n        },\n\n        /**\n         * Get image box width.\n         *\n         * @param {Object} row\n         * @return {Number}\n         */\n        getWidth: function (row) {\n            return this.getImage(row.images).width;\n        },\n\n        /**\n         * Get image box height.\n         *\n         * @param {Object} row\n         * @return {Number}\n         */\n        getHeight: function (row) {\n            return this.getImage(row.images).height;\n        },\n\n        /**\n         * Get resized image width.\n         *\n         * @param {Object} row\n         * @return {Number}\n         */\n        getResizedImageWidth: function (row) {\n            return this.getImage(row.images)['resized_width'];\n        },\n\n        /**\n         * Get resized image height.\n         *\n         * @param {Object} row\n         * @return {Number}\n         */\n        getResizedImageHeight: function (row) {\n            return this.getImage(row.images)['resized_height'];\n        },\n\n        /**\n         * Get image alt text.\n         *\n         * @param {Object} row\n         * @return {String}\n         */\n        getLabel: function (row) {\n            if (!this.imageExists(row)) {\n                return this._super();\n            }\n\n            return this.getImage(row.images).label;\n        },\n\n        /**\n         * Check if image exist.\n         *\n         * @param {Object} row\n         * @return {Boolean}\n         */\n        imageExists: function (row) {\n            return this.getImage(row.images) !== 'undefined';\n        },\n\n        /**\n         * Check if component must be shown.\n         *\n         * @return {Boolean}\n         */\n        isAllowed: function () {\n            return columnStatusValidator.isValid(this.source(), 'image', 'show_attributes');\n        }\n    });\n});\n","Magento_Catalog/js/product/list/columns/price-box.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'ko',\n    'underscore',\n    'uiRegistry',\n    'mageUtils',\n    'uiCollection',\n    'Magento_Catalog/js/product/list/column-status-validator',\n    'uiLayout'\n], function (ko, _, registry, utils, Collection, columnStatusValidator, layout) {\n    'use strict';\n\n    return Collection.extend({\n        defaults: {\n            label: '',\n            hasSpecialPrice: false,\n            showMinimalPrice: false,\n            useLinkForAsLowAs: false,\n            visible: true,\n            headerTmpl: 'ui/grid/columns/text',\n            bodyTmpl: 'Magento_Catalog/product/price/price_box',\n            disableAction: false,\n            controlVisibility: true,\n            sortable: false,\n            sorting: false,\n            draggable: true,\n            fieldClass: {},\n            renders: {\n                default: {}\n            },\n            ignoreTmpls: {\n                fieldAction: true\n            },\n            statefull: {\n                visible: true,\n                sorting: true\n            },\n            imports: {\n                exportSorting: 'sorting'\n            },\n            listens: {\n                elems: ''\n            },\n            modules: {\n                source: '${ $.provider }'\n            },\n            pricesInit: {}\n        },\n\n        /**\n         * Sort prices api\n         *\n         * @returns {exports}\n         */\n        sort: function () {\n            return this;\n        },\n\n        /**\n         * Check whether is allowed to render price or not\n         *\n         * @returns {*}\n         */\n        isAllowed: function () {\n            return columnStatusValidator.isValid(this.source(), 'price', 'show_attributes');\n        },\n\n        /**\n         * Retrieve array of prices, that should be rendered for specific product\n         *\n         * @param {Array} row\n         * @return {Array}\n         */\n        getPrices: function (row) {\n            var elems = this.elems() ? this.elems() : ko.getObservable(this, 'elems'),\n                result;\n\n            //we cant take type of product from row\n            this.initPrices(row);\n            result = _.filter(elems, function (elem) {\n                return elem.productType === row.type;\n            });\n\n            return result;\n        },\n\n        /**\n         * Recursive Merging of objects\n         *\n         * @param {Array} target\n         * @param {Array} source\n         * @returns {Array}\n         * @private\n         */\n        _deepObjectExtend: function (target, source) {\n            var _target = utils.copy(target);\n\n            _.each(source, function (value, key) {\n                if (_.keys(value).length && typeof _target[key] !== 'undefined') {\n                    _target[key] = this._deepObjectExtend(_target[key], value);\n                } else {\n                    _target[key] = value;\n                }\n            }, this);\n\n            return _target;\n        },\n\n        /**\n         * Init price type box, in cases when product type has custom component or bodyTmpl\n         *\n         * @param {String} productType\n         * @private\n         */\n        _initPriceWithCustomMetaData: function (productType) {\n            var price = this._deepObjectExtend(\n                this.renders.prices['default'],\n                this.renders.prices[productType]\n            );\n\n            price.name = productType + '.default';\n            price.parent = this.name;\n            price.source = this.source;\n            price.productType = productType;\n            layout([price]);\n        },\n\n        /**\n         * Init Prices by product type and add them to layout\n         *\n         * @param {Array} _priceData\n         * @param {String} productType\n         * @private\n         */\n        _initPricesForProductType: function (_priceData, productType) {\n            var prices = [];\n\n            this._setPriceNamesToPrices(_priceData, productType);\n            _.sortBy(_priceData, this._comparePrices);\n\n            _.each(_priceData, function (priceData) {\n                if (!priceData.component) {\n                    return;\n                }\n\n                priceData.parent = this.name;\n                priceData.provider = this.provider;\n                priceData.productType = productType;\n                priceData = utils.template(priceData, this);\n                prices.push(priceData);\n            }, this);\n\n            layout(prices);\n        },\n\n        /**\n         * Init dynamic price components\n         *\n         * @param {Array} row\n         * @returns {void}\n         */\n        initPrices: function (row) {\n            var _priceData = [],\n                productType = row.type,\n                defaultPrice = this.renders.prices['default'];\n\n            if (this.pricesInit[productType]) {\n                return true;\n            }\n\n            this.pricesInit[productType] = true;\n\n            if (this.renders.prices[productType] && this._needToApplyCustomTemplate(this.renders.prices[productType])) {\n                return this._initPriceWithCustomMetaData(productType);\n            }\n\n            if (this.renders.prices[productType] && this.renders.prices[productType].children) {\n                _priceData = this._deepObjectExtend(defaultPrice.children, this.renders.prices[productType].children);\n            } else {\n                _priceData = defaultPrice.children;\n            }\n\n            return this._initPricesForProductType(_priceData, productType);\n        },\n\n        /**\n         * Set name to all price components\n         *\n         * @param {Array} prices\n         * @param {String} productType\n         * @private\n         */\n        _setPriceNamesToPrices: function (prices, productType) {\n            _.each(prices, function (price, name) {\n                price.priceType = name;\n                price.name = name + '.' + productType;\n            });\n\n            return prices;\n        },\n\n        /**\n         * Sort callback to compare prices by sort order\n         *\n         * @param {Number} firstPrice\n         * @param {Number} secondPrice\n         * @returns {Number}\n         * @private\n         */\n        _comparePrices: function (firstPrice, secondPrice) {\n            if (firstPrice.sortOrder < secondPrice.sortOrder) {\n                return -1;\n            }\n\n            if (firstPrice.sortOrder > secondPrice.sortOrder) {\n                return 1;\n            }\n\n            return 0;\n        },\n\n        /**\n         * Check whether metadata of product type prices was changed, and we should\n         * to apply custom template or custom component\n         *\n         * @param {Array} productData\n         * @returns {*}\n         * @private\n         */\n        _needToApplyCustomTemplate: function (productData) {\n            return productData.bodyTmpl || productData.component;\n        },\n\n        /**\n         * Returns path to the columns' body template.\n         *\n         * @returns {String}\n         */\n        getBody: function () {\n            return this.bodyTmpl;\n        },\n\n        /**\n         * Get price label.\n         *\n         * @returns {String}\n         */\n        getLabel: function () {\n            return this.label;\n        }\n    });\n});\n","Magento_Catalog/js/product/list/columns/pricetype-box.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'ko',\n    'underscore',\n    'uiCollection'\n], function (ko, _, Collection) {\n    'use strict';\n\n    return Collection.extend({\n        /**\n         * Find from all price ui components, price with specific code, init source on it and set priceType\n         *\n         * @param {String} code\n         * @returns {*|T}\n         */\n        getPriceByCode: function (code) {\n            var elems = this.elems() ? this.elems() : ko.getObservable(this, 'elems'),\n                price;\n\n            price = _.filter(elems, function (elem) {\n                return elem.index.split('.').shift() === code;\n            }, this).pop();\n\n            price.source = this.source();\n            price.priceType = code;\n\n            return price;\n        },\n\n        /**\n         * Retrieve body template\n         *\n         * @returns {String}\n         */\n        getBody: function () {\n            return this.bodyTmpl;\n        },\n\n        /**\n         * Check whether price has price range, depends on different options, that can be choose\n         *\n         * @param {Object} row\n         * @returns {Boolean}\n         */\n        hasPriceRange: function (row) {\n            return row['price_info']['max_regular_price'] !== row['price_info']['min_regular_price'];\n        }\n    });\n});\n","Magento_Catalog/js/product/storage/data-storage.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'underscore',\n    'ko',\n    'mageUtils',\n    'Magento_Catalog/js/product/query-builder',\n    'Magento_Customer/js/customer-data',\n    'jquery/jquery-storageapi'\n], function ($, _, ko, utils, queryBuilder, customerData) {\n    'use strict';\n\n    /**\n     * Process data from API request\n     *\n     * @param {Object} data\n     * @returns {Object}\n     */\n    function getParsedDataFromServer(data) {\n        var result = {};\n\n        _.each(data.items, function (item) {\n                if (item.id) {\n                    result[item.id] = item;\n                }\n            }\n        );\n\n        return {\n            items: result\n        };\n    }\n\n    /**\n     * Set data to localStorage with support check.\n     *\n     * @param {String} namespace\n     * @param {Object} data\n     */\n    function setLocalStorageItem(namespace, data) {\n        try {\n            window.localStorage.setItem(namespace, JSON.stringify(data));\n        } catch (e) {\n            console.warn('localStorage is unavailable - skipping local caching of product data');\n            console.error(e);\n        }\n    }\n\n    return {\n\n        /**\n         * Class name\n         */\n        name: 'DataStorage',\n        request: {},\n        customerDataProvider: 'product_data_storage',\n\n        /**\n         * Initialize class\n         *\n         * @return Chainable.\n         */\n        initialize: function () {\n            if (!this.data) {\n                this.data = ko.observable({});\n            }\n\n            this.initLocalStorage()\n                .initCustomerDataReloadListener()\n                .cachesDataFromLocalStorage()\n                .initDataListener()\n                .initProvideStorage()\n                .initProviderListener();\n\n            return this;\n        },\n\n        /**\n         * Initialize listener to customer data reload\n         *\n         * @return Chainable.\n         */\n        initCustomerDataReloadListener: function () {\n            $(document).on('customer-data-invalidate', this._flushProductStorage.bind(this));\n\n            return this;\n        },\n\n        /**\n         * Flush product storage\n         *\n         * @private\n         * @return void\n         */\n        _flushProductStorage: function (event, sections) {\n            if (_.isEmpty(sections) || _.contains(sections, 'product_data_storage')) {\n                this.localStorage.removeAll();\n            }\n        },\n\n        /**\n         * Initialize listener to data property\n         *\n         * @return Chainable.\n         */\n        initDataListener: function () {\n            this.data.subscribe(this.dataHandler.bind(this));\n\n            return this;\n        },\n\n        /**\n         * Initialize provider storage\n         *\n         * @return Chainable.\n         */\n        initProvideStorage: function () {\n            this.providerHandler(customerData.get(this.customerDataProvider)());\n\n            return this;\n        },\n\n        /**\n         * Handler to update \"data\" property.\n         * Sets data to localStorage\n         *\n         * @param {Object} data\n         */\n        dataHandler: function (data) {\n            if (_.isEmpty(data)) {\n                this.localStorage.removeAll();\n            } else {\n                setLocalStorageItem(this.namespace, data);\n            }\n        },\n\n        /**\n         * Handler to update data in provider.\n         *\n         * @param {Object} data\n         */\n        providerHandler: function (data) {\n            var currentData = utils.copy(this.data()),\n                ids = _.keys(data.items);\n\n            if (data.items && ids.length) {\n                //we can extend only items\n                data = data.items;\n                this.data(_.extend(data, currentData));\n            }\n        },\n\n        /**\n         * Sets data ids\n         *\n         * @param {String} currency\n         * @param {String} store\n         * @param {Object} ids\n         */\n        setIds: function (currency, store, ids) {\n            if (!this.hasInCache(currency, store, ids)) {\n                this.loadDataFromServer(currency, store, ids);\n            } else {\n                this.data.valueHasMutated();\n            }\n        },\n\n        /**\n         * Gets data from \"data\" property by identifiers\n         *\n         * @param {String} currency\n         * @param {String} store\n         * @param {Object} productIdentifiers\n         *\n         * @return {Object} data.\n         */\n        getDataByIdentifiers: function (currency, store, productIdentifiers) {\n            var data = {},\n                dataCollection = this.data(),\n                id;\n\n            for (id in productIdentifiers) {\n                if (productIdentifiers.hasOwnProperty(id)) {\n                    data[id] = dataCollection[id];\n                }\n            }\n\n            return data;\n        },\n\n        /**\n         * Checks has cached data or not\n         *\n         * @param {String} currency\n         * @param {String} store\n         * @param {Object} ids\n         *\n         * @return {Boolean}\n         */\n        hasInCache: function (currency, store, ids) {\n            var data = this.data(),\n                id;\n\n            for (id in ids) {\n                if (!data.hasOwnProperty(id) ||\n                    data[id]['currency_code'] !== currency ||\n                    ~~data[id]['store_id'] !== ~~store\n                ) {\n                    return false;\n                }\n            }\n\n            return true;\n        },\n\n        /**\n         * Load data from server by ids\n         *\n         * @param {String} currency\n         * @param {String} store\n         * @param {Object} ids\n         *\n         * @return void\n         */\n        loadDataFromServer: function (currency, store, ids) {\n            var idsArray = _.keys(ids),\n                prepareAjaxParams = {\n                    'entity_id': idsArray.join(',')\n                };\n\n            if (this.request.sent && this.hasIdsInSentRequest(ids)) {\n                return;\n            }\n\n            this.request = {\n                sent: true,\n                data: ids\n            };\n\n            this.updateRequestConfig.data = queryBuilder.buildQuery(prepareAjaxParams);\n            this.updateRequestConfig.data['store_id'] = store;\n            this.updateRequestConfig.data['currency_code'] = currency;\n            $.ajax(this.updateRequestConfig).done(function (data) {\n                this.request = {};\n                this.providerHandler(getParsedDataFromServer(data));\n            }.bind(this));\n        },\n\n        /**\n         * Each product page consist product cache data,\n         * this function prepare those data to appropriate view, and save it\n         *\n         * @param {Object} data\n         */\n        addDataFromPageCache: function (data) {\n            this.providerHandler(getParsedDataFromServer(data));\n        },\n\n        /**\n         * @param {Object} ids\n         * @returns {Boolean}\n         */\n        hasIdsInSentRequest: function (ids) {\n            var sentDataIds,\n                currentDataIds;\n\n            if (this.request.data) {\n                sentDataIds = _.keys(this.request.data);\n                currentDataIds = _.keys(ids);\n\n                _.each(currentDataIds, function (id) {\n                    if (_.lastIndexOf(sentDataIds, id) === -1) {\n                        return false;\n                    }\n                });\n\n                return true;\n            }\n\n            return false;\n        },\n\n        /**\n         * Initialize provider listener\n         *\n         * @return Chainable.\n         */\n        initProviderListener: function () {\n            customerData.get(this.customerDataProvider).subscribe(this.providerHandler.bind(this));\n\n            return this;\n        },\n\n        /**\n         * Caches data from local storage to local scope\n         *\n         * @return Chainable.\n         */\n        cachesDataFromLocalStorage: function () {\n            this.data(this.getDataFromLocalStorage());\n\n            return this;\n        },\n\n        /**\n         * Gets data from local storage by current namespace\n         *\n         * @return {Object}.\n         */\n        getDataFromLocalStorage: function () {\n            return this.localStorage.get();\n        },\n\n        /**\n         * Initialize localStorage\n         *\n         * @return Chainable.\n         */\n        initLocalStorage: function () {\n            this.localStorage = $.initNamespaceStorage(this.namespace).localStorage;\n\n            return this;\n        }\n    };\n});\n","Magento_Catalog/js/product/storage/ids-storage-compare.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'ko',\n    'mageUtils',\n    'Magento_Customer/js/customer-data',\n    'Magento_Catalog/js/product/storage/ids-storage'\n], function (_, ko, utils, customerData, idsStorage) {\n    'use strict';\n\n    return _.extend(utils.copy(idsStorage), {\n\n        /**\n         * Class name\n         */\n        name: 'IdsStorageCompare',\n\n        /**\n         * Initializes class\n         *\n         * @return Chainable.\n         */\n        initialize: function () {\n            if (!this.data) {\n                this.data = ko.observable({});\n            }\n\n            if (this.provider && window.checkout && window.checkout.baseUrl) {\n                this.providerDataHandler(customerData.get(this.provider)());\n                this.initProviderListener();\n            }\n\n            this.initLocalStorage()\n                .cachesDataFromLocalStorage()\n                .initDataListener();\n\n            return this;\n        },\n\n        /**\n         * Initializes listener for external data provider\n         */\n        initProviderListener: function () {\n            customerData.get(this.provider).subscribe(this.providerDataHandler.bind(this));\n        },\n\n        /**\n         * Initializes handler for external data provider update\n         *\n         * @param {Object} data\n         */\n        providerDataHandler: function (data) {\n            data = data.items || data;\n            data = this.prepareData(data);\n\n            this.add(data);\n        },\n\n        /**\n         * Prepares data to correct interface\n         *\n         * @param {Object} data\n         *\n         * @returns {Object} data\n         */\n        prepareData: function (data) {\n            var result = {},\n                scopeId;\n\n            _.each(data, function (item) {\n                if (typeof item.productScope !== 'undefined') {\n                    scopeId = item.productScope === 'store' ? window.checkout.storeId :\n                        item.productScope === 'group' ? window.checkout.storeGroupId :\n                            window.checkout.websiteId;\n\n                    result[item.productScope + '-' + scopeId + '-' + item.id] = {\n                        'added_at': new Date().getTime() / 1000,\n                        'product_id': item.id,\n                        'scope_id': scopeId\n                    };\n                } else {\n                    result[item.id] = {\n                        'added_at': new Date().getTime() / 1000,\n                        'product_id': item.id\n                    };\n                }\n            });\n\n            return result;\n        }\n    });\n});\n","Magento_Catalog/js/product/storage/ids-storage.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'underscore',\n    'ko',\n    'mageUtils',\n    'jquery/jquery-storageapi'\n], function ($, _, ko, utils) {\n    'use strict';\n\n    /**\n     * Set data to localStorage with support check.\n     *\n     * @param {String} namespace\n     * @param {Object} data\n     */\n    function setLocalStorageItem(namespace, data) {\n        try {\n            window.localStorage.setItem(namespace, JSON.stringify(data));\n        } catch (e) {\n            console.warn('localStorage is unavailable - skipping local caching of product data');\n            console.error(e);\n        }\n    }\n\n    return {\n\n        /**\n         * Class name\n         */\n        name: 'IdsStorage',\n\n        /**\n         * Initializes class\n         *\n         * @return Chainable.\n         */\n        initialize: function () {\n            if (!this.data) {\n                this.data = ko.observable({});\n            }\n\n            this.initCustomerDataReloadListener()\n                .initLocalStorage()\n                .cachesDataFromLocalStorage()\n                .initDataListener();\n\n            return this;\n        },\n\n        /**\n         * Gets data from local storage by current namespace\n         *\n         * @return {Object}.\n         */\n        getDataFromLocalStorage: function () {\n            return this.localStorage.get();\n        },\n\n        /**\n         * Caches data from local storage to local scope\n         *\n         * @return Chainable.\n         */\n        cachesDataFromLocalStorage: function () {\n            this.data(this.getDataFromLocalStorage());\n\n            return this;\n        },\n\n        /**\n         * Initialize localStorage\n         *\n         * @return Chainable.\n         */\n        initLocalStorage: function () {\n            this.localStorage = $.initNamespaceStorage(this.namespace).localStorage;\n\n            return this;\n        },\n\n        /**\n         * Initializes listener to \"data\" property\n         */\n        initDataListener: function () {\n            this.data.subscribe(this.internalDataHandler.bind(this));\n        },\n\n        /**\n         * Initialize listener to customer data reload\n         *\n         * @return Chainable.\n         */\n        initCustomerDataReloadListener: function () {\n            $(document).on('customer-data-reload', function (event, sections) {\n                if ((_.isEmpty(sections) || _.contains(sections, this.namespace)) && ~~this.allowToSendRequest) {\n                    this.localStorage.removeAll();\n                    this.data();\n                }\n            }.bind(this));\n\n            return this;\n        },\n\n        /**\n         * Initializes handler to \"data\" property update\n         */\n        internalDataHandler: function (data) {\n            setLocalStorageItem(this.namespace, data);\n        },\n\n        /**\n         * Initializes handler to storage update\n         */\n        externalDataHandler: function (data) {\n            data = data.items ? data.items : data;\n\n            this.set(_.extend(utils.copy(this.data()), data));\n        }\n    };\n});\n\n","Magento_Catalog/js/product/storage/storage-service.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'underscore',\n    'mageUtils',\n    'mage/translate',\n    'Magento_Catalog/js/product/storage/ids-storage',\n    'Magento_Catalog/js/product/storage/data-storage',\n    'Magento_Catalog/js/product/storage/ids-storage-compare'\n], function ($, _, utils, $t, IdsStorage, DataStore, IdsStorageCompare) {\n    'use strict';\n\n    return (function () {\n\n        var /**\n             * {Object} storages - list of storages\n             */\n            storages = {},\n\n            /**\n             * {Object} classes - list of classes\n             */\n            classes = {},\n\n            /**\n             * {Object} prototype - methods that will be added to all storage classes to prototype property.\n             */\n            prototype = {\n\n                /**\n                 * Sets data to storage\n                 *\n                 * @param {*} data\n                 */\n                set: function (data) {\n                    if (!utils.compare(data, this.data()).equal) {\n                        this.data(data);\n                    }\n                },\n\n                /**\n                 * Adds some data to current storage data\n                 *\n                 * @param {*} data\n                 */\n                add: function (data) {\n                    if (!_.isEmpty(data)) {\n                        this.data(_.extend(utils.copy(this.data()), data));\n                    }\n                },\n\n                /**\n                 * Gets current storage data\n                 *\n                 * @returns {*} data\n                 */\n                get: function () {\n                    return this.data();\n                }\n            },\n\n            /**\n             * Required properties to storage\n             */\n            storagesInterface =  {\n                data: 'function',\n                initialize: 'function',\n                namespace: 'string'\n            },\n\n            /**\n             * Private service methods\n             */\n            _private = {\n\n                /**\n                 * Overrides class method and add ability use _super to call parent method\n                 *\n                 * @param {Object} extensionMethods\n                 * @param {Object} originInstance\n                 */\n                overrideClassMethods: function (extensionMethods, originInstance) {\n                    var methodsName = _.keys(extensionMethods),\n                        i = 0,\n                        length = methodsName.length;\n\n                    for (i; i < length; i++) {\n                        if (_.isFunction(originInstance[methodsName[i]])) {\n                            originInstance[methodsName[i]] = extensionMethods[methodsName[i]];\n                        }\n                    }\n\n                    return originInstance;\n                },\n\n                /**\n                 * Checks is storage implement interface\n                 *\n                 * @param {Object} classInstance\n                 *\n                 * @returns {Boolean}\n                 */\n                isImplementInterface: function (classInstance) {\n                    _.each(storagesInterface, function (key, value) {\n                        if (typeof classInstance[key] !== value) {\n                            return false;\n                        }\n                    });\n\n                    return true;\n                }\n            },\n\n            /**\n             * Subscribers list\n             */\n            subsctibers = {};\n\n        (function () {\n            /**\n             * @param {Object} config\n             * @return void\n             */\n            classes[IdsStorage.name] = function (config) {\n                _.extend(this, IdsStorage, config);\n            };\n\n            /**\n             * @param {Object} config\n             * @return void\n             */\n            classes[IdsStorageCompare.name] = function (config) {\n                _.extend(this, IdsStorageCompare, config);\n            };\n\n            /**\n             * @param {Object} config\n             * @return void\n             */\n            classes[DataStore.name] = function (config) {\n                _.extend(this, DataStore, config);\n            };\n\n            _.each(classes, function (classItem) {\n                classItem.prototype = prototype;\n            });\n        })();\n\n        return {\n\n            /**\n             * Creates new storage or returns if storage with declared namespace exist\n             *\n             * @param {Object} config - storage config\n             * @throws {Error}\n             * @returns {Object} storage instance\n             */\n            createStorage: function (config) {\n                var instance,\n                    initialized;\n\n                if (storages[config.namespace]) {\n                    return storages[config.namespace];\n                }\n\n                instance = new classes[config.className](config);\n\n                if (_private.isImplementInterface(instance)) {\n                    initialized = storages[config.namespace] = instance.initialize();\n                    this.processSubscribers(initialized, config);\n\n                    return initialized;\n                }\n\n                throw new Error('Class ' + config.className + $t('does not implement Storage Interface'));\n            },\n\n            /**\n             * Process subscribers\n             *\n             * Differentiate subscribers by their namespaces: recently_viewed or recently_compared\n             * and process callbacks. Callbacks can be add through onStorageInit function\n             *\n             * @param {Object} initialized\n             * @param {Object} config\n             * @return void\n             */\n            processSubscribers: function (initialized, config) {\n                if (subsctibers[config.namespace]) {\n                    _.each(subsctibers[config.namespace], function (callback) {\n                        callback(initialized);\n                    });\n\n                    delete subsctibers[config.namespace];\n                }\n            },\n\n            /**\n             * Listens storage creating by namespace\n             *\n             * @param {String} namespace\n             * @param {Function} callback\n             * @return void\n             */\n            onStorageInit: function (namespace, callback) {\n                if (storages[namespace]) {\n                    callback(storages[namespace]);\n                } else {\n                    subsctibers[namespace] ?\n                        subsctibers[namespace].push(callback) :\n                        subsctibers[namespace] = [callback];\n                }\n            },\n\n            /**\n             * Gets storage by namespace\n             *\n             * @param {String} namespace\n             *\n             * @returns {Object} storage insance\n             */\n            getStorage: function (namespace) {\n                return storages[namespace];\n            }\n        };\n    })();\n});\n\n","Magento_Catalog/js/product/view/product-ids-resolver.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'Magento_Catalog/js/product/view/product-ids'\n], function (_, productIds) {\n    'use strict';\n\n    /**\n     * Returns id's of products in form.\n     *\n     * @param {jQuery} $form\n     * @return {Array}\n     */\n    return function ($form) {\n        var idSet = productIds(),\n            product = _.findWhere($form.serializeArray(), {\n            name: 'product'\n        });\n\n        if (!_.isUndefined(product)) {\n            idSet.push(product.value);\n        }\n\n        return _.uniq(idSet);\n    };\n});\n","Magento_Catalog/js/product/view/product-ids.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko'\n], function (ko) {\n    'use strict';\n\n    return ko.observableArray([]);\n});\n","Magento_Catalog/js/product/view/product-info-resolver.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'Magento_Catalog/js/product/view/product-info'\n], function (_, productInfo) {\n    'use strict';\n\n    /**\n     * Returns info about products in form.\n     *\n     * @param {jQuery} $form\n     * @return {Array}\n     */\n    return function ($form) {\n        var product = _.findWhere($form.serializeArray(), {\n                name: 'product'\n            });\n\n        if (!_.isUndefined(product)) {\n            productInfo().push(\n                {\n                    'id': product.value\n                }\n            );\n        }\n\n        return _.uniq(productInfo(), function (item) {\n            return item.id;\n        });\n    };\n});\n\n","Magento_Catalog/js/product/view/product-info.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko'\n], function (ko) {\n    'use strict';\n\n    return ko.observableArray([]);\n});\n","Magento_Catalog/js/product/view/provider.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'uiElement',\n    'Magento_Catalog/js/product/storage/storage-service'\n], function (_, Element, storage) {\n    'use strict';\n\n    return Element.extend({\n        defaults: {\n            identifiersConfig: {\n                namespace: 'recently_viewed_product'\n            },\n            productStorageConfig: {\n                namespace: 'product_data_storage',\n                updateRequestConfig: {\n                    method: 'GET',\n                    dataType: 'json'\n                },\n                className: 'DataStorage'\n            }\n        },\n\n        /**\n         * Initializes\n         *\n         * @returns {Object} Chainable.\n         */\n        initialize: function () {\n            this._super();\n\n            if (window.checkout && window.checkout.baseUrl) {\n                this.initIdsStorage();\n            }\n\n            this.initDataStorage();\n\n            return this;\n        },\n\n        /**\n         * Init ids storage\n         *\n         * @returns {Object} Chainable.\n         */\n        initIdsStorage: function () {\n            storage.onStorageInit(this.identifiersConfig.namespace, this.idsStorageHandler.bind(this));\n\n            return this;\n        },\n\n        /**\n         * Init data storage\n         *\n         * @returns {Object} Chainable.\n         */\n        initDataStorage: function () {\n            storage.onStorageInit(this.productStorageConfig.namespace, this.dataStorageHandler.bind(this));\n\n            return this;\n        },\n\n        /**\n         * Init data storage handler\n         *\n         * @param {Object} dataStorage - storage instance\n         */\n        dataStorageHandler: function (dataStorage) {\n            this.productStorage = dataStorage;\n            this.productStorage.add(this.data.items);\n        },\n\n        /**\n         * Init ids storage handler\n         *\n         * @param {Object} idsStorage - storage instance\n         */\n        idsStorageHandler: function (idsStorage) {\n            this.idsStorage = idsStorage;\n            this.idsStorage.add(this.getIdentifiers());\n        },\n\n        /**\n         * Gets ids from items\n         *\n         * @returns {Object}\n         */\n        getIdentifiers: function () {\n            var result = {},\n                productCurrentScope = this.data.productCurrentScope,\n                scopeId = productCurrentScope === 'store' ? window.checkout.storeId :\n                    productCurrentScope === 'group' ? window.checkout.storeGroupId :\n                        window.checkout.websiteId;\n\n            _.each(this.data.items, function (item, key) {\n                result[productCurrentScope + '-' + scopeId + '-' + key] = {\n                    'added_at': new Date().getTime() / 1000,\n                    'product_id': key,\n                    'scope_id': scopeId\n                };\n            }, this);\n\n            return result;\n        }\n    });\n});\n","Magento_Catalog/js/view/compare-products.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Customer/js/customer-data',\n    'jquery',\n    'mage/mage',\n    'mage/decorate'\n], function (Component, customerData, $) {\n    'use strict';\n\n    var sidebarInitialized = false;\n\n    /**\n     * Initialize sidebar\n     */\n    function initSidebar() {\n        if (sidebarInitialized) {\n            return;\n        }\n\n        sidebarInitialized = true;\n        $('[data-role=compare-products-sidebar]').decorate('list', true);\n    }\n\n    return Component.extend({\n        /** @inheritdoc */\n        initialize: function () {\n            this._super();\n            this.compareProducts = customerData.get('compare-products');\n\n            initSidebar();\n        }\n    });\n});\n","Magento_Catalog/js/view/image.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        /** @inheritdoc */\n        initialize: function () {\n            this._super();\n\n            this.template = window.checkout.imageTemplate || this.template;\n        }\n    });\n});\n","Magento_Catalog/product/view/validation.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget',\n    'mage/validation/validation'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.validation', $.mage.validation, {\n        options: {\n            radioCheckboxClosest: 'ul, ol',\n\n            /**\n             * @param {*} error\n             * @param {HTMLElement} element\n             */\n            errorPlacement: function (error, element) {\n                var messageBox,\n                    dataValidate;\n\n                if ($(element).hasClass('datetime-picker')) {\n                    element = $(element).parent();\n\n                    if (element.parent().find('.mage-error').length) {\n                        return;\n                    }\n                }\n\n                if (element.attr('data-errors-message-box')) {\n                    messageBox = $(element.attr('data-errors-message-box'));\n                    messageBox.html(error);\n\n                    return;\n                }\n\n                dataValidate = element.attr('data-validate');\n\n                if (dataValidate && dataValidate.indexOf('validate-one-checkbox-required-by-name') > 0) {\n                    error.appendTo('#links-advice-container');\n                } else if (element.is(':radio, :checkbox')) {\n                    element.closest(this.radioCheckboxClosest).after(error);\n                } else {\n                    element.after(error);\n                }\n            },\n\n            /**\n             * @param {HTMLElement} element\n             * @param {String} errorClass\n             */\n            highlight: function (element, errorClass) {\n                var dataValidate = $(element).attr('data-validate');\n\n                if (dataValidate && dataValidate.indexOf('validate-required-datetime') > 0) {\n                    $(element).parent().find('.datetime-picker').each(function () {\n                        $(this).removeClass(errorClass);\n\n                        if ($(this).val().length === 0) {\n                            $(this).addClass(errorClass);\n                        }\n                    });\n                } else if ($(element).is(':radio, :checkbox')) {\n                    $(element).closest(this.radioCheckboxClosest).addClass(errorClass);\n                } else {\n                    $(element).addClass(errorClass);\n                }\n            },\n\n            /**\n             * @param {HTMLElement} element\n             * @param {String} errorClass\n             */\n            unhighlight: function (element, errorClass) {\n                var dataValidate = $(element).attr('data-validate');\n\n                if (dataValidate && dataValidate.indexOf('validate-required-datetime') > 0) {\n                    $(element).parent().find('.datetime-picker').removeClass(errorClass);\n                } else if ($(element).is(':radio, :checkbox')) {\n                    $(element).closest(this.radioCheckboxClosest).removeClass(errorClass);\n                } else {\n                    $(element).removeClass(errorClass);\n                }\n            }\n        }\n    });\n\n    return $.mage.validation;\n});\n","Magento_CatalogSearch/js/search-terms-log.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mageUtils'\n], function ($, utils) {\n    'use strict';\n\n    return function (data) {\n        $.ajax({\n            method: 'GET',\n            url: data.url,\n            data: {\n                'q': utils.getUrlParameters(window.location.href).q\n            }\n        });\n    };\n});\n","Magento_Checkout/js/checkout-data.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * Checkout adapter for customer data storage\n *\n * @api\n */\ndefine([\n    'jquery',\n    'Magento_Customer/js/customer-data',\n    'mageUtils',\n    'jquery/jquery-storageapi'\n], function ($, storage, utils) {\n    'use strict';\n\n    var cacheKey = 'checkout-data',\n\n        /**\n         * @param {Object} data\n         */\n        saveData = function (data) {\n            storage.set(cacheKey, data);\n        },\n\n        /**\n         * @return {*}\n         */\n        initData = function () {\n            return {\n                'selectedShippingAddress': null, //Selected shipping address pulled from persistence storage\n                'shippingAddressFromData': null, //Shipping address pulled from persistence storage\n                'newCustomerShippingAddress': null, //Shipping address pulled from persistence storage for customer\n                'selectedShippingRate': null, //Shipping rate pulled from persistence storage\n                'selectedPaymentMethod': null, //Payment method pulled from persistence storage\n                'selectedBillingAddress': null, //Selected billing address pulled from persistence storage\n                'billingAddressFromData': null, //Billing address pulled from persistence storage\n                'newCustomerBillingAddress': null //Billing address pulled from persistence storage for new customer\n            };\n        },\n\n        /**\n         * @return {*}\n         */\n        getData = function () {\n            var data = storage.get(cacheKey)();\n\n            if ($.isEmptyObject(data)) {\n                data = $.initNamespaceStorage('mage-cache-storage').localStorage.get(cacheKey);\n\n                if ($.isEmptyObject(data)) {\n                    data = initData();\n                    saveData(data);\n                }\n            }\n\n            return data;\n        };\n\n    return {\n        /**\n         * Setting the selected shipping address pulled from persistence storage\n         *\n         * @param {Object} data\n         */\n        setSelectedShippingAddress: function (data) {\n            var obj = getData();\n\n            obj.selectedShippingAddress = data;\n            saveData(obj);\n        },\n\n        /**\n         * Pulling the selected shipping address from persistence storage\n         *\n         * @return {*}\n         */\n        getSelectedShippingAddress: function () {\n            return getData().selectedShippingAddress;\n        },\n\n        /**\n         * Setting the shipping address pulled from persistence storage\n         *\n         * @param {Object} data\n         */\n        setShippingAddressFromData: function (data) {\n            var obj = getData();\n\n            obj.shippingAddressFromData = utils.filterFormData(data);\n            saveData(obj);\n        },\n\n        /**\n         * Pulling the shipping address from persistence storage\n         *\n         * @return {*}\n         */\n        getShippingAddressFromData: function () {\n            return getData().shippingAddressFromData;\n        },\n\n        /**\n         * Setting the shipping address pulled from persistence storage for new customer\n         *\n         * @param {Object} data\n         */\n        setNewCustomerShippingAddress: function (data) {\n            var obj = getData();\n\n            obj.newCustomerShippingAddress = data;\n            saveData(obj);\n        },\n\n        /**\n         * Pulling the shipping address from persistence storage for new customer\n         *\n         * @return {*}\n         */\n        getNewCustomerShippingAddress: function () {\n            return getData().newCustomerShippingAddress;\n        },\n\n        /**\n         * Setting the selected shipping rate pulled from persistence storage\n         *\n         * @param {Object} data\n         */\n        setSelectedShippingRate: function (data) {\n            var obj = getData();\n\n            obj.selectedShippingRate = data;\n            saveData(obj);\n        },\n\n        /**\n         * Pulling the selected shipping rate from local storage\n         *\n         * @return {*}\n         */\n        getSelectedShippingRate: function () {\n            return getData().selectedShippingRate;\n        },\n\n        /**\n         * Setting the selected payment method pulled from persistence storage\n         *\n         * @param {Object} data\n         */\n        setSelectedPaymentMethod: function (data) {\n            var obj = getData();\n\n            obj.selectedPaymentMethod = data;\n            saveData(obj);\n        },\n\n        /**\n         * Pulling the payment method from persistence storage\n         *\n         * @return {*}\n         */\n        getSelectedPaymentMethod: function () {\n            return getData().selectedPaymentMethod;\n        },\n\n        /**\n         * Setting the selected billing address pulled from persistence storage\n         *\n         * @param {Object} data\n         */\n        setSelectedBillingAddress: function (data) {\n            var obj = getData();\n\n            obj.selectedBillingAddress = data;\n            saveData(obj);\n        },\n\n        /**\n         * Pulling the selected billing address from persistence storage\n         *\n         * @return {*}\n         */\n        getSelectedBillingAddress: function () {\n            return getData().selectedBillingAddress;\n        },\n\n        /**\n         * Setting the billing address pulled from persistence storage\n         *\n         * @param {Object} data\n         */\n        setBillingAddressFromData: function (data) {\n            var obj = getData();\n\n            obj.billingAddressFromData = utils.filterFormData(data);\n            saveData(obj);\n        },\n\n        /**\n         * Pulling the billing address from persistence storage\n         *\n         * @return {*}\n         */\n        getBillingAddressFromData: function () {\n            return getData().billingAddressFromData;\n        },\n\n        /**\n         * Setting the billing address pulled from persistence storage for new customer\n         *\n         * @param {Object} data\n         */\n        setNewCustomerBillingAddress: function (data) {\n            var obj = getData();\n\n            obj.newCustomerBillingAddress = data;\n            saveData(obj);\n        },\n\n        /**\n         * Pulling the billing address from persistence storage for new customer\n         *\n         * @return {*}\n         */\n        getNewCustomerBillingAddress: function () {\n            return getData().newCustomerBillingAddress;\n        },\n\n        /**\n         * Pulling the email address from persistence storage\n         *\n         * @return {*}\n         */\n        getValidatedEmailValue: function () {\n            var obj = getData();\n\n            return obj.validatedEmailValue ? obj.validatedEmailValue : '';\n        },\n\n        /**\n         * Setting the email address pulled from persistence storage\n         *\n         * @param {String} email\n         */\n        setValidatedEmailValue: function (email) {\n            var obj = getData();\n\n            obj.validatedEmailValue = email;\n            saveData(obj);\n        },\n\n        /**\n         * Pulling the email input field value from persistence storage\n         *\n         * @return {*}\n         */\n        getInputFieldEmailValue: function () {\n            var obj = getData();\n\n            return obj.inputFieldEmailValue ? obj.inputFieldEmailValue : '';\n        },\n\n        /**\n         * Setting the email input field value pulled from persistence storage\n         *\n         * @param {String} email\n         */\n        setInputFieldEmailValue: function (email) {\n            var obj = getData();\n\n            obj.inputFieldEmailValue = email;\n            saveData(obj);\n        },\n\n        /**\n         * Pulling the checked email value from persistence storage\n         *\n         * @return {*}\n         */\n        getCheckedEmailValue: function () {\n            var obj = getData();\n\n            return obj.checkedEmailValue ? obj.checkedEmailValue : '';\n        },\n\n        /**\n         * Setting the checked email value pulled from persistence storage\n         *\n         * @param {String} email\n         */\n        setCheckedEmailValue: function (email) {\n            var obj = getData();\n\n            obj.checkedEmailValue = email;\n            saveData(obj);\n        }\n    };\n});\n","Magento_Checkout/js/checkout-loader.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'rjsResolver'\n], function (resolver) {\n    'use strict';\n\n    /**\n     * Removes provided loader element from DOM.\n     *\n     * @param {HTMLElement} $loader - Loader DOM element.\n     */\n    function hideLoader($loader) {\n        $loader.parentNode.removeChild($loader);\n    }\n\n    /**\n     * Initializes assets loading process listener.\n     *\n     * @param {Object} config - Optional configuration\n     * @param {HTMLElement} $loader - Loader DOM element.\n     */\n    function init(config, $loader) {\n        resolver(hideLoader.bind(null, $loader));\n    }\n\n    return init;\n});\n","Magento_Checkout/js/discount-codes.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.discountCode', {\n        options: {\n        },\n\n        /** @inheritdoc */\n        _create: function () {\n            this.couponCode = $(this.options.couponCodeSelector);\n            this.removeCoupon = $(this.options.removeCouponSelector);\n\n            $(this.options.applyButton).on('click', $.proxy(function () {\n                this.couponCode.attr('data-validate', '{required:true}');\n                this.removeCoupon.attr('value', '0');\n                $(this.element).validation().trigger('submit');\n            }, this));\n\n            $(this.options.cancelButton).on('click', $.proxy(function () {\n                this.couponCode.removeAttr('data-validate');\n                this.removeCoupon.attr('value', '1');\n                this.element.trigger('submit');\n            }, this));\n        }\n    });\n\n    return $.mage.discountCode;\n});\n","Magento_Checkout/js/empty-cart.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['Magento_Customer/js/customer-data'], function (customerData) {\n    'use strict';\n\n    return function () {\n        var cartData = customerData.get('cart');\n\n        customerData.getInitCustomerData().done(function () {\n            if (cartData().items && cartData().items.length !== 0) {\n                customerData.reload(['cart'], false);\n            }\n        });\n    };\n});\n","Magento_Checkout/js/proceed-to-checkout.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Customer/js/model/authentication-popup',\n    'Magento_Customer/js/customer-data'\n], function ($, authenticationPopup, customerData) {\n    'use strict';\n\n    return function (config, element) {\n        $(element).on('click', function (event) {\n            var cart = customerData.get('cart'),\n                customer = customerData.get('customer');\n\n            event.preventDefault();\n\n            if (!customer().firstname && cart().isGuestCheckoutAllowed === false) {\n                authenticationPopup.showModal();\n\n                return false;\n            }\n            $(element).attr('disabled', true);\n            location.href = config.checkoutUrl;\n        });\n\n    };\n});\n","Magento_Checkout/js/region-updater.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/template',\n    'underscore',\n    'jquery-ui-modules/widget',\n    'mage/validation'\n], function ($, mageTemplate, _) {\n    'use strict';\n\n    $.widget('mage.regionUpdater', {\n        options: {\n            regionTemplate:\n                '<option value=\"<%- data.value %>\" <% if (data.isSelected) { %>selected=\"selected\"<% } %>>' +\n                    '<%- data.title %>' +\n                '</option>',\n            isRegionRequired: true,\n            isZipRequired: true,\n            isCountryRequired: true,\n            currentRegion: null,\n            isMultipleCountriesAllowed: true\n        },\n\n        /**\n         *\n         * @private\n         */\n        _create: function () {\n            this._initCountryElement();\n\n            this.currentRegionOption = this.options.currentRegion;\n            this.regionTmpl = mageTemplate(this.options.regionTemplate);\n\n            this._updateRegion(this.element.find('option:selected').val());\n\n            $(this.options.regionListId).on('change', $.proxy(function (e) {\n                this.setOption = false;\n                this.currentRegionOption = $(e.target).val();\n            }, this));\n\n            $(this.options.regionInputId).on('focusout', $.proxy(function () {\n                this.setOption = true;\n            }, this));\n        },\n\n        /**\n         *\n         * @private\n         */\n        _initCountryElement: function () {\n\n            if (this.options.isMultipleCountriesAllowed) {\n                this.element.parents('div.field').show();\n                this.element.on('change', $.proxy(function (e) {\n                    // clear region inputs on country change\n                    $(this.options.regionListId).val('');\n                    $(this.options.regionInputId).val('');\n                    this._updateRegion($(e.target).val());\n                }, this));\n\n                if (this.options.isCountryRequired) {\n                    this.element.addClass('required-entry');\n                    this.element.parents('div.field').addClass('required');\n                }\n            } else {\n                this.element.parents('div.field').hide();\n            }\n        },\n\n        /**\n         * Remove options from dropdown list\n         *\n         * @param {Object} selectElement - jQuery object for dropdown list\n         * @private\n         */\n        _removeSelectOptions: function (selectElement) {\n            selectElement.find('option').each(function (index) {\n                if (index) {\n                    $(this).remove();\n                }\n            });\n        },\n\n        /**\n         * Render dropdown list\n         * @param {Object} selectElement - jQuery object for dropdown list\n         * @param {String} key - region code\n         * @param {Object} value - region object\n         * @private\n         */\n        _renderSelectOption: function (selectElement, key, value) {\n            selectElement.append($.proxy(function () {\n                var name = value.name.replace(/[!\"#$%&'()*+,.\\/:;<=>?@[\\\\\\]^`{|}~]/g, '\\\\$&'),\n                    tmplData,\n                    tmpl;\n\n                if (value.code && $(name).is('span')) {\n                    key = value.code;\n                    value.name = $(name).text();\n                }\n\n                tmplData = {\n                    value: key,\n                    title: value.name,\n                    isSelected: false\n                };\n\n                if (this.options.defaultRegion === key) {\n                    tmplData.isSelected = true;\n                }\n\n                tmpl = this.regionTmpl({\n                    data: tmplData\n                });\n\n                return $(tmpl);\n            }, this));\n        },\n\n        /**\n         * Takes clearError callback function as first option\n         * If no form is passed as option, look up the closest form and call clearError method.\n         * @private\n         */\n        _clearError: function () {\n            var args = ['clearError', this.options.regionListId, this.options.regionInputId, this.options.postcodeId];\n\n            if (this.options.clearError && typeof this.options.clearError === 'function') {\n                this.options.clearError.call(this);\n            } else {\n                if (!this.options.form) {\n                    this.options.form = this.element.closest('form').length ? $(this.element.closest('form')[0]) : null;\n                }\n\n                this.options.form = $(this.options.form);\n\n                this.options.form && this.options.form.data('validator') &&\n                    this.options.form.validation.apply(this.options.form, _.compact(args));\n\n                // Clean up errors on region & zip fix\n                $(this.options.regionInputId).removeClass('mage-error').parent().find('.mage-error').remove();\n                $(this.options.regionListId).removeClass('mage-error').parent().find('.mage-error').remove();\n                $(this.options.postcodeId).removeClass('mage-error').parent().find('.mage-error').remove();\n            }\n        },\n\n        /**\n         * Update dropdown list based on the country selected\n         *\n         * @param {String} country - 2 uppercase letter for country code\n         * @private\n         */\n        _updateRegion: function (country) {\n            // Clear validation error messages\n            var regionList = $(this.options.regionListId),\n                regionInput = $(this.options.regionInputId),\n                postcode = $(this.options.postcodeId),\n                label = regionList.parent().siblings('label'),\n                container = regionList.parents('div.field'),\n                regionsEntries,\n                regionId,\n                regionData;\n\n            this._clearError();\n            this._checkRegionRequired(country);\n\n            // Populate state/province dropdown list if available or use input box\n            if (this.options.regionJson[country]) {\n                this._removeSelectOptions(regionList);\n                regionsEntries = _.pairs(this.options.regionJson[country]);\n                regionsEntries.sort(function (a, b) {\n                    return a[1].name > b[1].name ? 1 : -1;\n                });\n                $.each(regionsEntries, $.proxy(function (key, value) {\n                    regionId = value[0];\n                    regionData = value[1];\n                    this._renderSelectOption(regionList, regionId, regionData);\n                }, this));\n\n                if (this.currentRegionOption) {\n                    regionList.val(this.currentRegionOption);\n                }\n\n                if (this.setOption) {\n                    regionList.find('option').filter(function () {\n                        return this.text === regionInput.val();\n                    }).attr('selected', true);\n                }\n\n                if (this.options.isRegionRequired) {\n                    regionList.addClass('required-entry').prop('disabled', false);\n                    container.addClass('required').show();\n                } else {\n                    regionList.removeClass('required-entry validate-select').removeAttr('data-validate');\n                    container.removeClass('required');\n\n                    if (!this.options.optionalRegionAllowed) { //eslint-disable-line max-depth\n                        regionList.hide();\n                        container.hide();\n                    } else {\n                        regionList.prop('disabled', false).show();\n                    }\n                }\n\n                regionList.show();\n                regionInput.hide();\n                label.attr('for', regionList.attr('id'));\n            } else {\n                this._removeSelectOptions(regionList);\n\n                if (this.options.isRegionRequired) {\n                    regionInput.addClass('required-entry').prop('disabled', false);\n                    container.addClass('required').show();\n                } else {\n                    if (!this.options.optionalRegionAllowed) { //eslint-disable-line max-depth\n                        regionInput.attr('disabled', 'disabled');\n                        container.hide();\n                    }\n                    container.removeClass('required');\n                    regionInput.removeClass('required-entry');\n                }\n\n                regionList.removeClass('required-entry').prop('disabled', 'disabled').hide();\n                regionInput.show();\n                label.attr('for', regionInput.attr('id'));\n            }\n\n            // If country is in optionalzip list, make postcode input not required\n            if (this.options.isZipRequired) {\n                $.inArray(country, this.options.countriesWithOptionalZip) >= 0 ?\n                    postcode.removeClass('required-entry').closest('.field').removeClass('required') :\n                    postcode.addClass('required-entry').closest('.field').addClass('required');\n            }\n\n            // Add defaultvalue attribute to state/province select element\n            regionList.attr('defaultvalue', this.options.defaultRegion);\n            this.options.form.find('[type=\"submit\"]').prop('disabled', false).show();\n        },\n\n        /**\n         * Check if the selected country has a mandatory region selection\n         *\n         * @param {String} country - Code of the country - 2 uppercase letter for country code\n         * @private\n         */\n        _checkRegionRequired: function (country) {\n            var self = this;\n\n            this.options.isRegionRequired = false;\n            $.each(this.options.regionJson.config['regions_required'], function (index, elem) {\n                if (elem === country) {\n                    self.options.isRegionRequired = true;\n                }\n            });\n        }\n    });\n\n    return $.mage.regionUpdater;\n});\n","Magento_Checkout/js/shopping-cart.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/confirm',\n    'jquery-ui-modules/widget',\n    'mage/translate'\n], function ($, confirm) {\n    'use strict';\n\n    $.widget('mage.shoppingCart', {\n        /** @inheritdoc */\n        _create: function () {\n            var items, i, reload;\n\n            $(this.options.emptyCartButton).on('click', $.proxy(function () {\n                this._confirmClearCart();\n            }, this));\n            items = $.find('[data-role=\"cart-item-qty\"]');\n\n            for (i = 0; i < items.length; i++) {\n                $(items[i]).on('keypress', $.proxy(function (event) { //eslint-disable-line no-loop-func\n                    var keyCode = event.keyCode ? event.keyCode : event.which;\n\n                    if (keyCode == 13) { //eslint-disable-line eqeqeq\n                        $(this.options.emptyCartButton).attr('name', 'update_cart_action_temp');\n                        $(this.options.updateCartActionContainer)\n                            .attr('name', 'update_cart_action').attr('value', 'update_qty');\n\n                    }\n                }, this));\n            }\n            $(this.options.continueShoppingButton).on('click', $.proxy(function () {\n                location.href = this.options.continueShoppingUrl;\n            }, this));\n\n            $(document).on('ajax:removeFromCart', $.proxy(function () {\n                reload = true;\n                $('div.block.block-minicart').on('dropdowndialogclose', $.proxy(function () {\n                    if (reload === true) {\n                        location.reload();\n                        reload = false;\n                    }\n                    $('div.block.block-minicart').off('dropdowndialogclose');\n                }));\n            }, this));\n            $(document).on('ajax:updateItemQty', $.proxy(function () {\n                reload = true;\n                $('div.block.block-minicart').on('dropdowndialogclose', $.proxy(function () {\n                    if (reload === true) {\n                        location.reload();\n                        reload = false;\n                    }\n                    $('div.block.block-minicart').off('dropdowndialogclose');\n                }));\n            }, this));\n        },\n\n        /**\n         * Display confirmation modal for clearing the cart\n         * @private\n         */\n        _confirmClearCart: function () {\n            var self = this;\n\n            confirm({\n                content: $.mage.__('Are you sure you want to remove all items from your shopping cart?'),\n                actions: {\n                    /**\n                     * Confirmation modal handler to execute clear cart action\n                     */\n                    confirm: function () {\n                        self.clearCart();\n                    }\n                }\n            });\n        },\n\n        /**\n         * Prepares the form and submit to clear the cart\n         * @public\n         */\n        clearCart: function () {\n            $(this.options.emptyCartButton).attr('name', 'update_cart_action_temp');\n            $(this.options.updateCartActionContainer)\n                .attr('name', 'update_cart_action').attr('value', 'empty_cart');\n\n            if ($(this.options.emptyCartButton).parents('form').length > 0) {\n                $(this.options.emptyCartButton).parents('form').trigger('submit');\n            }\n        }\n    });\n\n    return $.mage.shoppingCart;\n});\n","Magento_Checkout/js/sidebar.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Customer/js/model/authentication-popup',\n    'Magento_Customer/js/customer-data',\n    'Magento_Ui/js/modal/alert',\n    'Magento_Ui/js/modal/confirm',\n    'underscore',\n    'jquery-ui-modules/widget',\n    'mage/decorate',\n    'mage/collapsible',\n    'mage/cookies',\n    'jquery-ui-modules/effect-fade'\n], function ($, authenticationPopup, customerData, alert, confirm, _) {\n    'use strict';\n\n    $.widget('mage.sidebar', {\n        options: {\n            isRecursive: true,\n            minicart: {\n                maxItemsVisible: 3\n            }\n        },\n        scrollHeight: 0,\n        shoppingCartUrl: window.checkout.shoppingCartUrl,\n\n        /**\n         * Create sidebar.\n         * @private\n         */\n        _create: function () {\n            this._initContent();\n        },\n\n        /**\n         * Update sidebar block.\n         */\n        update: function () {\n            $(this.options.targetElement).trigger('contentUpdated');\n            this._calcHeight();\n        },\n\n        /**\n         * @private\n         */\n        _initContent: function () {\n            var self = this,\n                events = {};\n\n            this.element.decorate('list', this.options.isRecursive);\n\n            /**\n             * @param {jQuery.Event} event\n             */\n            events['click ' + this.options.button.close] = function (event) {\n                event.stopPropagation();\n                $(self.options.targetElement).dropdownDialog('close');\n            };\n            events['click ' + this.options.button.checkout] = $.proxy(function () {\n                var cart = customerData.get('cart'),\n                    customer = customerData.get('customer'),\n                    element = $(this.options.button.checkout);\n\n                if (!customer().firstname && cart().isGuestCheckoutAllowed === false) {\n                    // set URL for redirect on successful login/registration. It's postprocessed on backend.\n                    $.cookie('login_redirect', this.options.url.checkout);\n\n                    if (this.options.url.isRedirectRequired) {\n                        element.prop('disabled', true);\n                        location.href = this.options.url.loginUrl;\n                    } else {\n                        authenticationPopup.showModal();\n                    }\n\n                    return false;\n                }\n                element.prop('disabled', true);\n                location.href = this.options.url.checkout;\n            }, this);\n\n            /**\n             * @param {jQuery.Event} event\n             */\n            events['click ' + this.options.button.remove] =  function (event) {\n                event.stopPropagation();\n                confirm({\n                    content: self.options.confirmMessage,\n                    actions: {\n                        /** @inheritdoc */\n                        confirm: function () {\n                            self._removeItem($(event.currentTarget));\n                        },\n\n                        /** @inheritdoc */\n                        always: function (e) {\n                            e.stopImmediatePropagation();\n                        }\n                    }\n                });\n            };\n\n            /**\n             * @param {jQuery.Event} event\n             */\n            events['keyup ' + this.options.item.qty] = function (event) {\n                self._showItemButton($(event.target));\n            };\n\n            /**\n             * @param {jQuery.Event} event\n             */\n            events['change ' + this.options.item.qty] = function (event) {\n                self._showItemButton($(event.target));\n            };\n\n            /**\n             * @param {jQuery.Event} event\n             */\n            events['click ' + this.options.item.button] = function (event) {\n                event.stopPropagation();\n                self._updateItemQty($(event.currentTarget));\n            };\n\n            /**\n             * @param {jQuery.Event} event\n             */\n            events['focusout ' + this.options.item.qty] = function (event) {\n                self._validateQty($(event.currentTarget));\n            };\n\n            this._on(this.element, events);\n            this._calcHeight();\n        },\n\n        /**\n         * @param {HTMLElement} elem\n         * @private\n         */\n        _showItemButton: function (elem) {\n            var itemId = elem.data('cart-item'),\n                itemQty = elem.data('item-qty');\n\n            if (this._isValidQty(itemQty, elem.val())) {\n                $('#update-cart-item-' + itemId).show('fade', 300);\n            } else if (elem.val() == 0) { //eslint-disable-line eqeqeq\n                this._hideItemButton(elem);\n            } else {\n                this._hideItemButton(elem);\n            }\n        },\n\n        /**\n         * @param {*} origin - origin qty. 'data-item-qty' attribute.\n         * @param {*} changed - new qty.\n         * @returns {Boolean}\n         * @private\n         */\n        _isValidQty: function (origin, changed) {\n            return origin != changed && //eslint-disable-line eqeqeq\n                changed.length > 0 &&\n                changed - 0 == changed && //eslint-disable-line eqeqeq\n                changed - 0 > 0;\n        },\n\n        /**\n         * @param {Object} elem\n         * @private\n         */\n        _validateQty: function (elem) {\n            var itemQty = elem.data('item-qty');\n\n            if (!this._isValidQty(itemQty, elem.val())) {\n                elem.val(itemQty);\n            }\n        },\n\n        /**\n         * @param {HTMLElement} elem\n         * @private\n         */\n        _hideItemButton: function (elem) {\n            var itemId = elem.data('cart-item');\n\n            $('#update-cart-item-' + itemId).hide('fade', 300);\n        },\n\n        /**\n         * @param {HTMLElement} elem\n         * @private\n         */\n        _updateItemQty: function (elem) {\n            var itemId = elem.data('cart-item');\n\n            this._ajax(this.options.url.update, {\n                'item_id': itemId,\n                'item_qty': $('#cart-item-' + itemId + '-qty').val()\n            }, elem, this._updateItemQtyAfter);\n        },\n\n        /**\n         * Update content after update qty\n         *\n         * @param {HTMLElement} elem\n         */\n        _updateItemQtyAfter: function (elem) {\n            var productData = this._getProductById(Number(elem.data('cart-item')));\n\n            if (!_.isUndefined(productData)) {\n                $(document).trigger('ajax:updateCartItemQty');\n\n                if (window.location.href === this.shoppingCartUrl) {\n                    window.location.reload(false);\n                }\n            }\n            this._hideItemButton(elem);\n        },\n\n        /**\n         * @param {HTMLElement} elem\n         * @private\n         */\n        _removeItem: function (elem) {\n            var itemId = elem.data('cart-item');\n\n            this._ajax(this.options.url.remove, {\n                'item_id': itemId\n            }, elem, this._removeItemAfter);\n        },\n\n        /**\n         * Update content after item remove\n         *\n         * @param {Object} elem\n         * @private\n         */\n        _removeItemAfter: function (elem) {\n            var productData = this._getProductById(Number(elem.data('cart-item')));\n\n            if (!_.isUndefined(productData)) {\n                $(document).trigger('ajax:removeFromCart', {\n                    productIds: [productData['product_id']],\n                    productInfo: [\n                        {\n                            'id': productData['product_id']\n                        }\n                    ]\n                });\n\n                if (window.location.href.indexOf(this.shoppingCartUrl) === 0) {\n                    window.location.reload();\n                }\n            }\n        },\n\n        /**\n         * Retrieves product data by Id.\n         *\n         * @param {Number} productId - product Id\n         * @returns {Object|undefined}\n         * @private\n         */\n        _getProductById: function (productId) {\n            return _.find(customerData.get('cart')().items, function (item) {\n                return productId === Number(item['item_id']);\n            });\n        },\n\n        /**\n         * @param {String} url - ajax url\n         * @param {Object} data - post data for ajax call\n         * @param {Object} elem - element that initiated the event\n         * @param {Function} callback - callback method to execute after AJAX success\n         */\n        _ajax: function (url, data, elem, callback) {\n            $.extend(data, {\n                'form_key': $.mage.cookies.get('form_key')\n            });\n\n            $.ajax({\n                url: url,\n                data: data,\n                type: 'post',\n                dataType: 'json',\n                context: this,\n\n                /** @inheritdoc */\n                beforeSend: function () {\n                    elem.attr('disabled', 'disabled');\n                },\n\n                /** @inheritdoc */\n                complete: function () {\n                    elem.attr('disabled', null);\n                }\n            })\n                .done(function (response) {\n                    var msg;\n\n                    if (response.success) {\n                        callback.call(this, elem, response);\n                    } else {\n                        msg = response['error_message'];\n\n                        if (msg) {\n                            alert({\n                                content: msg\n                            });\n                        }\n                    }\n                })\n                .fail(function (error) {\n                    console.log(JSON.stringify(error));\n                });\n        },\n\n        /**\n         * Calculate height of minicart list\n         *\n         * @private\n         */\n        _calcHeight: function () {\n            var self = this,\n                height = 0,\n                counter = this.options.minicart.maxItemsVisible,\n                target = $(this.options.minicart.list),\n                outerHeight;\n\n            self.scrollHeight = 0;\n            target.children().each(function () {\n\n                if ($(this).find('.options').length > 0) {\n                    $(this).collapsible();\n                }\n                outerHeight = $(this).outerHeight(true);\n\n                if (counter-- > 0) {\n                    height += outerHeight;\n                }\n                self.scrollHeight += outerHeight;\n            });\n\n            target.parent().height(height);\n        }\n    });\n\n    return $.mage.sidebar;\n});\n","Magento_Checkout/js/action/create-billing-address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Checkout/js/model/address-converter'\n], function (addressConverter) {\n    'use strict';\n\n    return function (addressData) {\n        var address = addressConverter.formAddressDataToQuoteAddress(addressData);\n\n        /**\n         * Returns new customer billing address type.\n         *\n         * @returns {String}\n         */\n        address.getType = function () {\n            return 'new-customer-billing-address';\n        };\n\n        return address;\n    };\n});\n","Magento_Checkout/js/action/create-shipping-address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Customer/js/model/address-list',\n    'Magento_Checkout/js/model/address-converter'\n], function (addressList, addressConverter) {\n    'use strict';\n\n    return function (addressData) {\n        var address = addressConverter.formAddressDataToQuoteAddress(addressData),\n            isAddressUpdated = addressList().some(function (currentAddress, index, addresses) {\n                if (currentAddress.getKey() == address.getKey()) { //eslint-disable-line eqeqeq\n                    addresses[index] = address;\n\n                    return true;\n                }\n\n                return false;\n            });\n\n        if (!isAddressUpdated) {\n            addressList.push(address);\n        } else {\n            addressList.valueHasMutated();\n        }\n\n        return address;\n    };\n});\n","Magento_Checkout/js/action/get-payment-information.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/url-builder',\n    'mage/storage',\n    'Magento_Checkout/js/model/error-processor',\n    'Magento_Customer/js/model/customer',\n    'Magento_Checkout/js/model/payment/method-converter',\n    'Magento_Checkout/js/model/payment-service'\n], function ($, quote, urlBuilder, storage, errorProcessor, customer, methodConverter, paymentService) {\n    'use strict';\n\n    return function (deferred, messageContainer) {\n        var serviceUrl;\n\n        deferred = deferred || $.Deferred();\n\n        /**\n         * Checkout for guest and registered customer.\n         */\n        if (!customer.isLoggedIn()) {\n            serviceUrl = urlBuilder.createUrl('/guest-carts/:cartId/payment-information', {\n                cartId: quote.getQuoteId()\n            });\n        } else {\n            serviceUrl = urlBuilder.createUrl('/carts/mine/payment-information', {});\n        }\n\n        return storage.get(\n            serviceUrl, false\n        ).done(function (response) {\n            quote.setTotals(response.totals);\n            paymentService.setPaymentMethods(methodConverter(response['payment_methods']));\n            deferred.resolve();\n        }).fail(function (response) {\n            errorProcessor.process(response, messageContainer);\n            deferred.reject();\n        });\n    };\n});\n","Magento_Checkout/js/action/get-totals.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    '../model/quote',\n    'Magento_Checkout/js/model/resource-url-manager',\n    'Magento_Checkout/js/model/error-processor',\n    'mage/storage',\n    'Magento_Checkout/js/model/totals'\n], function ($, quote, resourceUrlManager, errorProcessor, storage, totals) {\n    'use strict';\n\n    return function (callbacks, deferred) {\n        deferred = deferred || $.Deferred();\n        totals.isLoading(true);\n\n        return storage.get(\n            resourceUrlManager.getUrlForCartTotals(quote),\n            false\n        ).done(function (response) {\n            var proceed = true;\n\n            totals.isLoading(false);\n\n            if (callbacks.length > 0) {\n                $.each(callbacks, function (index, callback) {\n                    proceed = proceed && callback();\n                });\n            }\n\n            if (proceed) {\n                quote.setTotals(response);\n                deferred.resolve();\n            }\n        }).fail(function (response) {\n            totals.isLoading(false);\n            deferred.reject();\n            errorProcessor.process(response);\n        }).always(function () {\n            totals.isLoading(false);\n        });\n    };\n});\n","Magento_Checkout/js/action/place-order.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/url-builder',\n    'Magento_Customer/js/model/customer',\n    'Magento_Checkout/js/model/place-order'\n], function (quote, urlBuilder, customer, placeOrderService) {\n    'use strict';\n\n    return function (paymentData, messageContainer) {\n        var serviceUrl, payload;\n\n        payload = {\n            cartId: quote.getQuoteId(),\n            billingAddress: quote.billingAddress(),\n            paymentMethod: paymentData\n        };\n\n        if (customer.isLoggedIn()) {\n            serviceUrl = urlBuilder.createUrl('/carts/mine/payment-information', {});\n        } else {\n            serviceUrl = urlBuilder.createUrl('/guest-carts/:quoteId/payment-information', {\n                quoteId: quote.getQuoteId()\n            });\n            payload.email = quote.guestEmail;\n        }\n\n        return placeOrderService(serviceUrl, payload, messageContainer);\n    };\n});\n","Magento_Checkout/js/action/recollect-shipping-rates.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/action/select-shipping-address',\n    'Magento_Checkout/js/model/shipping-rate-registry'\n], function (quote, selectShippingAddress, rateRegistry) {\n    'use strict';\n\n    return function () {\n        var shippingAddress = null;\n\n        if (!quote.isVirtual()) {\n            shippingAddress = quote.shippingAddress();\n\n            rateRegistry.set(shippingAddress.getCacheKey(), null);\n            selectShippingAddress(shippingAddress);\n        }\n    };\n});\n","Magento_Checkout/js/action/redirect-on-success.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine(\n    [\n        'mage/url',\n        'Magento_Checkout/js/model/full-screen-loader'\n    ],\n    function (url, fullScreenLoader) {\n        'use strict';\n\n        return {\n            redirectUrl: window.checkoutConfig.defaultSuccessPageUrl,\n\n            /**\n             * Provide redirect to page\n             */\n            execute: function () {\n                fullScreenLoader.startLoader();\n                this.redirectToSuccessPage();\n            },\n\n            /**\n             * Redirect to success page.\n             */\n            redirectToSuccessPage: function () {\n                window.location.replace(url.build(this.redirectUrl));\n            }\n        };\n    }\n);\n","Magento_Checkout/js/action/select-billing-address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    '../model/quote'\n], function ($, quote) {\n    'use strict';\n\n    return function (billingAddress) {\n        var address = null;\n\n        if (quote.shippingAddress() && billingAddress.getCacheKey() == //eslint-disable-line eqeqeq\n            quote.shippingAddress().getCacheKey()\n        ) {\n            address = $.extend(true, {}, billingAddress);\n            address.saveInAddressBook = null;\n        } else {\n            address = billingAddress;\n        }\n        quote.billingAddress(address);\n    };\n});\n","Magento_Checkout/js/action/select-payment-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Checkout/js/model/quote'\n], function (quote) {\n    'use strict';\n\n    return function (paymentMethod) {\n        if (paymentMethod) {\n            paymentMethod.__disableTmpl = {\n                title: true\n            };\n        }\n        quote.paymentMethod(paymentMethod);\n    };\n});\n","Magento_Checkout/js/action/select-shipping-address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Checkout/js/model/quote'\n], function (quote) {\n    'use strict';\n\n    return function (shippingAddress) {\n        quote.shippingAddress(shippingAddress);\n    };\n});\n","Magento_Checkout/js/action/select-shipping-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    '../model/quote'\n], function (quote) {\n    'use strict';\n\n    return function (shippingMethod) {\n        quote.shippingMethod(shippingMethod);\n    };\n});\n","Magento_Checkout/js/action/set-billing-address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine(\n    [\n        'jquery',\n        'Magento_Checkout/js/model/quote',\n        'Magento_Checkout/js/model/url-builder',\n        'mage/storage',\n        'Magento_Checkout/js/model/error-processor',\n        'Magento_Customer/js/model/customer',\n        'Magento_Checkout/js/model/full-screen-loader',\n        'Magento_Checkout/js/action/get-payment-information'\n    ],\n    function ($,\n              quote,\n              urlBuilder,\n              storage,\n              errorProcessor,\n              customer,\n              fullScreenLoader,\n              getPaymentInformationAction\n    ) {\n        'use strict';\n\n        return function (messageContainer) {\n            var serviceUrl,\n                payload;\n\n            /**\n             * Checkout for guest and registered customer.\n             */\n            if (!customer.isLoggedIn()) {\n                serviceUrl = urlBuilder.createUrl('/guest-carts/:cartId/billing-address', {\n                    cartId: quote.getQuoteId()\n                });\n                payload = {\n                    cartId: quote.getQuoteId(),\n                    address: quote.billingAddress()\n                };\n            } else {\n                serviceUrl = urlBuilder.createUrl('/carts/mine/billing-address', {});\n                payload = {\n                    cartId: quote.getQuoteId(),\n                    address: quote.billingAddress()\n                };\n            }\n\n            fullScreenLoader.startLoader();\n\n            return storage.post(\n                serviceUrl, JSON.stringify(payload)\n            ).done(\n                function () {\n                    var deferred = $.Deferred();\n\n                    getPaymentInformationAction(deferred);\n                    $.when(deferred).done(function () {\n                        fullScreenLoader.stopLoader();\n                    });\n                }\n            ).fail(\n                function (response) {\n                    errorProcessor.process(response, messageContainer);\n                    fullScreenLoader.stopLoader();\n                }\n            );\n        };\n    }\n);\n","Magento_Checkout/js/action/set-payment-information-extended.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/url-builder',\n    'mage/storage',\n    'Magento_Checkout/js/model/error-processor',\n    'Magento_Customer/js/model/customer',\n    'Magento_Checkout/js/action/get-totals',\n    'Magento_Checkout/js/model/full-screen-loader',\n    'underscore',\n    'Magento_Checkout/js/model/payment/place-order-hooks'\n], function (quote, urlBuilder, storage, errorProcessor, customer, getTotalsAction, fullScreenLoader, _, hooks) {\n    'use strict';\n\n    /**\n     * Filter template data.\n     *\n     * @param {Object|Array} data\n     */\n    var filterTemplateData = function (data) {\n        return _.each(data, function (value, key, list) {\n            if (_.isArray(value) || _.isObject(value)) {\n                list[key] = filterTemplateData(value);\n            }\n\n            if (key === '__disableTmpl' || key === 'title') {\n                delete list[key];\n            }\n        });\n    };\n\n    return function (messageContainer, paymentData, skipBilling) {\n        var serviceUrl,\n            payload,\n            headers = {};\n\n        paymentData = filterTemplateData(paymentData);\n        skipBilling = skipBilling || false;\n        payload = {\n            cartId: quote.getQuoteId(),\n            paymentMethod: paymentData\n        };\n\n        /**\n         * Checkout for guest and registered customer.\n         */\n        if (!customer.isLoggedIn()) {\n            serviceUrl = urlBuilder.createUrl('/guest-carts/:cartId/set-payment-information', {\n                cartId: quote.getQuoteId()\n            });\n            payload.email = quote.guestEmail;\n        } else {\n            serviceUrl = urlBuilder.createUrl('/carts/mine/set-payment-information', {});\n        }\n\n        if (skipBilling === false) {\n            payload.billingAddress = quote.billingAddress();\n        }\n\n        fullScreenLoader.startLoader();\n\n        _.each(hooks.requestModifiers, function (modifier) {\n            modifier(headers, payload);\n        });\n\n        return storage.post(\n            serviceUrl, JSON.stringify(payload), true, 'application/json', headers\n        ).fail(\n            function (response) {\n                errorProcessor.process(response, messageContainer);\n            }\n        ).always(\n            function () {\n                fullScreenLoader.stopLoader();\n                _.each(hooks.afterRequestListeners, function (listener) {\n                    listener();\n                });\n            }\n        );\n    };\n});\n","Magento_Checkout/js/action/set-payment-information.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Checkout/js/action/set-payment-information-extended'\n\n], function (setPaymentInformationExtended) {\n    'use strict';\n\n    return function (messageContainer, paymentData) {\n\n        return setPaymentInformationExtended(messageContainer, paymentData, false);\n    };\n});\n","Magento_Checkout/js/action/set-shipping-information.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    '../model/quote',\n    'Magento_Checkout/js/model/shipping-save-processor'\n], function (quote, shippingSaveProcessor) {\n    'use strict';\n\n    return function () {\n        return shippingSaveProcessor.saveShippingInformation(quote.shippingAddress().getType());\n    };\n});\n","Magento_Checkout/js/action/update-shopping-cart.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Ui/js/modal/alert',\n    'jquery',\n    'jquery-ui-modules/widget',\n    'mage/validation'\n], function (alert, $) {\n    'use strict';\n\n    $.widget('mage.updateShoppingCart', {\n        options: {\n            validationURL: '',\n            eventName: 'updateCartItemQty',\n            updateCartActionContainer: ''\n        },\n\n        /** @inheritdoc */\n        _create: function () {\n            this._on(this.element, {\n                'submit': this.onSubmit\n            });\n        },\n\n        /**\n         * Prevents default submit action and calls form validator.\n         *\n         * @param {Event} event\n         * @return {Boolean}\n         */\n        onSubmit: function (event) {\n            var action = this.element.find(this.options.updateCartActionContainer).val();\n\n            if (!this.options.validationURL || action === 'empty_cart') {\n                return true;\n            }\n\n            if (this.isValid()) {\n                event.preventDefault();\n                this.validateItems(this.options.validationURL, this.element.serialize());\n            }\n\n            return false;\n        },\n\n        /**\n         * Validates requested form.\n         *\n         * @return {Boolean}\n         */\n        isValid: function () {\n            return this.element.validation() && this.element.validation('isValid');\n        },\n\n        /**\n         * Validates updated shopping cart data.\n         *\n         * @param {String} url - request url\n         * @param {Object} data - post data for ajax call\n         */\n        validateItems: function (url, data) {\n            $.extend(data, {\n                'form_key': $.mage.cookies.get('form_key')\n            });\n\n            $.ajax({\n                url: url,\n                data: data,\n                type: 'post',\n                dataType: 'json',\n                context: this,\n\n                /** @inheritdoc */\n                beforeSend: function () {\n                    $(document.body).trigger('processStart');\n                },\n\n                /** @inheritdoc */\n                complete: function () {\n                    $(document.body).trigger('processStop');\n                }\n            })\n            .done(function (response) {\n                if (response.success) {\n                    this.onSuccess();\n                } else {\n                    this.onError(response);\n                }\n            })\n            .fail(function () {\n                this.submitForm();\n            });\n        },\n\n        /**\n         * Form validation succeed.\n         */\n        onSuccess: function () {\n            $(document).trigger('ajax:' + this.options.eventName);\n            this.submitForm();\n        },\n\n        /**\n         * Form validation failed.\n         */\n        onError: function (response) {\n            var that = this,\n                elm,\n                responseData = JSON.parse(response['error_message']);\n\n            if (response['error_message']) {\n                try {\n                    $.each(responseData, function (index, data) {\n\n                        if (data.itemId !== undefined) {\n                            elm = $('#cart-' + data.itemId + '-qty');\n                            elm.val(elm.attr('data-item-qty'));\n                        }\n                        response['error_message'] = data.error;\n                    });\n                } catch (e) {}\n                alert({\n                    content: response['error_message'],\n                    actions: {\n                        /** @inheritdoc */\n                        always: function () {\n                            that.submitForm();\n                        }\n                    }\n                });\n            } else {\n                this.submitForm();\n            }\n        },\n\n        /**\n         * Real submit of validated form.\n         */\n        submitForm: function () {\n            this.element\n                .off('submit', this.onSubmit)\n                .on('submit', function () {\n                    $(document.body).trigger('processStart');\n                })\n                .trigger('submit');\n        }\n    });\n\n    return $.mage.updateShoppingCart;\n});\n","Magento_Checkout/js/model/address-converter.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'Magento_Checkout/js/model/new-customer-address',\n    'Magento_Customer/js/customer-data',\n    'mage/utils/objects',\n    'underscore'\n], function ($, address, customerData, mageUtils, _) {\n    'use strict';\n\n    var countryData = customerData.get('directory-data');\n\n    return {\n        /**\n         * Convert address form data to Address object\n         *\n         * @param {Object} formData\n         * @returns {Object}\n         */\n        formAddressDataToQuoteAddress: function (formData) {\n            // clone address form data to new object\n            var addressData = $.extend(true, {}, formData),\n                region,\n                regionName = addressData.region,\n                customAttributes;\n\n            if (mageUtils.isObject(addressData.street)) {\n                addressData.street = this.objectToArray(addressData.street);\n            }\n\n            addressData.region = {\n                'region_id': addressData['region_id'],\n                'region_code': addressData['region_code'],\n                region: regionName\n            };\n\n            if (addressData['region_id'] &&\n                countryData()[addressData['country_id']] &&\n                countryData()[addressData['country_id']].regions\n            ) {\n                region = countryData()[addressData['country_id']].regions[addressData['region_id']];\n\n                if (region) {\n                    addressData.region['region_id'] = addressData['region_id'];\n                    addressData.region['region_code'] = region.code;\n                    addressData.region.region = region.name;\n                }\n            } else if (\n                !addressData['region_id'] &&\n                countryData()[addressData['country_id']] &&\n                countryData()[addressData['country_id']].regions\n            ) {\n                addressData.region['region_code'] = '';\n                addressData.region.region = '';\n            }\n            delete addressData['region_id'];\n\n            if (addressData['custom_attributes']) {\n                addressData['custom_attributes'] = _.map(\n                    addressData['custom_attributes'],\n                    function (value, key) {\n                        customAttributes = {\n                            'attribute_code': key,\n                            'value': value\n                        };\n\n                        if (typeof value === 'boolean') {\n                            customAttributes = {\n                                'attribute_code': key,\n                                'value': value,\n                                'label': value === true ? 'Yes' : 'No'\n                            };\n                        }\n\n                        return customAttributes;\n                    }\n                );\n            }\n\n            return address(addressData);\n        },\n\n        /**\n         * Convert Address object to address form data.\n         *\n         * @param {Object} addrs\n         * @returns {Object}\n         */\n        quoteAddressToFormAddressData: function (addrs) {\n            var self = this,\n                output = {},\n                streetObject,\n                customAttributesObject;\n\n            $.each(addrs, function (key) {\n                if (addrs.hasOwnProperty(key) && typeof addrs[key] !== 'function') {\n                    output[self.toUnderscore(key)] = addrs[key];\n                }\n            });\n\n            if (Array.isArray(addrs.street)) {\n                streetObject = {};\n                addrs.street.forEach(function (value, index) {\n                    streetObject[index] = value;\n                });\n                output.street = streetObject;\n            }\n\n            //jscs:disable requireCamelCaseOrUpperCaseIdentifiers\n            if (Array.isArray(addrs.customAttributes)) {\n                customAttributesObject = {};\n                addrs.customAttributes.forEach(function (value) {\n                    customAttributesObject[value.attribute_code] = value.value;\n                });\n                output.custom_attributes = customAttributesObject;\n            }\n            //jscs:enable requireCamelCaseOrUpperCaseIdentifiers\n\n            return output;\n        },\n\n        /**\n         * @param {String} string\n         */\n        toUnderscore: function (string) {\n            return string.replace(/([A-Z])/g, function ($1) {\n                return '_' + $1.toLowerCase();\n            });\n        },\n\n        /**\n         * @param {Object} formProviderData\n         * @param {String} formIndex\n         * @return {Object}\n         */\n        formDataProviderToFlatData: function (formProviderData, formIndex) {\n            var addressData = {};\n\n            $.each(formProviderData, function (path, value) {\n                var pathComponents = path.split('.'),\n                    dataObject = {};\n\n                pathComponents.splice(pathComponents.indexOf(formIndex), 1);\n                pathComponents.reverse();\n                $.each(pathComponents, function (index, pathPart) {\n                    var parent = {};\n\n                    if (index == 0) { //eslint-disable-line eqeqeq\n                        dataObject[pathPart] = value;\n                    } else {\n                        parent[pathPart] = dataObject;\n                        dataObject = parent;\n                    }\n                });\n                $.extend(true, addressData, dataObject);\n            });\n\n            return addressData;\n        },\n\n        /**\n         * Convert object to array\n         * @param {Object} object\n         * @returns {Array}\n         */\n        objectToArray: function (object) {\n            var convertedArray = [];\n\n            $.each(object, function (key) {\n                return typeof object[key] === 'string' ? convertedArray.push(object[key]) : false;\n            });\n\n            return convertedArray.slice(0);\n        },\n\n        /**\n         * @param {Object} addrs\n         * @return {*|Object}\n         */\n        addressToEstimationAddress: function (addrs) {\n            var self = this,\n                estimatedAddressData = {};\n\n            $.each(addrs, function (key) {\n                estimatedAddressData[self.toUnderscore(key)] = addrs[key];\n            });\n\n            return this.formAddressDataToQuoteAddress(estimatedAddressData);\n        }\n    };\n});\n","Magento_Checkout/js/model/authentication-messages.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko',\n    'Magento_Ui/js/model/messages'\n], function (ko, Messages) {\n    'use strict';\n\n    return new Messages();\n});\n","Magento_Checkout/js/model/billing-address-postcode-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n           'jquery',\n           'Magento_Checkout/js/model/postcode-validator',\n           'mage/translate',\n           'uiRegistry'\n       ], function (\n    $,\n    postcodeValidator,\n    $t,\n    uiRegistry\n) {\n    'use strict';\n\n    var postcodeElementName = 'postcode';\n\n    return {\n        validateZipCodeTimeout: 0,\n        validateDelay: 2000,\n\n        /**\n         * Perform postponed binding for fieldset elements\n         *\n         * @param {String} formPath\n         */\n        initFields: function (formPath) {\n            var self = this;\n\n            uiRegistry.async(formPath + '.' + postcodeElementName)(self.bindHandler.bind(self));\n        },\n\n        /**\n         * @param {Object} element\n         * @param {Number} delay\n         */\n        bindHandler: function (element, delay) {\n            var self = this;\n\n            delay = typeof delay === 'undefined' ? self.validateDelay : delay;\n\n            element.on('value', function () {\n                clearTimeout(self.validateZipCodeTimeout);\n                self.validateZipCodeTimeout = setTimeout(function () {\n                    self.postcodeValidation(element);\n                }, delay);\n            });\n        },\n\n        /**\n         * @param {Object} postcodeElement\n         * @return {*}\n         */\n        postcodeValidation: function (postcodeElement) {\n            var countryId = $('select[name=\"country_id\"]:visible').val(),\n                validationResult,\n                warnMessage;\n\n            if (postcodeElement == null || postcodeElement.value() == null) {\n                return true;\n            }\n\n            postcodeElement.warn(null);\n            validationResult = postcodeValidator.validate(postcodeElement.value(), countryId);\n\n            if (!validationResult) {\n                warnMessage = $t('Provided Zip/Postal Code seems to be invalid.');\n\n                if (postcodeValidator.validatedPostCodeExample.length) {\n                    warnMessage += $t(' Example: ') + postcodeValidator.validatedPostCodeExample.join('; ') + '. ';\n                }\n                warnMessage += $t('If you believe it is the right one you can ignore this notice.');\n                postcodeElement.warn(warnMessage);\n            }\n\n            return validationResult;\n        }\n    };\n});\n","Magento_Checkout/js/model/checkout-data-resolver.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * Checkout adapter for customer data storage\n */\ndefine([\n    'Magento_Customer/js/model/address-list',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/checkout-data',\n    'Magento_Checkout/js/action/create-shipping-address',\n    'Magento_Checkout/js/action/select-shipping-address',\n    'Magento_Checkout/js/action/select-shipping-method',\n    'Magento_Checkout/js/model/payment-service',\n    'Magento_Checkout/js/action/select-payment-method',\n    'Magento_Checkout/js/model/address-converter',\n    'Magento_Checkout/js/action/select-billing-address',\n    'Magento_Checkout/js/action/create-billing-address',\n    'underscore'\n], function (\n    addressList,\n    quote,\n    checkoutData,\n    createShippingAddress,\n    selectShippingAddress,\n    selectShippingMethodAction,\n    paymentService,\n    selectPaymentMethodAction,\n    addressConverter,\n    selectBillingAddress,\n    createBillingAddress,\n    _\n) {\n    'use strict';\n\n    var isBillingAddressResolvedFromBackend = false;\n\n    return {\n\n        /**\n         * Resolve estimation address. Used local storage\n         */\n        resolveEstimationAddress: function () {\n            var address;\n\n            if (quote.isVirtual()) {\n                if (checkoutData.getBillingAddressFromData()) {\n                    address = addressConverter.formAddressDataToQuoteAddress(\n                        checkoutData.getBillingAddressFromData()\n                    );\n                    selectBillingAddress(address);\n                } else {\n                    this.resolveBillingAddress();\n                }\n            } else if (checkoutData.getShippingAddressFromData()) {\n                address = addressConverter.formAddressDataToQuoteAddress(checkoutData.getShippingAddressFromData());\n                selectShippingAddress(address);\n            } else {\n                this.resolveShippingAddress();\n            }\n        },\n\n        /**\n         * Resolve shipping address. Used local storage\n         */\n        resolveShippingAddress: function () {\n            var newCustomerShippingAddress;\n\n            if (!checkoutData.getShippingAddressFromData() &&\n                window.checkoutConfig.shippingAddressFromData\n            ) {\n                checkoutData.setShippingAddressFromData(window.checkoutConfig.shippingAddressFromData);\n            }\n\n            newCustomerShippingAddress = checkoutData.getNewCustomerShippingAddress();\n\n            if (newCustomerShippingAddress) {\n                createShippingAddress(newCustomerShippingAddress);\n            }\n            this.applyShippingAddress();\n        },\n\n        /**\n         * Apply resolved estimated address to quote\n         *\n         * @param {Object} isEstimatedAddress\n         */\n        applyShippingAddress: function (isEstimatedAddress) {\n            var address,\n                shippingAddress,\n                isConvertAddress;\n\n            if (addressList().length === 0) {\n                address = addressConverter.formAddressDataToQuoteAddress(\n                    checkoutData.getShippingAddressFromData()\n                );\n                selectShippingAddress(address);\n            }\n            shippingAddress = quote.shippingAddress();\n            isConvertAddress = isEstimatedAddress || false;\n\n            if (!shippingAddress) {\n                shippingAddress = this.getShippingAddressFromCustomerAddressList();\n\n                if (shippingAddress) {\n                    selectShippingAddress(\n                        isConvertAddress ?\n                            addressConverter.addressToEstimationAddress(shippingAddress)\n                            : shippingAddress\n                    );\n                }\n            }\n        },\n\n        /**\n         * @param {Object} ratesData\n         */\n        resolveShippingRates: function (ratesData) {\n            var selectedShippingRate = checkoutData.getSelectedShippingRate(),\n                availableRate = false;\n\n            if (ratesData.length === 1 && !quote.shippingMethod()) {\n                //set shipping rate if we have only one available shipping rate\n                selectShippingMethodAction(ratesData[0]);\n\n                return;\n            }\n\n            if (quote.shippingMethod()) {\n                availableRate = _.find(ratesData, function (rate) {\n                    return rate['carrier_code'] == quote.shippingMethod()['carrier_code'] && //eslint-disable-line\n                        rate['method_code'] == quote.shippingMethod()['method_code']; //eslint-disable-line eqeqeq\n                });\n            }\n\n            if (!availableRate && selectedShippingRate) {\n                availableRate = _.find(ratesData, function (rate) {\n                    return rate['carrier_code'] + '_' + rate['method_code'] === selectedShippingRate;\n                });\n            }\n\n            if (!availableRate && window.checkoutConfig.selectedShippingMethod) {\n                availableRate = _.find(ratesData, function (rate) {\n                    var selectedShippingMethod = window.checkoutConfig.selectedShippingMethod;\n\n                    return rate['carrier_code'] == selectedShippingMethod['carrier_code'] && //eslint-disable-line\n                        rate['method_code'] == selectedShippingMethod['method_code']; //eslint-disable-line eqeqeq\n                });\n            }\n\n            //Unset selected shipping method if not available\n            if (!availableRate) {\n                selectShippingMethodAction(null);\n            } else {\n                selectShippingMethodAction(availableRate);\n            }\n        },\n\n        /**\n         * Resolve payment method. Used local storage\n         */\n        resolvePaymentMethod: function () {\n            var availablePaymentMethods = paymentService.getAvailablePaymentMethods(),\n                selectedPaymentMethod = checkoutData.getSelectedPaymentMethod();\n\n            if (selectedPaymentMethod) {\n                availablePaymentMethods.some(function (payment) {\n                    if (payment.method == selectedPaymentMethod) { //eslint-disable-line eqeqeq\n                        selectPaymentMethodAction(payment);\n                    }\n                });\n            }\n        },\n\n        /**\n         * Resolve billing address. Used local storage\n         */\n        resolveBillingAddress: function () {\n            var selectedBillingAddress,\n                newCustomerBillingAddressData;\n\n            selectedBillingAddress = checkoutData.getSelectedBillingAddress();\n            newCustomerBillingAddressData = checkoutData.getNewCustomerBillingAddress();\n\n            if (selectedBillingAddress) {\n                if (selectedBillingAddress === 'new-customer-billing-address' && newCustomerBillingAddressData) {\n                    selectBillingAddress(createBillingAddress(newCustomerBillingAddressData));\n                } else {\n                    addressList.some(function (address) {\n                        if (selectedBillingAddress === address.getKey()) {\n                            selectBillingAddress(address);\n                        }\n                    });\n                }\n            } else {\n                this.applyBillingAddress();\n            }\n\n            if (!isBillingAddressResolvedFromBackend &&\n                !checkoutData.getBillingAddressFromData() &&\n                !_.isEmpty(window.checkoutConfig.billingAddressFromData) &&\n                !quote.billingAddress()\n            ) {\n                if (window.checkoutConfig.isBillingAddressFromDataValid === true) {\n                    selectBillingAddress(createBillingAddress(window.checkoutConfig.billingAddressFromData));\n                } else {\n                    checkoutData.setBillingAddressFromData(window.checkoutConfig.billingAddressFromData);\n                }\n                isBillingAddressResolvedFromBackend = true;\n            }\n        },\n\n        /**\n         * Apply resolved billing address to quote\n         */\n        applyBillingAddress: function () {\n            var shippingAddress,\n                isBillingAddressInitialized;\n\n            if (quote.billingAddress()) {\n                selectBillingAddress(quote.billingAddress());\n\n                return;\n            }\n\n            if (quote.isVirtual() || !quote.billingAddress()) {\n                isBillingAddressInitialized = addressList.some(function (addrs) {\n                    if (addrs.isDefaultBilling()) {\n                        selectBillingAddress(addrs);\n\n                        return true;\n                    }\n\n                    return false;\n                });\n            }\n\n            shippingAddress = quote.shippingAddress();\n\n            if (!isBillingAddressInitialized &&\n                shippingAddress &&\n                shippingAddress.canUseForBilling() &&\n                (shippingAddress.isDefaultShipping() || !quote.isVirtual())\n            ) {\n                //set billing address same as shipping by default if it is not empty\n                selectBillingAddress(quote.shippingAddress());\n            }\n        },\n\n        /**\n         * Get shipping address from address list\n         *\n         * @return {Object|null}\n         */\n        getShippingAddressFromCustomerAddressList: function () {\n            var shippingAddress = _.find(\n                    addressList(),\n                    function (address) {\n                        return checkoutData.getSelectedShippingAddress() == address.getKey() //eslint-disable-line\n                    }\n                );\n\n            if (!shippingAddress) {\n                shippingAddress = _.find(\n                    addressList(),\n                    function (address) {\n                        return address.isDefaultShipping();\n                    }\n                );\n            }\n\n            if (!shippingAddress && addressList().length === 1) {\n                shippingAddress = addressList()[0];\n            }\n\n            return shippingAddress;\n        }\n    };\n});\n","Magento_Checkout/js/model/customer-email-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Customer/js/model/customer',\n    'mage/validation'\n], function ($, customer) {\n    'use strict';\n\n    return {\n        /**\n         * Validate checkout agreements\n         *\n         * @returns {Boolean}\n         */\n        validate: function () {\n            var emailValidationResult = customer.isLoggedIn(),\n                loginFormSelector = 'form[data-role=email-with-possible-login]';\n\n            if (!customer.isLoggedIn()) {\n                $(loginFormSelector).validation();\n                emailValidationResult = Boolean($(loginFormSelector + ' input[name=username]').valid());\n            }\n\n            return emailValidationResult;\n        }\n    };\n});\n","Magento_Checkout/js/model/default-post-code-resolver.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([], function () {\n    'use strict';\n\n    /**\n     * Define necessity of using default post code value\n     */\n    var useDefaultPostCode;\n\n    return {\n        /**\n         * Resolve default post code\n         *\n         * @returns {String|null}\n         */\n        resolve: function () {\n            return useDefaultPostCode ?  window.checkoutConfig.defaultPostcode : null;\n        },\n\n        /**\n         * Set state to useDefaultPostCode variable\n         *\n         * @param {Boolean} shouldUseDefaultPostCode\n         * @returns {underscore}\n         */\n        setUseDefaultPostCode: function (shouldUseDefaultPostCode) {\n            useDefaultPostCode = shouldUseDefaultPostCode;\n\n            return this;\n        }\n    };\n});\n","Magento_Checkout/js/model/default-validation-rules.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        /**\n         * @return {Object}\n         */\n        getRules: function () {\n            return {\n                'country_id': {\n                    'required': true\n                }\n            };\n        }\n    };\n});\n","Magento_Checkout/js/model/default-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mageUtils',\n    './default-validation-rules',\n    'mage/translate'\n], function ($, utils, validationRules, $t) {\n    'use strict';\n\n    return {\n        validationErrors: [],\n\n        /**\n         * @param {Object} address\n         * @return {Boolean}\n         */\n        validate: function (address) {\n            var self = this;\n\n            this.validationErrors = [];\n            $.each(validationRules.getRules(), function (field, rule) {\n                var message;\n\n                if (rule.required && utils.isEmpty(address[field])) {\n                    message = $t('Field ') + field + $t(' is required.');\n\n                    self.validationErrors.push(message);\n                }\n            });\n\n            return !this.validationErrors.length;\n        }\n    };\n});\n","Magento_Checkout/js/model/error-processor.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'mage/url',\n    'Magento_Ui/js/model/messageList',\n    'mage/translate'\n], function (url, globalMessageList, $t) {\n    'use strict';\n\n    return {\n        /**\n         * @param {Object} response\n         * @param {Object} messageContainer\n         */\n        process: function (response, messageContainer) {\n            var error;\n\n            messageContainer = messageContainer || globalMessageList;\n\n            if (response.status == 401) { //eslint-disable-line eqeqeq\n                this.redirectTo(url.build('customer/account/login/'));\n            } else {\n                try {\n                    error = JSON.parse(response.responseText);\n                } catch (exception) {\n                    error = {\n                        message: $t('Something went wrong with your request. Please try again later.')\n                    };\n                }\n                messageContainer.addErrorMessage(error);\n            }\n        },\n\n        /**\n         * Method to redirect by requested URL.\n         */\n        redirectTo: function (redirectUrl) {\n            window.location.replace(redirectUrl);\n        }\n    };\n});\n","Magento_Checkout/js/model/full-screen-loader.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'rjsResolver'\n], function ($, resolver) {\n    'use strict';\n\n    var containerId = '#checkout';\n\n    return {\n\n        /**\n         * Start full page loader action\n         */\n        startLoader: function () {\n            $(containerId).trigger('processStart');\n        },\n\n        /**\n         * Stop full page loader action\n         *\n         * @param {Boolean} [forceStop]\n         */\n        stopLoader: function (forceStop) {\n            var $elem = $(containerId),\n                stop = $elem.trigger.bind($elem, 'processStop'); //eslint-disable-line jquery-no-bind-unbind\n\n            forceStop ? stop() : resolver(stop);\n        }\n    };\n});\n","Magento_Checkout/js/model/new-customer-address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'Magento_Checkout/js/model/default-post-code-resolver'\n], function (_, DefaultPostCodeResolver) {\n    'use strict';\n\n    /**\n     * @param {Object} addressData\n     * Returns new address object\n     */\n    return function (addressData) {\n        var identifier = Date.now(),\n            countryId = addressData['country_id'] || addressData.countryId || window.checkoutConfig.defaultCountryId,\n            regionId;\n\n        if (addressData.region && addressData.region['region_id']) {\n            regionId = addressData.region['region_id'];\n        } else if (!addressData['region_id']) {\n            regionId = undefined;\n        } else if (\n            /* eslint-disable */\n            addressData['country_id'] && addressData['country_id'] == window.checkoutConfig.defaultCountryId ||\n            !addressData['country_id'] && countryId == window.checkoutConfig.defaultCountryId\n            /* eslint-enable */\n        ) {\n            regionId = window.checkoutConfig.defaultRegionId || undefined;\n        }\n\n        return {\n            email: addressData.email,\n            countryId: countryId,\n            regionId: regionId || addressData.regionId,\n            regionCode: addressData.region ? addressData.region['region_code'] : null,\n            region: addressData.region ? addressData.region.region : null,\n            customerId: addressData['customer_id'] || addressData.customerId,\n            street: addressData.street ? _.compact(addressData.street) : addressData.street,\n            company: addressData.company,\n            telephone: addressData.telephone,\n            fax: addressData.fax,\n            postcode: addressData.postcode ? addressData.postcode : DefaultPostCodeResolver.resolve(),\n            city: addressData.city,\n            firstname: addressData.firstname,\n            lastname: addressData.lastname,\n            middlename: addressData.middlename,\n            prefix: addressData.prefix,\n            suffix: addressData.suffix,\n            vatId: addressData['vat_id'],\n            saveInAddressBook: addressData['save_in_address_book'],\n            customAttributes: addressData['custom_attributes'],\n            extensionAttributes: addressData['extension_attributes'],\n\n            /**\n             * @return {*}\n             */\n            isDefaultShipping: function () {\n                return addressData['default_shipping'];\n            },\n\n            /**\n             * @return {*}\n             */\n            isDefaultBilling: function () {\n                return addressData['default_billing'];\n            },\n\n            /**\n             * @return {String}\n             */\n            getType: function () {\n                return 'new-customer-address';\n            },\n\n            /**\n             * @return {String}\n             */\n            getKey: function () {\n                return this.getType();\n            },\n\n            /**\n             * @return {String}\n             */\n            getCacheKey: function () {\n                return this.getType() + identifier;\n            },\n\n            /**\n             * @return {Boolean}\n             */\n            isEditable: function () {\n                return true;\n            },\n\n            /**\n             * @return {Boolean}\n             */\n            canUseForBilling: function () {\n                return true;\n            }\n        };\n    };\n});\n","Magento_Checkout/js/model/payment-service.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/payment/method-list',\n    'Magento_Checkout/js/action/select-payment-method'\n], function (_, quote, methodList, selectPaymentMethod) {\n    'use strict';\n\n    /**\n    * Free method filter\n    * @param {Object} paymentMethod\n    * @return {Boolean}\n    */\n    var isFreePaymentMethod = function (paymentMethod) {\n            return paymentMethod.method === 'free';\n        },\n\n        /**\n         * Grabs the grand total from quote\n         * @return {Number}\n         */\n        getGrandTotal = function () {\n            return quote.totals()['grand_total'];\n        };\n\n    return {\n        isFreeAvailable: false,\n\n        /**\n         * Populate the list of payment methods\n         * @param {Array} methods\n         */\n        setPaymentMethods: function (methods) {\n            var freeMethod,\n                filteredMethods,\n                methodIsAvailable,\n                methodNames;\n\n            freeMethod = _.find(methods, isFreePaymentMethod);\n            this.isFreeAvailable = !!freeMethod;\n\n            if (freeMethod && getGrandTotal() <= 0) {\n                methods.splice(0, methods.length, freeMethod);\n                selectPaymentMethod(freeMethod);\n            }\n\n            filteredMethods = _.without(methods, freeMethod);\n\n            if (filteredMethods.length === 1) {\n                selectPaymentMethod(filteredMethods[0]);\n            } else if (quote.paymentMethod()) {\n                methodIsAvailable = methods.some(function (item) {\n                    return item.method === quote.paymentMethod().method;\n                });\n                //Unset selected payment method if not available\n                if (!methodIsAvailable) {\n                    selectPaymentMethod(null);\n                }\n            }\n\n            /**\n             * Overwrite methods with existing methods to preserve ko array references.\n             * This prevent ko from re-rendering those methods.\n             */\n            methodNames = _.pluck(methods, 'method');\n            _.map(methodList(), function (existingMethod) {\n                var existingMethodIndex = methodNames.indexOf(existingMethod.method);\n\n                if (existingMethodIndex !== -1) {\n                    methods[existingMethodIndex] = existingMethod;\n                }\n            });\n\n            methodList(methods);\n        },\n\n        /**\n         * Get the list of available payment methods.\n         * @return {Array}\n         */\n        getAvailablePaymentMethods: function () {\n            var allMethods = methodList().slice(),\n                grandTotalOverZero = getGrandTotal() > 0;\n\n            if (!this.isFreeAvailable) {\n                return allMethods;\n            }\n\n            if (grandTotalOverZero) {\n                return _.reject(allMethods, isFreePaymentMethod);\n            }\n\n            return _.filter(allMethods, isFreePaymentMethod);\n        }\n    };\n});\n","Magento_Checkout/js/model/place-order.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/**\n * @api\n */\ndefine(\n    [\n        'mage/storage',\n        'Magento_Checkout/js/model/error-processor',\n        'Magento_Checkout/js/model/full-screen-loader',\n        'Magento_Customer/js/customer-data',\n        'Magento_Checkout/js/model/payment/place-order-hooks',\n        'underscore'\n    ],\n    function (storage, errorProcessor, fullScreenLoader, customerData, hooks, _) {\n        'use strict';\n\n        return function (serviceUrl, payload, messageContainer) {\n            var headers = {};\n\n            fullScreenLoader.startLoader();\n            _.each(hooks.requestModifiers, function (modifier) {\n                modifier(headers, payload);\n            });\n\n            return storage.post(\n                serviceUrl, JSON.stringify(payload), true, 'application/json', headers\n            ).fail(\n                function (response) {\n                    errorProcessor.process(response, messageContainer);\n                }\n            ).done(\n                function (response) {\n                    var clearData = {\n                        'selectedShippingAddress': null,\n                        'shippingAddressFromData': null,\n                        'newCustomerShippingAddress': null,\n                        'selectedShippingRate': null,\n                        'selectedPaymentMethod': null,\n                        'selectedBillingAddress': null,\n                        'billingAddressFromData': null,\n                        'newCustomerBillingAddress': null\n                    };\n\n                    if (response.responseType !== 'error') {\n                        customerData.set('checkout-data', clearData);\n                    }\n                }\n            ).always(\n                function () {\n                    fullScreenLoader.stopLoader();\n                    _.each(hooks.afterRequestListeners, function (listener) {\n                        listener();\n                    });\n                }\n            );\n        };\n    }\n);\n","Magento_Checkout/js/model/postcode-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'mageUtils'\n], function (utils) {\n    'use strict';\n\n    return {\n        validatedPostCodeExample: [],\n\n        /**\n         * @param {*} postCode\n         * @param {*} countryId\n         * @param {Array} postCodesPatterns\n         * @return {Boolean}\n         */\n        validate: function (postCode, countryId, postCodesPatterns) {\n            var pattern, regex,\n                patterns = postCodesPatterns ? postCodesPatterns[countryId] :\n                    window.checkoutConfig.postCodes[countryId];\n\n            this.validatedPostCodeExample = [];\n\n            if (!utils.isEmpty(postCode) && !utils.isEmpty(patterns)) {\n                for (pattern in patterns) {\n                    if (patterns.hasOwnProperty(pattern)) { //eslint-disable-line max-depth\n                        this.validatedPostCodeExample.push(patterns[pattern].example);\n                        regex = new RegExp(patterns[pattern].pattern);\n\n                        if (regex.test(postCode)) { //eslint-disable-line max-depth\n                            return true;\n                        }\n                    }\n                }\n\n                return false;\n            }\n\n            return true;\n        }\n    };\n});\n","Magento_Checkout/js/model/quote.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/**\n * @api\n */\ndefine([\n    'ko',\n    'underscore',\n    'domReady!'\n], function (ko, _) {\n    'use strict';\n\n    /**\n     * Get totals data from the extension attributes.\n     * @param {*} data\n     * @returns {*}\n     */\n    var proceedTotalsData = function (data) {\n            if (_.isObject(data) && _.isObject(data['extension_attributes'])) {\n                _.each(data['extension_attributes'], function (element, index) {\n                    data[index] = element;\n                });\n            }\n\n            return data;\n        },\n        billingAddress = ko.observable(null),\n        shippingAddress = ko.observable(null),\n        shippingMethod = ko.observable(null),\n        paymentMethod = ko.observable(null),\n        quoteData = window.checkoutConfig.quoteData,\n        basePriceFormat = window.checkoutConfig.basePriceFormat,\n        priceFormat = window.checkoutConfig.priceFormat,\n        storeCode = window.checkoutConfig.storeCode,\n        totalsData = proceedTotalsData(window.checkoutConfig.totalsData),\n        totals = ko.observable(totalsData),\n        collectedTotals = ko.observable({});\n\n    return {\n        totals: totals,\n        shippingAddress: shippingAddress,\n        shippingMethod: shippingMethod,\n        billingAddress: billingAddress,\n        paymentMethod: paymentMethod,\n        guestEmail: null,\n\n        /**\n         * @return {*}\n         */\n        getQuoteId: function () {\n            return quoteData['entity_id'];\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        isVirtual: function () {\n            return !!Number(quoteData['is_virtual']);\n        },\n\n        /**\n         * @return {*}\n         */\n        getPriceFormat: function () {\n            return priceFormat;\n        },\n\n        /**\n         * @return {*}\n         */\n        getBasePriceFormat: function () {\n            return basePriceFormat;\n        },\n\n        /**\n         * @return {*}\n         */\n        getItems: function () {\n            return window.checkoutConfig.quoteItemData;\n        },\n\n        /**\n         *\n         * @return {*}\n         */\n        getTotals: function () {\n            return totals;\n        },\n\n        /**\n         * @param {Object} data\n         */\n        setTotals: function (data) {\n            data = proceedTotalsData(data);\n            totals(data);\n            this.setCollectedTotals('subtotal_with_discount', parseFloat(data['subtotal_with_discount']));\n        },\n\n        /**\n         * @param {*} paymentMethodCode\n         */\n        setPaymentMethod: function (paymentMethodCode) {\n            paymentMethod(paymentMethodCode);\n        },\n\n        /**\n         * @return {*}\n         */\n        getPaymentMethod: function () {\n            return paymentMethod;\n        },\n\n        /**\n         * @return {*}\n         */\n        getStoreCode: function () {\n            return storeCode;\n        },\n\n        /**\n         * @param {String} code\n         * @param {*} value\n         */\n        setCollectedTotals: function (code, value) {\n            var colTotals = collectedTotals();\n\n            colTotals[code] = value;\n            collectedTotals(colTotals);\n        },\n\n        /**\n         * @return {Number}\n         */\n        getCalculatedTotal: function () {\n            var total = 0.; //eslint-disable-line no-floating-decimal\n\n            _.each(collectedTotals(), function (value) {\n                total += value;\n            });\n\n            return total;\n        }\n    };\n});\n","Magento_Checkout/js/model/resource-url-manager.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Customer/js/model/customer',\n    'Magento_Checkout/js/model/url-builder',\n    'mageUtils'\n], function (customer, urlBuilder, utils) {\n        'use strict';\n\n        return {\n            /**\n             * @param {Object} quote\n             * @return {*}\n             */\n            getUrlForTotalsEstimationForNewAddress: function (quote) {\n                var params = this.getCheckoutMethod() == 'guest' ? //eslint-disable-line eqeqeq\n                        {\n                            cartId: quote.getQuoteId()\n                        } : {},\n                    urls = {\n                        'guest': '/guest-carts/:cartId/totals-information',\n                        'customer': '/carts/mine/totals-information'\n                    };\n\n                return this.getUrl(urls, params);\n            },\n\n            /**\n             * @param {Object} quote\n             * @return {*}\n             */\n            getUrlForEstimationShippingMethodsForNewAddress: function (quote) {\n                var params = this.getCheckoutMethod() == 'guest' ? //eslint-disable-line eqeqeq\n                        {\n                            quoteId: quote.getQuoteId()\n                        } : {},\n                    urls = {\n                        'guest': '/guest-carts/:quoteId/estimate-shipping-methods',\n                        'customer': '/carts/mine/estimate-shipping-methods'\n                    };\n\n                return this.getUrl(urls, params);\n            },\n\n            /**\n             * @param {Object} quote\n             * @return {*}\n             */\n            getUrlForEstimationShippingMethodsByAddressId: function (quote) {\n                var params = this.getCheckoutMethod() == 'guest' ? //eslint-disable-line eqeqeq\n                        {\n                            quoteId: quote.getQuoteId()\n                        } : {},\n                    urls = {\n                        'default': '/carts/mine/estimate-shipping-methods-by-address-id'\n                    };\n\n                return this.getUrl(urls, params);\n            },\n\n            /**\n             * @param {String} couponCode\n             * @param {String} quoteId\n             * @return {*}\n             */\n            getApplyCouponUrl: function (couponCode, quoteId) {\n                var params = this.getCheckoutMethod() == 'guest' ? //eslint-disable-line eqeqeq\n                        {\n                            quoteId: quoteId\n                        } : {},\n                    urls = {\n                        'guest': '/guest-carts/' + quoteId + '/coupons/' + encodeURIComponent(couponCode),\n                        'customer': '/carts/mine/coupons/' + encodeURIComponent(couponCode)\n                    };\n\n                return this.getUrl(urls, params);\n            },\n\n            /**\n             * @param {String} quoteId\n             * @return {*}\n             */\n            getCancelCouponUrl: function (quoteId) {\n                var params = this.getCheckoutMethod() == 'guest' ? //eslint-disable-line eqeqeq\n                        {\n                            quoteId: quoteId\n                        } : {},\n                    urls = {\n                        'guest': '/guest-carts/' + quoteId + '/coupons/',\n                        'customer': '/carts/mine/coupons/'\n                    };\n\n                return this.getUrl(urls, params);\n            },\n\n            /**\n             * @param {Object} quote\n             * @return {*}\n             */\n            getUrlForCartTotals: function (quote) {\n                var params = this.getCheckoutMethod() == 'guest' ? //eslint-disable-line eqeqeq\n                        {\n                            quoteId: quote.getQuoteId()\n                        } : {},\n                    urls = {\n                        'guest': '/guest-carts/:quoteId/totals',\n                        'customer': '/carts/mine/totals'\n                    };\n\n                return this.getUrl(urls, params);\n            },\n\n            /**\n             * @param {Object} quote\n             * @return {*}\n             */\n            getUrlForSetShippingInformation: function (quote) {\n                var params = this.getCheckoutMethod() == 'guest' ? //eslint-disable-line eqeqeq\n                        {\n                            cartId: quote.getQuoteId()\n                        } : {},\n                    urls = {\n                        'guest': '/guest-carts/:cartId/shipping-information',\n                        'customer': '/carts/mine/shipping-information'\n                    };\n\n                return this.getUrl(urls, params);\n            },\n\n            /**\n             * Get url for service.\n             *\n             * @param {*} urls\n             * @param {*} urlParams\n             * @return {String|*}\n             */\n            getUrl: function (urls, urlParams) {\n                var url;\n\n                if (utils.isEmpty(urls)) {\n                    return 'Provided service call does not exist.';\n                }\n\n                if (!utils.isEmpty(urls['default'])) {\n                    url = urls['default'];\n                } else {\n                    url = urls[this.getCheckoutMethod()];\n                }\n\n                return urlBuilder.createUrl(url, urlParams);\n            },\n\n            /**\n             * @return {String}\n             */\n            getCheckoutMethod: function () {\n                return customer.isLoggedIn() ? 'customer' : 'guest';\n            }\n        };\n    }\n);\n","Magento_Checkout/js/model/shipping-rate-registry.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    var cache = [];\n\n    return {\n        /**\n         * @param {String} addressKey\n         * @return {*}\n         */\n        get: function (addressKey) {\n            if (cache[addressKey]) {\n                return cache[addressKey];\n            }\n\n            return false;\n        },\n\n        /**\n         * @param {String} addressKey\n         * @param {*} data\n         */\n        set: function (addressKey, data) {\n            cache[addressKey] = data;\n        }\n    };\n});\n","Magento_Checkout/js/model/shipping-rate-service.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/shipping-rate-processor/new-address',\n    'Magento_Checkout/js/model/shipping-rate-processor/customer-address'\n], function (quote, defaultProcessor, customerAddressProcessor) {\n    'use strict';\n\n    var processors = {};\n\n    processors.default =  defaultProcessor;\n    processors['customer-address'] = customerAddressProcessor;\n\n    quote.shippingAddress.subscribe(function () {\n        var type = quote.shippingAddress().getType();\n\n        if (processors[type]) {\n            processors[type].getRates(quote.shippingAddress());\n        } else {\n            processors.default.getRates(quote.shippingAddress());\n        }\n    });\n\n    return {\n        /**\n         * @param {String} type\n         * @param {*} processor\n         */\n        registerProcessor: function (type, processor) {\n            processors[type] = processor;\n        }\n    };\n});\n","Magento_Checkout/js/model/shipping-rates-validation-rules.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine(['jquery'], function ($) {\n    'use strict';\n\n    var ratesRules = {},\n        checkoutConfig = window.checkoutConfig;\n\n    return {\n        /**\n         * @param {String} carrier\n         * @param {Object} rules\n         */\n        registerRules: function (carrier, rules) {\n            if (checkoutConfig.activeCarriers.indexOf(carrier) !== -1) {\n                ratesRules[carrier] = rules.getRules();\n            }\n        },\n\n        /**\n         * @return {Object}\n         */\n        getRules: function () {\n            return ratesRules;\n        },\n\n        /**\n         * @return {Array}\n         */\n        getObservableFields: function () {\n            var self = this,\n                observableFields = [];\n\n            $.each(self.getRules(), function (carrier, fields) {\n                $.each(fields, function (field) {\n                    if (observableFields.indexOf(field) === -1) {\n                        observableFields.push(field);\n                    }\n                });\n            });\n\n            return observableFields;\n        }\n    };\n});\n","Magento_Checkout/js/model/shipping-rates-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'ko',\n    './shipping-rates-validation-rules',\n    '../model/address-converter',\n    '../action/select-shipping-address',\n    './postcode-validator',\n    './default-validator',\n    'mage/translate',\n    'uiRegistry',\n    'Magento_Checkout/js/model/shipping-address/form-popup-state',\n    'Magento_Checkout/js/model/quote'\n], function (\n    $,\n    ko,\n    shippingRatesValidationRules,\n    addressConverter,\n    selectShippingAddress,\n    postcodeValidator,\n    defaultValidator,\n    $t,\n    uiRegistry,\n    formPopUpState\n) {\n    'use strict';\n\n    var checkoutConfig = window.checkoutConfig,\n        validators = [],\n        observedElements = [],\n        postcodeElements = [],\n        postcodeElementName = 'postcode';\n\n    validators.push(defaultValidator);\n\n    return {\n        validateAddressTimeout: 0,\n        validateZipCodeTimeout: 0,\n        validateDelay: 2000,\n\n        /**\n         * @param {String} carrier\n         * @param {Object} validator\n         */\n        registerValidator: function (carrier, validator) {\n            if (checkoutConfig.activeCarriers.indexOf(carrier) !== -1) {\n                validators.push(validator);\n            }\n        },\n\n        /**\n         * @param {Object} address\n         * @return {Boolean}\n         */\n        validateAddressData: function (address) {\n            return validators.some(function (validator) {\n                return validator.validate(address);\n            });\n        },\n\n        /**\n         * Perform postponed binding for fieldset elements\n         *\n         * @param {String} formPath\n         */\n        initFields: function (formPath) {\n            var self = this,\n                elements = shippingRatesValidationRules.getObservableFields();\n\n            if ($.inArray(postcodeElementName, elements) === -1) {\n                // Add postcode field to observables if not exist for zip code validation support\n                elements.push(postcodeElementName);\n            }\n\n            $.each(elements, function (index, field) {\n                uiRegistry.async(formPath + '.' + field)(self.doElementBinding.bind(self));\n            });\n        },\n\n        /**\n         * Bind shipping rates request to form element\n         *\n         * @param {Object} element\n         * @param {Boolean} force\n         * @param {Number} delay\n         */\n        doElementBinding: function (element, force, delay) {\n            var observableFields = shippingRatesValidationRules.getObservableFields();\n\n            if (element && (observableFields.indexOf(element.index) !== -1 || force)) {\n                if (element.index !== postcodeElementName) {\n                    this.bindHandler(element, delay);\n                }\n            }\n\n            if (element.index === postcodeElementName) {\n                this.bindHandler(element, delay);\n                postcodeElements.push(element);\n            }\n        },\n\n        /**\n         * @param {*} elements\n         * @param {Boolean} force\n         * @param {Number} delay\n         */\n        bindChangeHandlers: function (elements, force, delay) {\n            var self = this;\n\n            $.each(elements, function (index, elem) {\n                self.doElementBinding(elem, force, delay);\n            });\n        },\n\n        /**\n         * @param {Object} element\n         * @param {Number} delay\n         */\n        bindHandler: function (element, delay) {\n            var self = this;\n\n            delay = typeof delay === 'undefined' ? self.validateDelay : delay;\n\n            if (element.component.indexOf('/group') !== -1) {\n                $.each(element.elems(), function (index, elem) {\n                    self.bindHandler(elem);\n                });\n            } else {\n                element.on('value', function () {\n                    clearTimeout(self.validateZipCodeTimeout);\n                    self.validateZipCodeTimeout = setTimeout(function () {\n                        if (element.index === postcodeElementName) {\n                            self.postcodeValidation(element);\n                        } else {\n                            $.each(postcodeElements, function (index, elem) {\n                                self.postcodeValidation(elem);\n                            });\n                        }\n                    }, delay);\n\n                    if (!formPopUpState.isVisible()) {\n                        clearTimeout(self.validateAddressTimeout);\n                        self.validateAddressTimeout = setTimeout(function () {\n                            self.validateFields();\n                        }, delay);\n                    }\n                });\n                observedElements.push(element);\n            }\n        },\n\n        /**\n         * @return {*}\n         */\n        postcodeValidation: function (postcodeElement) {\n            var countryId = $('select[name=\"country_id\"]:visible').val(),\n                validationResult,\n                warnMessage;\n\n            if (postcodeElement == null || postcodeElement.value() == null) {\n                return true;\n            }\n\n            postcodeElement.warn(null);\n            validationResult = postcodeValidator.validate(postcodeElement.value(), countryId);\n\n            if (!validationResult) {\n                warnMessage = $t('Provided Zip/Postal Code seems to be invalid.');\n\n                if (postcodeValidator.validatedPostCodeExample.length) {\n                    warnMessage += $t(' Example: ') + postcodeValidator.validatedPostCodeExample.join('; ') + '. ';\n                }\n                warnMessage += $t('If you believe it is the right one you can ignore this notice.');\n                postcodeElement.warn(warnMessage);\n            }\n\n            return validationResult;\n        },\n\n        /**\n         * Convert form data to quote address and validate fields for shipping rates\n         */\n        validateFields: function () {\n            var addressFlat = addressConverter.formDataProviderToFlatData(\n                this.collectObservedData(),\n                'shippingAddress'\n                ),\n                address;\n\n            if (this.validateAddressData(addressFlat)) {\n                addressFlat = uiRegistry.get('checkoutProvider').shippingAddress;\n                address = addressConverter.formAddressDataToQuoteAddress(addressFlat);\n                selectShippingAddress(address);\n            }\n        },\n\n        /**\n         * Collect observed fields data to object\n         *\n         * @returns {*}\n         */\n        collectObservedData: function () {\n            var observedValues = {};\n\n            $.each(observedElements, function (index, field) {\n                observedValues[field.dataScope] = field.value();\n            });\n\n            return observedValues;\n        }\n    };\n});\n","Magento_Checkout/js/model/shipping-save-processor.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_Checkout/js/model/shipping-save-processor/default'\n], function (defaultProcessor) {\n    'use strict';\n\n    var processors = {};\n\n    processors['default'] =  defaultProcessor;\n\n    return {\n        /**\n         * @param {String} type\n         * @param {*} processor\n         */\n        registerProcessor: function (type, processor) {\n            processors[type] = processor;\n        },\n\n        /**\n         * @param {String} type\n         * @return {Array}\n         */\n        saveShippingInformation: function (type) {\n            var rates = [];\n\n            if (processors[type]) {\n                rates = processors[type].saveShippingInformation();\n            } else {\n                rates = processors['default'].saveShippingInformation();\n            }\n\n            return rates;\n        }\n    };\n});\n","Magento_Checkout/js/model/shipping-service.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko',\n    'Magento_Checkout/js/model/checkout-data-resolver'\n], function (ko, checkoutDataResolver) {\n    'use strict';\n\n    var shippingRates = ko.observableArray([]);\n\n    return {\n        isLoading: ko.observable(false),\n\n        /**\n         * Set shipping rates\n         *\n         * @param {*} ratesData\n         */\n        setShippingRates: function (ratesData) {\n            shippingRates(ratesData);\n            shippingRates.valueHasMutated();\n            checkoutDataResolver.resolveShippingRates(ratesData);\n        },\n\n        /**\n         * Get shipping rates\n         *\n         * @returns {*}\n         */\n        getShippingRates: function () {\n            return shippingRates;\n        }\n    };\n});\n","Magento_Checkout/js/model/sidebar.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        popUp: false,\n\n        /**\n         * @param {Object} popUp\n         */\n        setPopup: function (popUp) {\n            this.popUp = popUp;\n        },\n\n        /**\n         * Show popup.\n         */\n        show: function () {\n            if (this.popUp) {\n                this.popUp.modal('openModal');\n            }\n        },\n\n        /**\n         * Hide popup.\n         */\n        hide: function () {\n            if (this.popUp) {\n                this.popUp.modal('closeModal');\n            }\n        }\n    };\n});\n","Magento_Checkout/js/model/step-navigator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'ko'\n], function ($, ko) {\n    'use strict';\n\n    var steps = ko.observableArray();\n\n    return {\n        steps: steps,\n        stepCodes: [],\n        validCodes: [],\n\n        /**\n         * @return {Boolean}\n         */\n        handleHash: function () {\n            var hashString = window.location.hash.replace('#', ''),\n                isRequestedStepVisible;\n\n            if (hashString === '') {\n                return false;\n            }\n\n            if ($.inArray(hashString, this.validCodes) === -1) {\n                window.location.href = window.checkoutConfig.pageNotFoundUrl;\n\n                return false;\n            }\n\n            isRequestedStepVisible = steps.sort(this.sortItems).some(function (element) {\n                return (element.code == hashString || element.alias == hashString) && element.isVisible(); //eslint-disable-line\n            });\n\n            //if requested step is visible, then we don't need to load step data from server\n            if (isRequestedStepVisible) {\n                return false;\n            }\n\n            steps().sort(this.sortItems).forEach(function (element) {\n                if (element.code == hashString || element.alias == hashString) { //eslint-disable-line eqeqeq\n                    element.navigate(element);\n                } else {\n                    element.isVisible(false);\n                }\n\n            });\n\n            return false;\n        },\n\n        /**\n         * @param {String} code\n         * @param {*} alias\n         * @param {*} title\n         * @param {Function} isVisible\n         * @param {*} navigate\n         * @param {*} sortOrder\n         */\n        registerStep: function (code, alias, title, isVisible, navigate, sortOrder) {\n            var hash, active;\n\n            if ($.inArray(code, this.validCodes) !== -1) {\n                throw new DOMException('Step code [' + code + '] already registered in step navigator');\n            }\n\n            if (alias != null) {\n                if ($.inArray(alias, this.validCodes) !== -1) {\n                    throw new DOMException('Step code [' + alias + '] already registered in step navigator');\n                }\n                this.validCodes.push(alias);\n            }\n            this.validCodes.push(code);\n            steps.push({\n                code: code,\n                alias: alias != null ? alias : code,\n                title: title,\n                isVisible: isVisible,\n                navigate: navigate,\n                sortOrder: sortOrder\n            });\n            active = this.getActiveItemIndex();\n            steps.each(function (elem, index) {\n                if (active !== index) {\n                    elem.isVisible(false);\n                }\n            });\n            this.stepCodes.push(code);\n            hash = window.location.hash.replace('#', '');\n\n            if (hash != '' && hash != code) { //eslint-disable-line eqeqeq\n                //Force hiding of not active step\n                isVisible(false);\n            }\n        },\n\n        /**\n         * @param {Object} itemOne\n         * @param {Object} itemTwo\n         * @return {Number}\n         */\n        sortItems: function (itemOne, itemTwo) {\n            return itemOne.sortOrder > itemTwo.sortOrder ? 1 : -1;\n        },\n\n        /**\n         * @return {Number}\n         */\n        getActiveItemIndex: function () {\n            var activeIndex = 0;\n\n            steps().sort(this.sortItems).some(function (element, index) {\n                if (element.isVisible()) {\n                    activeIndex = index;\n\n                    return true;\n                }\n\n                return false;\n            });\n\n            return activeIndex;\n        },\n\n        /**\n         * @param {*} code\n         * @return {Boolean}\n         */\n        isProcessed: function (code) {\n            var activeItemIndex = this.getActiveItemIndex(),\n                sortedItems = steps().sort(this.sortItems),\n                requestedItemIndex = -1;\n\n            sortedItems.forEach(function (element, index) {\n                if (element.code == code) { //eslint-disable-line eqeqeq\n                    requestedItemIndex = index;\n                }\n            });\n\n            return activeItemIndex > requestedItemIndex;\n        },\n\n        /**\n         * @param {*} code\n         * @param {*} scrollToElementId\n         */\n        navigateTo: function (code, scrollToElementId) {\n            var sortedItems = steps().sort(this.sortItems),\n                bodyElem = $('body');\n\n            scrollToElementId = scrollToElementId || null;\n\n            if (!this.isProcessed(code)) {\n                return;\n            }\n            sortedItems.forEach(function (element) {\n                if (element.code == code) { //eslint-disable-line eqeqeq\n                    element.isVisible(true);\n                    bodyElem.animate({\n                        scrollTop: $('#' + code).offset().top\n                    }, 0, function () {\n                        window.location = window.checkoutConfig.checkoutUrl + '#' + code;\n                    });\n\n                    if (scrollToElementId && $('#' + scrollToElementId).length) {\n                        bodyElem.animate({\n                            scrollTop: $('#' + scrollToElementId).offset().top\n                        }, 0);\n                    }\n                } else {\n                    element.isVisible(false);\n                }\n\n            });\n        },\n\n        /**\n         * Sets window location hash.\n         *\n         * @param {String} hash\n         */\n        setHash: function (hash) {\n            window.location.hash = hash;\n        },\n\n        /**\n         * Next step.\n         */\n        next: function () {\n            var activeIndex = 0,\n                code;\n\n            steps().sort(this.sortItems).forEach(function (element, index) {\n                if (element.isVisible()) {\n                    element.isVisible(false);\n                    activeIndex = index;\n                }\n            });\n\n            if (steps().length > activeIndex + 1) {\n                code = steps()[activeIndex + 1].code;\n                steps()[activeIndex + 1].isVisible(true);\n                this.setHash(code);\n                document.body.scrollTop = document.documentElement.scrollTop = 0;\n            }\n        }\n    };\n});\n","Magento_Checkout/js/model/totals.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'ko',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Customer/js/customer-data'\n], function (ko, quote, customerData) {\n    'use strict';\n\n    var quoteItems = ko.observable(quote.totals().items),\n        cartData = customerData.get('cart'),\n        quoteSubtotal = parseFloat(quote.totals().subtotal),\n        subtotalAmount = parseFloat(cartData().subtotalAmount);\n\n    quote.totals.subscribe(function (newValue) {\n        quoteItems(newValue.items);\n    });\n\n    if (!isNaN(subtotalAmount) && quoteSubtotal !== subtotalAmount && quoteSubtotal !== 0) {\n        customerData.reload(['cart'], false);\n    }\n\n    return {\n        totals: quote.totals,\n        isLoading: ko.observable(false),\n\n        /**\n         * @return {Function}\n         */\n        getItems: function () {\n            return quoteItems;\n        },\n\n        /**\n         * @param {*} code\n         * @return {*}\n         */\n        getSegment: function (code) {\n            var i, total;\n\n            if (!this.totals()) {\n                return null;\n            }\n\n            for (i in this.totals()['total_segments']) { //eslint-disable-line guard-for-in\n                total = this.totals()['total_segments'][i];\n\n                if (total.code == code) { //eslint-disable-line eqeqeq\n                    return total;\n                }\n            }\n\n            return null;\n        }\n    };\n});\n","Magento_Checkout/js/model/url-builder.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['jquery'], function ($) {\n    'use strict';\n\n    return {\n        method: 'rest',\n        storeCode: window.checkoutConfig.storeCode,\n        version: 'V1',\n        serviceUrl: ':method/:storeCode/:version',\n\n        /**\n         * @param {String} url\n         * @param {Object} params\n         * @return {*}\n         */\n        createUrl: function (url, params) {\n            var completeUrl = this.serviceUrl + url;\n\n            return this.bindParams(completeUrl, params);\n        },\n\n        /**\n         * @param {String} url\n         * @param {Object} params\n         * @return {*}\n         */\n        bindParams: function (url, params) {\n            var urlParts;\n\n            params.method = this.method;\n            params.storeCode = this.storeCode;\n            params.version = this.version;\n\n            urlParts = url.split('/');\n            urlParts = urlParts.filter(Boolean);\n\n            $.each(urlParts, function (key, part) {\n                part = part.replace(':', '');\n\n                if (params[part] != undefined) { //eslint-disable-line eqeqeq\n                    urlParts[key] = params[part];\n                }\n            });\n\n            return urlParts.join('/');\n        }\n    };\n});\n","Magento_Checkout/js/model/cart/cache.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * Cart adapter for customer data storage.\n * It stores cart data in customer data(localStorage) without saving on server.\n * Adapter is created for shipping rates and totals data caching. It eliminates unneeded calculations requests.\n */\ndefine([\n    'underscore',\n    'Magento_Customer/js/customer-data',\n    'mageUtils'\n], function (_, storage, utils) {\n    'use strict';\n\n    var cacheKey = 'cart-data',\n        cartData = {\n            totals: null,\n            address: null,\n            cartVersion: null,\n            shippingMethodCode: null,\n            shippingCarrierCode: null,\n            rates: null\n        },\n\n        /**\n         * Set data to local storage.\n         *\n         * @param {Object} checkoutData\n         */\n        setData = function (checkoutData) {\n            storage.set(cacheKey, checkoutData);\n        },\n\n        /**\n         * Get data from local storage.\n         *\n         * @param {String} [key]\n         * @returns {*}\n         */\n        getData = function (key) {\n            var data = key ? storage.get(cacheKey)()[key] : storage.get(cacheKey)();\n\n            if (_.isEmpty(storage.get(cacheKey)())) {\n                setData(utils.copy(cartData));\n            }\n\n            return data;\n        },\n\n        /**\n         * Build method name base on name, prefix and suffix.\n         *\n         * @param {String} name\n         * @param {String} prefix\n         * @param {String} suffix\n         * @return {String}\n         */\n        getMethodName = function (name, prefix, suffix) {\n            prefix = prefix || '';\n            suffix = suffix || '';\n\n            return prefix + name.charAt(0).toUpperCase() + name.slice(1) + suffix;\n        };\n\n    /**\n     * Provides get/set/isChanged/clear methods for work with cart data.\n     * Can be customized via mixin functionality.\n     */\n    return {\n        cartData: cartData,\n\n        /**\n         * Array of required address fields\n         */\n        requiredFields: ['countryId', 'region', 'regionId', 'postcode'],\n\n        /**\n         * Get data from customer data.\n         * Concatenate provided key with method name and call method if it exist or makes get by key.\n         *\n         * @param {String} key\n         * @return {*}\n         */\n        get: function (key) {\n            var methodName = getMethodName(key, '_get');\n\n            if (key === cacheKey) {\n                return getData();\n            }\n\n            if (this[methodName]) {\n                return this[methodName]();\n            }\n\n            return getData(key);\n        },\n\n        /**\n         * Set data to customer data.\n         * Concatenate provided key with method name and call method if it exist or makes set by key.\n         * @example _setCustomAddress method will be called, if it exists.\n         *  set('address', customAddressValue)\n         * @example Will set value by provided key.\n         *  set('rates', ratesToCompare)\n         *\n         * @param {String} key\n         * @param {*} value\n         */\n        set: function (key, value) {\n            var methodName = getMethodName(key, '_set'),\n                obj;\n\n            if (key === cacheKey) {\n                _.each(value, function (val, k) {\n                    this.set(k, val);\n                }, this);\n\n                return;\n            }\n\n            if (this[methodName]) {\n                this[methodName](value);\n            } else {\n                obj = getData();\n                obj[key] = value;\n                setData(obj);\n            }\n        },\n\n        /**\n         * Clear data in cache.\n         * Concatenate provided key with method name and call method if it exist or clear by key.\n         * @example _clearCustomAddress method will be called, if it exist.\n         *  clear('customAddress')\n         * @example Will clear data by provided key.\n         *  clear('rates')\n         *\n         * @param {String} key\n         */\n        clear: function (key) {\n            var methodName = getMethodName(key, '_clear');\n\n            if (key === cacheKey) {\n                setData(this.cartData);\n\n                return;\n            }\n\n            if (this[methodName]) {\n                this[methodName]();\n            } else {\n                this.set(key, null);\n            }\n        },\n\n        /**\n         * Check if provided data has difference with cached data.\n         * Concatenate provided key with method name and call method if it exist or makes strict equality.\n         * @example Will call existing _isAddressChanged.\n         *  isChanged('address', addressToCompare)\n         * @example Will get data by provided key and make strict equality with provided value.\n         *  isChanged('rates', ratesToCompare)\n         *\n         * @param {String} key\n         * @param {*} value\n         * @return {Boolean}\n         */\n        isChanged: function (key, value) {\n            var methodName = getMethodName(key, '_is', 'Changed');\n\n            if (this[methodName]) {\n                return this[methodName](value);\n            }\n\n            return this.get(key) !== value;\n        },\n\n        /**\n         * Compare cached address with provided.\n         * Custom method for check object equality.\n         *\n         * @param {Object} address\n         * @returns {Boolean}\n         */\n        _isAddressChanged: function (address) {\n            return JSON.stringify(_.pick(this.get('address'), this.requiredFields)) !==\n                JSON.stringify(_.pick(address, this.requiredFields));\n        },\n\n        /**\n         * Compare cached subtotal with provided.\n         * Custom method for check object equality.\n         *\n         * @param {float} subtotal\n         * @returns {Boolean}\n         */\n        _isSubtotalChanged: function (subtotal) {\n            var cached = parseFloat(this.get('totals').subtotal);\n\n            return subtotal !== cached;\n        }\n    };\n});\n","Magento_Checkout/js/model/cart/estimate-service.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/shipping-rate-processor/new-address',\n    'Magento_Checkout/js/model/cart/totals-processor/default',\n    'Magento_Checkout/js/model/shipping-service',\n    'Magento_Checkout/js/model/cart/cache',\n    'Magento_Customer/js/customer-data'\n], function (quote, defaultProcessor, totalsDefaultProvider, shippingService, cartCache, customerData) {\n    'use strict';\n\n    var rateProcessors = {},\n        totalsProcessors = {},\n\n        /**\n         * Estimate totals for shipping address and update shipping rates.\n         */\n        estimateTotalsAndUpdateRates = function () {\n            var type = quote.shippingAddress().getType();\n\n            if (\n                quote.isVirtual() ||\n                window.checkoutConfig.activeCarriers && window.checkoutConfig.activeCarriers.length === 0\n            ) {\n                // update totals block when estimated address was set\n                totalsProcessors['default'] = totalsDefaultProvider;\n                totalsProcessors[type] ?\n                    totalsProcessors[type].estimateTotals(quote.shippingAddress()) :\n                    totalsProcessors['default'].estimateTotals(quote.shippingAddress());\n            } else {\n                // check if user data not changed -> load rates from cache\n                if (!cartCache.isChanged('address', quote.shippingAddress()) &&\n                    !cartCache.isChanged('cartVersion', customerData.get('cart')()['data_id']) &&\n                    cartCache.get('rates')\n                ) {\n                    shippingService.setShippingRates(cartCache.get('rates'));\n\n                    return;\n                }\n\n                // update rates list when estimated address was set\n                rateProcessors['default'] = defaultProcessor;\n                rateProcessors[type] ?\n                    rateProcessors[type].getRates(quote.shippingAddress()) :\n                    rateProcessors['default'].getRates(quote.shippingAddress());\n\n                // save rates to cache after load\n                shippingService.getShippingRates().subscribe(function (rates) {\n                    cartCache.set('rates', rates);\n                });\n            }\n        },\n\n        /**\n         * Estimate totals for shipping address.\n         */\n        estimateTotalsShipping = function () {\n            totalsDefaultProvider.estimateTotals(quote.shippingAddress());\n        },\n\n        /**\n         * Estimate totals for billing address.\n         */\n        estimateTotalsBilling = function () {\n            var type = quote.billingAddress().getType();\n\n            if (quote.isVirtual()) {\n                // update totals block when estimated address was set\n                totalsProcessors['default'] = totalsDefaultProvider;\n                totalsProcessors[type] ?\n                    totalsProcessors[type].estimateTotals(quote.billingAddress()) :\n                    totalsProcessors['default'].estimateTotals(quote.billingAddress());\n            }\n        };\n\n    quote.shippingAddress.subscribe(estimateTotalsAndUpdateRates);\n    quote.shippingMethod.subscribe(estimateTotalsShipping);\n    quote.billingAddress.subscribe(estimateTotalsBilling);\n});\n","Magento_Checkout/js/model/cart/totals-processor/default.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'Magento_Checkout/js/model/resource-url-manager',\n    'Magento_Checkout/js/model/quote',\n    'mage/storage',\n    'Magento_Checkout/js/model/totals',\n    'Magento_Checkout/js/model/error-processor',\n    'Magento_Checkout/js/model/cart/cache',\n    'Magento_Customer/js/customer-data'\n], function (_, resourceUrlManager, quote, storage, totalsService, errorProcessor, cartCache, customerData) {\n    'use strict';\n\n    /**\n     * Load data from server.\n     *\n     * @param {Object} address\n     */\n    var loadFromServer = function (address) {\n        var serviceUrl,\n            payload;\n\n        // Start loader for totals block\n        totalsService.isLoading(true);\n        serviceUrl = resourceUrlManager.getUrlForTotalsEstimationForNewAddress(quote);\n        payload = {\n            addressInformation: {\n                address: _.pick(address, cartCache.requiredFields)\n            }\n        };\n\n        if (quote.shippingMethod() && quote.shippingMethod()['method_code']) {\n            payload.addressInformation['shipping_method_code'] = quote.shippingMethod()['method_code'];\n            payload.addressInformation['shipping_carrier_code'] = quote.shippingMethod()['carrier_code'];\n        }\n\n        return storage.post(\n            serviceUrl, JSON.stringify(payload), false\n        ).done(function (result) {\n            var data = {\n                totals: result,\n                address: address,\n                cartVersion: customerData.get('cart')()['data_id'],\n                shippingMethodCode: null,\n                shippingCarrierCode: null\n            };\n\n            if (quote.shippingMethod() && quote.shippingMethod()['method_code']) {\n                data.shippingMethodCode = quote.shippingMethod()['method_code'];\n                data.shippingCarrierCode = quote.shippingMethod()['carrier_code'];\n            }\n\n            quote.setTotals(result);\n            cartCache.set('cart-data', data);\n        }).fail(function (response) {\n            errorProcessor.process(response);\n        }).always(function () {\n            // Stop loader for totals block\n            totalsService.isLoading(false);\n        });\n    };\n\n    return {\n        /**\n         * Array of required address fields.\n         * @property {Array.String} requiredFields\n         * @deprecated Use cart cache.\n         */\n        requiredFields: cartCache.requiredFields,\n\n        /**\n         * Get shipping rates for specified address.\n         * @param {Object} address\n         */\n        estimateTotals: function (address) {\n            var data = {\n                shippingMethodCode: null,\n                shippingCarrierCode: null\n            };\n\n            if (quote.shippingMethod() && quote.shippingMethod()['method_code']) {\n                data.shippingMethodCode = quote.shippingMethod()['method_code'];\n                data.shippingCarrierCode = quote.shippingMethod()['carrier_code'];\n            }\n\n            return loadFromServer(address);\n        }\n    };\n});\n","Magento_Checkout/js/model/payment/additional-validators.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([], function () {\n    'use strict';\n\n    var validators = [];\n\n    return {\n        /**\n         * Register unique validator\n         *\n         * @param {*} validator\n         */\n        registerValidator: function (validator) {\n            validators.push(validator);\n        },\n\n        /**\n         * Returns array of registered validators\n         *\n         * @returns {Array}\n         */\n        getValidators: function () {\n            return validators;\n        },\n\n        /**\n         * Process validators\n         *\n         * @returns {Boolean}\n         */\n        validate: function (hideError) {\n            var validationResult = true;\n\n            hideError = hideError || false;\n\n            if (validators.length <= 0) {\n                return validationResult;\n            }\n\n            validators.forEach(function (item) {\n                if (item.validate(hideError) == false) { //eslint-disable-line eqeqeq\n                    validationResult = false;\n\n                    return false;\n                }\n            });\n\n            return validationResult;\n        }\n    };\n});\n","Magento_Checkout/js/model/payment/method-converter.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore'\n], function (_) {\n    'use strict';\n\n    return function (methods) {\n        _.each(methods, function (method) {\n            if (method.hasOwnProperty('code')) {\n                method.method = method.code;\n                delete method.code;\n            }\n        });\n\n        return methods;\n    };\n});\n","Magento_Checkout/js/model/payment/method-group.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n        'uiElement',\n        'mage/translate'\n], function (Element, $t) {\n    'use strict';\n\n    var DEFAULT_GROUP_ALIAS = 'default';\n\n    return Element.extend({\n        defaults: {\n            alias: DEFAULT_GROUP_ALIAS,\n            title: $t('Payment Method'),\n            sortOrder: 100,\n            displayArea: 'payment-methods-items-${ $.alias }'\n        },\n\n        /**\n         * Checks if group instance is default\n         *\n         * @returns {Boolean}\n         */\n        isDefault: function () {\n            return this.alias === DEFAULT_GROUP_ALIAS;\n        }\n    });\n});\n","Magento_Checkout/js/model/payment/method-list.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko'\n], function (ko) {\n    'use strict';\n\n    return ko.observableArray([]);\n});\n","Magento_Checkout/js/model/payment/place-order-hooks.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        requestModifiers: [],\n        afterRequestListeners: []\n    };\n});\n","Magento_Checkout/js/model/payment/renderer-list.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko'\n], function (ko) {\n    'use strict';\n\n    return ko.observableArray([]);\n});\n","Magento_Checkout/js/model/shipping-address/form-popup-state.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko'\n], function (ko) {\n    'use strict';\n\n    return {\n        isVisible: ko.observable(false)\n    };\n});\n","Magento_Checkout/js/model/shipping-rate-processor/customer-address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/model/resource-url-manager',\n    'Magento_Checkout/js/model/quote',\n    'mage/storage',\n    'Magento_Checkout/js/model/shipping-service',\n    'Magento_Checkout/js/model/shipping-rate-registry',\n    'Magento_Checkout/js/model/error-processor'\n], function (resourceUrlManager, quote, storage, shippingService, rateRegistry, errorProcessor) {\n    'use strict';\n\n    return {\n        /**\n         * @param {Object} address\n         */\n        getRates: function (address) {\n            var cache;\n\n            shippingService.isLoading(true);\n            cache = rateRegistry.get(address.getKey());\n\n            if (cache) {\n                shippingService.setShippingRates(cache);\n                shippingService.isLoading(false);\n            } else {\n                storage.post(\n                    resourceUrlManager.getUrlForEstimationShippingMethodsByAddressId(),\n                    JSON.stringify({\n                        addressId: address.customerAddressId\n                    }),\n                    false\n                ).done(function (result) {\n                    rateRegistry.set(address.getKey(), result);\n                    shippingService.setShippingRates(result);\n                }).fail(function (response) {\n                    shippingService.setShippingRates([]);\n                    errorProcessor.process(response);\n                }).always(function () {\n                    shippingService.isLoading(false);\n                }\n                );\n            }\n        }\n    };\n});\n","Magento_Checkout/js/model/shipping-rate-processor/new-address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'Magento_Checkout/js/model/resource-url-manager',\n    'Magento_Checkout/js/model/quote',\n    'mage/storage',\n    'Magento_Checkout/js/model/shipping-service',\n    'Magento_Checkout/js/model/shipping-rate-registry',\n    'Magento_Checkout/js/model/error-processor'\n], function (resourceUrlManager, quote, storage, shippingService, rateRegistry, errorProcessor) {\n    'use strict';\n\n    return {\n        /**\n         * Get shipping rates for specified address.\n         * @param {Object} address\n         */\n        getRates: function (address) {\n            var cache, serviceUrl, payload;\n\n            shippingService.isLoading(true);\n            cache = rateRegistry.get(address.getCacheKey());\n            serviceUrl = resourceUrlManager.getUrlForEstimationShippingMethodsForNewAddress(quote);\n            payload = JSON.stringify({\n                    address: {\n                        'street': address.street,\n                        'city': address.city,\n                        'region_id': address.regionId,\n                        'region': address.region,\n                        'country_id': address.countryId,\n                        'postcode': address.postcode,\n                        'email': address.email,\n                        'customer_id': address.customerId,\n                        'firstname': address.firstname,\n                        'lastname': address.lastname,\n                        'middlename': address.middlename,\n                        'prefix': address.prefix,\n                        'suffix': address.suffix,\n                        'vat_id': address.vatId,\n                        'company': address.company,\n                        'telephone': address.telephone,\n                        'fax': address.fax,\n                        'custom_attributes': address.customAttributes,\n                        'save_in_address_book': address.saveInAddressBook\n                    }\n                }\n            );\n\n            if (cache) {\n                shippingService.setShippingRates(cache);\n                shippingService.isLoading(false);\n            } else {\n                storage.post(\n                    serviceUrl, payload, false\n                ).done(function (result) {\n                    rateRegistry.set(address.getCacheKey(), result);\n                    shippingService.setShippingRates(result);\n                }).fail(function (response) {\n                    shippingService.setShippingRates([]);\n                    errorProcessor.process(response);\n                }).always(function () {\n                    shippingService.isLoading(false);\n                });\n            }\n        }\n    };\n});\n","Magento_Checkout/js/model/shipping-save-processor/default.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/resource-url-manager',\n    'mage/storage',\n    'Magento_Checkout/js/model/payment-service',\n    'Magento_Checkout/js/model/payment/method-converter',\n    'Magento_Checkout/js/model/error-processor',\n    'Magento_Checkout/js/model/full-screen-loader',\n    'Magento_Checkout/js/action/select-billing-address',\n    'Magento_Checkout/js/model/shipping-save-processor/payload-extender'\n], function (\n    ko,\n    quote,\n    resourceUrlManager,\n    storage,\n    paymentService,\n    methodConverter,\n    errorProcessor,\n    fullScreenLoader,\n    selectBillingAddressAction,\n    payloadExtender\n) {\n    'use strict';\n\n    return {\n        /**\n         * @return {jQuery.Deferred}\n         */\n        saveShippingInformation: function () {\n            var payload;\n\n            if (!quote.billingAddress() && quote.shippingAddress().canUseForBilling()) {\n                selectBillingAddressAction(quote.shippingAddress());\n            }\n\n            payload = {\n                addressInformation: {\n                    'shipping_address': quote.shippingAddress(),\n                    'billing_address': quote.billingAddress(),\n                    'shipping_method_code': quote.shippingMethod()['method_code'],\n                    'shipping_carrier_code': quote.shippingMethod()['carrier_code']\n                }\n            };\n\n            payloadExtender(payload);\n\n            fullScreenLoader.startLoader();\n\n            return storage.post(\n                resourceUrlManager.getUrlForSetShippingInformation(quote),\n                JSON.stringify(payload)\n            ).done(\n                function (response) {\n                    quote.setTotals(response.totals);\n                    paymentService.setPaymentMethods(methodConverter(response['payment_methods']));\n                    fullScreenLoader.stopLoader();\n                }\n            ).fail(\n                function (response) {\n                    errorProcessor.process(response);\n                    fullScreenLoader.stopLoader();\n                }\n            );\n        }\n    };\n});\n","Magento_Checkout/js/model/shipping-save-processor/payload-extender.js":"define([], function () {\n    'use strict';\n\n    return function (payload) {\n        payload.addressInformation['extension_attributes'] = {};\n\n        return payload;\n    };\n});\n","Magento_Checkout/js/view/authentication-messages.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Ui/js/view/messages',\n    'Magento_Checkout/js/model/authentication-messages'\n], function (Component, messageContainer) {\n    'use strict';\n\n    return Component.extend({\n        /** @inheritdoc */\n        initialize: function (config) {\n            return this._super(config, messageContainer);\n        }\n    });\n});\n","Magento_Checkout/js/view/authentication.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Ui/js/form/form',\n    'Magento_Customer/js/action/login',\n    'Magento_Customer/js/model/customer',\n    'mage/validation',\n    'Magento_Checkout/js/model/authentication-messages',\n    'Magento_Checkout/js/model/full-screen-loader'\n], function ($, Component, loginAction, customer, validation, messageContainer, fullScreenLoader) {\n    'use strict';\n\n    var checkoutConfig = window.checkoutConfig;\n\n    return Component.extend({\n        isGuestCheckoutAllowed: checkoutConfig.isGuestCheckoutAllowed,\n        isCustomerLoginRequired: checkoutConfig.isCustomerLoginRequired,\n        registerUrl: checkoutConfig.registerUrl,\n        forgotPasswordUrl: checkoutConfig.forgotPasswordUrl,\n        autocomplete: checkoutConfig.autocomplete,\n        defaults: {\n            template: 'Magento_Checkout/authentication'\n        },\n\n        /**\n         * Is login form enabled for current customer.\n         *\n         * @return {Boolean}\n         */\n        isActive: function () {\n            return !customer.isLoggedIn();\n        },\n\n        /**\n         * Provide login action.\n         *\n         * @param {HTMLElement} loginForm\n         */\n        login: function (loginForm) {\n            var loginData = {},\n                formDataArray = $(loginForm).serializeArray();\n\n            formDataArray.forEach(function (entry) {\n                loginData[entry.name] = entry.value;\n            });\n\n            if ($(loginForm).validation() &&\n                $(loginForm).validation('isValid')\n            ) {\n                fullScreenLoader.startLoader();\n                loginAction(loginData, checkoutConfig.checkoutUrl, undefined, messageContainer).always(function () {\n                    fullScreenLoader.stopLoader();\n                });\n            }\n        }\n    });\n});\n","Magento_Checkout/js/view/billing-address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko',\n    'underscore',\n    'Magento_Ui/js/form/form',\n    'Magento_Customer/js/model/customer',\n    'Magento_Customer/js/model/address-list',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/action/create-billing-address',\n    'Magento_Checkout/js/action/select-billing-address',\n    'Magento_Checkout/js/checkout-data',\n    'Magento_Checkout/js/model/checkout-data-resolver',\n    'Magento_Customer/js/customer-data',\n    'Magento_Checkout/js/action/set-billing-address',\n    'Magento_Ui/js/model/messageList',\n    'mage/translate',\n    'Magento_Checkout/js/model/billing-address-postcode-validator',\n    'Magento_Checkout/js/model/address-converter'\n],\nfunction (\n    ko,\n    _,\n    Component,\n    customer,\n    addressList,\n    quote,\n    createBillingAddress,\n    selectBillingAddress,\n    checkoutData,\n    checkoutDataResolver,\n    customerData,\n    setBillingAddressAction,\n    globalMessageList,\n    $t,\n    billingAddressPostcodeValidator,\n    addressConverter\n) {\n    'use strict';\n\n    var lastSelectedBillingAddress = null,\n        addressUpadated = false,\n        addressEdited = false,\n        countryData = customerData.get('directory-data'),\n        addressOptions = addressList().filter(function (address) {\n            return address.getType() === 'customer-address';\n        });\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/billing-address',\n            actionsTemplate: 'Magento_Checkout/billing-address/actions',\n            formTemplate: 'Magento_Checkout/billing-address/form',\n            detailsTemplate: 'Magento_Checkout/billing-address/details',\n            links: {\n                isAddressFormVisible: '${$.billingAddressListProvider}:isNewAddressSelected'\n            }\n        },\n        currentBillingAddress: quote.billingAddress,\n        customerHasAddresses: addressOptions.length > 0,\n\n        /**\n         * Init component\n         */\n        initialize: function () {\n            this._super();\n            quote.paymentMethod.subscribe(function () {\n                checkoutDataResolver.resolveBillingAddress();\n            }, this);\n            billingAddressPostcodeValidator.initFields(this.get('name') + '.form-fields');\n        },\n\n        /**\n         * @return {exports.initObservable}\n         */\n        initObservable: function () {\n            this._super()\n                .observe({\n                    selectedAddress: null,\n                    isAddressDetailsVisible: quote.billingAddress() != null,\n                    isAddressFormVisible: !customer.isLoggedIn() || !addressOptions.length,\n                    isAddressSameAsShipping: false,\n                    saveInAddressBook: 1\n                });\n\n            quote.billingAddress.subscribe(function (newAddress) {\n                if (quote.isVirtual()) {\n                    this.isAddressSameAsShipping(false);\n                } else {\n                    this.isAddressSameAsShipping(\n                        newAddress != null &&\n                        newAddress.getCacheKey() == quote.shippingAddress().getCacheKey() //eslint-disable-line eqeqeq\n                    );\n                }\n\n                if (newAddress != null && newAddress.saveInAddressBook !== undefined) {\n                    this.saveInAddressBook(newAddress.saveInAddressBook);\n                } else {\n                    this.saveInAddressBook(1);\n                }\n                this.isAddressDetailsVisible(true);\n            }, this);\n\n            return this;\n        },\n\n        canUseShippingAddress: ko.computed(function () {\n            return !quote.isVirtual() && quote.shippingAddress() && quote.shippingAddress().canUseForBilling();\n        }),\n\n        /**\n         * @param {Object} address\n         * @return {*}\n         */\n        addressOptionsText: function (address) {\n            return address.getAddressInline();\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        useShippingAddress: function () {\n            if (this.isAddressSameAsShipping()) {\n                selectBillingAddress(quote.shippingAddress());\n\n                this.updateAddresses();\n                this.isAddressDetailsVisible(true);\n            } else {\n                lastSelectedBillingAddress = quote.billingAddress();\n                quote.billingAddress(null);\n                this.isAddressDetailsVisible(false);\n            }\n            checkoutData.setSelectedBillingAddress(null);\n\n            return true;\n        },\n\n        /**\n         * Update address action\n         */\n        updateAddress: function () {\n            var addressData, newBillingAddress;\n\n            addressUpadated = true;\n\n            if (this.selectedAddress() && !this.isAddressFormVisible()) {\n                selectBillingAddress(this.selectedAddress());\n                checkoutData.setSelectedBillingAddress(this.selectedAddress().getKey());\n            } else {\n                this.source.set('params.invalid', false);\n                this.source.trigger(this.dataScopePrefix + '.data.validate');\n\n                if (this.source.get(this.dataScopePrefix + '.custom_attributes')) {\n                    this.source.trigger(this.dataScopePrefix + '.custom_attributes.data.validate');\n                }\n\n                if (!this.source.get('params.invalid')) {\n                    addressData = this.source.get(this.dataScopePrefix);\n\n                    if (customer.isLoggedIn() && !this.customerHasAddresses) { //eslint-disable-line max-depth\n                        this.saveInAddressBook(1);\n                    }\n                    addressData['save_in_address_book'] = this.saveInAddressBook() ? 1 : 0;\n                    newBillingAddress = createBillingAddress(addressData);\n                    // New address must be selected as a billing address\n                    selectBillingAddress(newBillingAddress);\n                    checkoutData.setSelectedBillingAddress(newBillingAddress.getKey());\n                    checkoutData.setNewCustomerBillingAddress(addressData);\n                }\n            }\n            setBillingAddressAction(globalMessageList);\n            this.updateAddresses();\n        },\n\n        /**\n         * Edit address action\n         */\n        editAddress: function () {\n            addressUpadated = false;\n            addressEdited = true;\n            lastSelectedBillingAddress = quote.billingAddress();\n            quote.billingAddress(null);\n            this.isAddressDetailsVisible(false);\n        },\n\n        /**\n         * Cancel address edit action\n         */\n        cancelAddressEdit: function () {\n            addressUpadated = true;\n            this.restoreBillingAddress();\n\n            if (quote.billingAddress()) {\n                // restore 'Same As Shipping' checkbox state\n                this.isAddressSameAsShipping(\n                    quote.billingAddress() != null &&\n                        quote.billingAddress().getCacheKey() == quote.shippingAddress().getCacheKey() && //eslint-disable-line\n                        !quote.isVirtual()\n                );\n                this.isAddressDetailsVisible(true);\n            }\n        },\n\n        /**\n         * Manage cancel button visibility\n         */\n        canUseCancelBillingAddress: ko.computed(function () {\n            return quote.billingAddress() || lastSelectedBillingAddress;\n        }),\n\n        /**\n         * Check if Billing Address Changes should be canceled\n         */\n        needCancelBillingAddressChanges: function () {\n            if (addressEdited && !addressUpadated) {\n                this.cancelAddressEdit();\n            }\n        },\n\n        /**\n         * Restore billing address\n         */\n        restoreBillingAddress: function () {\n            var lastBillingAddress;\n\n            if (lastSelectedBillingAddress != null) {\n                selectBillingAddress(lastSelectedBillingAddress);\n                lastBillingAddress = addressConverter.quoteAddressToFormAddressData(lastSelectedBillingAddress);\n\n                checkoutData.setNewCustomerBillingAddress(lastBillingAddress);\n            }\n        },\n\n        /**\n         * @param {Number} countryId\n         * @return {*}\n         */\n        getCountryName: function (countryId) {\n            return countryData()[countryId] != undefined ? countryData()[countryId].name : ''; //eslint-disable-line\n        },\n\n        /**\n         * Trigger action to update shipping and billing addresses\n         */\n        updateAddresses: function () {\n            if (window.checkoutConfig.reloadOnBillingAddress ||\n                !window.checkoutConfig.displayBillingOnPaymentMethod\n            ) {\n                setBillingAddressAction(globalMessageList);\n            }\n        },\n\n        /**\n         * Get code\n         * @param {Object} parent\n         * @returns {String}\n         */\n        getCode: function (parent) {\n            return _.isFunction(parent.getCode) ? parent.getCode() : 'shared';\n        },\n\n        /**\n         * Get customer attribute label\n         *\n         * @param {*} attribute\n         * @returns {*}\n         */\n        getCustomAttributeLabel: function (attribute) {\n            var label;\n\n            if (typeof attribute === 'string') {\n                return attribute;\n            }\n\n            if (attribute.label) {\n                return attribute.label;\n            }\n\n            if (_.isArray(attribute.value)) {\n                label = _.map(attribute.value, function (value) {\n                    return this.getCustomAttributeOptionLabel(attribute['attribute_code'], value) || value;\n                }, this).join(', ');\n            } else if (typeof attribute.value === 'object') {\n                label = _.map(Object.values(attribute.value)).join(', ');\n            } else {\n                label = this.getCustomAttributeOptionLabel(attribute['attribute_code'], attribute.value);\n            }\n\n            return label || attribute.value;\n        },\n\n        /**\n         * Get option label for given attribute code and option ID\n         *\n         * @param {String} attributeCode\n         * @param {String} value\n         * @returns {String|null}\n         */\n        getCustomAttributeOptionLabel: function (attributeCode, value) {\n            var option,\n                label,\n                options = this.source.get('customAttributes') || {};\n\n            if (options[attributeCode]) {\n                option = _.findWhere(options[attributeCode], {\n                    value: value\n                });\n\n                if (option) {\n                    label = option.label;\n                }\n            } else if (value.file !== null) {\n                label = value.file;\n            }\n\n            return label;\n        }\n    });\n});\n","Magento_Checkout/js/view/cart-item-renderer.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        /**\n         * Prepare the product name value to be rendered as HTML\n         *\n         * @param {String} productName\n         * @return {String}\n         */\n        getProductNameUnsanitizedHtml: function (productName) {\n            // product name has already escaped on backend\n            return productName;\n        },\n\n        /**\n         * Prepare the given option value to be rendered as HTML\n         *\n         * @param {String} optionValue\n         * @return {String}\n         */\n        getOptionValueUnsanitizedHtml: function (optionValue) {\n            // option value has already escaped on backend\n            return optionValue;\n        }\n    });\n});\n","Magento_Checkout/js/view/estimation.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Catalog/js/price-utils',\n    'Magento_Checkout/js/model/totals',\n    'Magento_Checkout/js/model/sidebar'\n], function (Component, quote, priceUtils, totals, sidebarModel) {\n    'use strict';\n\n    return Component.extend({\n        isLoading: totals.isLoading,\n\n        /**\n         * @return {Number}\n         */\n        getQuantity: function () {\n            if (totals.totals()) {\n                return parseFloat(totals.totals()['items_qty']);\n            }\n\n            return 0;\n        },\n\n        /**\n         * @return {Number}\n         */\n        getPureValue: function () {\n            if (totals.totals()) {\n                return parseFloat(totals.getSegment('grand_total').value);\n            }\n\n            return 0;\n        },\n\n        /**\n         * Show sidebar.\n         */\n        showSidebar: function () {\n            sidebarModel.show();\n        },\n\n        /**\n         * @param {*} price\n         * @return {*|String}\n         */\n        getFormattedPrice: function (price) {\n            return priceUtils.formatPriceLocale(price, quote.getPriceFormat());\n        },\n\n        /**\n         * @return {*|String}\n         */\n        getValue: function () {\n            return this.getFormattedPrice(this.getPureValue());\n        }\n    });\n});\n","Magento_Checkout/js/view/minicart.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Customer/js/customer-data',\n    'jquery',\n    'ko',\n    'underscore',\n    'sidebar',\n    'mage/translate',\n    'mage/dropdown'\n], function (Component, customerData, $, ko, _) {\n    'use strict';\n\n    var sidebarInitialized = false,\n        addToCartCalls = 0,\n        miniCart;\n\n    miniCart = $('[data-block=\\'minicart\\']');\n\n    /**\n     * @return {Boolean}\n     */\n    function initSidebar() {\n        if (miniCart.data('mageSidebar')) {\n            miniCart.sidebar('update');\n        }\n\n        if (!$('[data-role=product-item]').length) {\n            return false;\n        }\n        miniCart.trigger('contentUpdated');\n\n        if (sidebarInitialized) {\n            return false;\n        }\n        sidebarInitialized = true;\n        miniCart.sidebar({\n            'targetElement': 'div.block.block-minicart',\n            'url': {\n                'checkout': window.checkout.checkoutUrl,\n                'update': window.checkout.updateItemQtyUrl,\n                'remove': window.checkout.removeItemUrl,\n                'loginUrl': window.checkout.customerLoginUrl,\n                'isRedirectRequired': window.checkout.isRedirectRequired\n            },\n            'button': {\n                'checkout': '#top-cart-btn-checkout',\n                'remove': '#mini-cart a.action.delete',\n                'close': '#btn-minicart-close'\n            },\n            'showcart': {\n                'parent': 'span.counter',\n                'qty': 'span.counter-number',\n                'label': 'span.counter-label'\n            },\n            'minicart': {\n                'list': '#mini-cart',\n                'content': '#minicart-content-wrapper',\n                'qty': 'div.items-total',\n                'subtotal': 'div.subtotal span.price',\n                'maxItemsVisible': window.checkout.minicartMaxItemsVisible\n            },\n            'item': {\n                'qty': ':input.cart-item-qty',\n                'button': ':button.update-cart-item'\n            },\n            'confirmMessage': $.mage.__('Are you sure you would like to remove this item from the shopping cart?')\n        });\n    }\n\n    miniCart.on('dropdowndialogopen', function () {\n        initSidebar();\n    });\n\n    return Component.extend({\n        shoppingCartUrl: window.checkout.shoppingCartUrl,\n        maxItemsToDisplay: window.checkout.maxItemsToDisplay,\n        cart: {},\n\n        // jscs:disable requireCamelCaseOrUpperCaseIdentifiers\n        /**\n         * @override\n         */\n        initialize: function () {\n            var self = this,\n                cartData = customerData.get('cart');\n\n            this.update(cartData());\n            cartData.subscribe(function (updatedCart) {\n                addToCartCalls--;\n                this.isLoading(addToCartCalls > 0);\n                sidebarInitialized = false;\n                this.update(updatedCart);\n                initSidebar();\n            }, this);\n            $('[data-block=\"minicart\"]').on('contentLoading', function () {\n                addToCartCalls++;\n                self.isLoading(true);\n            });\n\n            if (\n                cartData().website_id !== window.checkout.websiteId && cartData().website_id !== undefined ||\n                cartData().storeId !== window.checkout.storeId && cartData().storeId !== undefined\n            ) {\n                customerData.reload(['cart'], false);\n            }\n\n            return this._super();\n        },\n        //jscs:enable requireCamelCaseOrUpperCaseIdentifiers\n\n        isLoading: ko.observable(false),\n        initSidebar: initSidebar,\n\n        /**\n         * Close mini shopping cart.\n         */\n        closeMinicart: function () {\n            $('[data-block=\"minicart\"]').find('[data-role=\"dropdownDialog\"]').dropdownDialog('close');\n        },\n\n        /**\n         * @param {String} productType\n         * @return {*|String}\n         */\n        getItemRenderer: function (productType) {\n            return this.itemRenderer[productType] || 'defaultRenderer';\n        },\n\n        /**\n         * Update mini shopping cart content.\n         *\n         * @param {Object} updatedCart\n         * @returns void\n         */\n        update: function (updatedCart) {\n            _.each(updatedCart, function (value, key) {\n                if (!this.cart.hasOwnProperty(key)) {\n                    this.cart[key] = ko.observable();\n                }\n                this.cart[key](value);\n            }, this);\n        },\n\n        /**\n         * Get cart param by name.\n         *\n         * @param {String} name\n         * @returns {*}\n         */\n        getCartParamUnsanitizedHtml: function (name) {\n            if (!_.isUndefined(name)) {\n                if (!this.cart.hasOwnProperty(name)) {\n                    this.cart[name] = ko.observable();\n                }\n            }\n\n            return this.cart[name]();\n        },\n\n        /**\n         * @deprecated please use getCartParamUnsanitizedHtml.\n         * @param {String} name\n         * @returns {*}\n         */\n        getCartParam: function (name) {\n            return this.getCartParamUnsanitizedHtml(name);\n        },\n\n        /**\n         * Returns array of cart items, limited by 'maxItemsToDisplay' setting\n         * @returns []\n         */\n        getCartItems: function () {\n            var items = this.getCartParamUnsanitizedHtml('items') || [];\n\n            items = items.slice(parseInt(-this.maxItemsToDisplay, 10));\n\n            return items;\n        },\n\n        /**\n         * Returns count of cart line items\n         * @returns {Number}\n         */\n        getCartLineItemsCount: function () {\n            var items = this.getCartParamUnsanitizedHtml('items') || [];\n\n            return parseInt(items.length, 10);\n        }\n    });\n});\n","Magento_Checkout/js/view/payment.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'uiComponent',\n    'ko',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/step-navigator',\n    'Magento_Checkout/js/model/payment-service',\n    'Magento_Checkout/js/model/payment/method-converter',\n    'Magento_Checkout/js/action/get-payment-information',\n    'Magento_Checkout/js/model/checkout-data-resolver',\n    'mage/translate'\n], function (\n    $,\n    _,\n    Component,\n    ko,\n    quote,\n    stepNavigator,\n    paymentService,\n    methodConverter,\n    getPaymentInformation,\n    checkoutDataResolver,\n    $t\n) {\n    'use strict';\n\n    /** Set payment methods to collection */\n    paymentService.setPaymentMethods(methodConverter(window.checkoutConfig.paymentMethods));\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/payment',\n            activeMethod: ''\n        },\n        isVisible: ko.observable(quote.isVirtual()),\n        quoteIsVirtual: quote.isVirtual(),\n        isPaymentMethodsAvailable: ko.computed(function () {\n            return paymentService.getAvailablePaymentMethods().length > 0;\n        }),\n\n        /** @inheritdoc */\n        initialize: function () {\n            this._super();\n            checkoutDataResolver.resolvePaymentMethod();\n            stepNavigator.registerStep(\n                'payment',\n                null,\n                $t('Review & Payments'),\n                this.isVisible,\n                _.bind(this.navigate, this),\n                this.sortOrder\n            );\n\n            return this;\n        },\n\n        /**\n         * Navigate method.\n         */\n        navigate: function () {\n            var self = this;\n\n            if (!self.hasShippingMethod()) {\n                this.isVisible(false);\n                stepNavigator.setHash('shipping');\n            } else {\n                getPaymentInformation().done(function () {\n                    self.isVisible(true);\n                });\n            }\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        hasShippingMethod: function () {\n            return window.checkoutConfig.selectedShippingMethod !== null;\n        },\n\n        /**\n         * @return {*}\n         */\n        getFormKey: function () {\n            return window.checkoutConfig.formKey;\n        }\n    });\n});\n","Magento_Checkout/js/view/progress-bar.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'ko',\n    'uiComponent',\n    'Magento_Checkout/js/model/step-navigator',\n    'Magento_Checkout/js/view/billing-address'\n], function ($, _, ko, Component, stepNavigator, billingAddress) {\n    'use strict';\n\n    var steps = stepNavigator.steps;\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/progress-bar',\n            visible: true\n        },\n        steps: steps,\n\n        /** @inheritdoc */\n        initialize: function () {\n            var stepsValue;\n\n            this._super();\n            window.addEventListener('hashchange', _.bind(stepNavigator.handleHash, stepNavigator));\n\n            if (!window.location.hash) {\n                stepsValue = stepNavigator.steps();\n\n                if (stepsValue.length) {\n                    stepNavigator.setHash(stepsValue.sort(stepNavigator.sortItems)[0].code);\n                }\n            }\n\n            stepNavigator.handleHash();\n        },\n\n        /**\n         * @param {*} itemOne\n         * @param {*} itemTwo\n         * @return {*|Number}\n         */\n        sortItems: function (itemOne, itemTwo) {\n            return stepNavigator.sortItems(itemOne, itemTwo);\n        },\n\n        /**\n         * @param {Object} step\n         */\n        navigateTo: function (step) {\n            if (step.code === 'shipping') {\n                billingAddress().needCancelBillingAddressChanges();\n            }\n            stepNavigator.navigateTo(step.code);\n        },\n\n        /**\n         * @param {Object} item\n         * @return {*|Boolean}\n         */\n        isProcessed: function (item) {\n            return stepNavigator.isProcessed(item.code);\n        }\n    });\n});\n","Magento_Checkout/js/view/registration.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'uiComponent',\n    'Magento_Ui/js/model/messageList'\n], function ($, Component, messageList) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/registration',\n            accountCreated: false,\n            creationStarted: false,\n            isFormVisible: true\n        },\n\n        /**\n         * @inheritdoc\n         */\n        initObservable: function () {\n            this._super()\n                .observe('accountCreated')\n                .observe('isFormVisible')\n                .observe('creationStarted');\n\n            return this;\n        },\n\n        /**\n         * @return {*}\n         */\n        getEmailAddress: function () {\n            return this.email;\n        },\n\n        /**\n         * @return String\n         */\n        getUrl: function () {\n            return this.registrationUrl;\n        },\n\n        /**\n         * Create new user account.\n         *\n         * @deprecated\n         */\n        createAccount: function () {\n            this.creationStarted(true);\n            $.post(\n                this.registrationUrl\n            ).done(\n                function (response) {\n\n                    if (response.errors == false) { //eslint-disable-line eqeqeq\n                        this.accountCreated(true);\n                    } else {\n                        messageList.addErrorMessage(response);\n                    }\n                    this.isFormVisible(false);\n                }.bind(this)\n            ).fail(\n                function (response) {\n                    this.accountCreated(false);\n                    this.isFormVisible(false);\n                    messageList.addErrorMessage(response);\n                }.bind(this)\n            );\n        }\n    });\n});\n","Magento_Checkout/js/view/shipping-information.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'uiComponent',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/step-navigator',\n    'Magento_Checkout/js/model/sidebar'\n], function ($, Component, quote, stepNavigator, sidebarModel) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/shipping-information'\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        isVisible: function () {\n            return !quote.isVirtual() && stepNavigator.isProcessed('shipping');\n        },\n\n        /**\n         * @return {String}\n         */\n        getShippingMethodTitle: function () {\n            var shippingMethod = quote.shippingMethod(),\n                shippingMethodTitle = '';\n\n            if (!shippingMethod) {\n                return '';\n            }\n\n            shippingMethodTitle = shippingMethod['carrier_title'];\n\n            if (typeof shippingMethod['method_title'] !== 'undefined') {\n                shippingMethodTitle += ' - ' + shippingMethod['method_title'];\n            }\n\n            return shippingMethodTitle;\n        },\n\n        /**\n         * Back step.\n         */\n        back: function () {\n            sidebarModel.hide();\n            stepNavigator.navigateTo('shipping');\n        },\n\n        /**\n         * Back to shipping method.\n         */\n        backToShippingMethod: function () {\n            sidebarModel.hide();\n            stepNavigator.navigateTo('shipping', 'opc-shipping_method');\n        }\n    });\n});\n","Magento_Checkout/js/view/shipping.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'Magento_Ui/js/form/form',\n    'ko',\n    'Magento_Customer/js/model/customer',\n    'Magento_Customer/js/model/address-list',\n    'Magento_Checkout/js/model/address-converter',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/action/create-shipping-address',\n    'Magento_Checkout/js/action/select-shipping-address',\n    'Magento_Checkout/js/model/shipping-rates-validator',\n    'Magento_Checkout/js/model/shipping-address/form-popup-state',\n    'Magento_Checkout/js/model/shipping-service',\n    'Magento_Checkout/js/action/select-shipping-method',\n    'Magento_Checkout/js/model/shipping-rate-registry',\n    'Magento_Checkout/js/action/set-shipping-information',\n    'Magento_Checkout/js/model/step-navigator',\n    'Magento_Ui/js/modal/modal',\n    'Magento_Checkout/js/model/checkout-data-resolver',\n    'Magento_Checkout/js/checkout-data',\n    'uiRegistry',\n    'mage/translate',\n    'Magento_Checkout/js/model/shipping-rate-service'\n], function (\n    $,\n    _,\n    Component,\n    ko,\n    customer,\n    addressList,\n    addressConverter,\n    quote,\n    createShippingAddress,\n    selectShippingAddress,\n    shippingRatesValidator,\n    formPopUpState,\n    shippingService,\n    selectShippingMethodAction,\n    rateRegistry,\n    setShippingInformationAction,\n    stepNavigator,\n    modal,\n    checkoutDataResolver,\n    checkoutData,\n    registry,\n    $t\n) {\n    'use strict';\n\n    var popUp = null;\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/shipping',\n            shippingFormTemplate: 'Magento_Checkout/shipping-address/form',\n            shippingMethodListTemplate: 'Magento_Checkout/shipping-address/shipping-method-list',\n            shippingMethodItemTemplate: 'Magento_Checkout/shipping-address/shipping-method-item',\n            imports: {\n                countryOptions: '${ $.parentName }.shippingAddress.shipping-address-fieldset.country_id:indexedOptions'\n            }\n        },\n        visible: ko.observable(!quote.isVirtual()),\n        errorValidationMessage: ko.observable(false),\n        isCustomerLoggedIn: customer.isLoggedIn,\n        isFormPopUpVisible: formPopUpState.isVisible,\n        isFormInline: addressList().length === 0,\n        isNewAddressAdded: ko.observable(false),\n        saveInAddressBook: 1,\n        quoteIsVirtual: quote.isVirtual(),\n\n        /**\n         * @return {exports}\n         */\n        initialize: function () {\n            var self = this,\n                hasNewAddress,\n                fieldsetName = 'checkout.steps.shipping-step.shippingAddress.shipping-address-fieldset';\n\n            this._super();\n\n            if (!quote.isVirtual()) {\n                stepNavigator.registerStep(\n                    'shipping',\n                    '',\n                    $t('Shipping'),\n                    this.visible, _.bind(this.navigate, this),\n                    this.sortOrder\n                );\n            }\n            checkoutDataResolver.resolveShippingAddress();\n\n            hasNewAddress = addressList.some(function (address) {\n                return address.getType() == 'new-customer-address'; //eslint-disable-line eqeqeq\n            });\n\n            this.isNewAddressAdded(hasNewAddress);\n\n            this.isFormPopUpVisible.subscribe(function (value) {\n                if (value) {\n                    self.getPopUp().openModal();\n                }\n            });\n\n            quote.shippingMethod.subscribe(function () {\n                self.errorValidationMessage(false);\n            });\n\n            registry.async('checkoutProvider')(function (checkoutProvider) {\n                var shippingAddressData = checkoutData.getShippingAddressFromData();\n\n                if (shippingAddressData) {\n                    checkoutProvider.set(\n                        'shippingAddress',\n                        $.extend(true, {}, checkoutProvider.get('shippingAddress'), shippingAddressData)\n                    );\n                }\n                checkoutProvider.on('shippingAddress', function (shippingAddrsData, changes) {\n                    var isStreetAddressDeleted, isStreetAddressNotEmpty;\n\n                    /**\n                     * In last modifying operation street address was deleted.\n                     * @return {Boolean}\n                     */\n                    isStreetAddressDeleted = function () {\n                        var change;\n\n                        if (!changes || changes.length === 0) {\n                            return false;\n                        }\n\n                        change = changes.pop();\n\n                        if (_.isUndefined(change.value) || _.isUndefined(change.oldValue)) {\n                            return false;\n                        }\n\n                        if (!change.path.startsWith('shippingAddress.street')) {\n                            return false;\n                        }\n\n                        return change.value.length === 0 && change.oldValue.length > 0;\n                    };\n\n                    isStreetAddressNotEmpty = shippingAddrsData.street && !_.isEmpty(shippingAddrsData.street[0]);\n\n                    if (isStreetAddressNotEmpty || isStreetAddressDeleted()) {\n                        checkoutData.setShippingAddressFromData(shippingAddrsData);\n                    }\n                });\n                shippingRatesValidator.initFields(fieldsetName);\n            });\n\n            return this;\n        },\n\n        /**\n         * Navigator change hash handler.\n         *\n         * @param {Object} step - navigation step\n         */\n        navigate: function (step) {\n            step && step.isVisible(true);\n        },\n\n        /**\n         * @return {*}\n         */\n        getPopUp: function () {\n            var self = this,\n                buttons;\n\n            if (!popUp) {\n                buttons = this.popUpForm.options.buttons;\n                this.popUpForm.options.buttons = [\n                    {\n                        text: buttons.save.text ? buttons.save.text : $t('Save Address'),\n                        class: buttons.save.class ? buttons.save.class : 'action primary action-save-address',\n                        click: self.saveNewAddress.bind(self)\n                    },\n                    {\n                        text: buttons.cancel.text ? buttons.cancel.text : $t('Cancel'),\n                        class: buttons.cancel.class ? buttons.cancel.class : 'action secondary action-hide-popup',\n\n                        /** @inheritdoc */\n                        click: this.onClosePopUp.bind(this)\n                    }\n                ];\n\n                /** @inheritdoc */\n                this.popUpForm.options.closed = function () {\n                    self.isFormPopUpVisible(false);\n                };\n\n                this.popUpForm.options.modalCloseBtnHandler = this.onClosePopUp.bind(this);\n                this.popUpForm.options.keyEventHandlers = {\n                    escapeKey: this.onClosePopUp.bind(this)\n                };\n\n                /** @inheritdoc */\n                this.popUpForm.options.opened = function () {\n                    // Store temporary address for revert action in case when user click cancel action\n                    self.temporaryAddress = $.extend(true, {}, checkoutData.getShippingAddressFromData());\n                };\n                popUp = modal(this.popUpForm.options, $(this.popUpForm.element));\n            }\n\n            return popUp;\n        },\n\n        /**\n         * Revert address and close modal.\n         */\n        onClosePopUp: function () {\n            checkoutData.setShippingAddressFromData($.extend(true, {}, this.temporaryAddress));\n            this.getPopUp().closeModal();\n        },\n\n        /**\n         * Show address form popup\n         */\n        showFormPopUp: function () {\n            this.isFormPopUpVisible(true);\n        },\n\n        /**\n         * Save new shipping address\n         */\n        saveNewAddress: function () {\n            var addressData,\n                newShippingAddress;\n\n            this.source.set('params.invalid', false);\n            this.triggerShippingDataValidateEvent();\n\n            if (!this.source.get('params.invalid')) {\n                addressData = this.source.get('shippingAddress');\n                // if user clicked the checkbox, its value is true or false. Need to convert.\n                addressData['save_in_address_book'] = this.saveInAddressBook ? 1 : 0;\n\n                // New address must be selected as a shipping address\n                newShippingAddress = createShippingAddress(addressData);\n                selectShippingAddress(newShippingAddress);\n                checkoutData.setSelectedShippingAddress(newShippingAddress.getKey());\n                checkoutData.setNewCustomerShippingAddress($.extend(true, {}, addressData));\n                this.getPopUp().closeModal();\n                this.isNewAddressAdded(true);\n            }\n        },\n\n        /**\n         * Shipping Method View\n         */\n        rates: shippingService.getShippingRates(),\n        isLoading: shippingService.isLoading,\n        isSelected: ko.computed(function () {\n            return quote.shippingMethod() ?\n                quote.shippingMethod()['carrier_code'] + '_' + quote.shippingMethod()['method_code'] :\n                null;\n        }),\n\n        /**\n         * @param {Object} shippingMethod\n         * @return {Boolean}\n         */\n        selectShippingMethod: function (shippingMethod) {\n            selectShippingMethodAction(shippingMethod);\n            checkoutData.setSelectedShippingRate(shippingMethod['carrier_code'] + '_' + shippingMethod['method_code']);\n\n            return true;\n        },\n\n        /**\n         * Set shipping information handler\n         */\n        setShippingInformation: function () {\n            if (this.validateShippingInformation()) {\n                quote.billingAddress(null);\n                checkoutDataResolver.resolveBillingAddress();\n                registry.async('checkoutProvider')(function (checkoutProvider) {\n                    var shippingAddressData = checkoutData.getShippingAddressFromData();\n\n                    if (shippingAddressData) {\n                        checkoutProvider.set(\n                            'shippingAddress',\n                            $.extend(true, {}, checkoutProvider.get('shippingAddress'), shippingAddressData)\n                        );\n                    }\n                });\n                setShippingInformationAction().done(\n                    function () {\n                        stepNavigator.next();\n                    }\n                );\n            }\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        validateShippingInformation: function () {\n            var shippingAddress,\n                addressData,\n                loginFormSelector = 'form[data-role=email-with-possible-login]',\n                emailValidationResult = customer.isLoggedIn(),\n                field,\n                option = _.isObject(this.countryOptions) && this.countryOptions[quote.shippingAddress().countryId],\n                messageContainer = registry.get('checkout.errors').messageContainer;\n\n            if (!quote.shippingMethod()) {\n                this.errorValidationMessage(\n                    $t('The shipping method is missing. Select the shipping method and try again.')\n                );\n\n                return false;\n            }\n\n            if (!customer.isLoggedIn()) {\n                $(loginFormSelector).validation();\n                emailValidationResult = Boolean($(loginFormSelector + ' input[name=username]').valid());\n            }\n\n            if (this.isFormInline) {\n                this.source.set('params.invalid', false);\n                this.triggerShippingDataValidateEvent();\n\n                if (!quote.shippingMethod()['method_code']) {\n                    this.errorValidationMessage(\n                        $t('The shipping method is missing. Select the shipping method and try again.')\n                    );\n                }\n\n                if (emailValidationResult &&\n                    this.source.get('params.invalid') ||\n                    !quote.shippingMethod()['method_code'] ||\n                    !quote.shippingMethod()['carrier_code']\n                ) {\n                    this.focusInvalid();\n\n                    return false;\n                }\n\n                shippingAddress = quote.shippingAddress();\n                addressData = addressConverter.formAddressDataToQuoteAddress(\n                    this.source.get('shippingAddress')\n                );\n\n                //Copy form data to quote shipping address object\n                for (field in addressData) {\n                    if (addressData.hasOwnProperty(field) &&  //eslint-disable-line max-depth\n                        shippingAddress.hasOwnProperty(field) &&\n                        typeof addressData[field] != 'function' &&\n                        _.isEqual(shippingAddress[field], addressData[field])\n                    ) {\n                        shippingAddress[field] = addressData[field];\n                    } else if (typeof addressData[field] != 'function' &&\n                        !_.isEqual(shippingAddress[field], addressData[field])) {\n                        shippingAddress = addressData;\n                        break;\n                    }\n                }\n\n                if (customer.isLoggedIn()) {\n                    shippingAddress['save_in_address_book'] = 1;\n                }\n                selectShippingAddress(shippingAddress);\n            } else if (customer.isLoggedIn() &&\n                option &&\n                option['is_region_required'] &&\n                !quote.shippingAddress().region\n            ) {\n                messageContainer.addErrorMessage({\n                    message: $t('Please specify a regionId in shipping address.')\n                });\n\n                return false;\n            }\n\n            if (!emailValidationResult) {\n                $(loginFormSelector + ' input[name=username]').trigger('focus');\n\n                return false;\n            }\n\n            return true;\n        },\n\n        /**\n         * Trigger Shipping data Validate Event.\n         */\n        triggerShippingDataValidateEvent: function () {\n            this.source.trigger('shippingAddress.data.validate');\n\n            if (this.source.get('shippingAddress.custom_attributes')) {\n                this.source.trigger('shippingAddress.custom_attributes.data.validate');\n            }\n        }\n    });\n});\n","Magento_Checkout/js/view/sidebar.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'ko',\n    'jquery',\n    'Magento_Checkout/js/model/sidebar'\n], function (Component, ko, $, sidebarModel) {\n    'use strict';\n\n    return Component.extend({\n        /**\n         * @param {HTMLElement} element\n         */\n        setModalElement: function (element) {\n            sidebarModel.setPopup($(element));\n        }\n    });\n});\n","Magento_Checkout/js/view/summary.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/totals'\n], function (Component, totals) {\n    'use strict';\n\n    return Component.extend({\n        isLoading: totals.isLoading\n    });\n});\n","Magento_Checkout/js/view/billing-address/list.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Customer/js/model/address-list',\n    'mage/translate',\n    'Magento_Customer/js/model/customer'\n], function (Component, addressList, $t, customer) {\n    'use strict';\n\n    var newAddressOption = {\n            /**\n             * Get new address label\n             * @returns {String}\n             */\n            getAddressInline: function () {\n                return $t('New Address');\n            },\n            customerAddressId: null\n        },\n        addressOptions = addressList().filter(function (address) {\n            return address.getType() === 'customer-address';\n        }),\n        addressDefaultIndex = addressOptions.findIndex(function (address) {\n            return address.isDefaultBilling();\n        });\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/billing-address',\n            selectedAddress: null,\n            isNewAddressSelected: false,\n            addressOptions: addressOptions,\n            exports: {\n                selectedAddress: '${ $.parentName }:selectedAddress'\n            }\n        },\n\n        /**\n         * @returns {Object} Chainable.\n         */\n        initConfig: function () {\n            this._super();\n            this.addressOptions.push(newAddressOption);\n\n            return this;\n        },\n\n        /**\n         * @return {exports.initObservable}\n         */\n        initObservable: function () {\n            this._super()\n                .observe('selectedAddress isNewAddressSelected')\n                .observe({\n                    isNewAddressSelected: !customer.isLoggedIn() || !addressOptions.length,\n                    selectedAddress: this.addressOptions[addressDefaultIndex]\n                });\n\n            return this;\n        },\n\n        /**\n         * @param {Object} address\n         * @return {*}\n         */\n        addressOptionsText: function (address) {\n            return address.getAddressInline();\n        },\n\n        /**\n         * @param {Object} address\n         */\n        onAddressChange: function (address) {\n            this.isNewAddressSelected(address === newAddressOption);\n        }\n    });\n});\n","Magento_Checkout/js/view/cart/shipping-estimation.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine(\n    [\n        'jquery',\n        'Magento_Ui/js/form/form',\n        'Magento_Checkout/js/action/select-shipping-address',\n        'Magento_Checkout/js/model/address-converter',\n        'Magento_Checkout/js/model/cart/estimate-service',\n        'Magento_Checkout/js/checkout-data',\n        'Magento_Checkout/js/model/shipping-rates-validator',\n        'uiRegistry',\n        'Magento_Checkout/js/model/quote',\n        'Magento_Checkout/js/model/checkout-data-resolver',\n        'Magento_Checkout/js/model/shipping-service',\n        'mage/validation'\n    ],\n    function (\n        $,\n        Component,\n        selectShippingAddress,\n        addressConverter,\n        estimateService,\n        checkoutData,\n        shippingRatesValidator,\n        registry,\n        quote,\n        checkoutDataResolver,\n        shippingService\n    ) {\n        'use strict';\n\n        return Component.extend({\n            defaults: {\n                template: 'Magento_Checkout/cart/shipping-estimation'\n            },\n            isVirtual: quote.isVirtual(),\n\n            /**\n             * @override\n             */\n            initialize: function () {\n                this._super();\n\n                // Prevent shipping methods showing none available whilst we resolve\n                shippingService.isLoading(true);\n\n                registry.async('checkoutProvider')(function (checkoutProvider) {\n                    var address, estimatedAddress;\n\n                    shippingService.isLoading(false);\n\n                    checkoutDataResolver.resolveEstimationAddress();\n                    address = quote.isVirtual() ? quote.billingAddress() : quote.shippingAddress();\n\n                    if (!address && quote.isVirtual()) {\n                        address = addressConverter.formAddressDataToQuoteAddress(\n                            checkoutData.getSelectedBillingAddress()\n                        );\n                    }\n\n                    if (address) {\n                        estimatedAddress = address.isEditable() ?\n                            addressConverter.quoteAddressToFormAddressData(address) :\n                            {\n                                // only the following fields must be used by estimation form data provider\n                                'country_id': address.countryId,\n                                region: address.region,\n                                'region_id': address.regionId,\n                                postcode: address.postcode\n                            };\n                        checkoutProvider.set(\n                            'shippingAddress',\n                            $.extend({}, checkoutProvider.get('shippingAddress'), estimatedAddress)\n                        );\n                    }\n\n                    if (!quote.isVirtual()) {\n                        checkoutProvider.on('shippingAddress', function (shippingAddressData) {\n                            //jscs:disable requireCamelCaseOrUpperCaseIdentifiers\n                            if (quote.shippingAddress().countryId !== shippingAddressData.country_id ||\n                                (shippingAddressData.postcode || shippingAddressData.region_id)\n                            ) {\n                                checkoutData.setShippingAddressFromData(shippingAddressData);\n                            }\n                            //jscs:enable requireCamelCaseOrUpperCaseIdentifiers\n                        });\n                    } else {\n                        checkoutProvider.on('shippingAddress', function (shippingAddressData) {\n                            checkoutData.setBillingAddressFromData(shippingAddressData);\n                        });\n                    }\n                });\n\n                return this;\n            },\n\n            /**\n             * @override\n             */\n            initElement: function (element) {\n                this._super();\n\n                if (element.index === 'address-fieldsets') {\n                    shippingRatesValidator.bindChangeHandlers(element.elems(), true, 500);\n                    element.elems.subscribe(function (elems) {\n                        shippingRatesValidator.doElementBinding(elems[elems.length - 1], true, 500);\n                    });\n                }\n\n                return this;\n            },\n\n            /**\n             * Returns shipping rates for address\n             * @returns void\n             */\n            getEstimationInfo: function () {\n                var addressData = null;\n\n                this.source.set('params.invalid', false);\n                this.source.trigger('shippingAddress.data.validate');\n\n                if (!this.source.get('params.invalid')) {\n                    addressData = this.source.get('shippingAddress');\n                    selectShippingAddress(addressConverter.formAddressDataToQuoteAddress(addressData));\n                }\n            }\n        });\n    }\n);\n","Magento_Checkout/js/view/cart/shipping-rates.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko',\n    'underscore',\n    'uiComponent',\n    'Magento_Checkout/js/model/shipping-service',\n    'Magento_Catalog/js/price-utils',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/action/select-shipping-method',\n    'Magento_Checkout/js/checkout-data'\n], function (ko, _, Component, shippingService, priceUtils, quote, selectShippingMethodAction, checkoutData) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/cart/shipping-rates'\n        },\n        isVisible: ko.observable(!quote.isVirtual()),\n        isLoading: shippingService.isLoading,\n        shippingRates: shippingService.getShippingRates(),\n        shippingRateGroups: ko.observableArray([]),\n        selectedShippingMethod: ko.computed(function () {\n            return quote.shippingMethod() ?\n                quote.shippingMethod()['carrier_code'] + '_' + quote.shippingMethod()['method_code'] :\n                null;\n        }),\n\n        /**\n         * @override\n         */\n        initObservable: function () {\n            var self = this;\n\n            this._super();\n\n            this.shippingRates.subscribe(function (rates) {\n                self.shippingRateGroups([]);\n                _.each(rates, function (rate) {\n                    var carrierTitle = rate['carrier_title'];\n\n                    if (self.shippingRateGroups.indexOf(carrierTitle) === -1) {\n                        self.shippingRateGroups.push(carrierTitle);\n                    }\n                });\n            });\n\n            return this;\n        },\n\n        /**\n         * Get shipping rates for specific group based on title.\n         * @returns Array\n         */\n        getRatesForGroup: function (shippingRateGroupTitle) {\n            return _.filter(this.shippingRates(), function (rate) {\n                return shippingRateGroupTitle === rate['carrier_title'];\n            });\n        },\n\n        /**\n         * Format shipping price.\n         * @returns {String}\n         */\n        getFormattedPrice: function (price) {\n            return priceUtils.formatPriceLocale(price, quote.getPriceFormat());\n        },\n\n        /**\n         * Set shipping method.\n         * @param {String} methodData\n         * @returns bool\n         */\n        selectShippingMethod: function (methodData) {\n            selectShippingMethodAction(methodData);\n            checkoutData.setSelectedShippingRate(methodData['carrier_code'] + '_' + methodData['method_code']);\n\n            return true;\n        }\n    });\n});\n","Magento_Checkout/js/view/cart/totals.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'uiComponent',\n    'Magento_Checkout/js/model/totals',\n    'Magento_Checkout/js/model/shipping-service'\n], function ($, Component, totalsService, shippingService) {\n    'use strict';\n\n    return Component.extend({\n        isLoading: totalsService.isLoading,\n\n        /**\n         * @override\n         */\n        initialize: function () {\n            this._super();\n            totalsService.totals.subscribe(function () {\n                $(window).trigger('resize');\n            });\n            shippingService.getShippingRates().subscribe(function () {\n                $(window).trigger('resize');\n            });\n        }\n    });\n});\n","Magento_Checkout/js/view/cart/totals/shipping.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/view/summary/shipping',\n    'Magento_Checkout/js/model/quote'\n], function (Component, quote) {\n    'use strict';\n\n    return Component.extend({\n\n        /**\n         * @override\n         */\n        isCalculated: function () {\n            return !!quote.shippingMethod();\n        }\n    });\n});\n","Magento_Checkout/js/view/checkout/placeOrderCaptcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Captcha/js/view/checkout/defaultCaptcha',\n    'Magento_Captcha/js/model/captchaList',\n    'underscore',\n    'Magento_Checkout/js/model/payment/place-order-hooks'\n],\nfunction (defaultCaptcha, captchaList, _, placeOrderHooks) {\n    'use strict';\n\n    return defaultCaptcha.extend({\n        /** @inheritdoc */\n        initialize: function () {\n            var self = this,\n                currentCaptcha;\n\n            this._super();\n            currentCaptcha = captchaList.getCaptchaByFormId(this.formId);\n\n            if (currentCaptcha != null) {\n                currentCaptcha.setIsVisible(true);\n                this.setCurrentCaptcha(currentCaptcha);\n                placeOrderHooks.requestModifiers.push(function (headers) {\n                    if (self.isRequired()) {\n                        headers['X-Captcha'] = self.captchaValue()();\n                    }\n                });\n                placeOrderHooks.afterRequestListeners.push(function () {\n                    self.refresh();\n                });\n            }\n        }\n    });\n});\n","Magento_Checkout/js/view/checkout/minicart/subtotal/totals.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'ko',\n    'uiComponent',\n    'Magento_Customer/js/customer-data'\n], function (ko, Component, customerData) {\n    'use strict';\n\n    return Component.extend({\n        displaySubtotal: ko.observable(true),\n\n        /**\n         * @override\n         */\n        initialize: function () {\n            this._super();\n            this.cart = customerData.get('cart');\n        }\n    });\n});\n","Magento_Checkout/js/view/configure/product-customer-data.js":"require([\n    'jquery',\n    'Magento_Customer/js/customer-data',\n    'underscore',\n    'domReady!'\n], function ($, customerData, _) {\n    'use strict';\n\n    var selectors = {\n        qtySelector: '#product_addtocart_form [name=\"qty\"]',\n        productIdSelector: '#product_addtocart_form [name=\"product\"]',\n        itemIdSelector: '#product_addtocart_form [name=\"item\"]'\n    },\n    cartData = customerData.get('cart'),\n    productId = $(selectors.productIdSelector).val(),\n    itemId = $(selectors.itemIdSelector).val(),\n    productQty,\n    productQtyInput,\n\n    /**\n    * Updates product's qty input value according to actual data\n    */\n    updateQty = function () {\n\n        if (productQty || productQty === 0) {\n            productQtyInput = productQtyInput || $(selectors.qtySelector);\n\n            if (productQtyInput && productQty.toString() !== productQtyInput.val()) {\n                productQtyInput.val(productQty);\n            }\n        }\n    },\n\n    /**\n    * Sets productQty according to cart data from customer-data\n    *\n    * @param {Object} data - cart data from customer-data\n    */\n    setProductQty = function (data) {\n        var product;\n\n        if (!(data && data.items && data.items.length && productId)) {\n            return;\n        }\n        product = _.find(data.items, function (item) {\n            if (item['item_id'] === itemId) {\n                return item['product_id'] === productId ||\n                    item['item_id'] === productId;\n            }\n        });\n\n        if (!product) {\n            return;\n        }\n        productQty = product.qty;\n    };\n\n    cartData.subscribe(function (updateCartData) {\n        setProductQty(updateCartData);\n        updateQty();\n    });\n\n    setProductQty(cartData());\n    updateQty();\n});\n","Magento_Checkout/js/view/form/element/email.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'uiComponent',\n    'ko',\n    'Magento_Customer/js/model/customer',\n    'Magento_Customer/js/action/check-email-availability',\n    'Magento_Customer/js/action/login',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/checkout-data',\n    'Magento_Checkout/js/model/full-screen-loader',\n    'mage/validation'\n], function ($, Component, ko, customer, checkEmailAvailability, loginAction, quote, checkoutData, fullScreenLoader) {\n    'use strict';\n\n    var validatedEmail;\n\n    if (!checkoutData.getValidatedEmailValue() &&\n        window.checkoutConfig.validatedEmailValue\n    ) {\n        checkoutData.setInputFieldEmailValue(window.checkoutConfig.validatedEmailValue);\n        checkoutData.setValidatedEmailValue(window.checkoutConfig.validatedEmailValue);\n    }\n\n    validatedEmail = checkoutData.getValidatedEmailValue();\n\n    if (validatedEmail && !customer.isLoggedIn()) {\n        quote.guestEmail = validatedEmail;\n    }\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/form/element/email',\n            email: checkoutData.getInputFieldEmailValue(),\n            emailFocused: false,\n            isLoading: false,\n            isPasswordVisible: false,\n            listens: {\n                email: 'emailHasChanged',\n                emailFocused: 'validateEmail'\n            },\n            ignoreTmpls: {\n                email: true\n            }\n        },\n        checkDelay: 2000,\n        checkRequest: null,\n        isEmailCheckComplete: null,\n        isCustomerLoggedIn: customer.isLoggedIn,\n        forgotPasswordUrl: window.checkoutConfig.forgotPasswordUrl,\n        emailCheckTimeout: 0,\n\n        /**\n         * Initializes regular properties of instance.\n         *\n         * @returns {Object} Chainable.\n         */\n        initConfig: function () {\n            this._super();\n\n            this.isPasswordVisible = this.resolveInitialPasswordVisibility();\n\n            return this;\n        },\n\n        /**\n         * Initializes observable properties of instance\n         *\n         * @returns {Object} Chainable.\n         */\n        initObservable: function () {\n            this._super()\n                .observe(['email', 'emailFocused', 'isLoading', 'isPasswordVisible']);\n\n            return this;\n        },\n\n        /**\n         * Callback on changing email property\n         */\n        emailHasChanged: function () {\n            var self = this;\n\n            clearTimeout(this.emailCheckTimeout);\n\n            if (self.validateEmail()) {\n                quote.guestEmail = self.email();\n                checkoutData.setValidatedEmailValue(self.email());\n            }\n            this.emailCheckTimeout = setTimeout(function () {\n                if (self.validateEmail()) {\n                    self.checkEmailAvailability();\n                } else {\n                    self.isPasswordVisible(false);\n                }\n            }, self.checkDelay);\n\n            checkoutData.setInputFieldEmailValue(self.email());\n        },\n\n        /**\n         * Check email existing.\n         */\n        checkEmailAvailability: function () {\n            this.validateRequest();\n            this.isEmailCheckComplete = $.Deferred();\n            this.isLoading(true);\n            this.checkRequest = checkEmailAvailability(this.isEmailCheckComplete, this.email());\n\n            $.when(this.isEmailCheckComplete).done(function () {\n                this.isPasswordVisible(false);\n                checkoutData.setCheckedEmailValue('');\n            }.bind(this)).fail(function () {\n                this.isPasswordVisible(true);\n                checkoutData.setCheckedEmailValue(this.email());\n            }.bind(this)).always(function () {\n                this.isLoading(false);\n            }.bind(this));\n        },\n\n        /**\n         * If request has been sent -> abort it.\n         * ReadyStates for request aborting:\n         * 1 - The request has been set up\n         * 2 - The request has been sent\n         * 3 - The request is in process\n         */\n        validateRequest: function () {\n            if (this.checkRequest != null && $.inArray(this.checkRequest.readyState, [1, 2, 3])) {\n                this.checkRequest.abort();\n                this.checkRequest = null;\n            }\n        },\n\n        /**\n         * Local email validation.\n         *\n         * @param {Boolean} focused - input focus.\n         * @returns {Boolean} - validation result.\n         */\n        validateEmail: function (focused) {\n            var loginFormSelector = 'form[data-role=email-with-possible-login]',\n                usernameSelector = loginFormSelector + ' input[name=username]',\n                loginForm = $(loginFormSelector),\n                validator,\n                valid;\n\n            loginForm.validation();\n\n            if (focused === false && !!this.email()) {\n                valid = !!$(usernameSelector).valid();\n\n                if (valid) {\n                    $(usernameSelector).removeAttr('aria-invalid aria-describedby');\n                }\n\n                return valid;\n            }\n\n            if (loginForm.is(':visible')) {\n                validator = loginForm.validate();\n\n                return validator.check(usernameSelector);\n            }\n\n            return true;\n        },\n\n        /**\n         * Log in form submitting callback.\n         *\n         * @param {HTMLElement} loginForm - form element.\n         */\n        login: function (loginForm) {\n            var loginData = {},\n                formDataArray = $(loginForm).serializeArray();\n\n            formDataArray.forEach(function (entry) {\n                loginData[entry.name] = entry.value;\n            });\n\n            if (this.isPasswordVisible() && $(loginForm).validation() && $(loginForm).validation('isValid')) {\n                fullScreenLoader.startLoader();\n                loginAction(loginData).always(function () {\n                    fullScreenLoader.stopLoader();\n                });\n            }\n        },\n\n        /**\n         * Resolves an initial state of a login form.\n         *\n         * @returns {Boolean} - initial visibility state.\n         */\n        resolveInitialPasswordVisibility: function () {\n            if (checkoutData.getInputFieldEmailValue() !== '' && checkoutData.getCheckedEmailValue() !== '') {\n                return true;\n            }\n\n            if (checkoutData.getInputFieldEmailValue() !== '') {\n                return checkoutData.getInputFieldEmailValue() === checkoutData.getCheckedEmailValue();\n            }\n\n            return false;\n        }\n    });\n});\n","Magento_Checkout/js/view/payment/default.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko',\n    'jquery',\n    'uiComponent',\n    'Magento_Checkout/js/action/place-order',\n    'Magento_Checkout/js/action/select-payment-method',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Customer/js/model/customer',\n    'Magento_Checkout/js/model/payment-service',\n    'Magento_Checkout/js/checkout-data',\n    'Magento_Checkout/js/model/checkout-data-resolver',\n    'uiRegistry',\n    'Magento_Checkout/js/model/payment/additional-validators',\n    'Magento_Ui/js/model/messages',\n    'uiLayout',\n    'Magento_Checkout/js/action/redirect-on-success'\n], function (\n    ko,\n    $,\n    Component,\n    placeOrderAction,\n    selectPaymentMethodAction,\n    quote,\n    customer,\n    paymentService,\n    checkoutData,\n    checkoutDataResolver,\n    registry,\n    additionalValidators,\n    Messages,\n    layout,\n    redirectOnSuccessAction\n) {\n    'use strict';\n\n    return Component.extend({\n        redirectAfterPlaceOrder: true,\n        isPlaceOrderActionAllowed: ko.observable(quote.billingAddress() != null),\n\n        /**\n         * After place order callback\n         */\n        afterPlaceOrder: function () {\n            // Override this function and put after place order logic here\n        },\n\n        /**\n         * Initialize view.\n         *\n         * @return {exports}\n         */\n        initialize: function () {\n            var billingAddressCode,\n                billingAddressData,\n                defaultAddressData;\n\n            this._super().initChildren();\n            quote.billingAddress.subscribe(function (address) {\n                this.isPlaceOrderActionAllowed(address !== null);\n            }, this);\n            checkoutDataResolver.resolveBillingAddress();\n\n            billingAddressCode = 'billingAddress' + this.getCode();\n            registry.async('checkoutProvider')(function (checkoutProvider) {\n                defaultAddressData = checkoutProvider.get(billingAddressCode);\n\n                if (defaultAddressData === undefined) {\n                    // Skip if payment does not have a billing address form\n                    return;\n                }\n                billingAddressData = checkoutData.getBillingAddressFromData();\n\n                if (billingAddressData) {\n                    checkoutProvider.set(\n                        billingAddressCode,\n                        $.extend(true, {}, defaultAddressData, billingAddressData)\n                    );\n                }\n                checkoutProvider.on(billingAddressCode, function (providerBillingAddressData) {\n                    checkoutData.setBillingAddressFromData(providerBillingAddressData);\n                }, billingAddressCode);\n            });\n\n            return this;\n        },\n\n        /**\n         * Initialize child elements\n         *\n         * @returns {Component} Chainable.\n         */\n        initChildren: function () {\n            this.messageContainer = new Messages();\n            this.createMessagesComponent();\n\n            return this;\n        },\n\n        /**\n         * Create child message renderer component\n         *\n         * @returns {Component} Chainable.\n         */\n        createMessagesComponent: function () {\n\n            var messagesComponent = {\n                parent: this.name,\n                name: this.name + '.messages',\n                displayArea: 'messages',\n                component: 'Magento_Ui/js/view/messages',\n                config: {\n                    messageContainer: this.messageContainer\n                }\n            };\n\n            layout([messagesComponent]);\n\n            return this;\n        },\n\n        /**\n         * Place order.\n         */\n        placeOrder: function (data, event) {\n            var self = this;\n\n            if (event) {\n                event.preventDefault();\n            }\n\n            if (this.validate() &&\n                additionalValidators.validate() &&\n                this.isPlaceOrderActionAllowed() === true\n            ) {\n                this.isPlaceOrderActionAllowed(false);\n\n                this.getPlaceOrderDeferredObject()\n                    .done(\n                        function () {\n                            self.afterPlaceOrder();\n\n                            if (self.redirectAfterPlaceOrder) {\n                                redirectOnSuccessAction.execute();\n                            }\n                        }\n                    ).always(\n                        function () {\n                            self.isPlaceOrderActionAllowed(true);\n                        }\n                    );\n\n                return true;\n            }\n\n            return false;\n        },\n\n        /**\n         * @return {*}\n         */\n        getPlaceOrderDeferredObject: function () {\n            return $.when(\n                placeOrderAction(this.getData(), this.messageContainer)\n            );\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        selectPaymentMethod: function () {\n            selectPaymentMethodAction(this.getData());\n            checkoutData.setSelectedPaymentMethod(this.item.method);\n\n            return true;\n        },\n\n        isChecked: ko.computed(function () {\n            return quote.paymentMethod() ? quote.paymentMethod().method : null;\n        }),\n\n        isRadioButtonVisible: ko.computed(function () {\n            return paymentService.getAvailablePaymentMethods().length !== 1;\n        }),\n\n        /**\n         * Get payment method data\n         */\n        getData: function () {\n            return {\n                'method': this.item.method,\n                'po_number': null,\n                'additional_data': null\n            };\n        },\n\n        /**\n         * Get payment method type.\n         */\n        getTitle: function () {\n            return this.item.title;\n        },\n\n        /**\n         * Get payment method code.\n         */\n        getCode: function () {\n            return this.item.method;\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        validate: function () {\n            return true;\n        },\n\n        /**\n         * @return {String}\n         */\n        getBillingAddressFormName: function () {\n            return 'billing-address-form-' + this.item.method;\n        },\n\n        /**\n         * Dispose billing address subscriptions\n         */\n        disposeSubscriptions: function () {\n            // dispose all active subscriptions\n            var billingAddressCode = 'billingAddress' + this.getCode();\n\n            registry.async('checkoutProvider')(function (checkoutProvider) {\n                checkoutProvider.off(billingAddressCode);\n            });\n        }\n    });\n});\n","Magento_Checkout/js/view/payment/email-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine(\n    [\n        'uiComponent',\n        'Magento_Checkout/js/model/payment/additional-validators',\n        'Magento_Checkout/js/model/customer-email-validator'\n    ],\n    function (Component, additionalValidators, agreementValidator) {\n        'use strict';\n\n        additionalValidators.registerValidator(agreementValidator);\n\n        return Component.extend({});\n    }\n);\n","Magento_Checkout/js/view/payment/list.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'ko',\n    'mageUtils',\n    'uiComponent',\n    'Magento_Checkout/js/model/payment/method-list',\n    'Magento_Checkout/js/model/payment/renderer-list',\n    'uiLayout',\n    'Magento_Checkout/js/model/checkout-data-resolver',\n    'mage/translate',\n    'uiRegistry'\n], function (_, ko, utils, Component, paymentMethods, rendererList, layout, checkoutDataResolver, $t, registry) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/payment-methods/list',\n            visible: paymentMethods().length > 0,\n            configDefaultGroup: {\n                name: 'methodGroup',\n                component: 'Magento_Checkout/js/model/payment/method-group'\n            },\n            paymentGroupsList: [],\n            defaultGroupTitle: $t('Select a new payment method')\n        },\n\n        /**\n         * Initialize view.\n         *\n         * @returns {Component} Chainable.\n         */\n        initialize: function () {\n            this._super().initDefaulGroup().initChildren();\n            paymentMethods.subscribe(\n                function (changes) {\n                    checkoutDataResolver.resolvePaymentMethod();\n                    //remove renderer for \"deleted\" payment methods\n                    _.each(changes, function (change) {\n                        if (change.status === 'deleted') {\n                            this.removeRenderer(change.value.method);\n                        }\n                    }, this);\n                    //add renderer for \"added\" payment methods\n                    _.each(changes, function (change) {\n                        if (change.status === 'added') {\n                            this.createRenderer(change.value);\n                        }\n                    }, this);\n                }, this, 'arrayChange');\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super().\n                observe(['paymentGroupsList']);\n\n            return this;\n        },\n\n        /**\n         * Creates default group\n         *\n         * @returns {Component} Chainable.\n         */\n        initDefaulGroup: function () {\n            layout([\n                this.configDefaultGroup\n            ]);\n\n            return this;\n        },\n\n        /**\n         * Create renders for child payment methods.\n         *\n         * @returns {Component} Chainable.\n         */\n        initChildren: function () {\n            var self = this;\n\n            _.each(paymentMethods(), function (paymentMethodData) {\n                self.createRenderer(paymentMethodData);\n            });\n\n            return this;\n        },\n\n        /**\n         * @returns\n         */\n        createComponent: function (payment) {\n            var rendererTemplate,\n                rendererComponent,\n                templateData;\n\n            templateData = {\n                parentName: this.name,\n                name: payment.name\n            };\n            rendererTemplate = {\n                parent: '${ $.$data.parentName }',\n                name: '${ $.$data.name }',\n                displayArea: payment.displayArea,\n                component: payment.component\n            };\n            rendererComponent = utils.template(rendererTemplate, templateData);\n            utils.extend(rendererComponent, {\n                item: payment.item,\n                config: payment.config\n            });\n\n            return rendererComponent;\n        },\n\n        /**\n         * Create renderer.\n         *\n         * @param {Object} paymentMethodData\n         */\n        createRenderer: function (paymentMethodData) {\n            var isRendererForMethod = false,\n                currentGroup;\n\n            registry.get(this.configDefaultGroup.name, function (defaultGroup) {\n                _.each(rendererList(), function (renderer) {\n\n                    if (renderer.hasOwnProperty('typeComparatorCallback') &&\n                        typeof renderer.typeComparatorCallback == 'function'\n                    ) {\n                        isRendererForMethod = renderer.typeComparatorCallback(renderer.type, paymentMethodData.method);\n                    } else {\n                        isRendererForMethod = renderer.type === paymentMethodData.method;\n                    }\n\n                    if (isRendererForMethod) {\n                        currentGroup = renderer.group ? renderer.group : defaultGroup;\n\n                        this.collectPaymentGroups(currentGroup);\n\n                        layout([\n                            this.createComponent(\n                                {\n                                    config: renderer.config,\n                                    component: renderer.component,\n                                    name: renderer.type,\n                                    method: paymentMethodData.method,\n                                    item: paymentMethodData,\n                                    displayArea: currentGroup.displayArea\n                                }\n                            )]);\n                    }\n                }.bind(this));\n            }.bind(this));\n        },\n\n        /**\n         * Collects unique groups of available payment methods\n         *\n         * @param {Object} group\n         */\n        collectPaymentGroups: function (group) {\n            var groupsList = this.paymentGroupsList(),\n                isGroupExists = _.some(groupsList, function (existsGroup) {\n                    return existsGroup.alias === group.alias;\n                });\n\n            if (!isGroupExists) {\n                groupsList.push(group);\n                groupsList = _.sortBy(groupsList, function (existsGroup) {\n                    return existsGroup.sortOrder;\n                });\n                this.paymentGroupsList(groupsList);\n            }\n        },\n\n        /**\n         * Returns payment group title\n         *\n         * @param {Object} group\n         * @returns {String}\n         */\n        getGroupTitle: function (group) {\n            var title = group().title;\n\n            if (group().isDefault() && this.paymentGroupsList().length > 1) {\n                title = this.defaultGroupTitle;\n            }\n\n            return title;\n        },\n\n        /**\n         * Checks if at least one payment method available\n         *\n         * @returns {String}\n         */\n        isPaymentMethodsAvailable: function () {\n            return _.some(this.paymentGroupsList(), function (group) {\n                return this.regionHasElements(group.displayArea);\n            }, this);\n        },\n\n        /**\n         * Remove view renderer.\n         *\n         * @param {String} paymentMethodCode\n         */\n        removeRenderer: function (paymentMethodCode) {\n            var items;\n\n            _.each(this.paymentGroupsList(), function (group) {\n                items = this.getRegion(group.displayArea);\n\n                _.find(items(), function (value) {\n                    if (value.item.method.indexOf(paymentMethodCode) === 0) {\n                        value.disposeSubscriptions();\n                        value.destroy();\n                    }\n                });\n            }, this);\n        }\n    });\n});\n","Magento_Checkout/js/view/shipping-address/list.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'ko',\n    'mageUtils',\n    'uiComponent',\n    'uiLayout',\n    'Magento_Customer/js/model/address-list'\n], function (_, ko, utils, Component, layout, addressList) {\n    'use strict';\n\n    var defaultRendererTemplate = {\n        parent: '${ $.$data.parentName }',\n        name: '${ $.$data.name }',\n        component: 'Magento_Checkout/js/view/shipping-address/address-renderer/default',\n        provider: 'checkoutProvider'\n    };\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/shipping-address/list',\n            visible: addressList().length > 0,\n            rendererTemplates: []\n        },\n\n        /** @inheritdoc */\n        initialize: function () {\n            this._super()\n                .initChildren();\n\n            addressList.subscribe(function (changes) {\n                    var self = this;\n\n                    changes.forEach(function (change) {\n                        if (change.status === 'added') {\n                            self.createRendererComponent(change.value, change.index);\n                        }\n                    });\n                },\n                this,\n                'arrayChange'\n            );\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initConfig: function () {\n            this._super();\n            // the list of child components that are responsible for address rendering\n            this.rendererComponents = [];\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initChildren: function () {\n            _.each(addressList(), this.createRendererComponent, this);\n\n            return this;\n        },\n\n        /**\n         * Create new component that will render given address in the address list\n         *\n         * @param {Object} address\n         * @param {*} index\n         */\n        createRendererComponent: function (address, index) {\n            var rendererTemplate, templateData, rendererComponent;\n\n            if (index in this.rendererComponents) {\n                this.rendererComponents[index].address(address);\n            } else {\n                // rendererTemplates are provided via layout\n                rendererTemplate = address.getType() != undefined && this.rendererTemplates[address.getType()] != undefined ? //eslint-disable-line\n                    utils.extend({}, defaultRendererTemplate, this.rendererTemplates[address.getType()]) :\n                    defaultRendererTemplate;\n                templateData = {\n                    parentName: this.name,\n                    name: index\n                };\n                rendererComponent = utils.template(rendererTemplate, templateData);\n                utils.extend(rendererComponent, {\n                    address: ko.observable(address)\n                });\n                layout([rendererComponent]);\n                this.rendererComponents[index] = rendererComponent;\n            }\n        }\n    });\n});\n","Magento_Checkout/js/view/shipping-address/address-renderer/default.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'ko',\n    'uiComponent',\n    'underscore',\n    'Magento_Checkout/js/action/select-shipping-address',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/shipping-address/form-popup-state',\n    'Magento_Checkout/js/checkout-data',\n    'Magento_Customer/js/customer-data'\n], function ($, ko, Component, _, selectShippingAddressAction, quote, formPopUpState, checkoutData, customerData) {\n    'use strict';\n\n    var countryData = customerData.get('directory-data');\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/shipping-address/address-renderer/default'\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super();\n            this.isSelected = ko.computed(function () {\n                var isSelected = false,\n                    shippingAddress = quote.shippingAddress();\n\n                if (shippingAddress) {\n                    isSelected = shippingAddress.getKey() == this.address().getKey(); //eslint-disable-line eqeqeq\n                }\n\n                return isSelected;\n            }, this);\n\n            return this;\n        },\n\n        /**\n         * @param {String} countryId\n         * @return {String}\n         */\n        getCountryName: function (countryId) {\n            return countryData()[countryId] != undefined ? countryData()[countryId].name : ''; //eslint-disable-line\n        },\n\n        /**\n         * Get customer attribute label\n         *\n         * @param {*} attribute\n         * @returns {*}\n         */\n        getCustomAttributeLabel: function (attribute) {\n            var label;\n\n            if (typeof attribute === 'string') {\n                return attribute;\n            }\n\n            if (attribute.label) {\n                return attribute.label;\n            }\n\n            if (_.isArray(attribute.value)) {\n                label = _.map(attribute.value, function (value) {\n                    return this.getCustomAttributeOptionLabel(attribute['attribute_code'], value) || value;\n                }, this).join(', ');\n            } else if (typeof attribute.value === 'object') {\n                label = _.map(Object.values(attribute.value)).join(', ');\n            } else {\n                label = this.getCustomAttributeOptionLabel(attribute['attribute_code'], attribute.value);\n            }\n\n            return label || attribute.value;\n        },\n\n        /**\n         * Get option label for given attribute code and option ID\n         *\n         * @param {String} attributeCode\n         * @param {String} value\n         * @returns {String|null}\n         */\n        getCustomAttributeOptionLabel: function (attributeCode, value) {\n            var option,\n                label,\n                options = this.source.get('customAttributes') || {};\n\n            if (options[attributeCode]) {\n                option = _.findWhere(options[attributeCode], {\n                    value: value\n                });\n\n                if (option) {\n                    label = option.label;\n                }\n            } else if (value.file !== null) {\n                label = value.file;\n            }\n\n            return label;\n        },\n\n        /** Set selected customer shipping address  */\n        selectAddress: function () {\n            selectShippingAddressAction(this.address());\n            checkoutData.setSelectedShippingAddress(this.address().getKey());\n        },\n\n        /**\n         * Edit address.\n         */\n        editAddress: function () {\n            formPopUpState.isVisible(true);\n            this.showPopup();\n\n        },\n\n        /**\n         * Show popup.\n         */\n        showPopup: function () {\n            $('[data-open-modal=\"opc-new-shipping-address\"]').trigger('click');\n        }\n    });\n});\n","Magento_Checkout/js/view/shipping-information/list.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'ko',\n    'mageUtils',\n    'uiComponent',\n    'uiLayout',\n    'Magento_Checkout/js/model/quote'\n], function ($, ko, utils, Component, layout, quote) {\n    'use strict';\n\n    var defaultRendererTemplate = {\n        parent: '${ $.$data.parentName }',\n        name: '${ $.$data.name }',\n        component: 'Magento_Checkout/js/view/shipping-information/address-renderer/default',\n        provider: 'checkoutProvider'\n    };\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/shipping-information/list',\n            rendererTemplates: {}\n        },\n\n        /** @inheritdoc */\n        initialize: function () {\n            var self = this;\n\n            this._super()\n                .initChildren();\n\n            quote.shippingAddress.subscribe(function (address) {\n                self.createRendererComponent(address);\n            });\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initConfig: function () {\n            this._super();\n            // the list of child components that are responsible for address rendering\n            this.rendererComponents = {};\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initChildren: function () {\n            return this;\n        },\n\n        /**\n         * Create new component that will render given address in the address list\n         *\n         * @param {Object} address\n         */\n        createRendererComponent: function (address) {\n            var rendererTemplate, templateData, rendererComponent;\n\n            $.each(this.rendererComponents, function (index, component) {\n                component.visible(false);\n            });\n\n            if (this.rendererComponents[address.getType()]) {\n                this.rendererComponents[address.getType()].address(address);\n                this.rendererComponents[address.getType()].visible(true);\n            } else {\n                // rendererTemplates are provided via layout\n                rendererTemplate = address.getType() != undefined && this.rendererTemplates[address.getType()] != undefined ? //eslint-disable-line\n                    utils.extend({}, defaultRendererTemplate, this.rendererTemplates[address.getType()]) :\n                    defaultRendererTemplate;\n                templateData = {\n                    parentName: this.name,\n                    name: address.getType()\n                };\n\n                rendererComponent = utils.template(rendererTemplate, templateData);\n                utils.extend(\n                    rendererComponent,\n                    {\n                        address: ko.observable(address),\n                        visible: ko.observable(true)\n                    }\n                );\n                layout([rendererComponent]);\n                this.rendererComponents[address.getType()] = rendererComponent;\n            }\n        }\n    });\n});\n","Magento_Checkout/js/view/shipping-information/address-renderer/default.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'underscore',\n    'Magento_Customer/js/customer-data'\n], function (Component, _, customerData) {\n    'use strict';\n\n    var countryData = customerData.get('directory-data');\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/shipping-information/address-renderer/default'\n        },\n\n        /**\n         * @param {*} countryId\n         * @return {String}\n         */\n        getCountryName: function (countryId) {\n            return countryData()[countryId] != undefined ? countryData()[countryId].name : ''; //eslint-disable-line\n        },\n\n        /**\n         * Get customer attribute label\n         *\n         * @param {*} attribute\n         * @returns {*}\n         */\n        getCustomAttributeLabel: function (attribute) {\n            var label;\n\n            if (typeof attribute === 'string') {\n                return attribute;\n            }\n\n            if (attribute.label) {\n                return attribute.label;\n            }\n\n            if (_.isArray(attribute.value)) {\n                label = _.map(attribute.value, function (value) {\n                    return this.getCustomAttributeOptionLabel(attribute['attribute_code'], value) || value;\n                }, this).join(', ');\n            } else if (typeof attribute.value === 'object') {\n                label = _.map(Object.values(attribute.value)).join(', ');\n            } else {\n                label = this.getCustomAttributeOptionLabel(attribute['attribute_code'], attribute.value);\n            }\n\n            return label || attribute.value;\n        },\n\n        /**\n         * Get option label for given attribute code and option ID\n         *\n         * @param {String} attributeCode\n         * @param {String} value\n         * @returns {String|null}\n         */\n        getCustomAttributeOptionLabel: function (attributeCode, value) {\n            var option,\n                label,\n                options = this.source.get('customAttributes') || {};\n\n            if (options[attributeCode]) {\n                option = _.findWhere(options[attributeCode], {\n                    value: value\n                });\n\n                if (option) {\n                    label = option.label;\n                }\n            } else if (value.file !== null) {\n                label = value.file;\n            }\n\n            return label;\n        }\n    });\n});\n","Magento_Checkout/js/view/summary/abstract-total.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Catalog/js/price-utils',\n    'Magento_Checkout/js/model/totals',\n    'Magento_Checkout/js/model/step-navigator'\n], function (Component, quote, priceUtils, totals, stepNavigator) {\n    'use strict';\n\n    return Component.extend({\n        /**\n         * @param {*} price\n         * @return {*|String}\n         */\n        getFormattedPrice: function (price) {\n            return priceUtils.formatPriceLocale(price, quote.getPriceFormat());\n        },\n\n        /**\n         * @return {*}\n         */\n        getTotals: function () {\n            return totals.totals();\n        },\n\n        /**\n         * @return {*}\n         */\n        isFullMode: function () {\n            if (!this.getTotals()) {\n                return false;\n            }\n\n            return stepNavigator.isProcessed('shipping');\n        }\n    });\n});\n","Magento_Checkout/js/view/summary/cart-items.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko',\n    'Magento_Checkout/js/model/totals',\n    'uiComponent',\n    'Magento_Checkout/js/model/step-navigator',\n    'Magento_Checkout/js/model/quote'\n], function (ko, totals, Component, stepNavigator, quote) {\n    'use strict';\n\n    var useQty = window.checkoutConfig.useQty;\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/summary/cart-items'\n        },\n        totals: totals.totals(),\n        items: ko.observable([]),\n        maxCartItemsToDisplay: window.checkoutConfig.maxCartItemsToDisplay,\n        cartUrl: window.checkoutConfig.cartUrl,\n\n        /**\n         * @deprecated Please use observable property (this.items())\n         */\n        getItems: totals.getItems(),\n\n        /**\n         * Returns cart items qty\n         *\n         * @returns {Number}\n         */\n        getItemsQty: function () {\n            return parseFloat(this.totals['items_qty']);\n        },\n\n        /**\n         * Returns count of cart line items\n         *\n         * @returns {Number}\n         */\n        getCartLineItemsCount: function () {\n            return parseInt(totals.getItems()().length, 10);\n        },\n\n        /**\n         * Returns shopping cart items summary (includes config settings)\n         *\n         * @returns {Number}\n         */\n        getCartSummaryItemsCount: function () {\n            return useQty ? this.getItemsQty() : this.getCartLineItemsCount();\n        },\n\n        /**\n         * @inheritdoc\n         */\n        initialize: function () {\n            this._super();\n            // Set initial items to observable field\n            this.setItems(totals.getItems()());\n            // Subscribe for items data changes and refresh items in view\n            totals.getItems().subscribe(function (items) {\n                this.setItems(items);\n            }.bind(this));\n        },\n\n        /**\n         * Set items to observable field\n         *\n         * @param {Object} items\n         */\n        setItems: function (items) {\n            if (items && items.length > 0) {\n                items = items.slice(parseInt(-this.maxCartItemsToDisplay, 10));\n            }\n            this.items(items);\n        },\n\n        /**\n         * Returns bool value for items block state (expanded or not)\n         *\n         * @returns {*|Boolean}\n         */\n        isItemsBlockExpanded: function () {\n            return quote.isVirtual() || stepNavigator.isProcessed('shipping');\n        }\n    });\n});\n","Magento_Checkout/js/view/summary/grand-total.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/view/summary/abstract-total',\n    'Magento_Checkout/js/model/quote'\n], function (Component, quote) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/summary/grand-total'\n        },\n\n        /**\n         * @return {*}\n         */\n        isDisplayed: function () {\n            return this.isFullMode();\n        },\n\n        /**\n         * Get pure value.\n         */\n        getPureValue: function () {\n            var totals = quote.getTotals()();\n\n            if (totals) {\n                return totals['grand_total'];\n            }\n\n            return quote['grand_total'];\n        },\n\n        /**\n         * @return {*|String}\n         */\n        getValue: function () {\n            return this.getFormattedPrice(this.getPureValue());\n        }\n    });\n});\n","Magento_Checkout/js/view/summary/shipping.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'Magento_Checkout/js/view/summary/abstract-total',\n    'Magento_Checkout/js/model/quote',\n    'Magento_SalesRule/js/view/summary/discount'\n], function ($, _, Component, quote, discountView) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/summary/shipping'\n        },\n        quoteIsVirtual: quote.isVirtual(),\n        totals: quote.getTotals(),\n\n        /**\n         * @return {*}\n         */\n        getShippingMethodTitle: function () {\n            var shippingMethod,\n                shippingMethodTitle = '';\n\n            if (!this.isCalculated()) {\n                return '';\n            }\n            shippingMethod = quote.shippingMethod();\n\n            if (!_.isArray(shippingMethod) && !_.isObject(shippingMethod)) {\n                return '';\n            }\n\n            if (typeof shippingMethod['method_title'] !== 'undefined') {\n                shippingMethodTitle = ' - ' + shippingMethod['method_title'];\n            }\n\n            return shippingMethodTitle ?\n                shippingMethod['carrier_title'] + shippingMethodTitle :\n                shippingMethod['carrier_title'];\n        },\n\n        /**\n         * @return {*|Boolean}\n         */\n        isCalculated: function () {\n            return this.totals() && this.isFullMode() && quote.shippingMethod() != null; //eslint-disable-line eqeqeq\n        },\n\n        /**\n         * @return {*}\n         */\n        getValue: function () {\n            var price;\n\n            if (!this.isCalculated()) {\n                return this.notCalculatedMessage;\n            }\n            price =  this.totals()['shipping_amount'];\n\n            return this.getFormattedPrice(price);\n        },\n\n        /**\n         * If is set coupon code, but there wasn't displayed discount view.\n         *\n         * @return {Boolean}\n         */\n        haveToShowCoupon: function () {\n            var couponCode = this.totals()['coupon_code'];\n\n            if (typeof couponCode === 'undefined') {\n                couponCode = false;\n            }\n\n            return couponCode && !discountView().isDisplayed();\n        },\n\n        /**\n         * Returns coupon code description.\n         *\n         * @return {String}\n         */\n        getCouponDescription: function () {\n            if (!this.haveToShowCoupon()) {\n                return '';\n            }\n\n            return '(' + this.totals()['coupon_code'] + ')';\n        }\n    });\n});\n","Magento_Checkout/js/view/summary/subtotal.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/view/summary/abstract-total',\n    'Magento_Checkout/js/model/quote'\n], function (Component, quote) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/summary/subtotal'\n        },\n\n        /**\n         * Get pure value.\n         *\n         * @return {*}\n         */\n        getPureValue: function () {\n            var totals = quote.getTotals()();\n\n            if (totals) {\n                return totals.subtotal;\n            }\n\n            return quote.subtotal;\n        },\n\n        /**\n         * @return {*|String}\n         */\n        getValue: function () {\n            return this.getFormattedPrice(this.getPureValue());\n        }\n\n    });\n});\n","Magento_Checkout/js/view/summary/totals.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/view/summary/abstract-total'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        /**\n         * @return {*}\n         */\n        isDisplayed: function () {\n            return this.isFullMode();\n        }\n    });\n});\n","Magento_Checkout/js/view/summary/item/details.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'escaper'\n], function (Component, escaper) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/summary/item/details',\n            allowedTags: ['b', 'strong', 'i', 'em', 'u']\n        },\n\n        /**\n         * @param {Object} quoteItem\n         * @return {String}\n         */\n        getNameUnsanitizedHtml: function (quoteItem) {\n            var txt = document.createElement('textarea');\n\n            txt.innerHTML = quoteItem.name;\n\n            return escaper.escapeHtml(txt.value, this.allowedTags);\n        },\n\n        /**\n         * @param {Object} quoteItem\n         * @return {String}Magento_Checkout/js/region-updater\n         */\n        getValue: function (quoteItem) {\n            return quoteItem.name;\n        }\n    });\n});\n","Magento_Checkout/js/view/summary/item/details/message.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['uiComponent'], function (Component) {\n    'use strict';\n\n    var quoteMessages = window.checkoutConfig.quoteMessages;\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/summary/item/details/message'\n        },\n        displayArea: 'item_message',\n        quoteMessages: quoteMessages,\n\n        /**\n         * @param {Object} item\n         * @return {null}\n         */\n        getMessage: function (item) {\n            if (this.quoteMessages[item['item_id']]) {\n                return this.quoteMessages[item['item_id']];\n            }\n\n            return null;\n        }\n    });\n});\n","Magento_Checkout/js/view/summary/item/details/subtotal.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/view/summary/abstract-total'\n], function (viewModel) {\n    'use strict';\n\n    return viewModel.extend({\n        defaults: {\n            displayArea: 'after_details',\n            template: 'Magento_Checkout/summary/item/details/subtotal'\n        },\n\n        /**\n         * @param {Object} quoteItem\n         * @return {*|String}\n         */\n        getValue: function (quoteItem) {\n            return this.getFormattedPrice(quoteItem['row_total']);\n        }\n    });\n});\n","Magento_Checkout/js/view/summary/item/details/thumbnail.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['uiComponent'], function (Component) {\n    'use strict';\n\n    var imageData = window.checkoutConfig.imageData;\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Checkout/summary/item/details/thumbnail'\n        },\n        displayArea: 'before_details',\n        imageData: imageData,\n\n        /**\n         * @param {Object} item\n         * @return {Array}\n         */\n        getImageItem: function (item) {\n            if (this.imageData[item['item_id']]) {\n                return this.imageData[item['item_id']];\n            }\n\n            return [];\n        },\n\n        /**\n         * @param {Object} item\n         * @return {null}\n         */\n        getSrc: function (item) {\n            if (this.imageData[item['item_id']]) {\n                return this.imageData[item['item_id']].src;\n            }\n\n            return null;\n        },\n\n        /**\n         * @param {Object} item\n         * @return {null}\n         */\n        getWidth: function (item) {\n            if (this.imageData[item['item_id']]) {\n                return this.imageData[item['item_id']].width;\n            }\n\n            return null;\n        },\n\n        /**\n         * @param {Object} item\n         * @return {null}\n         */\n        getHeight: function (item) {\n            if (this.imageData[item['item_id']]) {\n                return this.imageData[item['item_id']].height;\n            }\n\n            return null;\n        },\n\n        /**\n         * @param {Object} item\n         * @return {null}\n         */\n        getAlt: function (item) {\n            if (this.imageData[item['item_id']]) {\n                return this.imageData[item['item_id']].alt;\n            }\n\n            return null;\n        }\n    });\n});\n","Magento_CheckoutAgreements/js/model/agreement-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/validation'\n], function ($) {\n    'use strict';\n\n    var checkoutConfig = window.checkoutConfig,\n        agreementsConfig = checkoutConfig ? checkoutConfig.checkoutAgreements : {},\n        agreementsInputPath = '.payment-method._active div.checkout-agreements input';\n\n    return {\n        /**\n         * Validate checkout agreements\n         *\n         * @returns {Boolean}\n         */\n        validate: function (hideError) {\n            var isValid = true;\n\n            if (!agreementsConfig.isEnabled || $(agreementsInputPath).length === 0) {\n                return true;\n            }\n\n            $(agreementsInputPath).each(function (index, element) {\n                if (!$.validator.validateSingleElement(element, {\n                    errorElement: 'div',\n                    hideError: hideError || false\n                })) {\n                    isValid = false;\n                }\n            });\n\n            return isValid;\n        }\n    };\n});\n","Magento_CheckoutAgreements/js/model/agreements-assigner.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    var agreementsConfig = window.checkoutConfig.checkoutAgreements;\n\n    /** Override default place order action and add agreement_ids to request */\n    return function (paymentData) {\n        var agreementForm,\n            agreementData,\n            agreementIds;\n\n        if (!agreementsConfig.isEnabled) {\n            return;\n        }\n\n        agreementForm = $('.payment-method._active div[data-role=checkout-agreements] input');\n        agreementData = agreementForm.serializeArray();\n        agreementIds = [];\n\n        agreementData.forEach(function (item) {\n            agreementIds.push(item.value);\n        });\n\n        if (paymentData['extension_attributes'] === undefined) {\n            paymentData['extension_attributes'] = {};\n        }\n\n        paymentData['extension_attributes']['agreement_ids'] = agreementIds;\n    };\n});\n","Magento_CheckoutAgreements/js/model/agreements-modal.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/modal',\n    'mage/translate'\n], function ($, modal, $t) {\n    'use strict';\n\n    return {\n        modalWindow: null,\n\n        /**\n         * Create popUp window for provided element.\n         *\n         * @param {HTMLElement} element\n         */\n        createModal: function (element) {\n            var options;\n\n            this.modalWindow = element;\n            options = {\n                'type': 'popup',\n                'modalClass': 'agreements-modal',\n                'responsive': true,\n                'innerScroll': true,\n                'trigger': '.show-modal',\n                'buttons': [\n                    {\n                        text: $t('Close'),\n                        class: 'action secondary action-hide-popup',\n\n                        /** @inheritdoc */\n                        click: function () {\n                            this.closeModal();\n                        }\n                    }\n                ]\n            };\n            modal(options, $(this.modalWindow));\n        },\n\n        /** Show login popup window */\n        showModal: function () {\n            $(this.modalWindow).modal('openModal');\n        }\n    };\n});\n","Magento_CheckoutAgreements/js/model/place-order-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/utils/wrapper',\n    'Magento_CheckoutAgreements/js/model/agreements-assigner'\n], function ($, wrapper, agreementsAssigner) {\n    'use strict';\n\n    return function (placeOrderAction) {\n\n        /** Override default place order action and add agreement_ids to request */\n        return wrapper.wrap(placeOrderAction, function (originalAction, paymentData, messageContainer) {\n            agreementsAssigner(paymentData);\n\n            return originalAction(paymentData, messageContainer);\n        });\n    };\n});\n","Magento_CheckoutAgreements/js/model/set-payment-information-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/utils/wrapper',\n    'Magento_CheckoutAgreements/js/model/agreements-assigner'\n], function ($, wrapper, agreementsAssigner) {\n    'use strict';\n\n    return function (placeOrderAction) {\n\n        /** Override place-order-mixin for set-payment-information action as they differs only by method signature */\n        return wrapper.wrap(placeOrderAction, function (originalAction, messageContainer, paymentData) {\n            agreementsAssigner(paymentData);\n\n            return originalAction(messageContainer, paymentData);\n        });\n    };\n});\n","Magento_CheckoutAgreements/js/view/agreement-validation.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/payment/additional-validators',\n    'Magento_CheckoutAgreements/js/model/agreement-validator'\n], function (Component, additionalValidators, agreementValidator) {\n    'use strict';\n\n    additionalValidators.registerValidator(agreementValidator);\n\n    return Component.extend({});\n});\n","Magento_CheckoutAgreements/js/view/checkout-agreements.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko',\n    'jquery',\n    'uiComponent',\n    'Magento_CheckoutAgreements/js/model/agreements-modal'\n], function (ko, $, Component, agreementsModal) {\n    'use strict';\n\n    var checkoutConfig = window.checkoutConfig,\n        agreementManualMode = 1,\n        agreementsConfig = checkoutConfig ? checkoutConfig.checkoutAgreements : {};\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_CheckoutAgreements/checkout/checkout-agreements'\n        },\n        isVisible: agreementsConfig.isEnabled,\n        agreements: agreementsConfig.agreements,\n        modalTitle: ko.observable(null),\n        modalContent: ko.observable(null),\n        contentHeight: ko.observable(null),\n        modalWindow: null,\n\n        /**\n         * Checks if agreement required\n         *\n         * @param {Object} element\n         */\n        isAgreementRequired: function (element) {\n            return element.mode == agreementManualMode; //eslint-disable-line eqeqeq\n        },\n\n        /**\n         * Show agreement content in modal\n         *\n         * @param {Object} element\n         */\n        showContent: function (element) {\n            this.modalTitle(element.checkboxText);\n            this.modalContent(element.content);\n            this.contentHeight(element.contentHeight ? element.contentHeight : 'auto');\n            agreementsModal.showModal();\n        },\n\n        /**\n         * build a unique id for the term checkbox\n         *\n         * @param {Object} context - the ko context\n         * @param {Number} agreementId\n         */\n        getCheckboxId: function (context, agreementId) {\n            var paymentMethodName = '',\n                paymentMethodRenderer = context.$parents[1];\n\n            // corresponding payment method fetched from parent context\n            if (paymentMethodRenderer) {\n                // item looks like this: {title: \"Check / Money order\", method: \"checkmo\"}\n                paymentMethodName = paymentMethodRenderer.item ?\n                  paymentMethodRenderer.item.method : '';\n            }\n\n            return 'agreement_' + paymentMethodName + '_' + agreementId;\n        },\n\n        /**\n         * Init modal window for rendered element\n         *\n         * @param {Object} element\n         */\n        initModal: function (element) {\n            agreementsModal.createModal(element);\n        }\n    });\n});\n","Magento_ConfigurableProduct/js/catalog-add-to-cart-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'jquery',\n    'Magento_ConfigurableProduct/js/product/view/product-info-resolver'\n], function (_, $, productInfoResolver) {\n    'use strict';\n\n    return function (widget) {\n\n        $.widget('mage.catalogAddToCart', widget, {\n            /**\n             * @param {jQuery} form\n             */\n            ajaxSubmit: function (form) {\n                var isConfigurable = !!_.find(form.serializeArray(), function (item) {\n                    return item.name.indexOf('super_attribute') !== -1;\n                });\n\n                if (isConfigurable) {\n                    this.options.productInfoResolver = productInfoResolver;\n                }\n\n                return this._super(form);\n            }\n        });\n\n        return $.mage.catalogAddToCart;\n    };\n});\n","Magento_ConfigurableProduct/js/catalog-add-to-cart.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\nrequire([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    /**\n     * Add selected configurable attributes to redirect url\n     *\n     * @see Magento_Catalog/js/catalog-add-to-cart\n     */\n    $('body').on('catalogCategoryAddToCartRedirect', function (event, data) {\n        $(data.form).find('select[name*=\"super\"]').each(function (index, item) {\n            data.redirectParameters.push(item.config.id + '=' + $(item).val());\n        });\n    });\n});\n","Magento_ConfigurableProduct/js/configurable-customer-data.js":"require([\n    'jquery',\n    'Magento_ConfigurableProduct/js/options-updater'\n], function ($, Updater) {\n    'use strict';\n\n    var selectors = {\n            formSelector: '#product_addtocart_form'\n        },\n        configurableWidgetName = 'mageConfigurable',\n        widgetInitEvent = 'configurable.initialized',\n\n    /**\n    * Sets all configurable attribute's selected values\n    */\n    updateConfigurableOptions = function () {\n        var configurableWidget = $(selectors.formSelector).data(configurableWidgetName);\n\n        if (!configurableWidget) {\n            return;\n        }\n        configurableWidget.options.values = this.productOptions || {};\n        configurableWidget._configureForValues();\n    },\n    updater = new Updater(widgetInitEvent, updateConfigurableOptions);\n\n    updater.listen();\n});\n","Magento_ConfigurableProduct/js/configurable.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'underscore',\n    'mage/template',\n    'mage/translate',\n    'priceUtils',\n    'priceBox',\n    'jquery-ui-modules/widget',\n    'jquery/jquery.parsequery',\n    'fotoramaVideoEvents'\n], function ($, _, mageTemplate, $t, priceUtils) {\n    'use strict';\n\n    $.widget('mage.configurable', {\n        options: {\n            superSelector: '.super-attribute-select',\n            selectSimpleProduct: '[name=\"selected_configurable_option\"]',\n            priceHolderSelector: '.price-box',\n            spConfig: {},\n            state: {},\n            priceFormat: {},\n            optionTemplate: '<%- data.label %>' +\n            '<% if (typeof data.finalPrice.value !== \"undefined\") { %>' +\n            ' <%- data.finalPrice.formatted %>' +\n            '<% } %>',\n            mediaGallerySelector: '[data-gallery-role=gallery-placeholder]',\n            mediaGalleryInitial: null,\n            slyOldPriceSelector: '.sly-old-price',\n            normalPriceLabelSelector: '.product-info-main .normal-price .price-label',\n\n            /**\n             * Defines the mechanism of how images of a gallery should be\n             * updated when user switches between configurations of a product.\n             *\n             * As for now value of this option can be either 'replace' or 'prepend'.\n             *\n             * @type {String}\n             */\n            gallerySwitchStrategy: 'replace',\n            tierPriceTemplateSelector: '#tier-prices-template',\n            tierPriceBlockSelector: '[data-role=\"tier-price-block\"]',\n            tierPriceTemplate: '',\n            selectorProduct: '.product-info-main',\n            selectorProductPrice: '[data-role=priceBox]',\n            qtyInfo: '#qty'\n        },\n\n        /**\n         * Creates widget\n         * @private\n         */\n        _create: function () {\n            // Initial setting of various option values\n            this._initializeOptions();\n\n            // Override defaults with URL query parameters and/or inputs values\n            this._overrideDefaults();\n\n            // Change events to check select reloads\n            this._setupChangeEvents();\n\n            // Fill state\n            this._fillState();\n\n            // Setup child and prev/next settings\n            this._setChildSettings();\n\n            // Setup/configure values to inputs\n            this._configureForValues();\n\n            $(this.element).trigger('configurable.initialized');\n            $(this.options.qtyInfo).on('input', this._reloadPrice.bind(this));\n        },\n\n        /**\n         * Initialize tax configuration, initial settings, and options values.\n         * @private\n         */\n        _initializeOptions: function () {\n            var options = this.options,\n                gallery = $(options.mediaGallerySelector),\n                priceBoxOptions = $(this.options.priceHolderSelector).priceBox('option').priceConfig || null;\n\n            if (priceBoxOptions && priceBoxOptions.optionTemplate) {\n                options.optionTemplate = priceBoxOptions.optionTemplate;\n            }\n\n            if (priceBoxOptions && priceBoxOptions.priceFormat) {\n                options.priceFormat = priceBoxOptions.priceFormat;\n            }\n            options.optionTemplate = mageTemplate(options.optionTemplate);\n            options.tierPriceTemplate = $(this.options.tierPriceTemplateSelector).html();\n\n            options.settings = options.spConfig.containerId ?\n                $(options.spConfig.containerId).find(options.superSelector) :\n                $(options.superSelector);\n\n            options.values = options.spConfig.defaultValues || {};\n            options.parentImage = $('[data-role=base-image-container] img').attr('src');\n\n            this.inputSimpleProduct = this.element.find(options.selectSimpleProduct);\n\n            gallery.data('gallery') ?\n                this._onGalleryLoaded(gallery) :\n                gallery.on('gallery:loaded', this._onGalleryLoaded.bind(this, gallery));\n\n        },\n\n        /**\n         * Override default options values settings with either URL query parameters or\n         * initialized inputs values.\n         * @private\n         */\n        _overrideDefaults: function () {\n            var hashIndex = window.location.href.indexOf('#');\n\n            if (hashIndex !== -1) {\n                this._parseQueryParams(window.location.href.substr(hashIndex + 1));\n            }\n\n            if (this.options.spConfig.inputsInitialized) {\n                this._setValuesByAttribute();\n            }\n\n            this._setInitialOptionsLabels();\n        },\n\n        /**\n         * Parse query parameters from a query string and set options values based on the\n         * key value pairs of the parameters.\n         * @param {*} queryString - URL query string containing query parameters.\n         * @private\n         */\n        _parseQueryParams: function (queryString) {\n            var queryParams = $.parseQuery({\n                query: queryString\n            });\n\n            $.each(queryParams, $.proxy(function (key, value) {\n                if (this.options.spConfig.attributes[key] !== undefined &&\n                    _.find(this.options.spConfig.attributes[key].options, function (element) {\n                        return element.id === value;\n                    })) {\n                    this.options.values[key] = value;\n                }\n            }, this));\n        },\n\n        /**\n         * Override default options values with values based on each element's attribute\n         * identifier.\n         * @private\n         */\n        _setValuesByAttribute: function () {\n            this.options.values = {};\n            $.each(this.options.settings, $.proxy(function (index, element) {\n                var attributeId;\n\n                if (element.value) {\n                    attributeId = element.id.replace(/[a-z]*/, '');\n\n                    if (this.options.spConfig.attributes[attributeId] !== undefined &&\n                        _.find(this.options.spConfig.attributes[attributeId].options, function (optionElement) {\n                            return optionElement.id === element.value;\n                        })) {\n                        this.options.values[attributeId] = element.value;\n                    }\n                }\n            }, this));\n        },\n\n        /**\n         * Set additional field with initial label to be used when switching between options with different prices.\n         * @private\n         */\n        _setInitialOptionsLabels: function () {\n            $.each(this.options.spConfig.attributes, $.proxy(function (index, element) {\n                $.each(element.options, $.proxy(function (optIndex, optElement) {\n                    this.options.spConfig.attributes[index].options[optIndex].initialLabel = optElement.label;\n                }, this));\n            }, this));\n        },\n\n        /**\n         * Set up .on('change') events for each option element to configure the option.\n         * @private\n         */\n        _setupChangeEvents: function () {\n            $.each(this.options.settings, $.proxy(function (index, element) {\n                $(element).on('change', this, this._configure);\n            }, this));\n        },\n\n        /**\n         * Iterate through the option settings and set each option's element configuration,\n         * attribute identifier. Set the state based on the attribute identifier.\n         * @private\n         */\n        _fillState: function () {\n            $.each(this.options.settings, $.proxy(function (index, element) {\n                var attributeId = element.id.replace(/[a-z]*/, '');\n\n                if (attributeId && this.options.spConfig.attributes[attributeId]) {\n                    element.config = this.options.spConfig.attributes[attributeId];\n                    element.attributeId = attributeId;\n                    this.options.state[attributeId] = false;\n                }\n            }, this));\n        },\n\n        /**\n         * Set each option's child settings, and next/prev option setting. Fill (initialize)\n         * an option's list of selections as needed or disable an option's setting.\n         * @private\n         */\n        _setChildSettings: function () {\n            var childSettings = [],\n                settings = this.options.settings,\n                index = settings.length,\n                option;\n\n            while (index--) {\n                option = settings[index];\n\n                if (index) {\n                    option.disabled = true;\n                } else {\n                    this._fillSelect(option);\n                }\n\n                _.extend(option, {\n                    childSettings: childSettings.slice(),\n                    prevSetting: settings[index - 1],\n                    nextSetting: settings[index + 1]\n                });\n\n                childSettings.push(option);\n            }\n        },\n\n        /**\n         * Setup for all configurable option settings. Set the value of the option and configure\n         * the option, which sets its state, and initializes the option's choices, etc.\n         * @private\n         */\n        _configureForValues: function () {\n            if (this.options.values) {\n                this.options.settings.each($.proxy(function (index, element) {\n                    var attributeId = element.attributeId;\n\n                    element.value = this.options.values[attributeId] || '';\n                    this._configureElement(element);\n                }, this));\n            }\n        },\n\n        /**\n         * Event handler for configuring an option.\n         * @private\n         * @param {Object} event - Event triggered to configure an option.\n         */\n        _configure: function (event) {\n            event.data._configureElement(this);\n        },\n\n        /**\n         * Configure an option, initializing it's state and enabling related options, which\n         * populates the related option's selection and resets child option selections.\n         * @private\n         * @param {*} element - The element associated with a configurable option.\n         */\n        _configureElement: function (element) {\n            this.simpleProduct = this._getSimpleProductId(element);\n\n            if (element.value) {\n                this.options.state[element.config.id] = element.value;\n\n                if (element.nextSetting) {\n                    element.nextSetting.disabled = false;\n                    this._fillSelect(element.nextSetting);\n                    this._resetChildren(element.nextSetting);\n                } else {\n                    if (!!document.documentMode) { //eslint-disable-line\n                        this.inputSimpleProduct.val(element.options[element.selectedIndex].config.allowedProducts[0]);\n                    } else {\n                        this.inputSimpleProduct.val(element.selectedOptions[0].config.allowedProducts[0]);\n                    }\n                }\n            } else {\n                this._resetChildren(element);\n            }\n\n            this._reloadPrice();\n            this._displayRegularPriceBlock(this.simpleProduct);\n            this._displayTierPriceBlock(this.simpleProduct);\n            this._displayNormalPriceLabel();\n            this._changeProductImage();\n        },\n\n        /**\n         * Change displayed product image according to chosen options of configurable product\n         *\n         * @private\n         */\n        _changeProductImage: function () {\n            var images,\n                initialImages = this.options.mediaGalleryInitial,\n                gallery = $(this.options.mediaGallerySelector).data('gallery');\n\n            if (_.isUndefined(gallery)) {\n                $(this.options.mediaGallerySelector).on('gallery:loaded', function () {\n                    this._changeProductImage();\n                }.bind(this));\n\n                return;\n            }\n\n            images = this.options.spConfig.images[this.simpleProduct];\n\n            if (images) {\n                images = this._sortImages(images);\n\n                if (this.options.gallerySwitchStrategy === 'prepend') {\n                    images = images.concat(initialImages);\n                }\n\n                images = $.extend(true, [], images);\n                images = this._setImageIndex(images);\n\n                gallery.updateData(images);\n                this._addFotoramaVideoEvents(false);\n            } else {\n                gallery.updateData(initialImages);\n                this._addFotoramaVideoEvents(true);\n            }\n        },\n\n        /**\n         * Add video events\n         *\n         * @param {Boolean} isInitial\n         * @private\n         */\n        _addFotoramaVideoEvents: function (isInitial) {\n            if (_.isUndefined($.mage.AddFotoramaVideoEvents)) {\n                return;\n            }\n\n            if (isInitial) {\n                $(this.options.mediaGallerySelector).AddFotoramaVideoEvents();\n\n                return;\n            }\n\n            $(this.options.mediaGallerySelector).AddFotoramaVideoEvents({\n                selectedOption: this.simpleProduct,\n                dataMergeStrategy: this.options.gallerySwitchStrategy\n            });\n        },\n\n        /**\n         * Sorting images array\n         *\n         * @private\n         */\n        _sortImages: function (images) {\n            return _.sortBy(images, function (image) {\n                return image.position;\n            });\n        },\n\n        /**\n         * Set correct indexes for image set.\n         *\n         * @param {Array} images\n         * @private\n         */\n        _setImageIndex: function (images) {\n            var length = images.length,\n                i;\n\n            for (i = 0; length > i; i++) {\n                images[i].i = i + 1;\n            }\n\n            return images;\n        },\n\n        /**\n         * For a given option element, reset all of its selectable options. Clear any selected\n         * index, disable the option choice, and reset the option's state if necessary.\n         * @private\n         * @param {*} element - The element associated with a configurable option.\n         */\n        _resetChildren: function (element) {\n            if (element.childSettings) {\n                _.each(element.childSettings, function (set) {\n                    set.selectedIndex = 0;\n                    set.disabled = true;\n                });\n\n                if (element.config) {\n                    this.options.state[element.config.id] = false;\n                }\n            }\n        },\n\n        /**\n         * Populates an option's selectable choices.\n         * @private\n         * @param {*} element - Element associated with a configurable option.\n         */\n        _fillSelect: function (element) {\n            var attributeId = element.id.replace(/[a-z]*/, ''),\n                options = this._getAttributeOptions(attributeId),\n                prevConfig,\n                index = 1,\n                allowedProducts,\n                allowedProductsByOption,\n                allowedProductsAll,\n                i,\n                j,\n                finalPrice = parseFloat(this.options.spConfig.prices.finalPrice.amount),\n                optionFinalPrice,\n                optionPriceDiff,\n                optionPrices = this.options.spConfig.optionPrices,\n                allowedOptions = [],\n                indexKey,\n                allowedProductMinPrice,\n                allowedProductsAllMinPrice,\n                canDisplayOutOfStockProducts = false,\n                filteredSalableProducts;\n\n            this._clearSelect(element);\n            element.options[0] = new Option('', '');\n            element.options[0].innerHTML = this.options.spConfig.chooseText;\n            prevConfig = false;\n\n            if (element.prevSetting) {\n                prevConfig = element.prevSetting.options[element.prevSetting.selectedIndex];\n            }\n\n            if (options) {\n                for (indexKey in this.options.spConfig.index) {\n                    /* eslint-disable max-depth */\n                    if (this.options.spConfig.index.hasOwnProperty(indexKey)) {\n                        allowedOptions = allowedOptions.concat(_.values(this.options.spConfig.index[indexKey]));\n                    }\n                }\n\n                if (prevConfig) {\n                    allowedProductsByOption = {};\n                    allowedProductsAll = [];\n\n                    for (i = 0; i < options.length; i++) {\n                        /* eslint-disable max-depth */\n                        for (j = 0; j < options[i].products.length; j++) {\n                            // prevConfig.config can be undefined\n                            if (prevConfig.config &&\n                                prevConfig.config.allowedProducts &&\n                                prevConfig.config.allowedProducts.indexOf(options[i].products[j]) > -1) {\n                                if (!allowedProductsByOption[i]) {\n                                    allowedProductsByOption[i] = [];\n                                }\n                                allowedProductsByOption[i].push(options[i].products[j]);\n                                allowedProductsAll.push(options[i].products[j]);\n                            }\n                        }\n                    }\n\n                    if (typeof allowedProductsAll[0] !== 'undefined' &&\n                        typeof optionPrices[allowedProductsAll[0]] !== 'undefined') {\n                        allowedProductsAllMinPrice = this._getAllowedProductWithMinPrice(allowedProductsAll);\n                        finalPrice = parseFloat(optionPrices[allowedProductsAllMinPrice].finalPrice.amount);\n                    }\n                }\n\n                for (i = 0; i < options.length; i++) {\n                    if (prevConfig && typeof allowedProductsByOption[i] === 'undefined') {\n                        continue; //jscs:ignore disallowKeywords\n                    }\n\n                    allowedProducts = prevConfig ? allowedProductsByOption[i] : options[i].products.slice(0);\n                    optionPriceDiff = 0;\n\n                    if (typeof allowedProducts[0] !== 'undefined' &&\n                        typeof optionPrices[allowedProducts[0]] !== 'undefined') {\n                        allowedProductMinPrice = this._getAllowedProductWithMinPrice(allowedProducts);\n                        optionFinalPrice = parseFloat(optionPrices[allowedProductMinPrice].finalPrice.amount);\n                        optionPriceDiff = optionFinalPrice - finalPrice;\n                        options[i].label = options[i].initialLabel;\n\n                        if (optionPriceDiff !== 0) {\n                            options[i].label += ' ' + priceUtils.formatPriceLocale(\n                                optionPriceDiff,\n                                this.options.priceFormat,\n                                true\n                            );\n                        }\n                    }\n\n                    if (allowedProducts.length > 0 || _.include(allowedOptions, options[i].id)) {\n                        options[i].allowedProducts = allowedProducts;\n                        element.options[index] = new Option(this._getOptionLabel(options[i]), options[i].id);\n\n                        if (this.options.spConfig.canDisplayShowOutOfStockStatus) {\n                            filteredSalableProducts = $(this.options.spConfig.salable[attributeId][options[i].id]).\n                            filter(options[i].allowedProducts);\n                            canDisplayOutOfStockProducts = filteredSalableProducts.length === 0;\n                        }\n\n                        if (typeof options[i].price !== 'undefined') {\n                            element.options[index].setAttribute('price', options[i].price);\n                        }\n\n                        if (allowedProducts.length === 0 || canDisplayOutOfStockProducts) {\n                            element.options[index].disabled = true;\n                        }\n\n                        element.options[index].config = options[i];\n                        index++;\n                    }\n\n                    /* eslint-enable max-depth */\n                }\n            }\n        },\n\n        /**\n         * Generate the label associated with a configurable option. This includes the option's\n         * label or value and the option's price.\n         * @private\n         * @param {*} option - A single choice among a group of choices for a configurable option.\n         * @return {String} The option label with option value and price (e.g. Black +1.99)\n         */\n        _getOptionLabel: function (option) {\n            return option.label;\n        },\n\n        /**\n         * Removes an option's selections.\n         * @private\n         * @param {*} element - The element associated with a configurable option.\n         */\n        _clearSelect: function (element) {\n            var i;\n\n            for (i = element.options.length - 1; i >= 0; i--) {\n                element.remove(i);\n            }\n        },\n\n        /**\n         * Retrieve the attribute options associated with a specific attribute Id.\n         * @private\n         * @param {Number} attributeId - The id of the attribute whose configurable options are sought.\n         * @return {Object} Object containing the attribute options.\n         */\n        _getAttributeOptions: function (attributeId) {\n            if (this.options.spConfig.attributes[attributeId]) {\n                return this.options.spConfig.attributes[attributeId].options;\n            }\n        },\n\n        /**\n         * Reload the price of the configurable product incorporating the prices of all of the\n         * configurable product's option selections.\n         */\n        _reloadPrice: function () {\n            $(this.options.priceHolderSelector).trigger('updatePrice', this._getPrices());\n        },\n\n        /**\n         * Get product various prices\n         * @returns {{}}\n         * @private\n         */\n        _getPrices: function () {\n            var prices = {},\n                elements = _.toArray(this.options.settings),\n                allowedProduct;\n\n            _.each(elements, function (element) {\n                var selected = element.options[element.selectedIndex],\n                    config = selected && selected.config,\n                    priceValue = this._calculatePrice({});\n\n                if (config && config.allowedProducts.length === 1) {\n                    priceValue = this._calculatePrice(config);\n                } else if (element.value) {\n                    allowedProduct = this._getAllowedProductWithMinPrice(config.allowedProducts);\n                    priceValue = this._calculatePrice({\n                        'allowedProducts': [\n                            allowedProduct\n                        ]\n                    });\n                }\n\n                if (!_.isEmpty(priceValue)) {\n                    prices.prices = priceValue;\n                }\n            }, this);\n\n            return prices;\n        },\n\n        /**\n         * Get product with minimum price from selected options.\n         *\n         * @param {Array} allowedProducts\n         * @returns {String}\n         * @private\n         */\n        _getAllowedProductWithMinPrice: function (allowedProducts) {\n            var optionPrices = this.options.spConfig.optionPrices,\n                product = {},\n                optionMinPrice, optionFinalPrice;\n\n            _.each(allowedProducts, function (allowedProduct) {\n                optionFinalPrice = parseFloat(optionPrices[allowedProduct].finalPrice.amount);\n\n                if (_.isEmpty(product) || optionFinalPrice < optionMinPrice) {\n                    optionMinPrice = optionFinalPrice;\n                    product = allowedProduct;\n                }\n            }, this);\n\n            return product;\n        },\n\n        /**\n         * Returns prices for configured products\n         *\n         * @param {*} config - Products configuration\n         * @returns {*}\n         * @private\n         */\n        _calculatePrice: function (config) {\n            var displayPrices = $(this.options.priceHolderSelector).priceBox('option').prices,\n                newPrices = this.options.spConfig.optionPrices[_.first(config.allowedProducts)] || {};\n\n            _.each(displayPrices, function (price, code) {\n                displayPrices[code].amount = newPrices[code] ? newPrices[code].amount - displayPrices[code].amount : 0;\n            });\n\n            return displayPrices;\n        },\n\n        /**\n         * Returns Simple product Id\n         *  depending on current selected option.\n         *\n         * @private\n         * @param {HTMLElement} element\n         * @returns {String|undefined}\n         */\n        _getSimpleProductId: function (element) {\n            // TODO: Rewrite algorithm. It should return ID of\n            //        simple product based on selected options.\n            var allOptions = element.config.options,\n                value = element.value,\n                config;\n\n            config = _.filter(allOptions, function (option) {\n                return option.id === value;\n            });\n            config = _.first(config);\n\n            return _.isEmpty(config) ?\n                undefined :\n                _.first(config.allowedProducts);\n\n        },\n\n        /**\n         * Show or hide regular price block\n         *\n         * @param {*} optionId\n         * @private\n         */\n        _displayRegularPriceBlock: function (optionId) {\n            var shouldBeShown = true,\n                $priceBox = this.element.parents(this.options.selectorProduct)\n                    .find(this.options.selectorProductPrice);\n\n            _.each(this.options.settings, function (element) {\n                if (element.value === '') {\n                    shouldBeShown = false;\n                }\n            });\n\n            if (shouldBeShown &&\n                this.options.spConfig.optionPrices[optionId].oldPrice.amount !==\n                this.options.spConfig.optionPrices[optionId].finalPrice.amount\n            ) {\n                $(this.options.slyOldPriceSelector).show();\n            } else {\n                $(this.options.slyOldPriceSelector).hide();\n            }\n\n            $(document).trigger('updateMsrpPriceBlock',\n                [\n                    optionId,\n                    this.options.spConfig.optionPrices,\n                    $priceBox\n                ]\n            );\n        },\n\n        /**\n         * Show or hide normal price label\n         *\n         * @private\n         */\n        _displayNormalPriceLabel: function () {\n            var shouldBeShown = false;\n\n            _.each(this.options.settings, function (element) {\n                if (element.value === '') {\n                    shouldBeShown = true;\n                }\n            });\n\n            if (shouldBeShown) {\n                $(this.options.normalPriceLabelSelector).show();\n            } else {\n                $(this.options.normalPriceLabelSelector).hide();\n            }\n        },\n\n        /**\n         * Callback which fired after gallery gets initialized.\n         *\n         * @param {HTMLElement} element - DOM element associated with gallery.\n         */\n        _onGalleryLoaded: function (element) {\n            var galleryObject = element.data('gallery');\n\n            this.options.mediaGalleryInitial = galleryObject.returnCurrentImages();\n        },\n\n        /**\n         * Show or hide tier price block\n         *\n         * @param {*} optionId\n         * @private\n         */\n        _displayTierPriceBlock: function (optionId) {\n            var tierPrices = typeof optionId != 'undefined' && this.options.spConfig.optionPrices[optionId].tierPrices;\n\n            if (_.isArray(tierPrices) && tierPrices.length > 0) {\n\n                if (this.options.tierPriceTemplate) {\n                    $(this.options.tierPriceBlockSelector).html(\n                        mageTemplate(this.options.tierPriceTemplate, {\n                            'tierPrices': tierPrices,\n                            '$t': $t,\n                            'currencyFormat': this.options.spConfig.currencyFormat,\n                            'priceUtils': priceUtils\n                        })\n                    ).show();\n                }\n            } else {\n                $(this.options.tierPriceBlockSelector).hide();\n            }\n        }\n    });\n\n    return $.mage.configurable;\n});\n","Magento_ConfigurableProduct/js/options-updater.js":"define([\n    'jquery',\n    'underscore',\n    'Magento_Customer/js/customer-data'\n], function ($, _, customerData) {\n    'use strict';\n\n    var selectors = {\n        formSelector: '#product_addtocart_form',\n        productIdSelector: '#product_addtocart_form [name=\"product\"]',\n        itemIdSelector: '#product_addtocart_form [name=\"item\"]'\n    },\n    cartData = customerData.get('cart'),\n    productId = $(selectors.productIdSelector).val(),\n    itemId = $(selectors.itemIdSelector).val(),\n\n    /**\n    * set productOptions according to cart data from customer-data\n    *\n    * @param {Object} data - cart data from customer-data\n    * @returns {Boolean} - whether the new options differ from previous\n    */\n    setProductOptions = function (data) {\n        var changedProductOptions;\n\n        if (!(data && data.items && data.items.length && productId)) {\n            return false;\n        }\n        changedProductOptions = _.find(data.items, function (item) {\n            if (item['item_id'] === itemId) {\n                return item['product_id'] === productId;\n            }\n        });\n        changedProductOptions = changedProductOptions && changedProductOptions.options &&\n            changedProductOptions.options.reduce(function (obj, val) {\n                obj[val['option_id']] = val['option_value'];\n\n                return obj;\n            }, {});\n\n        if (JSON.stringify(this.productOptions || {}) === JSON.stringify(changedProductOptions || {})) {\n            return false;\n        }\n\n        this.productOptions = changedProductOptions;\n\n        return true;\n    },\n\n    /**\n    * Listens to update of cart data or options initialization and update selected option according to customer data\n    *\n    */\n    listen = function () {\n        cartData.subscribe(function (updateCartData) {\n            if (this.setProductOptions(updateCartData)) {\n                this.updateOptions();\n            }\n        }.bind(this));\n        $(selectors.formSelector).on(this.eventName, function () {\n            this.setProductOptions(cartData());\n            this.updateOptions();\n        }.bind(this));\n    },\n\n    /**\n    * Updater constructor function\n    *\n    */\n    Updater = function (eventName, updateOptionsCallback) {\n        if (this instanceof Updater) {\n            this.eventName = eventName;\n            this.updateOptions = updateOptionsCallback;\n            this.productOptions = {};\n        }\n    };\n\n    Updater.prototype.setProductOptions = setProductOptions;\n    Updater.prototype.listen = listen;\n\n    return Updater;\n});\n","Magento_ConfigurableProduct/js/product/view/product-info-resolver.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'Magento_Catalog/js/product/view/product-info'\n], function (_, productInfo) {\n    'use strict';\n\n    /**\n     * Returns info about configurable products in form.\n     *\n     * @param {jQuery} $form\n     * @return {Array}\n     */\n    return function ($form) {\n        var optionValues = [],\n            product = _.findWhere($form.serializeArray(), {\n                name: 'product'\n            }),\n            productId;\n\n        if (!_.isUndefined(product)) {\n            productId = product.value;\n            _.each($form.serializeArray(), function (item) {\n                if (item.name.indexOf('super_attribute') !== -1) {\n                    optionValues.push(item.value);\n                }\n            });\n            optionValues.sort();\n            productInfo().push(\n                {\n                    'id': productId,\n                    'optionValues': optionValues\n                }\n            );\n        }\n\n        return _.uniq(productInfo(), function (item) {\n            var optionValuesStr = item.optionValues ? item.optionValues.join() : '';\n\n            return item.id + optionValuesStr;\n        });\n    };\n});\n\n","Magento_Cookie/js/notices.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget',\n    'mage/cookies'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.cookieNotices', {\n        /** @inheritdoc */\n        _create: function () {\n            if ($.mage.cookies.get(this.options.cookieName)) {\n                this.element.hide();\n            } else {\n                this.element.show();\n            }\n            $(this.options.cookieAllowButtonSelector).on('click', $.proxy(function () {\n                var cookieExpires = new Date(new Date().getTime() + this.options.cookieLifetime * 1000);\n\n                $.mage.cookies.set(this.options.cookieName, JSON.stringify(this.options.cookieValue), {\n                    expires: cookieExpires\n                });\n\n                if ($.mage.cookies.get(this.options.cookieName)) {\n                    this.element.hide();\n                    $(document).trigger('user:allowed:save:cookie');\n                } else {\n                    window.location.href = this.options.noCookiesUrl;\n                }\n            }, this));\n        }\n    });\n\n    return $.mage.cookieNotices;\n});\n","Magento_Cookie/js/require-cookie.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/alert',\n    'jquery-ui-modules/widget',\n    'mage/mage',\n    'mage/translate'\n], function ($, alert) {\n    'use strict';\n\n    $.widget('mage.requireCookie', {\n        options: {\n            event: 'click',\n            noCookieUrl: 'enable-cookies',\n            triggers: ['.action.login', '.action.submit'],\n            isRedirectCmsPage: true\n        },\n\n        /**\n         * Constructor\n         * @private\n         */\n        _create: function () {\n            this._bind();\n        },\n\n        /**\n         * This method binds elements found in this widget.\n         * @private\n         */\n        _bind: function () {\n            var events = {};\n\n            $.each(this.options.triggers, function (index, value) {\n                events['click ' + value] = '_checkCookie';\n            });\n            this._on(events);\n        },\n\n        /**\n         * This method set the url for the redirect.\n         * @param {jQuery.Event} event\n         * @private\n         */\n        _checkCookie: function (event) {\n            if (navigator.cookieEnabled) {\n                return;\n            }\n\n            event.preventDefault();\n\n            if (this.options.isRedirectCmsPage) {\n                window.location = this.options.noCookieUrl;\n            } else {\n                alert({\n                    content: $.mage.__('Cookies are disabled in your browser.')\n                });\n            }\n        }\n    });\n\n    return $.mage.requireCookie;\n});\n","Magento_Customer/js/address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/confirm',\n    'jquery-ui-modules/widget',\n    'mage/translate'\n], function ($, confirm) {\n    'use strict';\n\n    $.widget('mage.address', {\n        /**\n         * Options common to all instances of this widget.\n         * @type {Object}\n         */\n        options: {\n            deleteConfirmMessage: $.mage.__('Are you sure you want to delete this address?')\n        },\n\n        /**\n         * Bind event handlers for adding and deleting addresses.\n         * @private\n         */\n        _create: function () {\n            var options         = this.options,\n                addAddress      = options.addAddress,\n                deleteAddress   = options.deleteAddress;\n\n            if (addAddress) {\n                $(document).on('click', addAddress, this._addAddress.bind(this));\n            }\n\n            if (deleteAddress) {\n                $(document).on('click', deleteAddress, this._deleteAddress.bind(this));\n            }\n        },\n\n        /**\n         * Add a new address.\n         * @private\n         */\n        _addAddress: function () {\n            window.location = this.options.addAddressLocation;\n        },\n\n        /**\n         * Delete the address whose id is specified in a data attribute after confirmation from the user.\n         * @private\n         * @param {jQuery.Event} e\n         * @return {Boolean}\n         */\n        _deleteAddress: function (e) {\n            var self = this;\n\n            confirm({\n                content: this.options.deleteConfirmMessage,\n                actions: {\n\n                    /** @inheritdoc */\n                    confirm: function () {\n                        if (typeof $(e.target).parent().data('address') !== 'undefined') {\n                            window.location = self.options.deleteUrlPrefix + $(e.target).parent().data('address') +\n                                '/form_key/' + $.mage.cookies.get('form_key');\n                        } else {\n                            window.location = self.options.deleteUrlPrefix + $(e.target).data('address') +\n                                '/form_key/' + $.mage.cookies.get('form_key');\n                        }\n                    }\n                }\n            });\n\n            return false;\n        }\n    });\n\n    return $.mage.address;\n});\n","Magento_Customer/js/addressValidation.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'mageUtils',\n    'mage/translate',\n    'Magento_Checkout/js/model/postcode-validator',\n    'jquery-ui-modules/widget',\n    'validation'\n], function ($, __, utils, $t, postCodeValidator) {\n    'use strict';\n\n    $.widget('mage.addressValidation', {\n        options: {\n            selectors: {\n                button: '[data-action=save-address]',\n                zip: '#zip',\n                country: 'select[name=\"country_id\"]:visible'\n            }\n        },\n\n        zipInput: null,\n        countrySelect: null,\n\n        /**\n         * Validation creation\n         *\n         * @protected\n         */\n        _create: function () {\n            var button = $(this.options.selectors.button, this.element);\n\n            this.zipInput = $(this.options.selectors.zip, this.element);\n            this.countrySelect = $(this.options.selectors.country, this.element);\n\n            this.element.validation({\n\n                /**\n                 * Submit Handler\n                 * @param {Element} form - address form\n                 */\n                submitHandler: function (form) {\n\n                    button.attr('disabled', true);\n                    form.submit();\n                }\n            });\n\n            this._addPostCodeValidation();\n        },\n\n        /**\n         * Add postcode validation\n         *\n         * @protected\n         */\n        _addPostCodeValidation: function () {\n            var self = this;\n\n            this.zipInput.on('keyup', __.debounce(function (event) {\n                    var valid = self._validatePostCode(event.target.value);\n\n                    self._renderValidationResult(valid);\n                }, 500)\n            );\n\n            this.countrySelect.on('change', function () {\n                var valid = self._validatePostCode(self.zipInput.val());\n\n                self._renderValidationResult(valid);\n            });\n        },\n\n        /**\n         * Validate post code value.\n         *\n         * @protected\n         * @param {String} postCode - post code\n         * @return {Boolean} Whether is post code valid\n         */\n        _validatePostCode: function (postCode) {\n            var countryId = this.countrySelect.val();\n\n            if (postCode === null) {\n                return true;\n            }\n\n            return postCodeValidator.validate(postCode, countryId, this.options.postCodes);\n        },\n\n        /**\n         * Renders warning messages for invalid post code.\n         *\n         * @protected\n         * @param {Boolean} valid\n         */\n        _renderValidationResult: function (valid) {\n            var warnMessage,\n                alertDiv = this.zipInput.next();\n\n            if (!valid) {\n                warnMessage = $t('Provided Zip/Postal Code seems to be invalid.');\n\n                if (postCodeValidator.validatedPostCodeExample.length) {\n                    warnMessage += $t(' Example: ') + postCodeValidator.validatedPostCodeExample.join('; ') + '. ';\n                }\n                warnMessage += $t('If you believe it is the right one you can ignore this notice.');\n            }\n\n            alertDiv.children(':first').text(warnMessage);\n\n            if (valid) {\n                alertDiv.hide();\n            } else {\n                alertDiv.show();\n            }\n        }\n    });\n\n    return $.mage.addressValidation;\n});\n","Magento_Customer/js/block-submit-on-send.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/mage'\n], function ($) {\n    'use strict';\n\n    return function (config) {\n        var dataForm = $('#' + config.formId);\n\n        dataForm.on('submit', function () {\n            $(this).find(':submit').attr('disabled', 'disabled');\n\n            if (this.isValid === false) {\n                $(this).find(':submit').prop('disabled', false);\n            }\n            this.isValid = true;\n        });\n        dataForm.on('invalid-form.validate', function () {\n            $(this).find(':submit').prop('disabled', false);\n            this.isValid = false;\n        });\n    };\n});\n","Magento_Customer/js/change-email-password.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.changeEmailPassword', {\n        options: {\n            changeEmailSelector: '[data-role=change-email]',\n            changePasswordSelector: '[data-role=change-password]',\n            mainContainerSelector: '[data-container=change-email-password]',\n            titleSelector: '[data-title=change-email-password]',\n            emailContainerSelector: '[data-container=change-email]',\n            newPasswordContainerSelector: '[data-container=new-password]',\n            confirmPasswordContainerSelector: '[data-container=confirm-password]',\n            currentPasswordSelector: '[data-input=current-password]',\n            emailSelector: '[data-input=change-email]',\n            newPasswordSelector: '[data-input=new-password]',\n            confirmPasswordSelector: '[data-input=confirm-password]'\n        },\n\n        /**\n         * Create widget\n         * @private\n         */\n        _create: function () {\n            this.element.on('change', $.proxy(function () {\n                this._checkChoice();\n            }, this));\n\n            this._checkChoice();\n            this._bind();\n        },\n\n        /**\n         * Event binding, will monitor change, keyup and paste events.\n         * @private\n         */\n        _bind: function () {\n            this._on($(this.options.emailSelector), {\n                'change': this._updatePasswordFieldWithEmailValue,\n                'keyup': this._updatePasswordFieldWithEmailValue,\n                'paste': this._updatePasswordFieldWithEmailValue\n            });\n        },\n\n        /**\n         * Check choice\n         * @private\n         */\n        _checkChoice: function () {\n            if ($(this.options.changeEmailSelector).is(':checked') &&\n                $(this.options.changePasswordSelector).is(':checked')) {\n                this._showAll();\n            } else if ($(this.options.changeEmailSelector).is(':checked')) {\n                this._showEmail();\n            } else if ($(this.options.changePasswordSelector).is(':checked')) {\n                this._showPassword();\n            } else {\n                this._hideAll();\n            }\n        },\n\n        /**\n         * Show email and password input fields\n         * @private\n         */\n        _showAll: function () {\n            $(this.options.titleSelector).html(this.options.titleChangeEmailAndPassword);\n\n            $(this.options.mainContainerSelector).show();\n            $(this.options.emailContainerSelector).show();\n            $(this.options.newPasswordContainerSelector).show();\n            $(this.options.confirmPasswordContainerSelector).show();\n\n            $(this.options.currentPasswordSelector).attr('data-validate', '{required:true}').prop('disabled', false);\n            $(this.options.emailSelector).attr('data-validate', '{required:true}').prop('disabled', false);\n            this._updatePasswordFieldWithEmailValue();\n            $(this.options.confirmPasswordSelector).attr(\n                'data-validate',\n                '{required:true, equalTo:\"' + this.options.newPasswordSelector + '\"}'\n            ).prop('disabled', false);\n        },\n\n        /**\n         * Hide email and password input fields\n         * @private\n         */\n        _hideAll: function () {\n            $(this.options.mainContainerSelector).hide();\n            $(this.options.emailContainerSelector).hide();\n            $(this.options.newPasswordContainerSelector).hide();\n            $(this.options.confirmPasswordContainerSelector).hide();\n\n            $(this.options.currentPasswordSelector).removeAttr('data-validate').prop('disabled', true);\n            $(this.options.emailSelector).removeAttr('data-validate').prop('disabled', true);\n            $(this.options.newPasswordSelector).removeAttr('data-validate').prop('disabled', true);\n            $(this.options.confirmPasswordSelector).removeAttr('data-validate').prop('disabled', true);\n        },\n\n        /**\n         * Show email input fields\n         * @private\n         */\n        _showEmail: function () {\n            this._showAll();\n            $(this.options.titleSelector).html(this.options.titleChangeEmail);\n\n            $(this.options.newPasswordContainerSelector).hide();\n            $(this.options.confirmPasswordContainerSelector).hide();\n\n            $(this.options.newPasswordSelector).removeAttr('data-validate').prop('disabled', true);\n            $(this.options.confirmPasswordSelector).removeAttr('data-validate').prop('disabled', true);\n        },\n\n        /**\n         * Show password input fields\n         * @private\n         */\n        _showPassword: function () {\n            this._showAll();\n            $(this.options.titleSelector).html(this.options.titleChangePassword);\n\n            $(this.options.emailContainerSelector).hide();\n\n            $(this.options.emailSelector).removeAttr('data-validate').prop('disabled', true);\n        },\n\n        /**\n         * Update password validation rules with email input field value\n         * @private\n         */\n        _updatePasswordFieldWithEmailValue: function () {\n            $(this.options.newPasswordSelector).attr(\n                'data-validate',\n                '{required:true, ' +\n                '\\'validate-customer-password\\':true, ' +\n                '\\'password-not-equal-to-user-name\\':\\'' + $(this.options.emailSelector).val() + '\\'}'\n            ).prop('disabled', false);\n        }\n    });\n\n    return $.mage.changeEmailPassword;\n});\n","Magento_Customer/js/checkout-balance.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.checkoutBalance', {\n        /**\n         * Initialize store credit events\n         * @private\n         */\n        _create: function () {\n            this.eventData = {\n                price: this.options.balance,\n                totalPrice: 0\n            };\n            this.element.on('change', $.proxy(function (e) {\n                if ($(e.target).is(':checked')) {\n                    this.eventData.price = -1 * this.options.balance;\n                } else {\n                    if (this.options.amountSubstracted) { //eslint-disable-line no-lonely-if\n                        this.eventData.price = parseFloat(this.options.usedAmount);\n                        this.options.amountSubstracted = false;\n                    } else {\n                        this.eventData.price = parseFloat(this.options.balance);\n                    }\n                }\n                this.element.trigger('updateCheckoutPrice', this.eventData);\n            }, this));\n        }\n    });\n\n    return $.mage.checkoutBalance;\n});\n","Magento_Customer/js/customer-data.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'underscore',\n    'ko',\n    'Magento_Customer/js/section-config',\n    'mage/url',\n    'mage/storage',\n    'jquery/jquery-storageapi'\n], function ($, _, ko, sectionConfig, url) {\n    'use strict';\n\n    var options = {},\n        storage,\n        storageInvalidation,\n        invalidateCacheBySessionTimeOut,\n        invalidateCacheByCloseCookieSession,\n        dataProvider,\n        buffer,\n        customerData,\n        deferred = $.Deferred();\n\n    url.setBaseUrl(window.BASE_URL);\n    options.sectionLoadUrl = url.build('customer/section/load');\n\n    /**\n     * @param {Object} invalidateOptions\n     */\n    invalidateCacheBySessionTimeOut = function (invalidateOptions) {\n        var date;\n\n        if (new Date($.localStorage.get('mage-cache-timeout')) < new Date()) {\n            storage.removeAll();\n        }\n        date = new Date(Date.now() + parseInt(invalidateOptions.cookieLifeTime, 10) * 1000);\n        $.localStorage.set('mage-cache-timeout', date);\n    };\n\n    /**\n     * Invalidate Cache By Close Cookie Session\n     */\n    invalidateCacheByCloseCookieSession = function () {\n        if (!$.cookieStorage.isSet('mage-cache-sessid')) {\n            storage.removeAll();\n        }\n\n        $.cookieStorage.set('mage-cache-sessid', true);\n    };\n\n    dataProvider = {\n\n        /**\n         * @param {Object} sectionNames\n         * @return {Object}\n         */\n        getFromStorage: function (sectionNames) {\n            var result = {};\n\n            _.each(sectionNames, function (sectionName) {\n                result[sectionName] = storage.get(sectionName);\n            });\n\n            return result;\n        },\n\n        /**\n         * @param {Object} sectionNames\n         * @param {Boolean} forceNewSectionTimestamp\n         * @return {*}\n         */\n        getFromServer: function (sectionNames, forceNewSectionTimestamp) {\n            var parameters;\n\n            sectionNames = sectionConfig.filterClientSideSections(sectionNames);\n            parameters = _.isArray(sectionNames) && sectionNames.indexOf('*') < 0 ? {\n                sections: sectionNames.join(',')\n            } : [];\n            parameters['force_new_section_timestamp'] = forceNewSectionTimestamp;\n\n            return $.getJSON(options.sectionLoadUrl, parameters).fail(function (jqXHR) {\n                throw new Error(jqXHR);\n            });\n        }\n    };\n\n    /**\n     * @param {Function} target\n     * @param {String} sectionName\n     * @return {*}\n     */\n    ko.extenders.disposableCustomerData = function (target, sectionName) {\n        var sectionDataIds, newSectionDataIds = {};\n\n        target.subscribe(function () {\n            setTimeout(function () {\n                storage.remove(sectionName);\n                sectionDataIds = $.cookieStorage.get('section_data_ids') || {};\n                _.each(sectionDataIds, function (data, name) {\n                    if (name !== sectionName) {\n                        newSectionDataIds[name] = data;\n                    }\n                });\n                $.cookieStorage.set('section_data_ids', newSectionDataIds);\n            }, 3000);\n        });\n\n        return target;\n    };\n\n    buffer = {\n        data: {},\n\n        /**\n         * @param {String} sectionName\n         */\n        bind: function (sectionName) {\n            this.data[sectionName] = ko.observable({});\n        },\n\n        /**\n         * @param {String} sectionName\n         * @return {Object}\n         */\n        get: function (sectionName) {\n            if (!this.data[sectionName]) {\n                this.bind(sectionName);\n            }\n\n            return this.data[sectionName];\n        },\n\n        /**\n         * @return {Array}\n         */\n        keys: function () {\n            return _.keys(this.data);\n        },\n\n        /**\n         * @param {String} sectionName\n         * @param {Object} sectionData\n         */\n        notify: function (sectionName, sectionData) {\n            if (!this.data[sectionName]) {\n                this.bind(sectionName);\n            }\n            this.data[sectionName](sectionData);\n        },\n\n        /**\n         * @param {Object} sections\n         */\n        update: function (sections) {\n            var sectionId = 0,\n                sectionDataIds = $.cookieStorage.get('section_data_ids') || {};\n\n            _.each(sections, function (sectionData, sectionName) {\n                sectionId = sectionData['data_id'];\n                sectionDataIds[sectionName] = sectionId;\n                storage.set(sectionName, sectionData);\n                storageInvalidation.remove(sectionName);\n                buffer.notify(sectionName, sectionData);\n            });\n            $.cookieStorage.set('section_data_ids', sectionDataIds);\n        },\n\n        /**\n         * @param {Object} sections\n         */\n        remove: function (sections) {\n            _.each(sections, function (sectionName) {\n                storage.remove(sectionName);\n\n                if (!sectionConfig.isClientSideSection(sectionName)) {\n                    storageInvalidation.set(sectionName, true);\n                }\n            });\n        }\n    };\n\n    customerData = {\n\n        /**\n         * Customer data initialization\n         */\n        init: function () {\n            var expiredSectionNames = this.getExpiredSectionNames();\n\n            if (expiredSectionNames.length > 0) {\n                _.each(dataProvider.getFromStorage(storage.keys()), function (sectionData, sectionName) {\n                    buffer.notify(sectionName, sectionData);\n                });\n                this.reload(expiredSectionNames, false);\n            } else {\n                _.each(dataProvider.getFromStorage(storage.keys()), function (sectionData, sectionName) {\n                    buffer.notify(sectionName, sectionData);\n                });\n\n                if (!_.isEmpty(storageInvalidation.keys())) {\n                    this.reload(storageInvalidation.keys(), false);\n                }\n            }\n\n            if (!_.isEmpty($.cookieStorage.get('section_data_clean'))) {\n                this.reload(sectionConfig.getSectionNames(), true);\n                $.cookieStorage.set('section_data_clean', '');\n            }\n        },\n\n        /**\n         * Storage init\n         */\n        initStorage: function () {\n            $.cookieStorage.setConf({\n                path: '/',\n                expires: new Date(Date.now() + parseInt(options.cookieLifeTime, 10) * 1000)\n            });\n            storage = $.initNamespaceStorage('mage-cache-storage').localStorage;\n            storageInvalidation = $.initNamespaceStorage('mage-cache-storage-section-invalidation').localStorage;\n        },\n\n        /**\n         * Retrieve the list of sections that has expired since last page reload.\n         *\n         * Sections can expire due to lifetime constraints or due to inconsistent storage information\n         * (validated by cookie data).\n         *\n         * @return {Array}\n         */\n        getExpiredSectionNames: function () {\n            var expiredSectionNames = [],\n                cookieSectionTimestamps = $.cookieStorage.get('section_data_ids') || {},\n                sectionLifetime = options.expirableSectionLifetime * 60,\n                currentTimestamp = Math.floor(Date.now() / 1000),\n                sectionData;\n\n            // process sections that can expire due to lifetime constraints\n            _.each(options.expirableSectionNames, function (sectionName) {\n                sectionData = storage.get(sectionName);\n\n                if (typeof sectionData === 'object' && sectionData['data_id'] + sectionLifetime <= currentTimestamp) {\n                    expiredSectionNames.push(sectionName);\n                }\n            });\n\n            // process sections that can expire due to storage information inconsistency\n            _.each(cookieSectionTimestamps, function (cookieSectionTimestamp, sectionName) {\n                sectionData = storage.get(sectionName);\n\n                if (typeof sectionData === 'undefined' ||\n                    typeof sectionData === 'object' &&\n                    cookieSectionTimestamp !== sectionData['data_id']\n                ) {\n                    expiredSectionNames.push(sectionName);\n                }\n            });\n\n            //remove expired section names of previously installed/enable modules\n            expiredSectionNames = _.intersection(expiredSectionNames, sectionConfig.getSectionNames());\n\n            return _.uniq(expiredSectionNames);\n        },\n\n        /**\n         * Check if some sections have to be reloaded.\n         *\n         * @deprecated Use getExpiredSectionNames instead.\n         *\n         * @return {Boolean}\n         */\n        needReload: function () {\n            var expiredSectionNames = this.getExpiredSectionNames();\n\n            return expiredSectionNames.length > 0;\n        },\n\n        /**\n         * Retrieve the list of expired keys.\n         *\n         * @deprecated Use getExpiredSectionNames instead.\n         *\n         * @return {Array}\n         */\n        getExpiredKeys: function () {\n            return this.getExpiredSectionNames();\n        },\n\n        /**\n         * @param {String} sectionName\n         * @return {*}\n         */\n        get: function (sectionName) {\n            return buffer.get(sectionName);\n        },\n\n        /**\n         * @param {String} sectionName\n         * @param {Object} sectionData\n         */\n        set: function (sectionName, sectionData) {\n            var data = {};\n\n            data[sectionName] = sectionData;\n            buffer.update(data);\n        },\n\n        /**\n         * Avoid using this function directly 'cause of possible performance drawbacks.\n         * Each customer section reload brings new non-cached ajax request.\n         *\n         * @param {Array} sectionNames\n         * @param {Boolean} forceNewSectionTimestamp\n         * @return {*}\n         */\n        reload: function (sectionNames, forceNewSectionTimestamp) {\n            return dataProvider.getFromServer(sectionNames, forceNewSectionTimestamp).done(function (sections) {\n                $(document).trigger('customer-data-reload', [sectionNames]);\n                buffer.update(sections);\n            });\n        },\n\n        /**\n         * @param {Array} sectionNames\n         */\n        invalidate: function (sectionNames) {\n            var sectionDataIds,\n                sectionsNamesForInvalidation;\n\n            sectionsNamesForInvalidation = _.contains(sectionNames, '*') ? sectionConfig.getSectionNames() :\n                sectionNames;\n\n            $(document).trigger('customer-data-invalidate', [sectionsNamesForInvalidation]);\n            buffer.remove(sectionsNamesForInvalidation);\n            sectionDataIds = $.cookieStorage.get('section_data_ids') || {};\n\n            // Invalidate section in cookie (increase version of section with 1000)\n            _.each(sectionsNamesForInvalidation, function (sectionName) {\n                if (!sectionConfig.isClientSideSection(sectionName)) {\n                    sectionDataIds[sectionName] += 1000;\n                }\n            });\n            $.cookieStorage.set('section_data_ids', sectionDataIds);\n        },\n\n        /**\n         * Checks if customer data is initialized.\n         *\n         * @returns {jQuery.Deferred}\n         */\n        getInitCustomerData: function () {\n            return deferred.promise();\n        },\n\n        /**\n         * Reload sections on ajax complete\n         *\n         * @param {Object} jsonResponse\n         * @param {Object} settings\n         */\n        onAjaxComplete: function (jsonResponse, settings) {\n            var sections,\n                redirects;\n\n            if (settings.type.match(/post|put|delete/i)) {\n                sections = sectionConfig.getAffectedSections(settings.url);\n\n                if (sections && sections.length) {\n                    this.invalidate(sections);\n                    redirects = ['redirect', 'backUrl'];\n\n                    if (_.isObject(jsonResponse) && !_.isEmpty(_.pick(jsonResponse, redirects))) { //eslint-disable-line\n                        return;\n                    }\n                    this.reload(sections, true);\n                }\n            }\n        },\n\n        /**\n         * @param {Object} settings\n         * @constructor\n         */\n        'Magento_Customer/js/customer-data': function (settings) {\n            options = settings;\n            customerData.initStorage();\n            invalidateCacheBySessionTimeOut(settings);\n            invalidateCacheByCloseCookieSession();\n            customerData.init();\n            deferred.resolve();\n        }\n    };\n\n    /**\n     * Events listener\n     */\n    $(document).on('ajaxComplete', function (event, xhr, settings) {\n        customerData.onAjaxComplete(xhr.responseJSON, settings);\n    });\n\n    /**\n     * Events listener\n     */\n    $(document).on('submit', function (event) {\n        var sections;\n\n        if (event.target.method.match(/post|put|delete/i)) {\n            sections = sectionConfig.getAffectedSections(event.target.action);\n\n            if (sections) {\n                customerData.invalidate(sections);\n            }\n        }\n    });\n\n    return customerData;\n});\n","Magento_Customer/js/customer-global-session-loader.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Customer/js/customer-data'\n], function ($, customerData) {\n    'use strict';\n\n    return function () {\n        var customer;\n\n        // When the session is available, this customer menu will be available\n        if ($('.customer-menu').length > 0) {\n            customer = customerData.get('customer');\n\n            customerData.getInitCustomerData().done(function () {\n                // Check if the customer data is set in local storage, if not reload data from server\n                if (!customer().firstname) {\n                    customerData.reload([], false);\n                }\n            });\n        }\n    };\n});\n","Magento_Customer/js/invalidation-processor.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'uiElement',\n    'Magento_Customer/js/customer-data'\n], function (_, Element, customerData) {\n    'use strict';\n\n    return Element.extend({\n        /**\n         * Initialize object\n         */\n        initialize: function () {\n            this._super();\n            this.process(customerData);\n        },\n\n        /**\n         * Process all rules in loop, each rule can invalidate some sections in customer data\n         *\n         * @param {Object} customerDataObject\n         */\n        process: function (customerDataObject) {\n            _.each(this.invalidationRules, function (rule, ruleName) {\n                _.each(rule, function (ruleArgs, rulePath) {\n                    require([rulePath], function (Rule) {\n                        var currentRule = new Rule(ruleArgs);\n\n                        if (!_.isFunction(currentRule.process)) {\n                            throw new Error('Rule ' + ruleName + ' should implement invalidationProcessor interface');\n                        }\n                        currentRule.process(customerDataObject);\n                    });\n                });\n            });\n        }\n    });\n});\n","Magento_Customer/js/logout-redirect.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/mage'\n], function ($) {\n    'use strict';\n\n    return function (data) {\n        $($.mage.redirect(data.url, 'assign', 5000));\n    };\n});\n","Magento_Customer/js/password-strength-indicator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Customer/js/zxcvbn',\n    'mage/translate',\n    'mage/validation'\n], function ($, zxcvbn, $t) {\n    'use strict';\n\n    $.widget('mage.passwordStrengthIndicator', {\n        options: {\n            cache: {},\n            passwordSelector: '[type=password]',\n            passwordStrengthMeterSelector: '[data-role=password-strength-meter]',\n            passwordStrengthMeterLabelSelector: '[data-role=password-strength-meter-label]',\n            formSelector: 'form',\n            emailSelector: 'input[type=\"email\"]'\n        },\n\n        /**\n         * Widget initialization\n         * @private\n         */\n        _create: function () {\n            this.options.cache.input = $(this.options.passwordSelector, this.element);\n            this.options.cache.meter = $(this.options.passwordStrengthMeterSelector, this.element);\n            this.options.cache.label = $(this.options.passwordStrengthMeterLabelSelector, this.element);\n\n            // We need to look outside the module for backward compatibility, since someone can already use the module.\n            // @todo Narrow this selector in 2.3 so it doesn't accidentally finds the email field from the\n            // newsletter email field or any other \"email\" field.\n            this.options.cache.email = $(this.options.formSelector).find(this.options.emailSelector);\n            this._bind();\n        },\n\n        /**\n         * Event binding, will monitor change, keyup and paste events.\n         * @private\n         */\n        _bind: function () {\n            this._on(this.options.cache.input, {\n                'change': this._calculateStrength,\n                'keyup': this._calculateStrength,\n                'paste': this._calculateStrength\n            });\n\n            if (this.options.cache.email.length) {\n                this._on(this.options.cache.email, {\n                    'change': this._calculateStrength,\n                    'keyup': this._calculateStrength,\n                    'paste': this._calculateStrength\n                });\n            }\n        },\n\n        /**\n         * Calculate password strength\n         * @private\n         */\n        _calculateStrength: function () {\n            var password = this._getPassword(),\n                isEmpty = password.length === 0,\n                zxcvbnScore,\n                displayScore,\n                isValid;\n\n            // Display score is based on combination of whether password is empty, valid, and zxcvbn strength\n            if (isEmpty) {\n                displayScore = 0;\n            } else {\n                this.options.cache.input.rules('add', {\n                    'password-not-equal-to-user-name': this.options.cache.email.val()\n                });\n\n                // We should only perform this check in case there is an email field on screen\n                if (this.options.cache.email.length &&\n                    password.toLowerCase() === this.options.cache.email.val().toLowerCase()) {\n                    displayScore = 1;\n                } else {\n                    isValid = $.validator.validateSingleElement(this.options.cache.input);\n                    zxcvbnScore = zxcvbn(password).score;\n                    displayScore = isValid && zxcvbnScore > 0 ? zxcvbnScore : 1;\n                }\n            }\n\n            // Update label\n            this._displayStrength(displayScore);\n        },\n\n        /**\n         * Display strength\n         * @param {Number} displayScore\n         * @private\n         */\n        _displayStrength: function (displayScore) {\n            var strengthLabel = '',\n                className;\n\n            switch (displayScore) {\n                case 0:\n                    strengthLabel = $t('No Password');\n                    className = 'password-none';\n                    break;\n\n                case 1:\n                    strengthLabel = $t('Weak');\n                    className = 'password-weak';\n                    break;\n\n                case 2:\n                    strengthLabel = $t('Medium');\n                    className = 'password-medium';\n                    break;\n\n                case 3:\n                    strengthLabel = $t('Strong');\n                    className = 'password-strong';\n                    break;\n\n                case 4:\n                    strengthLabel = $t('Very Strong');\n                    className = 'password-very-strong';\n                    break;\n            }\n\n            this.options.cache.meter\n                .removeClass()\n                .addClass(className);\n            this.options.cache.label.text(strengthLabel);\n        },\n\n        /**\n         * Get password value\n         * @returns {*}\n         * @private\n         */\n        _getPassword: function () {\n            return this.options.cache.input.val();\n        }\n    });\n\n    return $.mage.passwordStrengthIndicator;\n});\n","Magento_Customer/js/section-config.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['underscore'], function (_) {\n    'use strict';\n\n    var baseUrls = [],\n        sections = [],\n        clientSideSections = [],\n        sectionNames = [],\n        canonize;\n\n    /**\n     * @param {String} url\n     * @return {String}\n     */\n    canonize = function (url) {\n        var route = url;\n\n        _.some(baseUrls, function (baseUrl) {\n            route = url.replace(baseUrl, '');\n\n            return route !== url;\n        });\n\n        return route.replace(/^\\/?index.php\\/?/, '').toLowerCase();\n    };\n\n    return {\n        /**\n         * Returns a list of sections which should be invalidated for given URL.\n         * @param {String} url - URL which was requested.\n         * @return {Object} - List of sections to invalidate.\n         */\n        getAffectedSections: function (url) {\n            var route = canonize(url),\n                actions = _.find(sections, function (val, section) {\n                    var matched;\n\n                    // Covers the case where \"*\" works as a glob pattern.\n                    if (section.indexOf('*') >= 0) {\n                        section = section.replace(/\\*/g, '[^/]+') + '$';\n                        matched = route.match(section);\n\n                        return matched && matched[0] === route;\n                    }\n\n                    return route.indexOf(section) === 0;\n                });\n\n            return _.union(_.toArray(actions), sections['*']);\n        },\n\n        /**\n         * Filters the list of given sections to the ones defined as client side.\n         * @param {Object} allSections - List of sections to check.\n         * @return {Object} - List of filtered sections.\n         */\n        filterClientSideSections: function (allSections) {\n            return _.difference(allSections, clientSideSections);\n        },\n\n        /**\n         * Tells if section is defined as client side.\n         * @param {String} sectionName - Name of the section to check.\n         * @return {Boolean}\n         */\n        isClientSideSection: function (sectionName) {\n            return _.contains(clientSideSections, sectionName);\n        },\n\n        /**\n         * Returns array of section names.\n         * @returns {Array}\n         */\n        getSectionNames: function () {\n            return sectionNames;\n        },\n\n        /**\n         * @param {Object} options\n         * @constructor\n         */\n        'Magento_Customer/js/section-config': function (options) {\n            baseUrls = options.baseUrls;\n            sections = options.sections;\n            clientSideSections = options.clientSideSections;\n            sectionNames = options.sectionNames;\n        }\n    };\n});\n","Magento_Customer/js/show-password.js":"/**\n* Copyright \u00a9 Magento, Inc. All rights reserved.\n* See COPYING.txt for license details.\n*/\n\ndefine([\n    'jquery',\n    'uiComponent'\n], function ($, Component) {\n    'use strict';\n\n    return Component.extend({\n        passwordSelector: '',\n        passwordInputType: 'password',\n        textInputType: 'text',\n\n        defaults: {\n            template: 'Magento_Customer/show-password',\n            isPasswordVisible: false\n        },\n\n        /**\n         * @return {Object}\n         */\n        initObservable: function () {\n            this._super()\n                .observe(['isPasswordVisible']);\n\n            this.isPasswordVisible.subscribe(function (isChecked) {\n                this._showPassword(isChecked);\n            }.bind(this));\n\n            return this;\n        },\n\n        /**\n         * Show/Hide password\n         * @private\n         */\n        _showPassword: function (isChecked) {\n            $(this.passwordSelector).attr('type',\n                isChecked ? this.textInputType : this.passwordInputType\n            );\n        }\n    });\n});\n","Magento_Customer/js/validation.js":"define([\n    'jquery',\n    'moment',\n    'mageUtils',\n    'jquery/validate',\n    'validation',\n    'mage/translate'\n], function ($, moment, utils) {\n    'use strict';\n\n    $.validator.addMethod(\n        'validate-date',\n        function (value, element, params) {\n            var dateFormat = utils.normalizeDate(params.dateFormat);\n\n            if (value === '') {\n                return true;\n            }\n\n            return moment(value, dateFormat, true).isValid();\n        },\n        $.mage.__('Invalid date')\n    );\n\n    $.validator.addMethod(\n        'validate-dob',\n        function (value, element, params) {\n            var dateFormat = utils.convertToMomentFormat(params.dateFormat);\n\n            if (value === '') {\n                return true;\n            }\n\n            return moment(value, dateFormat).isBefore(moment());\n        },\n        $.mage.__('The Date of Birth should not be greater than today.')\n    );\n});\n","Magento_Customer/js/action/check-email-availability.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'mage/storage',\n    'Magento_Checkout/js/model/url-builder'\n], function (storage, urlBuilder) {\n    'use strict';\n\n    return function (deferred, email) {\n        return storage.post(\n            urlBuilder.createUrl('/customers/isEmailAvailable', {}),\n            JSON.stringify({\n                customerEmail: email\n            }),\n            false\n        ).done(function (isEmailAvailable) {\n            if (isEmailAvailable) {\n                deferred.resolve();\n            } else {\n                deferred.reject();\n            }\n        }).fail(function () {\n            deferred.reject();\n        });\n    };\n});\n","Magento_Customer/js/action/login.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/storage',\n    'Magento_Ui/js/model/messageList',\n    'Magento_Customer/js/customer-data',\n    'mage/translate'\n], function ($, storage, globalMessageList, customerData, $t) {\n    'use strict';\n\n    var callbacks = [],\n\n        /**\n         * @param {Object} loginData\n         * @param {String} redirectUrl\n         * @param {*} isGlobal\n         * @param {Object} messageContainer\n         */\n        action = function (loginData, redirectUrl, isGlobal, messageContainer) {\n            messageContainer = messageContainer || globalMessageList;\n            let customerLoginUrl = 'customer/ajax/login';\n\n            if (loginData.customerLoginUrl) {\n                customerLoginUrl = loginData.customerLoginUrl;\n                delete loginData.customerLoginUrl;\n            }\n\n            return storage.post(\n                customerLoginUrl,\n                JSON.stringify(loginData),\n                isGlobal\n            ).done(function (response) {\n                if (response.errors) {\n                    messageContainer.addErrorMessage(response);\n                    callbacks.forEach(function (callback) {\n                        callback(loginData);\n                    });\n                } else {\n                    callbacks.forEach(function (callback) {\n                        callback(loginData);\n                    });\n                    customerData.invalidate(['customer']);\n\n                    if (response.redirectUrl) {\n                        window.location.href = response.redirectUrl;\n                    } else if (redirectUrl) {\n                        window.location.href = redirectUrl;\n                    } else {\n                        location.reload();\n                    }\n                }\n            }).fail(function () {\n                messageContainer.addErrorMessage({\n                    'message': $t('Could not authenticate. Please try again later')\n                });\n                callbacks.forEach(function (callback) {\n                    callback(loginData);\n                });\n            });\n        };\n\n    /**\n     * @param {Function} callback\n     */\n    action.registerLoginCallback = function (callback) {\n        callbacks.push(callback);\n    };\n\n    return action;\n});\n","Magento_Customer/js/invalidation-rules/website-rule.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'uiClass'\n], function (Element) {\n    'use strict';\n\n    return Element.extend({\n\n        defaults: {\n            scopeConfig: {}\n        },\n\n        /**\n         * Takes website id from current customer data and compare it with current website id\n         * If customer belongs to another scope, we need to invalidate current section\n         *\n         * @param {Object} customerData\n         */\n        process: function (customerData) {\n            var customer = customerData.get('customer');\n\n            if (this.scopeConfig && customer() &&\n                ~~customer().websiteId !== ~~this.scopeConfig.websiteId && ~~customer().websiteId !== 0) {\n                customerData.reload(['customer']);\n            }\n        }\n    });\n});\n","Magento_Customer/js/model/address-list.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'ko',\n    './customer-addresses'\n], function (ko, defaultProvider) {\n    'use strict';\n\n    return ko.observableArray(defaultProvider.getAddressItems());\n});\n","Magento_Customer/js/model/authentication-popup.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/modal'\n], function ($, modal) {\n    'use strict';\n\n    return {\n        modalWindow: null,\n\n        /**\n         * Create popUp window for provided element\n         *\n         * @param {HTMLElement} element\n         */\n        createPopUp: function (element) {\n            var options = {\n                'type': 'popup',\n                'modalClass': 'popup-authentication',\n                'focus': '[name=username]',\n                'responsive': true,\n                'innerScroll': true,\n                'trigger': '.proceed-to-checkout',\n                'buttons': []\n            };\n\n            this.modalWindow = element;\n            modal(options, $(this.modalWindow));\n        },\n\n        /** Show login popup window */\n        showModal: function () {\n            $(this.modalWindow).modal('openModal').trigger('contentUpdated');\n        }\n    };\n});\n","Magento_Customer/js/model/customer-addresses.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'ko',\n    './customer/address'\n], function ($, ko, Address) {\n    'use strict';\n\n    var isLoggedIn = ko.observable(window.isCustomerLoggedIn);\n\n    return {\n        /**\n         * @return {Array}\n         */\n        getAddressItems: function () {\n            var items = [],\n                customerData = window.customerData;\n\n            if (isLoggedIn()) {\n                if (Object.keys(customerData).length) {\n                    $.each(customerData.addresses, function (key, item) {\n                        items.push(new Address(item));\n                    });\n                }\n            }\n\n            return items;\n        }\n    };\n});\n","Magento_Customer/js/model/customer.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'ko',\n    'underscore',\n    './address-list'\n], function ($, ko, _, addressList) {\n    'use strict';\n\n    var isLoggedIn = ko.observable(window.isCustomerLoggedIn),\n        customerData = {};\n\n    if (isLoggedIn()) {\n        customerData = window.customerData;\n    } else {\n        customerData = {};\n    }\n\n    return {\n        customerData: customerData,\n        customerDetails: {},\n        isLoggedIn: isLoggedIn,\n\n        /**\n         * @param {Boolean} flag\n         */\n        setIsLoggedIn: function (flag) {\n            isLoggedIn(flag);\n        },\n\n        /**\n         * @return {Array}\n         */\n        getBillingAddressList: function () {\n            return addressList();\n        },\n\n        /**\n         * @return {Array}\n         */\n        getShippingAddressList: function () {\n            return addressList();\n        },\n\n        /**\n         * @param {String} fieldName\n         * @param {*} value\n         */\n        setDetails: function (fieldName, value) {\n            if (fieldName) {\n                this.customerDetails[fieldName] = value;\n            }\n        },\n\n        /**\n         * @param {String} fieldName\n         * @return {*}\n         */\n        getDetails: function (fieldName) {\n            if (fieldName) {\n                if (this.customerDetails.hasOwnProperty(fieldName)) {\n                    return this.customerDetails[fieldName];\n                }\n\n                return undefined;\n            }\n\n            return this.customerDetails;\n        },\n\n        /**\n         * @param {Array} address\n         * @return {Number}\n         */\n        addCustomerAddress: function (address) {\n            var fields = [\n                    'customer_id', 'country_id', 'street', 'company', 'telephone', 'fax', 'postcode', 'city',\n                    'firstname', 'lastname', 'middlename', 'prefix', 'suffix', 'vat_id', 'default_billing',\n                    'default_shipping'\n                ],\n                customerAddress = {},\n                hasAddress = 0,\n                existingAddress;\n\n            if (!this.customerData.addresses) {\n                this.customerData.addresses = [];\n            }\n\n            customerAddress = _.pick(address, fields);\n\n            if (address.hasOwnProperty('region_id')) {\n                customerAddress.region = {\n                    'region_id': address['region_id'],\n                    region: address.region\n                };\n            }\n\n            for (existingAddress in this.customerData.addresses) {\n                if (this.customerData.addresses.hasOwnProperty(existingAddress)) {\n                    if (_.isEqual(this.customerData.addresses[existingAddress], customerAddress)) { //eslint-disable-line\n                        hasAddress = existingAddress;\n                        break;\n                    }\n                }\n            }\n\n            if (hasAddress === 0) {\n                return this.customerData.addresses.push(customerAddress) - 1;\n            }\n\n            return hasAddress;\n        },\n\n        /**\n         * @param {*} addressId\n         * @return {Boolean}\n         */\n        setAddressAsDefaultBilling: function (addressId) {\n            if (this.customerData.addresses[addressId]) {\n                this.customerData.addresses[addressId]['default_billing'] = 1;\n\n                return true;\n            }\n\n            return false;\n        },\n\n        /**\n         * @param {*} addressId\n         * @return {Boolean}\n         */\n        setAddressAsDefaultShipping: function (addressId) {\n            if (this.customerData.addresses[addressId]) {\n                this.customerData.addresses[addressId]['default_shipping'] = 1;\n\n                return true;\n            }\n\n            return false;\n        }\n    };\n});\n","Magento_Customer/js/model/customer/address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine(['underscore'], function (_) {\n    'use strict';\n\n    /**\n     * Returns new address object.\n     *\n     * @param {Object} addressData\n     * @return {Object}\n     */\n    return function (addressData) {\n        var regionId;\n\n        if (addressData.region['region_id'] && addressData.region['region_id'] !== '0') {\n            regionId = addressData.region['region_id'] + '';\n        }\n\n        return {\n            customerAddressId: addressData.id,\n            email: addressData.email,\n            countryId: addressData['country_id'],\n            regionId: regionId,\n            regionCode: addressData.region['region_code'],\n            region: addressData.region.region,\n            customerId: addressData['customer_id'],\n            street: addressData.street,\n            company: addressData.company,\n            telephone: addressData.telephone,\n            fax: addressData.fax,\n            postcode: addressData.postcode,\n            city: addressData.city,\n            firstname: addressData.firstname,\n            lastname: addressData.lastname,\n            middlename: addressData.middlename,\n            prefix: addressData.prefix,\n            suffix: addressData.suffix,\n            vatId: addressData['vat_id'],\n            sameAsBilling: addressData['same_as_billing'],\n            saveInAddressBook: addressData['save_in_address_book'],\n            customAttributes: _.toArray(addressData['custom_attributes']).reverse(),\n\n            /**\n             * @return {*}\n             */\n            isDefaultShipping: function () {\n                return addressData['default_shipping'];\n            },\n\n            /**\n             * @return {*}\n             */\n            isDefaultBilling: function () {\n                return addressData['default_billing'];\n            },\n\n            /**\n             * @return {*}\n             */\n            getAddressInline: function () {\n                return addressData.inline;\n            },\n\n            /**\n             * @return {String}\n             */\n            getType: function () {\n                return 'customer-address';\n            },\n\n            /**\n             * @return {String}\n             */\n            getKey: function () {\n                return this.getType() + this.customerAddressId;\n            },\n\n            /**\n             * @return {String}\n             */\n            getCacheKey: function () {\n                return this.getKey();\n            },\n\n            /**\n             * @return {Boolean}\n             */\n            isEditable: function () {\n                return false;\n            },\n\n            /**\n             * @return {Boolean}\n             */\n            canUseForBilling: function () {\n                return true;\n            }\n        };\n    };\n});\n","Magento_Customer/js/view/authentication-popup.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'ko',\n    'Magento_Ui/js/form/form',\n    'Magento_Customer/js/action/login',\n    'Magento_Customer/js/customer-data',\n    'Magento_Customer/js/model/authentication-popup',\n    'mage/translate',\n    'mage/url',\n    'Magento_Ui/js/modal/alert',\n    'mage/validation'\n], function ($, ko, Component, loginAction, customerData, authenticationPopup, $t, url, alert) {\n    'use strict';\n\n    return Component.extend({\n        registerUrl: window.authenticationPopup.customerRegisterUrl,\n        forgotPasswordUrl: window.authenticationPopup.customerForgotPasswordUrl,\n        autocomplete: window.authenticationPopup.autocomplete,\n        modalWindow: null,\n        isLoading: ko.observable(false),\n\n        defaults: {\n            template: 'Magento_Customer/authentication-popup'\n        },\n\n        /**\n         * Init\n         */\n        initialize: function () {\n            var self = this;\n\n            this._super();\n            url.setBaseUrl(window.authenticationPopup.baseUrl);\n            loginAction.registerLoginCallback(function () {\n                self.isLoading(false);\n            });\n        },\n\n        /** Init popup login window */\n        setModalElement: function (element) {\n            if (authenticationPopup.modalWindow == null) {\n                authenticationPopup.createPopUp(element);\n            }\n        },\n\n        /** Is login form enabled for current customer */\n        isActive: function () {\n            var customer = customerData.get('customer');\n\n            return customer() == false; //eslint-disable-line eqeqeq\n        },\n\n        /** Show login popup window */\n        showModal: function () {\n            if (this.modalWindow) {\n                $(this.modalWindow).modal('openModal');\n            } else {\n                alert({\n                    content: $t('Guest checkout is disabled.')\n                });\n            }\n        },\n\n        /**\n         * Provide login action\n         *\n         * @return {Boolean}\n         */\n        login: function (formUiElement, event) {\n            var loginData = {},\n                formElement = $(event.currentTarget),\n                formDataArray = formElement.serializeArray();\n\n            event.stopPropagation();\n            formDataArray.forEach(function (entry) {\n                loginData[entry.name] = entry.value;\n            });\n            loginData['customerLoginUrl'] = window.authenticationPopup.customerLoginUrl;\n            if (formElement.validation() &&\n                formElement.validation('isValid')\n            ) {\n                this.isLoading(true);\n                loginAction(loginData);\n            }\n\n            return false;\n        }\n    });\n});\n","Magento_Customer/js/view/customer.js":"/**\n* Copyright \u00a9 Magento, Inc. All rights reserved.\n* See COPYING.txt for license details.\n*/\n\ndefine([\n    'uiComponent',\n    'Magento_Customer/js/customer-data'\n], function (Component, customerData) {\n    'use strict';\n\n    return Component.extend({\n        /** @inheritdoc */\n        initialize: function () {\n            this._super();\n\n            this.customer = customerData.get('customer');\n        }\n    });\n});\n","Magento_Dhl/js/model/shipping-rates-validation-rules.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        /**\n         * @return {Object}\n         */\n        getRules: function () {\n            return {\n                'postcode': {\n                    'required': true\n                },\n                'country_id': {\n                    'required': true\n                },\n                'city': {\n                    'required': true\n                }\n            };\n        }\n    };\n});\n","Magento_Dhl/js/model/shipping-rates-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mageUtils',\n    'Magento_Dhl/js/model/shipping-rates-validation-rules',\n    'mage/translate'\n], function ($, utils, validationRules, $t) {\n    'use strict';\n\n    return {\n        validationErrors: [],\n\n        /**\n         * @param {Object} address\n         * @return {Boolean}\n         */\n        validate: function (address) {\n            var self = this;\n\n            this.validationErrors = [];\n            $.each(validationRules.getRules(), function (field, rule) {\n                var message;\n\n                if (rule.required && utils.isEmpty(address[field])) {\n                    message = $t('Field ') + field + $t(' is required.');\n\n                    self.validationErrors.push(message);\n                }\n            });\n\n            return !this.validationErrors.length;\n        }\n    };\n});\n","Magento_Dhl/js/view/shipping-rates-validation.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/shipping-rates-validator',\n    'Magento_Checkout/js/model/shipping-rates-validation-rules',\n    'Magento_Dhl/js/model/shipping-rates-validator',\n    'Magento_Dhl/js/model/shipping-rates-validation-rules'\n], function (\n    Component,\n    defaultShippingRatesValidator,\n    defaultShippingRatesValidationRules,\n    dhlShippingRatesValidator,\n    dhlShippingRatesValidationRules\n) {\n    'use strict';\n\n    defaultShippingRatesValidator.registerValidator('dhl', dhlShippingRatesValidator);\n    defaultShippingRatesValidationRules.registerRules('dhl', dhlShippingRatesValidationRules);\n\n    return Component;\n});\n","Magento_Directory/js/region-updater.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/template',\n    'underscore',\n    'jquery-ui-modules/widget',\n    'mage/validation'\n], function ($, mageTemplate, _) {\n    'use strict';\n\n    $.widget('mage.directoryRegionUpdater', {\n        options: {\n            regionTemplate:\n                '<option value=\"<%- data.value %>\" <% if (data.isSelected) { %>selected=\"selected\"<% } %>>' +\n                '<%- data.title %>' +\n                '</option>',\n            isRegionRequired: true,\n            isZipRequired: true,\n            isCountryRequired: true,\n            currentRegion: null,\n            isMultipleCountriesAllowed: true\n        },\n\n        /**\n         *\n         * @private\n         */\n        _create: function () {\n            this._initCountryElement();\n\n            this.currentRegionOption = this.options.currentRegion;\n            this.regionTmpl = mageTemplate(this.options.regionTemplate);\n\n            this._updateRegion(this.element.find('option:selected').val());\n\n            $(this.options.regionListId).on('change', $.proxy(function (e) {\n                this.setOption = false;\n                this.currentRegionOption = $(e.target).val();\n            }, this));\n\n            $(this.options.regionInputId).on('focusout', $.proxy(function () {\n                this.setOption = true;\n            }, this));\n        },\n\n        /**\n         *\n         * @private\n         */\n        _initCountryElement: function () {\n\n            if (this.options.isMultipleCountriesAllowed) {\n                this.element.parents('div.field').show();\n                this.element.on('change', $.proxy(function (e) {\n                    // clear region inputs on country change\n                    $(this.options.regionListId).val('');\n                    $(this.options.regionInputId).val('');\n                    this._updateRegion($(e.target).val());\n                }, this));\n\n                if (this.options.isCountryRequired) {\n                    this.element.addClass('required-entry');\n                    this.element.parents('div.field').addClass('required');\n                }\n            } else {\n                this.element.parents('div.field').hide();\n            }\n        },\n\n        /**\n         * Remove options from dropdown list\n         *\n         * @param {Object} selectElement - jQuery object for dropdown list\n         * @private\n         */\n        _removeSelectOptions: function (selectElement) {\n            selectElement.find('option').each(function (index) {\n                if (index) {\n                    $(this).remove();\n                }\n            });\n        },\n\n        /**\n         * Render dropdown list\n         * @param {Object} selectElement - jQuery object for dropdown list\n         * @param {String} key - region code\n         * @param {Object} value - region object\n         * @private\n         */\n        _renderSelectOption: function (selectElement, key, value) {\n            selectElement.append($.proxy(function () {\n                var name = value.name.replace(/[!\"#$%&'()*+,.\\/:;<=>?@[\\\\\\]^`{|}~]/g, '\\\\$&'),\n                    tmplData,\n                    tmpl;\n\n                if (value.code && $(name).is('span')) {\n                    key = value.code;\n                    value.name = $(name).text();\n                }\n\n                tmplData = {\n                    value: key,\n                    title: value.name,\n                    isSelected: false\n                };\n\n                if (this.options.defaultRegion === key) {\n                    tmplData.isSelected = true;\n                }\n\n                tmpl = this.regionTmpl({\n                    data: tmplData\n                });\n\n                return $(tmpl);\n            }, this));\n        },\n\n        /**\n         * Takes clearError callback function as first option\n         * If no form is passed as option, look up the closest form and call clearError method.\n         * @private\n         */\n        _clearError: function () {\n            var args = ['clearError', this.options.regionListId, this.options.regionInputId, this.options.postcodeId];\n\n            if (this.options.clearError && typeof this.options.clearError === 'function') {\n                this.options.clearError.call(this);\n            } else {\n                if (!this.options.form) {\n                    this.options.form = this.element.closest('form').length ? $(this.element.closest('form')[0]) : null;\n                }\n\n                this.options.form = $(this.options.form);\n\n                this.options.form && this.options.form.data('validator') &&\n                this.options.form.validation.apply(this.options.form, _.compact(args));\n\n                // Clean up errors on region & zip fix\n                $(this.options.regionInputId).removeClass('mage-error').parent().find('[generated]').remove();\n                $(this.options.regionListId).removeClass('mage-error').parent().find('[generated]').remove();\n                $(this.options.postcodeId).removeClass('mage-error').parent().find('[generated]').remove();\n            }\n        },\n\n        /**\n         * Update dropdown list based on the country selected\n         *\n         * @param {String} country - 2 uppercase letter for country code\n         * @private\n         */\n        _updateRegion: function (country) {\n            // Clear validation error messages\n            var regionList = $(this.options.regionListId),\n                regionInput = $(this.options.regionInputId),\n                postcode = $(this.options.postcodeId),\n                label = regionList.parent().siblings('label'),\n                container = regionList.parents('div.field'),\n                regionsEntries,\n                regionId,\n                regionData;\n\n            this._clearError();\n            this._checkRegionRequired(country);\n\n            // Populate state/province dropdown list if available or use input box\n            if (this.options.regionJson[country]) {\n                this._removeSelectOptions(regionList);\n                regionsEntries = _.pairs(this.options.regionJson[country]);\n                $.each(regionsEntries, $.proxy(function (key, value) {\n                    regionData = value[1];\n                    regionId = regionData.id;\n                    this._renderSelectOption(regionList, regionId.toString(), regionData);\n                }, this));\n\n                if (this.currentRegionOption) {\n                    regionList.val(this.currentRegionOption);\n                }\n\n                if (this.setOption) {\n                    regionList.find('option').filter(function () {\n                        return this.text === regionInput.val();\n                    }).attr('selected', true);\n                }\n\n                if (this.options.isRegionRequired) {\n                    regionList.addClass('required-entry').removeAttr('disabled');\n                    container.addClass('required').show();\n                } else {\n                    regionList.removeClass('required-entry validate-select').removeAttr('data-validate');\n                    container.removeClass('required');\n\n                    if (!this.options.optionalRegionAllowed) { //eslint-disable-line max-depth\n                        regionList.hide();\n                        container.hide();\n                    } else {\n                        regionList.removeAttr('disabled').show();\n                    }\n                }\n\n                regionList.show();\n                regionInput.hide();\n                label.attr('for', regionList.attr('id'));\n            } else {\n                this._removeSelectOptions(regionList);\n\n                if (this.options.isRegionRequired) {\n                    regionInput.addClass('required-entry').removeAttr('disabled');\n                    container.addClass('required').show();\n                } else {\n                    if (!this.options.optionalRegionAllowed) { //eslint-disable-line max-depth\n                        regionInput.attr('disabled', 'disabled');\n                        container.hide();\n                    }\n                    container.removeClass('required');\n                    regionInput.removeClass('required-entry');\n                }\n\n                regionList.removeClass('required-entry').prop('disabled', 'disabled').hide();\n                regionInput.show();\n                label.attr('for', regionInput.attr('id'));\n            }\n\n            // If country is in optionalzip list, make postcode input not required\n            if (this.options.isZipRequired) {\n                $.inArray(country, this.options.countriesWithOptionalZip) >= 0 ?\n                    postcode.removeClass('required-entry').closest('.field').removeClass('required') :\n                    postcode.addClass('required-entry').closest('.field').addClass('required');\n            }\n\n            // Add defaultvalue attribute to state/province select element\n            regionList.attr('defaultvalue', this.options.defaultRegion);\n            this.options.form.find('[type=\"submit\"]').removeAttr('disabled').show();\n        },\n\n        /**\n         * Check if the selected country has a mandatory region selection\n         *\n         * @param {String} country - Code of the country - 2 uppercase letter for country code\n         * @private\n         */\n        _checkRegionRequired: function (country) {\n            var self = this;\n\n            this.options.isRegionRequired = false;\n            $.each(this.options.regionJson.config['regions_required'], function (index, elem) {\n                if (elem === country) {\n                    self.options.isRegionRequired = true;\n                }\n            });\n        }\n    });\n\n    return $.mage.directoryRegionUpdater;\n});\n","Magento_Downloadable/js/downloadable.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget',\n    'Magento_Catalog/js/price-box'\n], function ($) {\n    'use strict';\n\n    /**\n     * Downloadable widget\n     */\n    $.widget('mage.downloadable', {\n        options: {\n            priceHolderSelector: '.price-box',\n            linkElement: '',\n            allElements: ''\n        },\n\n        /**\n         * @inheritdoc\n         */\n        _init: function initLinks() {\n            var element = this.element,\n                options = $(this.options.linkElement, element);\n\n            options.trigger('change');\n        },\n\n        /**\n         *  @inheritdoc\n         */\n        _create: function () {\n            var self = this;\n\n            this.element.find(this.options.linkElement).on('change', $.proxy(function () {\n                this._reloadPrice();\n            }, this));\n\n            this.element.find(this.options.allElements).on('change', function () {\n                if (this.checked) {\n                    $('label[for=\"' + this.id + '\"] > span').text($(this).attr('data-checked'));\n                    self.element.find(self.options.linkElement + ':not(:checked)').each(function () {\n                        $(this).trigger('click');\n                    });\n                } else {\n                    $('[for=\"' + this.id + '\"] > span').text($(this).attr('data-notchecked'));\n                    self.element.find(self.options.linkElement + ':checked').each(function () {\n                        $(this).trigger('click');\n                    });\n                }\n            });\n\n            this._reloadPrice();\n        },\n\n        /**\n         * Reload product price with selected link price included\n         * @private\n         */\n        _reloadPrice: function () {\n            var finalPrice = 0,\n                basePrice = 0;\n\n            this.element.find(this.options.linkElement + ':checked').each($.proxy(function (index, element) {\n                finalPrice += this.options.config.links[$(element).val()].finalPrice;\n                basePrice += this.options.config.links[$(element).val()].basePrice;\n            }, this));\n\n            $(this.options.priceHolderSelector).trigger('updatePrice', {\n                'prices': {\n                    'finalPrice': {\n                        'amount': finalPrice\n                    },\n                    'basePrice': {\n                        'amount': basePrice\n                    }\n                }\n            });\n\n            this.reloadAllCheckText();\n        },\n\n        /**\n         * Reload all-elements-checkbox's label\n         * @private\n         */\n        reloadAllCheckText: function () {\n            var allChecked = true,\n                allElementsCheck = $(this.options.allElements),\n                allElementsLabel = $('label[for=\"' + allElementsCheck.attr('id') + '\"] > span');\n\n            $(this.options.linkElement).each(function () {\n                if (!this.checked) {\n                    allChecked = false;\n                }\n            });\n\n            if (allChecked) {\n                allElementsLabel.text(allElementsCheck.attr('data-checked'));\n                allElementsCheck.prop('checked', true);\n            } else {\n                allElementsLabel.text(allElementsCheck.attr('data-notchecked'));\n                allElementsCheck.prop('checked', false);\n            }\n        }\n    });\n\n    return $.mage.downloadable;\n});\n","Magento_Fedex/js/model/shipping-rates-validation-rules.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        /**\n         * @return {Object}\n         */\n        getRules: function () {\n            return {\n                'postcode': {\n                    'required': true\n                },\n                'country_id': {\n                    'required': true\n                },\n                'city': {\n                    'required': true\n                }\n            };\n        }\n    };\n});\n","Magento_Fedex/js/model/shipping-rates-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mageUtils',\n    'Magento_Fedex/js/model/shipping-rates-validation-rules',\n    'mage/translate'\n], function ($, utils, validationRules, $t) {\n    'use strict';\n\n    return {\n        validationErrors: [],\n\n        /**\n         * @param {Object} address\n         * @return {Boolean}\n         */\n        validate: function (address) {\n            var self = this;\n\n            this.validationErrors = [];\n            $.each(validationRules.getRules(), function (field, rule) {\n                var message;\n\n                if (rule.required && utils.isEmpty(address[field])) {\n                    message = $t('Field ') + field + $t(' is required.');\n\n                    self.validationErrors.push(message);\n                }\n            });\n\n            return !this.validationErrors.length;\n        }\n    };\n});\n","Magento_Fedex/js/view/shipping-rates-validation.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/shipping-rates-validator',\n    'Magento_Checkout/js/model/shipping-rates-validation-rules',\n    'Magento_Fedex/js/model/shipping-rates-validator',\n    'Magento_Fedex/js/model/shipping-rates-validation-rules'\n], function (\n    Component,\n    defaultShippingRatesValidator,\n    defaultShippingRatesValidationRules,\n    fedexShippingRatesValidator,\n    fedexShippingRatesValidationRules\n) {\n    'use strict';\n\n    defaultShippingRatesValidator.registerValidator('fedex', fedexShippingRatesValidator);\n    defaultShippingRatesValidationRules.registerRules('fedex', fedexShippingRatesValidationRules);\n\n    return Component;\n});\n","Magento_GiftMessage/js/gift-options.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.giftOptions', {\n        options: {\n            mageError: 'mage-error',\n            noDisplay: 'no-display',\n            requiredEntry: 'required-entry'\n        },\n\n        /**\n         * Initial toggle of the various gift options after widget instantiation.\n         * @private\n         */\n        _init: function () {\n            this._toggleVisibility();\n        },\n\n        /**\n         * Bind a click handler to the widget's context element.\n         * @private\n         */\n        _create: function () {\n            this.element.on('click', $.proxy(this._toggleVisibility, this));\n            $(this.element.data('selector').id).find('.giftmessage-area')\n                .on('change', $.proxy(this._toggleRequired, this));\n        },\n\n        /**\n         * Toggle the visibility of the widget's context element's selector(s).\n         * @private\n         * @param {jQuery.Event} event - Click event. Target is a checkbox.\n         */\n        _toggleVisibility: function (event) {\n            var checkbox = event ? $(event.target) : this.element,\n                container = $(checkbox.data('selector').id),\n                _this;\n\n            if (checkbox.is(':checked')) {\n                container.show()\n                    .find('.giftmessage-area:not(:visible)').each(function (x, element) {\n                        if ($(element).val().length > 0) {\n                            $(element).trigger('change');\n                            container.find('a').trigger('click');\n                        }\n                    });\n            } else {\n                _this = this;\n                container.hide()\n                    .find('.input-text:not(.giftmessage-area)').each(function (x, element) {\n                        $(element).val(element.defaultValue).removeClass(_this.options.mageError)\n                            .next('div.' + _this.options.mageError).remove();\n                    }).end()\n                    .find('.giftmessage-area').val('').change().end()\n                    .find('.select').val('').change().end()\n                    .find('.checkbox:checked').prop('checked', false).trigger('click').prop('checked', false).end()\n                    .find('.price-box').addClass(this.options.noDisplay).end();\n            }\n        },\n\n        /**\n         * Make the From and To input fields required if a gift message has been written.\n         * @private\n         * @param {jQuery.Event} event - Change event. Target is a textarea.\n         */\n        _toggleRequired: function (event) {\n            var textArea = $(event.target),\n                length = textArea.val().length;\n\n            textArea.closest('li').prev('.fields')\n                .find('.input-text').toggleClass(this.options.requiredEntry, length > 0);\n        }\n    });\n\n    return $.mage.giftOptions;\n});\n","Magento_GiftMessage/js/action/gift-options.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'Magento_GiftMessage/js/model/url-builder',\n    'mage/storage',\n    'Magento_Ui/js/model/messageList',\n    'Magento_Checkout/js/model/error-processor',\n    'mage/url',\n    'Magento_Checkout/js/model/quote',\n    'underscore'\n], function (urlBuilder, storage, messageList, errorProcessor, url, quote, _) {\n    'use strict';\n\n    return function (giftMessage, remove) {\n        var serviceUrl;\n\n        url.setBaseUrl(giftMessage.getConfigValue('baseUrl'));\n\n        if (giftMessage.getConfigValue('isCustomerLoggedIn')) {\n            serviceUrl = urlBuilder.createUrl('/carts/mine/gift-message', {});\n\n            if (giftMessage.itemId != 'orderLevel') { //eslint-disable-line eqeqeq\n                serviceUrl = urlBuilder.createUrl('/carts/mine/gift-message/:itemId', {\n                    itemId: giftMessage.itemId\n                });\n            }\n        } else {\n            serviceUrl = urlBuilder.createUrl('/guest-carts/:cartId/gift-message', {\n                cartId: quote.getQuoteId()\n            });\n\n            if (giftMessage.itemId != 'orderLevel') { //eslint-disable-line eqeqeq\n                serviceUrl = urlBuilder.createUrl(\n                    '/guest-carts/:cartId/gift-message/:itemId',\n                    {\n                        cartId: quote.getQuoteId(), itemId: giftMessage.itemId\n                    }\n                );\n            }\n        }\n        messageList.clear();\n\n        storage.post(\n            serviceUrl,\n            JSON.stringify({\n                'gift_message': giftMessage.getSubmitParams(remove)\n            })\n        ).done(function () {\n            giftMessage.reset();\n            _.each(giftMessage.getAfterSubmitCallbacks(), function (callback) {\n                if (_.isFunction(callback)) {\n                    callback();\n                }\n            });\n        }).fail(function (response) {\n            errorProcessor.process(response);\n        });\n    };\n});\n","Magento_GiftMessage/js/model/gift-message.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'uiElement',\n    'underscore',\n    'mage/url'\n], function (uiElement, _, url) {\n    'use strict';\n\n    var provider = uiElement();\n\n    return function (itemId) {\n        var model = {\n            id: 'message-' + itemId,\n            itemId: itemId,\n            observables: {},\n            additionalOptions: [],\n            submitParams: [\n                'recipient',\n                'sender',\n                'message'\n            ],\n\n            /**\n             * Initialize.\n             */\n            initialize: function () {\n                var message = false;\n\n                this.getObservable('alreadyAdded')(false);\n\n                if (this.itemId == 'orderLevel') { //eslint-disable-line eqeqeq\n                    message = window.giftOptionsConfig.giftMessage.hasOwnProperty(this.itemId) ?\n                        window.giftOptionsConfig.giftMessage[this.itemId] :\n                        null;\n                } else {\n                    message =\n                        window.giftOptionsConfig.giftMessage.hasOwnProperty('itemLevel') &&\n                        window.giftOptionsConfig.giftMessage.itemLevel.hasOwnProperty(this.itemId) ?\n                            window.giftOptionsConfig.giftMessage.itemLevel[this.itemId].message :\n                            null;\n                }\n\n                if (_.isObject(message)) {\n                    this.getObservable('recipient')(message.recipient);\n                    this.getObservable('sender')(message.sender);\n                    this.getObservable('message')(message.message);\n                    this.getObservable('alreadyAdded')(true);\n                }\n            },\n\n            /**\n             * @param {String} key\n             * @return {*}\n             */\n            getObservable: function (key) {\n                this.initObservable(this.id, key);\n\n                return provider[this.getUniqueKey(this.id, key)];\n            },\n\n            /**\n             * @param {String} node\n             * @param {String} key\n             */\n            initObservable: function (node, key) {\n                if (node && !this.observables.hasOwnProperty(node)) {\n                    this.observables[node] = [];\n                }\n\n                if (key && this.observables[node].indexOf(key) === -1) {\n                    this.observables[node].push(key);\n                    provider.observe(this.getUniqueKey(node, key));\n                }\n            },\n\n            /**\n             * @param {String} node\n             * @param {String} key\n             * @return {String}\n             */\n            getUniqueKey: function (node, key) {\n                return node + '-' + key;\n            },\n\n            /**\n             * @param {String} key\n             * @return {null}\n             */\n            getConfigValue: function (key) {\n                return window.giftOptionsConfig.hasOwnProperty(key) ?\n                    window.giftOptionsConfig[key]\n                    : null;\n            },\n\n            /**\n             * Reset.\n             */\n            reset: function () {\n                this.getObservable('isClear')(true);\n            },\n\n            /**\n             * @return {Array}\n             */\n            getAfterSubmitCallbacks: function () {\n                var callbacks = [];\n\n                callbacks.push(this.afterSubmit);\n                _.each(this.additionalOptions, function (option) {\n                    if (_.isFunction(option.afterSubmit)) {\n                        callbacks.push(option.afterSubmit);\n                    }\n                });\n\n                return callbacks;\n            },\n\n            /**\n             * After submit.\n             */\n            afterSubmit: function () {\n                window.location.href = url.build('checkout/cart/updatePost') +\n                    '?form_key=' + window.checkoutConfig.formKey +\n                    '&cart[]';\n            },\n\n            /**\n             * @param {Boolean} remove\n             * @return {Object}\n             */\n            getSubmitParams: function (remove) {\n                var params = {},\n                    self = this;\n\n                _.each(this.submitParams, function (key) {\n                    var observable = provider[self.getUniqueKey(self.id, key)];\n\n                    if (_.isFunction(observable)) {\n                        params[key] = remove ? null : observable();\n                    }\n                });\n\n                if (this.additionalOptions.length) {\n                    params['extension_attributes'] = {};\n                }\n                _.each(this.additionalOptions, function (option) {\n                    if (_.isFunction(option.getSubmitParams)) {\n                        params['extension_attributes'] = _.extend(\n                            params['extension_attributes'],\n                            option.getSubmitParams(remove)\n                        );\n                    }\n                });\n\n                return params;\n            },\n\n            /**\n             * Check if gift message can be displayed\n             *\n             * @returns {Boolean}\n             */\n            isGiftMessageAvailable: function () {\n                var isGloballyAvailable,\n                    giftMessageConfig,\n                    itemConfig;\n\n                // itemId represent gift message level: 'orderLevel' constant or cart item ID\n                if (this.itemId === 'orderLevel') {\n                    return this.getConfigValue('isOrderLevelGiftOptionsEnabled');\n                }\n\n                // gift message product configuration must override system configuration\n                isGloballyAvailable = this.getConfigValue('isItemLevelGiftOptionsEnabled');\n                giftMessageConfig = window.giftOptionsConfig.giftMessage;\n                itemConfig = giftMessageConfig.hasOwnProperty('itemLevel') &&\n                    giftMessageConfig.itemLevel.hasOwnProperty(this.itemId) ?\n                    giftMessageConfig.itemLevel[this.itemId] :\n                    {};\n\n                return itemConfig.hasOwnProperty('is_available') ? itemConfig['is_available'] : isGloballyAvailable;\n            }\n        };\n\n        model.initialize();\n\n        return model;\n    };\n});\n","Magento_GiftMessage/js/model/gift-options.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'ko'\n], function (_, ko) {\n    'use strict';\n\n    return {\n        options: ko.observableArray([]),\n\n        /**\n         * @param {Object} option\n         */\n        addOption: function (option) {\n            if (!this.options().hasOwnProperty(option.itemId)) {\n                this.options.push({\n                        id: option.itemId, value: option\n                    }\n                );\n            }\n        },\n\n        /**\n         * @param {*} itemId\n         * @return {*}\n         */\n        getOptionByItemId: function (itemId) {\n            var option = null;\n\n            _.each(this.options(), function (data) {\n                if (data.id === itemId) {\n                    option = data.value;\n\n                    return false;\n                }\n            });\n\n            return option;\n        }\n    };\n});\n","Magento_GiftMessage/js/model/url-builder.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Checkout/js/model/url-builder'\n], function ($, urlBuilder) {\n    'use strict';\n\n    return $.extend(urlBuilder, {\n        storeCode: window.giftOptionsConfig.storeCode\n    });\n});\n","Magento_GiftMessage/js/view/gift-message.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_GiftMessage/js/model/gift-message',\n    'Magento_GiftMessage/js/model/gift-options',\n    'Magento_GiftMessage/js/action/gift-options'\n], function (Component, GiftMessage, giftOptions, giftOptionsService) {\n    'use strict';\n\n    return Component.extend({\n        formBlockVisibility: null,\n        resultBlockVisibility: null,\n        model: {},\n\n        /**\n         * Component init\n         */\n        initialize: function () {\n            var self = this,\n                model;\n\n            this._super()\n                .observe('formBlockVisibility')\n                .observe({\n                    'resultBlockVisibility': false\n                });\n\n            this.itemId = this.itemId || 'orderLevel';\n            model = new GiftMessage(this.itemId);\n            this.model = model;\n            this.isResultBlockVisible();\n            giftOptions.addOption(model);\n\n            this.model.getObservable('isClear').subscribe(function (value) {\n                if (value == true) { //eslint-disable-line eqeqeq\n                    self.formBlockVisibility(false);\n                    self.model.getObservable('alreadyAdded')(true);\n                }\n            });\n        },\n\n        /**\n         * Is reslt block visible\n         */\n        isResultBlockVisible: function () {\n            var self = this;\n\n            if (this.model.getObservable('alreadyAdded')()) {\n                this.resultBlockVisibility(true);\n            }\n            this.model.getObservable('additionalOptionsApplied').subscribe(function (value) {\n                if (value == true) { //eslint-disable-line eqeqeq\n                    self.resultBlockVisibility(true);\n                }\n            });\n        },\n\n        /**\n         * @param {String} key\n         * @return {*}\n         */\n        getObservable: function (key) {\n            return this.model.getObservable(key);\n        },\n\n        /**\n         * Hide\\Show form block\n         */\n        toggleFormBlockVisibility: function () {\n            if (!this.model.getObservable('alreadyAdded')()) {\n                this.formBlockVisibility(!this.formBlockVisibility());\n            } else {\n                this.resultBlockVisibility(!this.resultBlockVisibility());\n            }\n        },\n\n        /**\n         * Edit options\n         */\n        editOptions: function () {\n            this.resultBlockVisibility(false);\n            this.formBlockVisibility(true);\n        },\n\n        /**\n         * Delete options\n         */\n        deleteOptions: function () {\n            giftOptionsService(this.model, true);\n        },\n\n        /**\n         * Hide form block\n         */\n        hideFormBlock: function () {\n            this.formBlockVisibility(false);\n\n            if (this.model.getObservable('alreadyAdded')()) {\n                this.resultBlockVisibility(true);\n            }\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        hasActiveOptions: function () {\n            var regionData = this.getRegion('additionalOptions'),\n                options = regionData(),\n                i;\n\n            for (i = 0; i < options.length; i++) {\n                if (options[i].isActive()) {\n                    return true;\n                }\n            }\n\n            return false;\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        isActive: function () {\n            return this.model.isGiftMessageAvailable();\n        },\n\n        /**\n         * Submit options\n         */\n        submitOptions: function () {\n            giftOptionsService(this.model);\n        }\n    });\n});\n","Magento_GoogleAnalytics/js/google-analytics.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/* jscs:disable */\n/* eslint-disable */\ndefine([\n    'jquery',\n    'mage/cookies'\n], function ($) {\n    'use strict';\n\n    /**\n     * @param {Object} config\n     */\n    return function (config) {\n        var allowServices = false,\n            allowedCookies,\n            allowedWebsites;\n\n        if (config.isCookieRestrictionModeEnabled) {\n            allowedCookies = $.mage.cookies.get(config.cookieName);\n\n            if (allowedCookies !== null) {\n                allowedWebsites = JSON.parse(allowedCookies);\n\n                if (allowedWebsites[config.currentWebsite] === 1) {\n                    allowServices = true;\n                }\n            }\n        } else {\n            allowServices = true;\n        }\n\n        if (allowServices) {\n            (function (i, s, o, g, r, a, m) {\n                i.GoogleAnalyticsObject = r;\n                i[r] = i[r] || function () {\n                        (i[r].q = i[r].q || []).push(arguments)\n                    }, i[r].l = 1 * new Date();\n                a = s.createElement(o),\n                    m = s.getElementsByTagName(o)[0];\n                a.async = 1;\n                a.src = g;\n                m.parentNode.insertBefore(a, m)\n            })(window, document, 'script', '//www.google-analytics.com/analytics.js', 'ga');\n\n            // Process page info\n            ga('create', config.pageTrackingData.accountId, 'auto');\n\n            if (config.pageTrackingData.isAnonymizedIpActive) {\n                ga('set', 'anonymizeIp', true);\n            }\n\n            // Process orders data\n            if (config.ordersTrackingData.hasOwnProperty('currency')) {\n                ga('require', 'ec', 'ec.js');\n\n                ga('set', 'currencyCode', config.ordersTrackingData.currency);\n\n                // Collect product data for GA\n                if (config.ordersTrackingData.products) {\n                    $.each(config.ordersTrackingData.products, function (index, value) {\n                        ga('ec:addProduct', value);\n                    });\n                }\n\n                // Collect orders data for GA\n                if (config.ordersTrackingData.orders) {\n                    $.each(config.ordersTrackingData.orders, function (index, value) {\n                        ga('ec:setAction', 'purchase', value);\n                    });\n                }\n\n                ga('send', 'pageview');\n            } else {\n                // Process Data if not orders\n                ga('send', 'pageview' + config.pageTrackingData.optPageUrl);\n            }\n        }\n    }\n});\n","Magento_GoogleGtag/js/google-adwords.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/* jscs:disable */\n/* eslint-disable */\ndefine([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    /**\n     * @param {Object} config\n     */\n    return function (config) {\n        if (!window.gtag) {\n            // Inject Global Site Tag\n            var gtagScript = document.createElement('script');\n            gtagScript.type = 'text/javascript';\n            gtagScript.async = true;\n            gtagScript.src = config.gtagSiteSrc;\n            document.head.appendChild(gtagScript);\n\n            window.dataLayer = window.dataLayer || [];\n\n            function gtag(){dataLayer.push(arguments);}\n            gtag('js', new Date());\n            gtag('set', 'developer_id.dYjhlMD', true);\n            if (config.conversionLabel) {\n                gtag(\n                    'event',\n                    'conversion',\n                    {'send_to': config.conversionId + '/'\n                            + config.conversionLabel}\n                );\n            }\n        } else {\n            gtag('config', config.conversionId);\n        }\n    }\n});\n","Magento_GoogleGtag/js/google-analytics.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/* jscs:disable */\n/* eslint-disable */\ndefine([\n    'jquery',\n    'mage/cookies'\n], function ($) {\n    'use strict';\n\n    /**\n     * @param {Object} config\n     */\n    return function (config) {\n        var allowServices = false,\n            allowedCookies,\n            allowedWebsites,\n            measurementId;\n\n        if (config.isCookieRestrictionModeEnabled) {\n            allowedCookies = $.mage.cookies.get(config.cookieName);\n\n            if (allowedCookies !== null) {\n                allowedWebsites = JSON.parse(allowedCookies);\n\n                if (allowedWebsites[config.currentWebsite] === 1) {\n                    allowServices = true;\n                }\n            }\n        } else {\n            allowServices = true;\n        }\n\n        if (allowServices) {\n            /* Global site tag (gtag.js) - Google Analytics */\n            measurementId = config.pageTrackingData.measurementId;\n            if (window.gtag) {\n                gtag('config', measurementId, { 'anonymize_ip': true });\n                // Purchase Event\n                if (config.ordersTrackingData.hasOwnProperty('currency')) {\n                    var purchaseObject = config.ordersTrackingData.orders[0];\n                    purchaseObject['items'] = config.ordersTrackingData.products;\n                    gtag('event', 'purchase', purchaseObject);\n                }\n            } else {\n                (function(d,s,u){\n                    var gtagScript = d.createElement(s);\n                    gtagScript.type = 'text/javascript';\n                    gtagScript.async = true;\n                    gtagScript.src = u;\n                    d.head.insertBefore(gtagScript, d.head.children[0]);\n                })(document, 'script', 'https://www.googletagmanager.com/gtag/js?id=' + measurementId);\n                window.dataLayer = window.dataLayer || [];\n                function gtag(){dataLayer.push(arguments);}\n                gtag('js', new Date());\n                gtag('set', 'developer_id.dYjhlMD', true);\n                gtag('config', measurementId, { 'anonymize_ip': true });\n                // Purchase Event\n                if (config.ordersTrackingData.hasOwnProperty('currency')) {\n                    var purchaseObject = config.ordersTrackingData.orders[0];\n                    purchaseObject['items'] = config.ordersTrackingData.products;\n                    gtag('event', 'purchase', purchaseObject);\n                }\n            }\n        }\n    }\n});\n","Magento_GroupedProduct/js/product-ids-resolver.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'Magento_Catalog/js/product/view/product-ids',\n    'Magento_Catalog/js/product/view/product-info'\n], function ($, productIds, productInfo) {\n    'use strict';\n\n    /**\n     * Returns id's of products in form.\n     *\n     * @param {Object} config\n     * @param {HTMLElement} element\n     * @return {Array}\n     */\n    return function (config, element) {\n        $(element).find('div[data-product-id]').each(function () {\n            productIds.push($(this).data('productId').toString());\n            productInfo.push(\n                {\n                    'id': $(this).data('productId').toString()\n                }\n            );\n        });\n\n        return productIds();\n    };\n});\n","Magento_InstantPurchase/js/view/instant-purchase.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'ko',\n    'jquery',\n    'underscore',\n    'uiComponent',\n    'Magento_Ui/js/modal/confirm',\n    'Magento_Customer/js/customer-data',\n    'mage/url',\n    'mage/template',\n    'mage/translate',\n    'text!Magento_InstantPurchase/template/confirmation.html',\n    'mage/validation'\n], function (ko, $, _, Component, confirm, customerData, urlBuilder, mageTemplate, $t, confirmationTemplate) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_InstantPurchase/instant-purchase',\n            buttonText: $t('Instant Purchase'),\n            purchaseUrl: urlBuilder.build('instantpurchase/button/placeOrder'),\n            showButton: false,\n            paymentToken: null,\n            shippingAddress: null,\n            billingAddress: null,\n            shippingMethod: null,\n            productFormSelector: '#product_addtocart_form',\n            confirmationTitle: $t('Instant Purchase Confirmation'),\n            confirmationData: {\n                message: $t('Are you sure you want to place order and pay?'),\n                shippingAddressTitle: $t('Shipping Address'),\n                billingAddressTitle: $t('Billing Address'),\n                paymentMethodTitle: $t('Payment Method'),\n                shippingMethodTitle: $t('Shipping Method')\n            }\n        },\n\n        /** @inheritdoc */\n        initialize: function () {\n            var instantPurchase = customerData.get('instant-purchase');\n\n            this._super();\n\n            this.setPurchaseData(instantPurchase());\n            instantPurchase.subscribe(this.setPurchaseData, this);\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super()\n                .observe('showButton paymentToken shippingAddress billingAddress shippingMethod');\n\n            return this;\n        },\n\n        /**\n         * Set data from customerData.\n         *\n         * @param {Object} data\n         */\n        setPurchaseData: function (data) {\n            this.showButton(data.available);\n            this.paymentToken(data.paymentToken);\n            this.shippingAddress(data.shippingAddress);\n            this.billingAddress(data.billingAddress);\n            this.shippingMethod(data.shippingMethod);\n        },\n\n        /**\n         * Confirmation method\n         */\n        instantPurchase: function () {\n            var form = $(this.productFormSelector),\n                confirmTemplate = mageTemplate(confirmationTemplate),\n                confirmData = _.extend({}, this.confirmationData, {\n                    paymentToken: this.paymentToken().summary,\n                    shippingAddress: this.shippingAddress().summary,\n                    billingAddress: this.billingAddress().summary,\n                    shippingMethod: this.shippingMethod().summary\n                });\n\n            if (!(form.validation() && form.validation('isValid'))) {\n                return;\n            }\n\n            confirm({\n                title: this.confirmationTitle,\n                content: confirmTemplate({\n                    data: confirmData\n                }),\n                actions: {\n                    /** @inheritdoc */\n                    confirm: function () {\n                        $.ajax({\n                            url: this.purchaseUrl,\n                            data: form.serialize(),\n                            type: 'post',\n                            dataType: 'json',\n\n                            /** Show loader before send */\n                            beforeSend: function () {\n                                $('body').trigger('processStart');\n                            }\n                        }).always(function () {\n                            $('body').trigger('processStop');\n                        });\n                    }.bind(this)\n                }\n            });\n        }\n    });\n});\n","Magento_InventoryConfigurableProductFrontendUi/js/configurable-variation-qty.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * Configurable variation left qty.\n */\ndefine([\n    'jquery',\n    'underscore',\n    'mage/url'\n], function ($, _, urlBuilder) {\n    'use strict';\n\n    return function (productSku, salesChannel, salesChannelCode) {\n        var selectorInfoStockSkuQty = '.availability.only',\n            selectorInfoStockSkuQtyValue = '.availability.only > strong',\n            productQtyInfoBlock = $(selectorInfoStockSkuQty),\n            productQtyInfo = $(selectorInfoStockSkuQtyValue);\n\n        if (!_.isUndefined(productSku) && productSku !== null) {\n            $.ajax({\n                url: urlBuilder.build('inventory_catalog/product/getQty/'),\n                dataType: 'json',\n                data: {\n                    'sku': productSku,\n                    'channel': salesChannel,\n                    'salesChannelCode': salesChannelCode\n                }\n            }).done(function (response) {\n                if (response.qty !== null && response.qty > 0) {\n                    productQtyInfo.text(response.qty);\n                    productQtyInfoBlock.show();\n                } else {\n                    productQtyInfoBlock.hide();\n                }\n            }).fail(function () {\n                productQtyInfoBlock.hide();\n            });\n        } else {\n            productQtyInfoBlock.hide();\n        }\n    };\n});\n","Magento_InventoryConfigurableProductFrontendUi/js/configurable.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'configurableVariationQty',\n    'jquery-ui-modules/widget'\n], function ($, configurableVariationQty) {\n    'use strict';\n\n    return function (configurable) {\n        $.widget('mage.configurable', configurable, {\n\n            /** @inheritdoc */\n            _configureElement: function (element) {\n                var salesChannel = this.options.spConfig.channel,\n                    salesChannelCode = this.options.spConfig.salesChannelCode,\n                    productVariationsSku = this.options.spConfig.sku;\n\n                this._super(element);\n                configurableVariationQty(productVariationsSku[this.simpleProduct], salesChannel, salesChannelCode);\n            }\n        });\n\n        return $.mage.configurable;\n    };\n});\n","Magento_InventoryInStorePickupFrontend/js/checkout-data-ext.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Customer/js/customer-data'\n], function (\n    storage\n) {\n    'use strict';\n\n    return function (checkoutData) {\n\n        var cacheKey = 'checkout-data',\n\n            /**\n             * @param {Object} data\n             */\n            saveData = function (data) {\n                storage.set(cacheKey, data);\n            },\n\n            /**\n             * @return {Object}\n             */\n            getData = function () {\n                //Makes sure that checkout storage is initiated (any method can be used)\n                checkoutData.getSelectedShippingAddress();\n\n                return storage.get(cacheKey)();\n            };\n\n        /**\n         * Save the pickup address in persistence storage\n         *\n         * @param {Object} data\n         */\n        checkoutData.setSelectedPickupAddress = function (data) {\n            var obj = getData();\n\n            obj.selectedPickupAddress = data;\n            saveData(obj);\n        };\n\n        /**\n         * Get the pickup address from persistence storage\n         *\n         * @return {*}\n         */\n        checkoutData.getSelectedPickupAddress = function () {\n            return getData().selectedPickupAddress || null;\n        };\n\n        return checkoutData;\n    };\n});\n","Magento_InventoryInStorePickupFrontend/js/model/checkout-data-resolver-ext.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'mage/utils/wrapper',\n    'Magento_Checkout/js/checkout-data',\n    'Magento_Checkout/js/action/select-shipping-address',\n    'Magento_Checkout/js/model/address-converter',\n    'Magento_InventoryInStorePickupFrontend/js/model/pickup-address-converter'\n], function (\n    wrapper,\n    checkoutData,\n    selectShippingAddress,\n    addressConverter,\n    pickupAddressConverter\n) {\n    'use strict';\n\n    return function (checkoutDataResolver) {\n        checkoutDataResolver.resolveShippingAddress = wrapper.wrapSuper(\n            checkoutDataResolver.resolveShippingAddress,\n            function () {\n                var shippingAddress,\n                    pickUpAddress;\n\n                if (checkoutData.getSelectedPickupAddress() && checkoutData.getSelectedShippingAddress()) {\n                    shippingAddress = addressConverter.formAddressDataToQuoteAddress(\n                        checkoutData.getSelectedPickupAddress()\n                    );\n                    pickUpAddress = pickupAddressConverter.formatAddressToPickupAddress(\n                        shippingAddress\n                    );\n\n                    if (pickUpAddress.getKey() === checkoutData.getSelectedShippingAddress()) {\n                        selectShippingAddress(pickUpAddress);\n\n                        return;\n                    }\n                }\n                this._super();\n            });\n\n        return checkoutDataResolver;\n    };\n});\n","Magento_InventoryInStorePickupFrontend/js/model/pickup-address-converter.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine(['underscore'], function (_) {\n    'use strict';\n\n    return {\n        /**\n         * Format address to use in store pickup\n         *\n         * @param {Object} address\n         * @return {*}\n         */\n        formatAddressToPickupAddress: function (address) {\n            var sourceCode = _.findWhere(address.customAttributes, {\n                'attribute_code': 'sourceCode'\n            });\n\n            if (!sourceCode &&\n                !_.isEmpty(address.extensionAttributes) &&\n                address.extensionAttributes['pickup_location_code']\n            ) {\n                sourceCode = {\n                    value: address.extensionAttributes['pickup_location_code']\n                };\n            }\n\n            if (sourceCode && address.getType() !== 'store-pickup-address') {\n                address = _.extend({}, address, {\n                    saveInAddressBook: 0,\n\n                    /**\n                     * Is address can be used for billing\n                     *\n                     * @return {Boolean}\n                     */\n                    canUseForBilling: function () {\n                        return false;\n                    },\n\n                    /**\n                     * Returns address type\n                     *\n                     * @return {String}\n                     */\n                    getType: function () {\n                        return 'store-pickup-address';\n                    },\n\n                    /**\n                     * Returns address key\n                     *\n                     * @return {*}\n                     */\n                    getKey: function () {\n                        return this.getType() + sourceCode.value;\n                    }\n                });\n            }\n\n            return address;\n        }\n    };\n});\n","Magento_InventoryInStorePickupFrontend/js/model/pickup-locations-service.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'knockout',\n    'Magento_InventoryInStorePickupFrontend/js/model/resource-url-manager',\n    'mage/storage',\n    'Magento_Customer/js/customer-data',\n    'Magento_Checkout/js/checkout-data',\n    'Magento_Checkout/js/model/address-converter',\n    'Magento_Checkout/js/action/select-shipping-address',\n    'underscore',\n    'mage/translate',\n    'mage/url',\n    'Magento_InventoryInStorePickupFrontend/js/model/pickup-address-converter'\n], function (\n    $,\n    ko,\n    resourceUrlManager,\n    storage,\n    customerData,\n    checkoutData,\n    addressConverter,\n    selectShippingAddressAction,\n    _,\n    $t,\n    url,\n    pickupAddressConverter\n) {\n    'use strict';\n\n    var websiteCode = window.checkoutConfig.websiteCode,\n        countryData = customerData.get('directory-data');\n\n    return {\n        isLoading: ko.observable(false),\n        selectedLocation: ko.observable(null),\n        locationsCache: [],\n\n        /**\n         * Get shipping rates for specified address.\n         *\n         * @param {String} sourceCode\n         */\n        getLocation: function (sourceCode) {\n            var serviceUrl = resourceUrlManager.getUrlForPickupLocation(websiteCode, sourceCode);\n\n            this.isLoading(true);\n\n            return storage\n                .get(serviceUrl, {}, false)\n                .then(function (result) {\n                    var addresses = result.items || [],\n                        address = addresses[0] || {};\n\n                    return this.formatAddress(address);\n                }.bind(this))\n                .fail(function (response) {\n                    this.processError(response);\n\n                    return [];\n                }.bind(this))\n                .always(function () {\n                    this.isLoading(false);\n                }.bind(this));\n        },\n\n        /**\n         * Get nearby pickup locations based on given search criteria.\n         *\n         * @param {Object} searchCriteria - Search criteria object.\n         * @see Magento/InventoryInStorePickup/Model/SearchCriteria/GetNearbyLocationsCriteria.php\n         */\n        getNearbyLocations: function (searchCriteria) {\n            var self = this,\n                serviceUrl = resourceUrlManager.getUrlForNearbyPickupLocations(websiteCode, searchCriteria);\n\n            if (self.locationsCache[serviceUrl]) {\n                return $.Deferred().resolve(self.locationsCache[serviceUrl]).promise();\n            }\n\n            self.isLoading(true);\n\n            return storage\n                .get(serviceUrl, {}, false)\n                .then(function (result) {\n                    self.locationsCache[serviceUrl] = _.map(result.items, function (address) {\n                        return self.formatAddress(address);\n                    });\n\n                    return self.locationsCache[serviceUrl];\n                })\n                .fail(function (response) {\n                    self.processError(response);\n\n                    return [];\n                })\n                .always(function () {\n                    self.isLoading(false);\n                });\n        },\n\n        /**\n         * Select location for sipping.\n         *\n         * @param {Object} location\n         * @returns void\n         */\n        selectForShipping: function (location) {\n            var address = $.extend(\n                {},\n                addressConverter.formAddressDataToQuoteAddress({\n                    firstname: location.name,\n                    lastname: 'Store',\n                    street: location.street,\n                    city: location.city,\n                    postcode: location.postcode,\n                    'country_id': location['country_id'],\n                    telephone: location.telephone,\n                    'region_id': location['region_id'],\n                    'save_in_address_book': 0,\n                    'extension_attributes': {\n                        'pickup_location_code': location['pickup_location_code']\n                    }\n                }));\n\n            address = pickupAddressConverter.formatAddressToPickupAddress(address);\n            this.selectedLocation(location);\n            selectShippingAddressAction(address);\n            checkoutData.setSelectedShippingAddress(address.getKey());\n            checkoutData.setSelectedPickupAddress(\n                addressConverter.quoteAddressToFormAddressData(address)\n            );\n        },\n\n        /**\n         * Formats address returned by REST endpoint to match checkout address field naming.\n         *\n         * @param {Object} address - Address object returned by REST endpoint.\n         */\n        formatAddress: function (address) {\n            return {\n                name: address.name,\n                description: address.description,\n                latitude: address.latitude,\n                longitude: address.longitude,\n                street: [address.street],\n                city: address.city,\n                postcode: address.postcode,\n                'country_id': address['country_id'],\n                country: this.getCountryName(address['country_id']),\n                telephone: address.phone,\n                'region_id': address['region_id'],\n                region: this.getRegionName(\n                    address['country_id'],\n                    address['region_id']\n                ),\n                'pickup_location_code': address['pickup_location_code']\n            };\n        },\n\n        /**\n         * Get country name by id.\n         *\n         * @param {*} countryId\n         * @return {String}\n         */\n        getCountryName: function (countryId) {\n            return countryData()[countryId] !== undefined ?\n                countryData()[countryId].name\n                : ''; //eslint-disable-line\n        },\n\n        /**\n         * Returns region name based on given country and region identifiers.\n         *\n         * @param {String} countryId - Country identifier.\n         * @param {String} regionId - Region identifier.\n         */\n        getRegionName: function (countryId, regionId) {\n            var regions = countryData()[countryId] ?\n                countryData()[countryId].regions\n                : null;\n\n            return regions && regions[regionId] ? regions[regionId].name : '';\n        },\n\n        /**\n         * Process response errors.\n         *\n         * @param {Object} response\n         * @returns void\n         */\n        processError: function (response) {\n            var expr = /([%])\\w+/g,\n                error;\n\n            if (response.status === 401) {\n                //eslint-disable-line eqeqeq\n                window.location.replace(url.build('customer/account/login/'));\n\n                return;\n            }\n\n            try {\n                error = JSON.parse(response.responseText);\n            } catch (exception) {\n                error = $t(\n                    'Something went wrong with your request. Please try again later.'\n                );\n            }\n\n            if (error.hasOwnProperty('parameters')) {\n                error = error.message.replace(expr, function (varName) {\n                    varName = varName.substr(1);\n\n                    if (error.parameters.hasOwnProperty(varName)) {\n                        return error.parameters[varName];\n                    }\n\n                    return error.parameters.shift();\n                });\n            }\n        },\n\n        /**\n         * Returns selected pick up address from local storage\n         *\n         * @returns {Object|null}\n         */\n        getSelectedPickupAddress: function () {\n            var shippingAddress,\n                pickUpAddress;\n\n            if (checkoutData.getSelectedPickupAddress()) {\n                shippingAddress = addressConverter.formAddressDataToQuoteAddress(\n                    checkoutData.getSelectedPickupAddress()\n                );\n                pickUpAddress = pickupAddressConverter.formatAddressToPickupAddress(\n                    shippingAddress\n                );\n\n                return pickUpAddress;\n            }\n\n            return null;\n        }\n    };\n});\n","Magento_InventoryInStorePickupFrontend/js/model/quote-ext.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko',\n    'Magento_InventoryInStorePickupFrontend/js/model/pickup-address-converter'\n], function (ko, pickupAddressConverter) {\n    'use strict';\n\n    return function (quote) {\n        var shippingAddress = quote.shippingAddress;\n\n        /**\n         * Makes sure that shipping address gets appropriate type when it points\n         * to a store pickup location.\n         */\n        quote.shippingAddress = ko.pureComputed({\n            /**\n             * Return quote shipping address\n             */\n            read: function () {\n                return shippingAddress();\n            },\n\n            /**\n             * Set quote shipping address\n             */\n            write: function (address) {\n                shippingAddress(\n                    pickupAddressConverter.formatAddressToPickupAddress(address)\n                );\n            }\n        });\n\n        return quote;\n    };\n});\n","Magento_InventoryInStorePickupFrontend/js/model/resource-url-manager.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['jquery', 'Magento_Checkout/js/model/resource-url-manager'], function (\n    $,\n    resourceUrlManager\n) {\n    'use strict';\n\n    return {\n        /**\n         * Returns URL for REST API to fetch nearby pickup locations defined for given sales channel.\n         *\n         * @param {String} salesChannelCode - Code of the sales channel.\n         * @param {Object} searchCriteria\n         */\n        getUrlForNearbyPickupLocations: function (\n            salesChannelCode,\n            searchCriteria\n        ) {\n            var urls = {\n                    default: '/inventory/in-store-pickup/pickup-locations/'\n                },\n                criteria = {\n                    searchRequest: {\n                        scopeCode: salesChannelCode\n                    }\n                };\n\n            searchCriteria = {\n                searchRequest: searchCriteria\n            };\n\n            return (\n                resourceUrlManager.getUrl(urls, {}) +\n                '?' +\n                $.param($.extend(true, criteria, searchCriteria))\n            );\n        },\n\n        /**\n         * Returns URL for REST API to fetch all pickup locations defined for given sales channel.\n         *\n         * @param {String} salesChannelType - Type of the sales channel, e.g. website.\n         * @param {String} salesChannelCode - Code of the sales channel.\n         */\n        getUrlForPickupLocationsAssignedToSalesChannel: function (\n            salesChannelType,\n            salesChannelCode\n        ) {\n            var params = {\n                    salesChannelType: salesChannelType,\n                    salesChannelCode: salesChannelCode\n                },\n                urls = {\n                    default: '/inventory/in-store-pickup/pickup-locations-assigned-to-sales-channel/' +\n                        ':salesChannelType/:salesChannelCode'\n                };\n\n            return resourceUrlManager.getUrl(urls, params);\n        },\n\n        /**\n         * Returns URL for REST API to fetch pickup location with given code defined for given sales channel.\n         *\n         * @param {String} salesChannelCode - Code of the sales channel.\n         * @param {String} pickupLocationCode - Code of the pickup location.\n         */\n        getUrlForPickupLocation: function (\n            salesChannelCode,\n            pickupLocationCode\n        ) {\n            var urls = {\n                    default: '/inventory/in-store-pickup/pickup-locations/'\n                },\n                searchRequest = {\n                    searchRequest: {\n                        filters: {\n                            pickupLocationCode: {\n                                value: pickupLocationCode\n                            }\n                        },\n                        scopeCode: salesChannelCode\n                    }\n                };\n\n            return resourceUrlManager.getUrl(urls, {}) +\n                '?' +\n                $.param(searchRequest);\n        }\n    };\n});\n","Magento_InventoryInStorePickupFrontend/js/model/shipping-rate-processor/store-pickup-address.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        /*eslint-disable no-unused-vars*/\n        /**\n         * @param {Object} address\n         */\n        getRates: function (address) {}\n    };\n});\n","Magento_InventoryInStorePickupFrontend/js/view/shipping-information-ext.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/model/quote'\n], function (quote) {\n    'use strict';\n\n    var storePickupShippingInformation = {\n        defaults: {\n            template: 'Magento_InventoryInStorePickupFrontend/shipping-information'\n        },\n\n        /**\n         * Get shipping method title based on delivery method.\n         *\n         * @return {String}\n         */\n        getShippingMethodTitle: function () {\n            var shippingMethod = quote.shippingMethod(),\n                locationName = '',\n                title;\n\n            if (!this.isStorePickup()) {\n\n                return this._super();\n            }\n\n            title = shippingMethod['carrier_title'] + ' - ' + shippingMethod['method_title'];\n\n            if (quote.shippingAddress().firstname !== undefined) {\n                locationName = quote.shippingAddress().firstname + ' ' + quote.shippingAddress().lastname;\n                title += ' \"' + locationName + '\"';\n            }\n\n            return title;\n        },\n\n        /**\n         * Get is store pickup delivery method selected.\n         *\n         * @returns {Boolean}\n         */\n        isStorePickup: function () {\n            var shippingMethod = quote.shippingMethod(),\n                isStorePickup = false;\n\n            if (shippingMethod !== null) {\n                isStorePickup = shippingMethod['carrier_code'] === 'instore' &&\n                    shippingMethod['method_code'] === 'pickup';\n            }\n\n            return isStorePickup;\n        }\n    };\n\n    return function (shippingInformation) {\n        return shippingInformation.extend(storePickupShippingInformation);\n    };\n});\n","Magento_InventoryInStorePickupFrontend/js/view/store-pickup.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'underscore',\n    'jquery',\n    'knockout',\n    'uiRegistry',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/action/select-shipping-method',\n    'Magento_Checkout/js/checkout-data',\n    'Magento_Checkout/js/model/shipping-service',\n    'Magento_Checkout/js/model/step-navigator',\n    'Magento_Checkout/js/model/shipping-rate-service',\n    'Magento_InventoryInStorePickupFrontend/js/model/shipping-rate-processor/store-pickup-address',\n    'Magento_InventoryInStorePickupFrontend/js/model/pickup-locations-service',\n    'Magento_InventoryInStorePickupFrontend/js/model/pickup-address-converter',\n    'Magento_Checkout/js/model/checkout-data-resolver',\n    'Magento_Checkout/js/action/select-shipping-address'\n], function (\n    Component,\n    _,\n    $,\n    ko,\n    registry,\n    quote,\n    selectShippingMethodAction,\n    checkoutData,\n    shippingService,\n    stepNavigator,\n    shippingRateService,\n    shippingRateProcessor,\n    pickupLocationsService,\n    pickupAddressConverter,\n    checkoutDataResolver,\n    selectShippingAddress\n) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_InventoryInStorePickupFrontend/store-pickup',\n            deliveryMethodSelectorTemplate: 'Magento_InventoryInStorePickupFrontend/delivery-method-selector',\n            isVisible: false,\n            isAvailable: false,\n            isStorePickupSelected: false,\n            rate: {\n                'carrier_code': 'instore',\n                'method_code': 'pickup'\n            },\n            nearbySearchLimit: 50,\n            defaultCountry: window.checkoutConfig.defaultCountryId,\n            delimiter: window.checkoutConfig.storePickupApiSearchTermDelimiter,\n            rates: shippingService.getShippingRates(),\n            inStoreMethod: null,\n            lastSelectedNonPickUpShippingAddress: null\n        },\n\n        /**\n         * @inheritdoc\n         */\n        initialize: function () {\n            this._super();\n\n            shippingRateService.registerProcessor('store-pickup-address', shippingRateProcessor);\n\n            this.convertAddressType(quote.shippingAddress());\n\n            this.isStorePickupSelected.subscribe(function () {\n                this.preselectLocation();\n            }, this);\n            this.preselectLocation();\n\n            this.syncWithShipping();\n        },\n\n        /**\n         * Init component observable variables\n         *\n         * @return {exports}\n         */\n        initObservable: function () {\n            this._super().observe(['isVisible']);\n\n            this.isStorePickupSelected = ko.pureComputed(function () {\n                return _.isMatch(quote.shippingMethod(), this.rate);\n            }, this);\n\n            this.isAvailable = ko.pureComputed(function () {\n                return _.findWhere(this.rates(), {\n                    'carrier_code': this.rate['carrier_code'],\n                    'method_code': this.rate['method_code']\n                });\n            }, this);\n\n            return this;\n        },\n\n        /**\n         * Synchronize store pickup visibility with shipping step.\n         *\n         * @returns void\n         */\n        syncWithShipping: function () {\n            var shippingStep = _.findWhere(stepNavigator.steps(), {\n                code: 'shipping'\n            });\n\n            shippingStep.isVisible.subscribe(function (isShippingVisible) {\n                this.isVisible(this.isAvailable && isShippingVisible);\n            }, this);\n            this.isVisible(this.isAvailable && shippingStep.isVisible());\n        },\n\n        /**\n         * @returns void\n         */\n        selectShipping: function () {\n            var nonPickupShippingMethod = _.find(\n                this.rates(),\n                function (rate) {\n                    return (\n                        rate['carrier_code'] !== this.rate['carrier_code'] &&\n                        rate['method_code'] !== this.rate['method_code']\n                    );\n                },\n                this\n            ),\n                nonPickupShippingAddress;\n\n            checkoutData.setSelectedShippingAddress(this.lastSelectedNonPickUpShippingAddress);\n            this.selectShippingMethod(nonPickupShippingMethod);\n\n            if (this.isStorePickupAddress(quote.shippingAddress())) {\n                nonPickupShippingAddress = checkoutDataResolver.getShippingAddressFromCustomerAddressList();\n\n                if (nonPickupShippingAddress) {\n                    selectShippingAddress(nonPickupShippingAddress);\n                }\n            }\n        },\n\n        /**\n         * @returns void\n         */\n        selectStorePickup: function () {\n            var pickupShippingMethod = _.findWhere(\n                this.rates(),\n                {\n                    'carrier_code': this.rate['carrier_code'],\n                    'method_code': this.rate['method_code']\n                },\n                this\n            );\n\n            this.lastSelectedNonPickUpShippingAddress = checkoutData.getSelectedShippingAddress();\n            checkoutData.setSelectedShippingAddress(null);\n            this.preselectLocation();\n            this.selectShippingMethod(pickupShippingMethod);\n        },\n\n        /**\n         * @param {Object} shippingMethod\n         */\n        selectShippingMethod: function (shippingMethod) {\n            selectShippingMethodAction(shippingMethod);\n            checkoutData.setSelectedShippingRate(\n                shippingMethod ? shippingMethod['carrier_code'] + '_' + shippingMethod['method_code'] : null\n            );\n        },\n\n        /**\n         * @param {Object} shippingAddress\n         * @returns void\n         */\n        convertAddressType: function (shippingAddress) {\n            var pickUpAddress;\n\n            if (\n                !this.isStorePickupAddress(shippingAddress) &&\n                this.isStorePickupSelected()\n            ) {\n                pickUpAddress = pickupAddressConverter.formatAddressToPickupAddress(shippingAddress);\n\n                if (quote.shippingAddress() !== pickUpAddress) {\n                    quote.shippingAddress(pickUpAddress);\n                }\n            }\n        },\n\n        /**\n         * @returns void\n         */\n        preselectLocation: function () {\n            var selectedLocation,\n                shippingAddress,\n                selectedPickupAddress,\n                customAttributes,\n                selectedSource,\n                selectedSourceCode,\n                nearestLocation,\n                productsInfo = [],\n                items = quote.getItems();\n\n            if (!this.isStorePickupSelected()) {\n                return;\n            }\n\n            selectedLocation = pickupLocationsService.selectedLocation();\n\n            if (selectedLocation) {\n                pickupLocationsService.selectForShipping(selectedLocation);\n\n                return;\n            }\n\n            shippingAddress = quote.shippingAddress();\n            customAttributes = shippingAddress.customAttributes || [];\n            selectedSource = _.findWhere(customAttributes, {\n                'attribute_code': 'sourceCode'\n            });\n\n            if (selectedSource) {\n                selectedSourceCode = selectedSource.value;\n            }\n\n            if (!selectedSourceCode) {\n                selectedSourceCode = this.getPickupLocationCodeFromAddress(shippingAddress);\n            }\n\n            if (!selectedSourceCode) {\n                selectedPickupAddress = pickupLocationsService.getSelectedPickupAddress();\n                selectedSourceCode = this.getPickupLocationCodeFromAddress(selectedPickupAddress);\n            }\n\n            if (selectedSourceCode) {\n                pickupLocationsService\n                    .getLocation(selectedSourceCode)\n                    .then(function (location) {\n                        pickupLocationsService.selectForShipping(location);\n                    });\n            } else if (shippingAddress.city && shippingAddress.postcode) {\n                _.each(items, function (item) {\n                    if (item['qty_options'] === undefined || item['qty_options'].length === 0) {\n                        productsInfo.push(\n                            {\n                                sku: item.sku\n                            }\n                        );\n                    }\n                });\n                pickupLocationsService\n                    .getNearbyLocations({\n                        area: {\n                            radius: this.nearbySearchRadius,\n                            searchTerm: shippingAddress.postcode + this.delimiter +\n                                        shippingAddress.countryId || this.defaultCountry\n                        },\n                        extensionAttributes: {\n                            productsInfo: productsInfo\n                        },\n                        pageSize: this.nearbySearchLimit\n                    })\n                    .then(function (locations) {\n                        nearestLocation = locations[0];\n\n                        if (nearestLocation) {\n                            pickupLocationsService.selectForShipping(\n                                nearestLocation\n                            );\n                        }\n                    });\n            }\n        },\n\n        /**\n         * @param {Object} address\n         * @returns {Boolean}\n         */\n        isStorePickupAddress: function (address) {\n            return address.getType() === 'store-pickup-address';\n        },\n\n        /**\n         * @param {Object} address\n         * @returns {String|null}\n         */\n        getPickupLocationCodeFromAddress: function (address) {\n            if (address &&\n                !_.isEmpty(address.extensionAttributes) &&\n                address.extensionAttributes['pickup_location_code']\n            ) {\n                return address.extensionAttributes['pickup_location_code'];\n            }\n\n            return null;\n        }\n    });\n});\n","Magento_InventoryInStorePickupFrontend/js/view/store-selector.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'uiComponent',\n    'uiRegistry',\n    'Magento_Ui/js/modal/modal',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Customer/js/model/customer',\n    'Magento_Checkout/js/model/step-navigator',\n    'Magento_Checkout/js/model/address-converter',\n    'Magento_Checkout/js/action/set-shipping-information',\n    'Magento_InventoryInStorePickupFrontend/js/model/pickup-locations-service'\n], function (\n    $,\n    _,\n    Component,\n    registry,\n    modal,\n    quote,\n    customer,\n    stepNavigator,\n    addressConverter,\n    setShippingInformationAction,\n    pickupLocationsService\n) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_InventoryInStorePickupFrontend/store-selector',\n            selectedLocationTemplate:\n                'Magento_InventoryInStorePickupFrontend/store-selector/selected-location',\n            storeSelectorPopupTemplate:\n                'Magento_InventoryInStorePickupFrontend/store-selector/popup',\n            storeSelectorPopupItemTemplate:\n                'Magento_InventoryInStorePickupFrontend/store-selector/popup-item',\n            loginFormSelector:\n                '#store-selector form[data-role=email-with-possible-login]',\n            defaultCountryId: window.checkoutConfig.defaultCountryId,\n            delimiter: window.checkoutConfig.storePickupApiSearchTermDelimiter,\n            selectedLocation: pickupLocationsService.selectedLocation,\n            quoteIsVirtual: quote.isVirtual,\n            searchQuery: '',\n            nearbyLocations: null,\n            isLoading: pickupLocationsService.isLoading,\n            popup: null,\n            searchDebounceTimeout: 300,\n            imports: {\n                nearbySearchRadius: '${ $.parentName }:nearbySearchRadius',\n                nearbySearchLimit: '${ $.parentName }:nearbySearchLimit'\n            }\n        },\n\n        /**\n         * Init component\n         *\n         * @return {exports}\n         */\n        initialize: function () {\n            var updateNearbyLocations, country;\n\n            this._super();\n\n            updateNearbyLocations = _.debounce(function (searchQuery) {\n                country = quote.shippingAddress() && quote.shippingAddress().countryId ?\n                    quote.shippingAddress().countryId : this.defaultCountryId;\n                searchQuery = this.getSearchTerm(searchQuery, country);\n                this.updateNearbyLocations(searchQuery);\n            }, this.searchDebounceTimeout).bind(this);\n            this.searchQuery.subscribe(updateNearbyLocations);\n\n            return this;\n        },\n\n        /**\n         * Init component observable variables\n         *\n         * @return {exports}\n         */\n        initObservable: function () {\n            return this._super().observe(['nearbyLocations', 'searchQuery']);\n        },\n\n        /**\n         * Set shipping information handler\n         */\n        setPickupInformation: function () {\n            if (this.validatePickupInformation()) {\n                setShippingInformationAction().done(function () {\n                    stepNavigator.next();\n                });\n            }\n        },\n\n        /**\n         * @return {*}\n         */\n        getPopup: function () {\n            if (!this.popup) {\n                this.popup = modal(\n                    this.popUpList.options,\n                    $(this.popUpList.element)\n                );\n            }\n\n            return this.popup;\n        },\n\n        /**\n         * Get Search Term from search query and country.\n         *\n         * @param {String} searchQuery\n         * @param {String} country\n         * @returns {String}\n         */\n        getSearchTerm: function (searchQuery, country) {\n            return searchQuery ? searchQuery + this.delimiter + country : searchQuery;\n        },\n\n        /**\n         * @returns void\n         */\n        openPopup: function () {\n            var shippingAddress = quote.shippingAddress(),\n                country = shippingAddress.countryId ? shippingAddress.countryId :\n                this.defaultCountryId,\n                searchTerm = '';\n\n            this.getPopup().openModal();\n\n            if (shippingAddress.city && shippingAddress.postcode) {\n                searchTerm = this.getSearchTerm(shippingAddress.postcode, country);\n            }\n\n            this.updateNearbyLocations(searchTerm);\n        },\n\n        /**\n         * @param {Object} location\n         * @returns void\n         */\n        selectPickupLocation: function (location) {\n            pickupLocationsService.selectForShipping(location);\n            this.getPopup().closeModal();\n        },\n\n        /**\n         * @param {Object} location\n         * @returns {*|Boolean}\n         */\n        isPickupLocationSelected: function (location) {\n            return _.isEqual(this.selectedLocation(), location);\n        },\n\n        /**\n         * @param {String} searchQuery\n         * @returns {*}\n         */\n        updateNearbyLocations: function (searchQuery) {\n            var self = this,\n                productsInfo = [],\n                items = quote.getItems(),\n                searchCriteria;\n\n            _.each(items, function (item) {\n                if (item['qty_options'] === undefined || item['qty_options'].length === 0) {\n                    productsInfo.push(\n                        {\n                            sku: item.sku\n                        }\n                    );\n                }\n            });\n\n            searchCriteria = {\n                extensionAttributes: {\n                    productsInfo: productsInfo\n                },\n                pageSize: this.nearbySearchLimit\n            };\n\n            if (searchQuery) {\n                searchCriteria.area = {\n                    radius: this.nearbySearchRadius,\n                    searchTerm: searchQuery\n                };\n            }\n\n            return pickupLocationsService\n                .getNearbyLocations(searchCriteria)\n                .then(function (locations) {\n                    self.nearbyLocations(locations);\n                })\n                .fail(function () {\n                    self.nearbyLocations([]);\n                });\n        },\n\n        /**\n         * @returns {Boolean}\n         */\n        validatePickupInformation: function () {\n            var emailValidationResult,\n                loginFormSelector = this.loginFormSelector;\n\n            if (!customer.isLoggedIn()) {\n                $(loginFormSelector).validation();\n                emailValidationResult = $(loginFormSelector + ' input[name=username]').valid() ? true : false;\n\n                if (!emailValidationResult) {\n                    $(this.loginFormSelector + ' input[name=username]').trigger('focus');\n\n                    return false;\n                }\n            }\n\n            return true;\n        }\n    });\n});\n","Magento_InventoryInStorePickupFrontend/js/view/form/element/email.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['jquery', 'Magento_Checkout/js/view/form/element/email'], function (\n    $,\n    Component\n) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template:\n                'Magento_InventoryInStorePickupFrontend/form/element/email',\n            links: {\n                email:\n                    'checkout.steps.shipping-step.shippingAddress.customer-email:email'\n            }\n        }\n    });\n});\n","Magento_InventorySwatchesFrontendUi/js/swatch-renderer.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'configurableVariationQty',\n    'jquery-ui-modules/widget'\n], function ($, configurableVariationQty) {\n    'use strict';\n\n    return function (SwatchRenderer) {\n        $.widget('mage.SwatchRenderer', SwatchRenderer, {\n\n            /** @inheritdoc */\n            _OnClick: function ($this, widget) {\n                var salesChannel = this.options.jsonConfig.channel,\n                    salesChannelCode = this.options.jsonConfig.salesChannelCode,\n                    productVariationsSku = this.options.jsonConfig.sku;\n\n                this._super($this, widget);\n                configurableVariationQty(productVariationsSku[widget.getProductId()], salesChannel, salesChannelCode);\n            }\n        });\n\n        return $.mage.SwatchRenderer;\n    };\n});\n","Magento_LoginAsCustomerAssistance/js/opt-in.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    return function (config, element) {\n        $(element).on('submit', function () {\n            this.elements['assistance_allowed'].value =\n                this.elements['assistance_allowed_checkbox'].checked ?\n                    config.allowAccess : config.denyAccess;\n        });\n    };\n});\n","Magento_LoginAsCustomerFrontendUi/js/login.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Customer/js/customer-data',\n    'Magento_Customer/js/section-config'\n], function ($, customerData, sectionConfig) {\n\n    'use strict';\n\n    return function (config) {\n        customerData.reload(sectionConfig.getSectionNames()).done(function () {\n            window.location.href = config.redirectUrl;\n        });\n    };\n});\n","Magento_LoginAsCustomerFrontendUi/js/view/loginAsCustomer.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'uiComponent',\n    'Magento_Customer/js/customer-data',\n    'mage/translate'\n], function ($, _, Component, customer) {\n    'use strict';\n\n    return Component.extend({\n\n        defaults: {\n            isVisible: false\n        },\n\n        /** @inheritdoc */\n        initialize: function () {\n            var customerData, loggedAsCustomerData;\n\n            this._super();\n\n            customerData = customer.get('customer');\n            loggedAsCustomerData = customer.get('loggedAsCustomer');\n\n            customerData.subscribe(function (data) {\n                this.fullname = data.fullname;\n                this.updateBanner();\n            }.bind(this));\n            loggedAsCustomerData.subscribe(function (data) {\n                this.adminUserId = data.adminUserId;\n                this.websiteName = data.websiteName;\n                this.updateBanner();\n            }.bind(this));\n\n            this.fullname = customerData().fullname;\n            this.adminUserId = loggedAsCustomerData().adminUserId;\n            this.websiteName = loggedAsCustomerData().websiteName;\n\n            this.updateBanner();\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super()\n                .observe(['isVisible', 'notificationText']);\n\n            return this;\n        },\n\n        /**\n         * Update banner area\n         *\n         * @returns void\n         */\n        updateBanner: function () {\n            if (this.adminUserId !== undefined) {\n                this.isVisible(this.adminUserId);\n            }\n\n            if (this.fullname !== undefined && this.websiteName !== undefined) {\n                this.notificationText($.mage.__('You are connected as <strong>%1</strong> on %2')\n                    .replace('%1', _.escape(this.fullname))\n                    .replace('%2', _.escape(this.websiteName)));\n            }\n        }\n    });\n});\n","Magento_Msrp/js/msrp.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'Magento_Catalog/js/price-utils',\n    'underscore',\n    'jquery-ui-modules/widget',\n    'mage/dropdown',\n    'mage/template'\n], function ($, priceUtils, _) {\n    'use strict';\n\n    $.widget('mage.addToCart', {\n        options: {\n            showAddToCart: true,\n            submitUrl: '',\n            cartButtonId: '',\n            singleOpenDropDown: true,\n            dialog: {}, // Options for mage/dropdown\n            dialogDelay: 500, // Delay in ms after resize dropdown shown again\n            origin: '', //Required, type of popup: 'msrp', 'tier' or 'info' popup\n\n            // Selectors\n            cartForm: '.form.map.checkout',\n            msrpLabelId: '#map-popup-msrp',\n            msrpPriceElement: '#map-popup-msrp .price-wrapper',\n            priceLabelId: '#map-popup-price',\n            priceElement: '#map-popup-price .price',\n            mapInfoLinks: '.map-show-info',\n            displayPriceElement: '.old-price.map-old-price .price-wrapper',\n            fallbackPriceElement: '.normal-price.map-fallback-price .price-wrapper',\n            displayPriceContainer: '.old-price.map-old-price',\n            fallbackPriceContainer: '.normal-price.map-fallback-price',\n            popUpAttr: '[data-role=msrp-popup-template]',\n            popupCartButtonId: '#map-popup-button',\n            paypalCheckoutButons: '[data-action=checkout-form-submit]',\n            popupId: '',\n            realPrice: '',\n            isSaleable: '',\n            msrpPrice: '',\n            helpLinkId: '',\n            addToCartButton: '',\n\n            // Text options\n            productName: '',\n            addToCartUrl: ''\n        },\n\n        openDropDown: null,\n        triggerClass: 'dropdown-active',\n\n        popUpOptions: {\n            appendTo: 'body',\n            dialogContentClass: 'active',\n            closeOnMouseLeave: false,\n            autoPosition: true,\n            closeOnClickOutside: false,\n            'dialogClass': 'popup map-popup-wrapper',\n            position: {\n                my: 'left top',\n                collision: 'fit none',\n                at: 'left bottom',\n                within: 'body'\n            },\n            shadowHinter: 'popup popup-pointer'\n        },\n        popupOpened: false,\n        wasOpened: false,\n\n        /**\n         * Creates widget instance\n         *\n         * @private\n         */\n        _create: function () {\n            if (this.options.origin === 'msrp') {\n                this.initMsrpPopup();\n            } else if (this.options.origin === 'info') {\n                this.initInfoPopup();\n            } else if (this.options.origin === 'tier') {\n                this.initTierPopup();\n            }\n            $(this.options.cartButtonId).on('click', this._addToCartSubmit.bind(this));\n            $(document).on('updateMsrpPriceBlock', this.onUpdateMsrpPrice.bind(this));\n            $(this.options.cartForm).on('submit', this._onSubmitForm.bind(this));\n        },\n\n        /**\n         * Init msrp popup\n         *\n         * @private\n         */\n        initMsrpPopup: function () {\n            var popupDOM = $(this.options.popUpAttr)[0],\n                $msrpPopup = $(popupDOM.innerHTML.trim());\n\n            $msrpPopup.find(this.options.productIdInput).val(this.options.productId);\n            $('body').append($msrpPopup);\n            $msrpPopup.trigger('contentUpdated');\n\n            $msrpPopup.find('button')\n                .on('click',\n                    this.handleMsrpAddToCart.bind(this))\n                .filter(this.options.popupCartButtonId)\n                .text($(this.options.addToCartButton).text());\n\n            $msrpPopup.find(this.options.paypalCheckoutButons).on('click',\n                this.handleMsrpPaypalCheckout.bind(this));\n\n            $(this.options.popupId).on('click',\n                this.openPopup.bind(this));\n\n            this.$popup = $msrpPopup;\n        },\n\n        /**\n         * Init info popup\n         *\n         * @private\n         */\n        initInfoPopup: function () {\n            var infoPopupDOM = $('[data-role=msrp-info-template]')[0],\n                $infoPopup = $(infoPopupDOM.innerHTML.trim());\n\n            $('body').append($infoPopup);\n\n            $(this.options.helpLinkId).on('click', function (e) {\n                this.popUpOptions.position.of = $(e.target);\n                $infoPopup.dropdownDialog(this.popUpOptions).dropdownDialog('open');\n                this._toggle($infoPopup);\n            }.bind(this));\n\n            this.$popup = $infoPopup;\n        },\n\n        /**\n         * Init tier price popup\n         * @private\n         */\n        initTierPopup: function () {\n            var popupDOM = $(this.options.popUpAttr)[0],\n                $tierPopup = $(popupDOM.innerHTML.trim());\n\n            $('body').append($tierPopup);\n            $tierPopup.find(this.options.productIdInput).val(this.options.productId);\n            this.popUpOptions.position.of = $(this.options.helpLinkId);\n\n            $tierPopup.find('button').on('click',\n                this.handleTierAddToCart.bind(this))\n                .filter(this.options.popupCartButtonId)\n                .text($(this.options.addToCartButton).text());\n\n            $tierPopup.find(this.options.paypalCheckoutButons).on('click',\n                this.handleTierPaypalCheckout.bind(this));\n\n            $(this.options.attr).on('click', function (e) {\n                this.$popup = $tierPopup;\n                this.tierOptions = $(e.target).data('tier-price');\n                this.openPopup(e);\n            }.bind(this));\n        },\n\n        /**\n         * handle 'AddToCart' click on Msrp popup\n         * @param {Object} ev\n         *\n         * @private\n         */\n        handleMsrpAddToCart: function (ev) {\n            ev.preventDefault();\n\n            if (this.options.addToCartButton) {\n                $(this.options.addToCartButton).trigger('click');\n                this.closePopup(this.$popup);\n            }\n        },\n\n        /**\n         * handle 'paypal checkout buttons' click on Msrp popup\n         *\n         * @private\n         */\n        handleMsrpPaypalCheckout: function () {\n            this.closePopup(this.$popup);\n        },\n\n        /**\n         * handle 'AddToCart' click on Tier popup\n         *\n         * @param {Object} ev\n         * @private\n         */\n        handleTierAddToCart: function (ev) {\n            ev.preventDefault();\n\n            if (this.options.addToCartButton &&\n                this.options.inputQty && !isNaN(this.tierOptions.qty)\n            ) {\n                $(this.options.inputQty).val(this.tierOptions.qty);\n                $(this.options.addToCartButton).trigger('click');\n                this.closePopup(this.$popup);\n            }\n        },\n\n        /**\n         * handle 'paypal checkout buttons' click on Tier popup\n         *\n         * @private\n         */\n        handleTierPaypalCheckout: function () {\n            if (this.options.inputQty && !isNaN(this.tierOptions.qty)\n            ) {\n                $(this.options.inputQty).val(this.tierOptions.qty);\n                this.closePopup(this.$popup);\n            }\n        },\n\n        /**\n         * Open and set up popup\n         *\n         * @param {Object} event\n         */\n        openPopup: function (event) {\n            var options = this.tierOptions || this.options;\n\n            this.popUpOptions.position.of = $(event.target);\n\n            if (!this.wasOpened) {\n                this.$popup.find(this.options.msrpLabelId).html(options.msrpPrice);\n                this.$popup.find(this.options.priceLabelId).html(options.realPrice);\n                this.wasOpened = true;\n            }\n            this.$popup.dropdownDialog(this.popUpOptions).dropdownDialog('open');\n            this._toggle(this.$popup);\n\n            if (!this.options.isSaleable) {\n                this.$popup.find('form').hide();\n            }\n        },\n\n        /**\n         * Toggle MAP popup visibility\n         *\n         * @param {HTMLElement} $elem\n         * @private\n         */\n        _toggle: function ($elem) {\n            $(document).on('mouseup.msrp touchend.msrp', function (e) {\n                if (!$elem.is(e.target) && $elem.has(e.target).length === 0) {\n                    this.closePopup($elem);\n                }\n            }.bind(this));\n            $(window).on('resize', function () {\n                this.closePopup($elem);\n            }.bind(this));\n        },\n\n        /**\n         * Close MAP information popup\n         *\n         * @param {HTMLElement} $elem\n         */\n        closePopup: function ($elem) {\n            $elem.dropdownDialog('close');\n            $(document).off('mouseup.msrp touchend.msrp');\n        },\n\n        /**\n         * Handler for addToCart action\n         *\n         * @param {Object} e\n         */\n        _addToCartSubmit: function (e) {\n            this.element.trigger('addToCart', this.element);\n\n            if (this.element.data('stop-processing')) {\n                return false;\n            }\n\n            if (this.options.addToCartButton) {\n                $(this.options.addToCartButton).trigger('click');\n\n                return false;\n            }\n\n            if (this.options.addToCartUrl) {\n                $('.mage-dropdown-dialog > .ui-dialog-content').dropdownDialog('close');\n            }\n\n            e.preventDefault();\n            $(this.options.cartForm).trigger('submit');\n        },\n\n        /**\n         * Call on event updatePrice. Proxy to updateMsrpPrice method.\n         *\n         * @param {Event} event\n         * @param {mixed} priceIndex\n         * @param {Object} prices\n         * @param {Object|undefined} $priceBox\n         */\n        onUpdateMsrpPrice: function onUpdateMsrpPrice(event, priceIndex, prices, $priceBox) {\n\n            var defaultMsrp,\n                defaultPrice,\n                msrpPrice,\n                finalPrice;\n\n            defaultMsrp = _.chain(prices).map(function (price) {\n                return price.msrpPrice.amount;\n            }).reject(function (p) {\n                return p === null;\n            }).max().value();\n\n            defaultPrice = _.chain(prices).map(function (p) {\n                return p.finalPrice.amount;\n            }).min().value();\n\n            if (typeof priceIndex !== 'undefined') {\n                msrpPrice = prices[priceIndex].msrpPrice.amount;\n                finalPrice = prices[priceIndex].finalPrice.amount;\n\n                if (msrpPrice === null || msrpPrice <= finalPrice) {\n                    this.updateNonMsrpPrice(priceUtils.formatPriceLocale(finalPrice), $priceBox);\n                } else {\n                    this.updateMsrpPrice(\n                        priceUtils.formatPriceLocale(finalPrice),\n                        priceUtils.formatPriceLocale(msrpPrice),\n                        false,\n                        $priceBox);\n                }\n            } else {\n                this.updateMsrpPrice(\n                    priceUtils.formatPriceLocale(defaultPrice),\n                    priceUtils.formatPriceLocale(defaultMsrp),\n                    true,\n                    $priceBox);\n            }\n        },\n\n        /**\n         * Update prices for configurable product with MSRP enabled\n         *\n         * @param {String} finalPrice\n         * @param {String} msrpPrice\n         * @param {Boolean} useDefaultPrice\n         * @param {Object|undefined} $priceBox\n         */\n        updateMsrpPrice: function (finalPrice, msrpPrice, useDefaultPrice, $priceBox) {\n            var options = this.tierOptions || this.options;\n\n            $(this.options.fallbackPriceContainer, $priceBox).hide();\n            $(this.options.displayPriceContainer, $priceBox).show();\n            $(this.options.mapInfoLinks, $priceBox).show();\n\n            if (useDefaultPrice || !this.wasOpened) {\n                if (this.$popup) {\n                    this.$popup.find(this.options.msrpLabelId).html(options.msrpPrice);\n                    this.$popup.find(this.options.priceLabelId).html(options.realPrice);\n                }\n\n                $(this.options.displayPriceElement, $priceBox).html(msrpPrice);\n                this.wasOpened = true;\n            }\n\n            if (!useDefaultPrice) {\n                this.$popup.find(this.options.msrpPriceElement).html(msrpPrice);\n                this.$popup.find(this.options.priceElement).html(finalPrice);\n                $(this.options.displayPriceElement, $priceBox).html(msrpPrice);\n            }\n        },\n\n        /**\n         * Display non MAP price for irrelevant products\n         *\n         * @param {String} price\n         * @param {Object|undefined} $priceBox\n         */\n        updateNonMsrpPrice: function (price, $priceBox) {\n            $(this.options.fallbackPriceElement, $priceBox).html(price);\n            $(this.options.displayPriceContainer, $priceBox).hide();\n            $(this.options.mapInfoLinks, $priceBox).hide();\n            $(this.options.fallbackPriceContainer, $priceBox).show();\n        },\n\n        /**\n         * Handler for submit form\n         *\n         * @private\n         */\n        _onSubmitForm: function () {\n            if ($(this.options.cartForm).valid()) {\n                $(this.options.cartButtonId).prop('disabled', true);\n            }\n        }\n\n    });\n\n    return $.mage.addToCart;\n});\n","Magento_Msrp/js/product/list/columns/msrp-price.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'underscore',\n    'Magento_Catalog/js/product/list/columns/price-box',\n    'Magento_Catalog/js/product/addtocart-button',\n    'mage/dropdown'\n], function ($, _, PriceBox) {\n    'use strict';\n\n    return PriceBox.extend({\n        defaults: {\n            priceBoxSelector: '[data-role=msrp-price-box]',\n            popupTmpl: 'Magento_Msrp/product/item/popup',\n            popupTriggerSelector: '[data-role=msrp-popup-trigger]',\n            popupSelector: '[data-role=msrp-popup]',\n            popupOptions: {\n                appendTo: 'body',\n                dialogContentClass: 'active',\n                closeOnMouseLeave: false,\n                autoPosition: true,\n                dialogClass: 'popup map-popup-wrapper',\n                position: {\n                    my: 'left top',\n                    collision: 'fit none',\n                    at: 'left bottom',\n                    within: 'body'\n                },\n                shadowHinter: 'popup popup-pointer'\n            }\n        },\n\n        /**\n         * Create and open popup with Msrp information.\n         *\n         * @param {Object} data - element data\n         * @param {DOMElement} elem - element\n         * @param {Event} event - event object\n         */\n        openPopup: function (data, elem, event) {\n            var $elem = $(elem),\n                $popup = $elem.find(this.popupSelector),\n                $trigger = $elem.find(this.popupTriggerSelector);\n\n            event.stopPropagation();\n\n            this.popupOptions.position.of = $trigger;\n            this.popupOptions.triggerTarget = $trigger;\n\n            $popup.dropdownDialog(this.popupOptions)\n                  .dropdownDialog('open');\n        },\n\n        /**\n         * Set listeners.\n         *\n         * @param {DOMElement} elem - DOM element\n         * @param {Object} data - element data\n         */\n        initListeners: function (elem, data) {\n            var $trigger = $(elem).find(this.popupTriggerSelector);\n\n            $trigger.on('click', this.openPopup.bind(this, data, elem));\n        },\n\n        /**\n         * Check whether we can apply msrp, or should use standard price.\n         *\n         * @param {Object} row\n         * @returns {Bool}\n         */\n        isMsrpApplicable: function (row) {\n            return this.getPrice(row)['is_applicable'];\n        },\n\n        /**\n         * Retrieve msrp formatted price\n         *\n         * @param {Object} row\n         * @returns {String}\n         */\n        getPrice: function (row) {\n            return row['price_info']['extension_attributes'].msrp;\n        },\n\n        /**\n         * UnsanitizedHtml version of getPrice.\n         *\n         * @param {Object} row\n         * @returns {String}\n         */\n        getPriceUnsanitizedHtml: function (row) {\n            return this.getPrice(row);\n        },\n\n        /**\n         * Get msrp_price property of a price.\n         *\n         * @param {Object} row\n         * @return {HTMLElement} final price html\n         */\n        getMsrpPriceUnsanitizedHtml: function (row) {\n            return this.getPrice(row)['msrp_price'];\n        },\n\n        /**\n         * Returns path to the columns' body template.\n         *\n         * @returns {String}\n         */\n        getBody: function () {\n            return this.bodyTmpl;\n        },\n\n        /**\n         * Check if popup with actual price must be shown.\n         *\n         * @returns {Boolean}\n         */\n        isShowPriceOnGesture: function (row) {\n            return this.getPrice(row)['is_shown_price_on_gesture'];\n        },\n\n        /**\n         * Get msrp price supporting text.\n         *\n         * @returns {String}\n         */\n        getMsrpPriceMessage: function (row) {\n            return this.getPrice(row)['msrp_message'];\n        },\n\n        /**\n         * UnsanitizedHtml version of getMsrpPriceMessage.\n         *\n         * @returns {String}\n         */\n        getMsrpPriceMessageUnsanitizedHtml: function (row) {\n            return this.getMsrpPriceMessage(row);\n        },\n\n        /**\n         * Get msrp price supporting text, when actual price is hidden.\n         *\n         * @returns {String}\n         */\n        getExplanationMessage: function (row) {\n            return this.getPrice(row)['explanation_message'];\n        },\n\n        /**\n         * UnsanitizedHtml version of getExplanationMessage.\n         *\n         * @returns {String}\n         */\n        getExplanationMessageUnsanitizedHtml: function (row) {\n            return this.getExplanationMessage(row);\n        }\n    });\n});\n","Magento_Msrp/js/view/checkout/minicart/subtotal/totals.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\n\ndefine([\n    'Magento_Tax/js/view/checkout/minicart/subtotal/totals',\n    'underscore'\n], function (Component, _) {\n    'use strict';\n\n    return Component.extend({\n\n        /**\n         * @override\n         */\n        initialize: function () {\n            this._super();\n            this.displaySubtotal(this.isMsrpApplied(this.cart().items));\n            this.cart.subscribe(function (updatedCart) {\n\n                this.displaySubtotal(this.isMsrpApplied(updatedCart.items));\n            }, this);\n        },\n\n        /**\n         * Determine if subtotal should be hidden.\n         * @param {Array} cartItems\n         * @return {Boolean}\n         */\n        isMsrpApplied: function (cartItems) {\n            return !_.find(cartItems, function (item) {\n                if (_.has(item, 'canApplyMsrp')) {\n                    return item.canApplyMsrp;\n                }\n\n                return false;\n            });\n        }\n    });\n});\n","Magento_Multishipping/js/multi-shipping-balance.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/dataPost',\n    'jquery-ui-modules/widget'\n], function ($, dataPost) {\n    'use strict';\n\n    $.widget('mage.multiShippingBalance', {\n        options: {\n            changeUrl: ''\n        },\n\n        /**\n         * Initialize balance checkbox events.\n         *\n         * @private\n         */\n        _create: function () {\n            this.element.on('change', $.proxy(function (event) {\n                dataPost().postData({\n                    action: this.options.changeUrl,\n                    data: {\n                        useBalance: +$(event.target).is(':checked')\n                    }\n                });\n            }, this));\n        }\n    });\n\n    return $.mage.multiShippingBalance;\n});\n","Magento_Multishipping/js/multi-shipping.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Customer/js/customer-data',\n    'jquery-ui-modules/widget'\n], function ($, customerData) {\n    'use strict';\n\n    $.widget('mage.multiShipping', {\n        options: {\n            itemsQty: 0,\n            addNewAddressBtn: 'button[data-role=\"add-new-address\"]', // Add a new multishipping address.\n            addNewAddressFlag: '#add_new_address_flag', // Hidden input field with value 0 or 1.\n            canContinueBtn: 'button[data-role=\"can-continue\"]', // Continue (update quantity or go to shipping).\n            canContinueFlag: '#can_continue_flag' // Hidden input field with value 0 or 1.\n        },\n\n        /**\n         * Bind event handlers to click events for corresponding buttons.\n         * @private\n         */\n        _create: function () {\n            this._prepareCartData();\n            $(this.options.addNewAddressBtn).on('click', $.proxy(this._addNewAddress, this));\n            $(this.options.canContinueBtn).on('click', $.proxy(this._canContinue, this));\n        },\n\n        /**\n         * Takes cart items qty from current cart data and compare it with current items qty\n         * Reloads cart data if cart items qty is wrong\n         * @private\n         */\n        _prepareCartData: function () {\n            var cartData = customerData.get('cart');\n\n            if (cartData()['summary_count'] !== this.options.itemsQty) {\n                customerData.reload(['cart'], false);\n            }\n        },\n\n        /**\n         * Add a new address. Set the hidden input field and submit the form. Then enter a new shipping address.\n         * @private\n         */\n        _addNewAddress: function () {\n            $(this.options.addNewAddressFlag).val(1);\n            this.element.submit();\n        },\n\n        /**\n         * Can the user continue to the next step? The data-flag attribute holds either 0 (no) or 1 (yes).\n         * @private\n         * @param {Event} event - Click event on the corresponding button.\n         */\n        _canContinue: function (event) {\n            $(this.options.canContinueFlag).val(parseInt($(event.currentTarget).data('flag'), 10));\n        }\n    });\n\n    return $.mage.multiShipping;\n});\n","Magento_Multishipping/js/overview.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget',\n    'mage/translate'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.orderOverview', {\n        options: {\n            opacity: 0.5, // CSS opacity for the 'Place Order' button when it's clicked and then disabled.\n            pleaseWaitLoader: 'span.please-wait', // 'Submitting order information...' Ajax loader.\n            placeOrderSubmit: 'button[type=\"submit\"]', // The 'Place Order' button.\n            agreements: '.checkout-agreements' // Container for all of the checkout agreements and terms/conditions\n        },\n\n        /**\n         * Bind a submit handler to the form.\n         * @private\n         */\n        _create: function () {\n            this.element.on('submit', $.proxy(this._showLoader, this));\n        },\n\n        /**\n         * Verify that all agreements and terms/conditions are checked. Show the Ajax loader. Disable\n         * the submit button (i.e. Place Order).\n         * @return {Boolean}\n         * @private\n         */\n        _showLoader: function () {\n            if ($(this.options.agreements).find('input[type=\"checkbox\"]:not(:checked)').length > 0) {\n                return false;\n            }\n            this.element.find(this.options.pleaseWaitLoader).show().end()\n                .find(this.options.placeOrderSubmit).prop('disabled', true).css('opacity', this.options.opacity);\n\n            return true;\n        }\n    });\n\n    return $.mage.orderOverview;\n});\n","Magento_Multishipping/js/payment.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/template',\n    'Magento_Ui/js/modal/alert',\n    'jquery-ui-modules/widget',\n    'mage/translate'\n], function ($, mageTemplate, alert) {\n    'use strict';\n\n    $.widget('mage.payment', {\n        options: {\n            continueSelector: '#payment-continue',\n            methodsContainer: '#payment-methods',\n            minBalance: 0,\n            tmpl: '<input id=\"hidden-free\" type=\"hidden\" name=\"payment[method]\" value=\"free\">'\n        },\n\n        /** @inheritdoc */\n        _create: function () {\n            this.element.find('dd [name^=\"payment[\"]').prop('disabled', true).end()\n                .on('click', this.options.continueSelector, $.proxy(this._submitHandler, this))\n                .on('updateCheckoutPrice', $.proxy(function (event, data) {\n                    //updating the checkoutPrice\n                    if (data.price) {\n                        this.options.checkoutPrice += data.price;\n                    }\n\n                    //updating total price\n                    if (data.totalPrice) {\n                        data.totalPrice = this.options.checkoutPrice;\n                    }\n\n                    if (this.options.checkoutPrice <= this.options.minBalance) {\n                        // Add free input field, hide and disable unchecked\n                        // checkbox payment method and all radio button payment methods\n                        this._disablePaymentMethods();\n                    } else {\n                        // Remove free input field, show all payment method\n                        this._enablePaymentMethods();\n                    }\n                }, this))\n                .on('click', 'dt input:radio', $.proxy(this._paymentMethodHandler, this));\n\n            if (this.options.checkoutPrice < this.options.minBalance) {\n                this._disablePaymentMethods();\n            } else {\n                this._enablePaymentMethods();\n            }\n        },\n\n        /**\n         * Display payment details when payment method radio button is checked\n         * @private\n         * @param {EventObject} e\n         */\n        _paymentMethodHandler: function (e) {\n            var element = $(e.target),\n                parentsDl = element.closest('dl');\n\n            parentsDl.find('dt input:radio').prop('checked', false);\n            parentsDl.find('dd').addClass('no-display').end()\n                .find('.items').hide()\n                .find('[name^=\"payment[\"]').prop('disabled', true);\n            element.prop('checked', true).parent()\n                .next('dd').removeClass('no-display')\n                .find('.items').show().find('[name^=\"payment[\"]').prop('disabled', false);\n        },\n\n        /**\n         * make sure one payment method is selected\n         * @private\n         * @return {Boolean}\n         */\n        _validatePaymentMethod: function () {\n            var methods = this.element.find('[name^=\"payment[\"]'),\n                isValid = false;\n\n            if (methods.length === 0) {\n                alert({\n                    content: $.mage.__('We can\\'t complete your order because you don\\'t have a payment method set up.')\n                });\n            } else if (this.options.checkoutPrice <= this.options.minBalance) {\n                isValid = true;\n            } else if (methods.filter('input:radio:checked').length) {\n                isValid = true;\n            } else {\n                alert({\n                    content: $.mage.__('Please choose a payment method.')\n                });\n            }\n\n            return isValid;\n        },\n\n        /**\n         * Disable and enable payment methods\n         * @private\n         */\n        _disablePaymentMethods: function () {\n            var tmpl = mageTemplate(this.options.tmpl, {\n                data: {}\n            });\n\n            this.element.find('input[name=\"payment[method]\"]').prop('disabled', true).end()\n                .find('input[id^=\"use\"][name^=\"payment[use\"]:not(:checked)').prop('disabled', true).parent().hide();\n            this.element.find('[name=\"payment[method]\"][value=\"free\"]').parent('dt').remove();\n            this.element.find(this.options.methodsContainer).hide().find('[name^=\"payment[\"]').prop('disabled', true);\n\n            $(tmpl).appendTo(this.element);\n        },\n\n        /**\n         * Enable and enable payment methods\n         * @private\n         */\n        _enablePaymentMethods: function () {\n            this.element.find('input[name=\"payment[method]\"]').prop('disabled', false).end()\n                .find('dt input:radio:checked').trigger('click').end()\n                .find('input[id^=\"use\"][name^=\"payment[use\"]:not(:checked)').prop('disabled', false).parent().show();\n            this.element.find(this.options.methodsContainer).show();\n        },\n\n        /**\n         * Returns checked payment method.\n         *\n         * @private\n         */\n        _getSelectedPaymentMethod: function () {\n            return this.element.find('input[name=\\'payment[method]\\']:checked');\n        },\n\n        /**\n         * Validate  before form submit\n         * @private\n         * @param {EventObject} e\n         */\n        _submitHandler: function (e) {\n            var currentMethod,\n                submitButton;\n\n            e.preventDefault();\n\n            if (this._validatePaymentMethod()) {\n                currentMethod = this._getSelectedPaymentMethod();\n                submitButton = currentMethod.parent().next('dd').find('button[type=submit]');\n\n                if (submitButton.length) {\n                    submitButton.first().trigger('click');\n                } else {\n                    this.element.trigger('submit');\n                }\n            }\n        }\n    });\n\n    return $.mage.payment;\n});\n","Magento_Newsletter/js/newsletter-sign-up.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'uiElement',\n    'mage/url',\n    'subscriptionStatusResolver',\n    'mage/validation'\n], function ($, Component, urlBuilder, subscriptionStatusResolver) {\n    'use strict';\n\n    return Component.extend({\n\n        defaults: {\n            signUpElement: '',\n            submitButton: '',\n            element: null\n        },\n\n        /** @inheritdoc */\n        initialize: function (config, element) {\n            this._super();\n            this.element = element;\n            $(element).on('change', $.proxy(this.updateSignUpStatus, this));\n            this.updateSignUpStatus();\n        },\n\n        /**\n         * Send status request and update subscription element according to result.\n         */\n        updateSignUpStatus: function () {\n            var element = $(this.element),\n                email = element.val(),\n                self = this,\n                newsletterSubscription;\n\n            if ($(self.signUpElement).is(':checked')) {\n                return;\n            }\n\n            if (!email || !$.validator.methods['validate-email'].call(this, email, element)) {\n                return;\n            }\n\n            newsletterSubscription = $.Deferred();\n\n            $(self.submitButton).prop('disabled', true);\n\n            subscriptionStatusResolver(email, newsletterSubscription);\n\n            $.when(newsletterSubscription).done(function (isSubscribed) {\n                if (isSubscribed) {\n                    $(self.signUpElement).prop('checked', true);\n                }\n            }).always(function () {\n                $(self.submitButton).prop('disabled', false);\n            });\n        }\n    });\n});\n","Magento_Newsletter/js/subscription-status-resolver.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/url'\n], function ($, urlBuilder) {\n    'use strict';\n\n    return function (email, deferred) {\n        return $.getJSON(\n            urlBuilder.build('newsletter/ajax/status'),\n            {\n                email: email\n            }\n        ).done(function (response) {\n            if (response.errors) {\n                deferred.reject();\n            } else {\n                deferred.resolve(response.subscribed);\n            }\n        }).fail(function () {\n            deferred.reject();\n        });\n    };\n});\n","Magento_OfflinePayments/js/view/payment/offline-payments.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/payment/renderer-list'\n], function (Component, rendererList) {\n    'use strict';\n\n    rendererList.push(\n        {\n            type: 'checkmo',\n            component: 'Magento_OfflinePayments/js/view/payment/method-renderer/checkmo-method'\n        },\n        {\n            type: 'banktransfer',\n            component: 'Magento_OfflinePayments/js/view/payment/method-renderer/banktransfer-method'\n        },\n        {\n            type: 'cashondelivery',\n            component: 'Magento_OfflinePayments/js/view/payment/method-renderer/cashondelivery-method'\n        },\n        {\n            type: 'purchaseorder',\n            component: 'Magento_OfflinePayments/js/view/payment/method-renderer/purchaseorder-method'\n        }\n    );\n\n    /** Add view logic here if needed */\n    return Component.extend({});\n});\n","Magento_OfflinePayments/js/view/payment/method-renderer/banktransfer-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'ko',\n    'Magento_Checkout/js/view/payment/default'\n], function (ko, Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_OfflinePayments/payment/banktransfer'\n        },\n\n        /**\n         * Get value of instruction field.\n         * @returns {String}\n         */\n        getInstructions: function () {\n            return window.checkoutConfig.payment.instructions[this.item.method];\n        }\n    });\n});\n","Magento_OfflinePayments/js/view/payment/method-renderer/cashondelivery-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'Magento_Checkout/js/view/payment/default'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_OfflinePayments/payment/cashondelivery'\n        },\n\n        /**\n         * Returns payment method instructions.\n         *\n         * @return {*}\n         */\n        getInstructions: function () {\n            return window.checkoutConfig.payment.instructions[this.item.method];\n        }\n    });\n});\n","Magento_OfflinePayments/js/view/payment/method-renderer/checkmo-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'Magento_Checkout/js/view/payment/default'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_OfflinePayments/payment/checkmo'\n        },\n\n        /**\n         * Returns send check to info.\n         *\n         * @return {*}\n         */\n        getMailingAddress: function () {\n            return window.checkoutConfig.payment.checkmo.mailingAddress;\n        },\n\n        /**\n         * Returns payable to info.\n         *\n         * @return {*}\n         */\n        getPayableTo: function () {\n            return window.checkoutConfig.payment.checkmo.payableTo;\n        }\n    });\n});\n","Magento_OfflinePayments/js/view/payment/method-renderer/purchaseorder-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'Magento_Checkout/js/view/payment/default',\n    'jquery',\n    'mage/validation'\n], function (Component, $) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_OfflinePayments/payment/purchaseorder-form',\n            purchaseOrderNumber: ''\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super()\n                .observe('purchaseOrderNumber');\n\n            return this;\n        },\n\n        /**\n         * @return {Object}\n         */\n        getData: function () {\n            return {\n                method: this.item.method,\n                'po_number': this.purchaseOrderNumber(),\n                'additional_data': null\n            };\n        },\n\n        /**\n         * @return {jQuery}\n         */\n        validate: function () {\n            var form = 'form[data-role=purchaseorder-form]';\n\n            return $(form).validation() && $(form).validation('isValid');\n        }\n    });\n});\n","Magento_OfflineShipping/js/model/shipping-rates-validation-rules/flatrate.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        /**\n         * @return {Object}\n         */\n        getRules: function () {\n            return {\n                'country_id': {\n                    'required': true\n                }\n            };\n        }\n    };\n});\n","Magento_OfflineShipping/js/model/shipping-rates-validation-rules/freeshipping.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        /**\n         * @return {Object}\n         */\n        getRules: function () {\n            return {\n                'country_id': {\n                    'required': true\n                }\n            };\n        }\n    };\n});\n","Magento_OfflineShipping/js/model/shipping-rates-validation-rules/tablerate.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        /**\n         * @return {Object}\n         */\n        getRules: function () {\n            return {\n                'postcode': {\n                    'required': true\n                },\n                'country_id': {\n                    'required': true\n                },\n                'region_id': {\n                    'required': true\n                },\n                'region_id_input': {\n                    'required': true\n                }\n            };\n        }\n    };\n});\n","Magento_OfflineShipping/js/model/shipping-rates-validator/flatrate.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mageUtils',\n    '../shipping-rates-validation-rules/flatrate',\n    'mage/translate'\n], function ($, utils, validationRules, $t) {\n    'use strict';\n\n    return {\n        validationErrors: [],\n\n        /**\n         * @param {Object} address\n         * @return {Boolean}\n         */\n        validate: function (address) {\n            var self = this;\n\n            this.validationErrors = [];\n            $.each(validationRules.getRules(), function (field, rule) {\n                var message;\n\n                if (rule.required && utils.isEmpty(address[field])) {\n                    message = $t('Field ') + field + $t(' is required.');\n                    self.validationErrors.push(message);\n                }\n            });\n\n            return !this.validationErrors.length;\n        }\n    };\n});\n","Magento_OfflineShipping/js/model/shipping-rates-validator/freeshipping.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mageUtils',\n    '../shipping-rates-validation-rules/freeshipping',\n    'mage/translate'\n], function ($, utils, validationRules, $t) {\n    'use strict';\n\n    return {\n        validationErrors: [],\n\n        /**\n         * @param {Object} address\n         * @return {Boolean}\n         */\n        validate: function (address) {\n            var self = this;\n\n            this.validationErrors = [];\n            $.each(validationRules.getRules(), function (field, rule) {\n                var message;\n\n                if (rule.required && utils.isEmpty(address[field])) {\n                    message = $t('Field ') + field + $t(' is required.');\n                    self.validationErrors.push(message);\n                }\n            });\n\n            return !this.validationErrors.length;\n        }\n    };\n});\n","Magento_OfflineShipping/js/model/shipping-rates-validator/tablerate.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mageUtils',\n    '../shipping-rates-validation-rules/tablerate',\n    'mage/translate'\n], function ($, utils, validationRules, $t) {\n    'use strict';\n\n    return {\n        validationErrors: [],\n\n        /**\n         * @param {Object} address\n         * @return {Boolean}\n         */\n        validate: function (address) {\n            var self = this;\n\n            this.validationErrors = [];\n            $.each(validationRules.getRules(), function (field, rule) {\n                var message, regionFields;\n\n                if (rule.required && utils.isEmpty(address[field])) {\n                    message = $t('Field ') + field + $t(' is required.');\n                    regionFields = ['region', 'region_id', 'region_id_input'];\n\n                    if (\n                        $.inArray(field, regionFields) === -1 ||\n                        utils.isEmpty(address.region) && utils.isEmpty(address['region_id'])\n                    ) {\n                        self.validationErrors.push(message);\n                    }\n                }\n            });\n\n            return !this.validationErrors.length;\n        }\n    };\n});\n","Magento_OfflineShipping/js/view/shipping-rates-validation/flatrate.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/shipping-rates-validator',\n    'Magento_Checkout/js/model/shipping-rates-validation-rules',\n    '../../model/shipping-rates-validator/flatrate',\n    '../../model/shipping-rates-validation-rules/flatrate'\n], function (\n    Component,\n    defaultShippingRatesValidator,\n    defaultShippingRatesValidationRules,\n    flatrateShippingRatesValidator,\n    flatrateShippingRatesValidationRules\n) {\n    'use strict';\n\n    defaultShippingRatesValidator.registerValidator('flatrate', flatrateShippingRatesValidator);\n    defaultShippingRatesValidationRules.registerRules('flatrate', flatrateShippingRatesValidationRules);\n\n    return Component;\n});\n","Magento_OfflineShipping/js/view/shipping-rates-validation/freeshipping.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/shipping-rates-validator',\n    'Magento_Checkout/js/model/shipping-rates-validation-rules',\n    '../../model/shipping-rates-validator/freeshipping',\n    '../../model/shipping-rates-validation-rules/freeshipping'\n], function (\n    Component,\n    defaultShippingRatesValidator,\n    defaultShippingRatesValidationRules,\n    freeshippingShippingRatesValidator,\n    freeshippingShippingRatesValidationRules\n) {\n    'use strict';\n\n    defaultShippingRatesValidator.registerValidator('freeshipping', freeshippingShippingRatesValidator);\n    defaultShippingRatesValidationRules.registerRules('freeshipping', freeshippingShippingRatesValidationRules);\n\n    return Component;\n});\n","Magento_OfflineShipping/js/view/shipping-rates-validation/tablerate.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/shipping-rates-validator',\n    'Magento_Checkout/js/model/shipping-rates-validation-rules',\n    '../../model/shipping-rates-validator/tablerate',\n    '../../model/shipping-rates-validation-rules/tablerate'\n], function (\n    Component,\n    defaultShippingRatesValidator,\n    defaultShippingRatesValidationRules,\n    tablerateShippingRatesValidator,\n    tablerateShippingRatesValidationRules\n) {\n    'use strict';\n\n    defaultShippingRatesValidator.registerValidator('tablerate', tablerateShippingRatesValidator);\n    defaultShippingRatesValidationRules.registerRules('tablerate', tablerateShippingRatesValidationRules);\n\n    return Component;\n});\n","Magento_PageBuilder/js/events.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['uiEvents'], function (uiEvents) {\n    'use strict';\n\n    return {\n\n        /**\n         * Calls callback when name event is triggered\n         *\n         * @param {String} events\n         * @param {Function} callback\n         * @param {Function} ns\n         * @return {Object}\n         */\n        on: function (events, callback, ns) {\n            uiEvents.on('pagebuilder:' + events, callback, 'pagebuilder:' + ns);\n\n            return this;\n        },\n\n        /**\n         * Removed callback from listening to target event\n         *\n         * @param {String} ns\n         * @return {Object}\n         */\n        off: function (ns) {\n            uiEvents.off('pagebuilder:' + ns);\n\n            return this;\n        },\n\n        /**\n         * Triggers event and executes all attached callbacks\n         *\n         * @param {String} name\n         * @param {any} args\n         * @returns {Boolean}\n         */\n        trigger: function (name, args) {\n            return uiEvents.trigger('pagebuilder:' + name, args);\n        }\n    };\n});\n","Magento_PageBuilder/js/widget-initializer.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'underscore',\n    'jquery',\n    'mage/apply/main',\n    'Magento_Ui/js/lib/view/utils/dom-observer'\n], function (_, $, mage, domObserver) {\n    'use strict';\n\n    /**\n     * Initializes components assigned to HTML elements.\n     *\n     *\n     * @param {HTMLElement} el\n     * @param {Array} data\n     * @param {Object} breakpoints\n     * @param {Object} currentViewport\n     */\n    function initializeWidget(el, data, breakpoints, currentViewport) {\n        _.each(data, function (config, component) {\n            config = config || {};\n            config.breakpoints = breakpoints;\n            config.currentViewport = currentViewport;\n            mage.applyFor(el, config, component);\n        });\n    }\n\n    return function (data, contextElement) {\n        _.each(data.config, function (componentConfiguration, elementPath) {\n            domObserver.get(\n                elementPath,\n                function (element) {\n                    var $element = $(element);\n\n                    if (contextElement) {\n                        $element = $(contextElement).find(element);\n                    }\n\n                    if ($element.length) {\n                        initializeWidget($element, componentConfiguration, data.breakpoints, data.currentViewport);\n                    }\n                }\n            );\n        });\n    };\n});\n","Magento_PageBuilder/js/content-type/banner/appearance/default/widget.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'Magento_PageBuilder/js/widget/show-on-hover',\n    'Magento_PageBuilder/js/widget/video-background'\n], function (showOnHover, videoBackground) {\n    'use strict';\n\n    return function (config, element) {\n        var videoElement = element[0].querySelector('[data-background-type=video]');\n\n        showOnHover(config);\n\n        if (videoElement) {\n            videoBackground(config, videoElement);\n        }\n    };\n});\n","Magento_PageBuilder/js/content-type/buttons/appearance/inline/widget.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'Magento_PageBuilder/js/events'\n], function ($, events) {\n    'use strict';\n\n    /**\n     * Equalize the width of a list of button-item components\n     *\n     * @param {JQuery} buttonList\n     */\n    var equalizeButtonWidth = function (buttonList) {\n        var buttonMinWidth = 0;\n\n        buttonList.css('min-width', buttonMinWidth);\n        buttonList.each(function () {\n            var buttonWidth = this.offsetWidth;\n\n            if (buttonWidth > buttonMinWidth) {\n                buttonMinWidth = buttonWidth;\n            }\n        });\n        buttonList.css('min-width', buttonMinWidth);\n    };\n\n    return function (config, element) {\n        var $element = $(element),\n            buttonSelector = '[data-element=\"link\"], [data-element=\"empty_link\"]';\n\n        if ($element.data('sameWidth')) {\n            equalizeButtonWidth($element.find(buttonSelector));\n            $(window).on('resize', function () {\n                equalizeButtonWidth($element.find(buttonSelector));\n            });\n            events.on('contentType:redrawAfter', function (eventData) {\n                if ($element.closest(eventData.element).length > 0) {\n                    equalizeButtonWidth($element.find(buttonSelector));\n                }\n            });\n        }\n    };\n});\n","Magento_PageBuilder/js/content-type/map/appearance/default/widget.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'Magento_PageBuilder/js/utils/map'\n], function ($, GoogleMap) {\n    'use strict';\n\n    return function (config, element) {\n        var locations,\n            controls,\n            mapOptions = {};\n\n        element = element[0];\n\n        if (element !== undefined && element.hasAttribute('data-locations')) {\n\n            /**\n             * Set map display to none if no locations\n             */\n            if (element.getAttribute('data-locations') === '[]') {\n                $(element).hide();\n\n                return;\n            }\n            locations = JSON.parse(element.getAttribute('data-locations'));\n            locations.forEach(function (location) {\n                location.position.latitude = parseFloat(location.position.latitude);\n                location.position.longitude = parseFloat(location.position.longitude);\n            });\n            controls = element.getAttribute('data-show-controls');\n            mapOptions.disableDefaultUI = controls !== 'true';\n            mapOptions.mapTypeControl = controls === 'true';\n            new GoogleMap(element, locations, mapOptions);\n        }\n    };\n});\n","Magento_PageBuilder/js/content-type/products/appearance/carousel/widget.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'matchMedia',\n    'Magento_PageBuilder/js/utils/breakpoints',\n    'Magento_PageBuilder/js/events',\n    'slick'\n], function ($, _, mediaCheck, breakpointsUtils, events) {\n    'use strict';\n\n    /**\n     * Build slick\n     *\n     * @param {jQuery} $carouselElement\n     * @param {Object} config\n     */\n    function buildSlick($carouselElement, config) {\n        /**\n         * Prevent each slick slider from being initialized more than once which could throw an error.\n         */\n        if ($carouselElement.hasClass('slick-initialized')) {\n            $carouselElement.slick('unslick');\n        }\n\n        config.slidesToScroll = config.slidesToShow;\n        $carouselElement.slick(config);\n    }\n\n    /**\n     * Initialize slider.\n     *\n     * @param {jQuery} $element\n     * @param {Object} slickConfig\n     * @param {Object} breakpoint\n     */\n    function initSlider($element, slickConfig, breakpoint) {\n        var productCount = $element.find('.product-item').length,\n            $carouselElement = $($element.children()),\n            centerModeClass = 'center-mode',\n            carouselMode = $element.data('carousel-mode'),\n            slidesToShow = breakpoint.options.products[carouselMode] ?\n                breakpoint.options.products[carouselMode].slidesToShow :\n                breakpoint.options.products.default.slidesToShow;\n\n        slickConfig.slidesToShow = parseFloat(slidesToShow);\n\n        if (carouselMode === 'continuous' && productCount > slickConfig.slidesToShow) {\n            $element.addClass(centerModeClass);\n            slickConfig.centerPadding = $element.data('center-padding');\n            slickConfig.centerMode = true;\n        } else {\n            $element.removeClass(centerModeClass);\n            slickConfig.infinite = $element.data('infinite-loop');\n        }\n\n        buildSlick($carouselElement, slickConfig);\n    }\n\n    return function (config, element) {\n        var $element = $(element),\n            $carouselElement = $($element.children()),\n            currentViewport = config.currentViewport,\n            currentBreakpoint = config.breakpoints[currentViewport],\n            slickConfig = {\n                autoplay: $element.data('autoplay'),\n                autoplaySpeed: $element.data('autoplay-speed') || 0,\n                arrows: $element.data('show-arrows'),\n                dots: $element.data('show-dots')\n            };\n\n        _.each(config.breakpoints, function (breakpoint) {\n            mediaCheck({\n                media: breakpointsUtils.buildMedia(breakpoint.conditions),\n\n                /** @inheritdoc */\n                entry: function () {\n                    initSlider($element, slickConfig, breakpoint);\n                }\n            });\n        });\n\n        //initialize slider when content type is added in mobile viewport\n        if (currentViewport === 'mobile') {\n            initSlider($element, slickConfig, currentBreakpoint);\n        }\n\n        // Redraw slide after content type gets redrawn\n        events.on('contentType:redrawAfter', function (args) {\n            if ($carouselElement.closest(args.element).length) {\n                $carouselElement.slick('setPosition');\n            }\n        });\n\n        events.on('stage:viewportChangeAfter', function (args) {\n            var breakpoint = config.breakpoints[args.viewport];\n\n            initSlider($element, slickConfig, breakpoint);\n        });\n    };\n});\n","Magento_PageBuilder/js/content-type/row/appearance/default/widget.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'Magento_PageBuilder/js/widget/video-background',\n    'jarallax'\n], function ($, videoBackground) {\n    'use strict';\n\n    return function (config, element) {\n        var $element = $(element),\n            parallaxSpeed = null,\n            elementStyle = null;\n\n        if ($element.data('appearance') === 'contained') {\n            $element = $(element).find('[data-element=\"inner\"]');\n        }\n\n        if ($element.data('background-type') === 'video') {\n            videoBackground(config, $element[0]);\n\n            return;\n        }\n\n        if ($element.data('enableParallax') !== 1) {\n            return;\n        }\n\n        $element.addClass('jarallax');\n        $element.attr('data-jarallax', '');\n\n        parallaxSpeed = parseFloat($element.data('parallaxSpeed'));\n        elementStyle = window.getComputedStyle($element[0]);\n\n        window.jarallax($element[0], {\n            imgPosition: elementStyle.backgroundPosition || '50% 50%',\n            imgRepeat: elementStyle.backgroundRepeat || 'no-repeat',\n            imgSize: elementStyle.backgroundSize || 'cover',\n            speed: !isNaN(parallaxSpeed) ? parallaxSpeed : 0.5\n        });\n    };\n});\n","Magento_PageBuilder/js/content-type/slide/appearance/default/widget.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'underscore',\n    'Magento_PageBuilder/js/widget/show-on-hover',\n    'Magento_PageBuilder/js/widget/video-background'\n], function ($, _, showOnHover, videoBackground) {\n    'use strict';\n\n    return function (config, element) {\n        var videoElement = element[0].querySelector('[data-background-type=video]'),\n            viewportElement = document.createElement('div'),\n            $slider = null;\n\n        showOnHover(config);\n\n        if (videoElement) {\n            $slider = $(element).closest('[data-content-type=slider]');\n            viewportElement.classList.add('jarallax-viewport-element');\n            videoElement.setAttribute('data-element-in-viewport', '.jarallax-viewport-element');\n            videoElement.appendChild(viewportElement);\n            videoBackground(config, videoElement);\n\n            if ($slider.data('afterChangeIsSet')) {\n                return;\n            }\n\n            $slider.on('afterChange init', function () {\n                var videoSlides = $slider[0].querySelectorAll('.jarallax');\n\n                _.each(videoSlides, function (videoSlide) {\n                    videoSlide.jarallax && videoSlide.jarallax.onScroll();\n                });\n            });\n            $slider.data('afterChangeIsSet', true);\n        }\n    };\n});\n","Magento_PageBuilder/js/content-type/slider/appearance/default/widget.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_PageBuilder/js/events',\n    'slick'\n], function ($, events) {\n    'use strict';\n\n    return function (config, sliderElement) {\n        var $element = $(sliderElement);\n\n        /**\n         * Prevent each slick slider from being initialized more than once which could throw an error.\n         */\n        if ($element.hasClass('slick-initialized')) {\n            $element.slick('unslick');\n        }\n\n        $element.slick({\n            autoplay: $element.data('autoplay'),\n            autoplaySpeed: $element.data('autoplay-speed') || 0,\n            fade: $element.data('fade'),\n            infinite: $element.data('infinite-loop'),\n            arrows: $element.data('show-arrows'),\n            dots: $element.data('show-dots')\n        });\n\n        // Redraw slide after content type gets redrawn\n        events.on('contentType:redrawAfter', function (args) {\n            if ($element.closest(args.element).length) {\n                $element.slick('setPosition');\n            }\n        });\n        // eslint-disable-next-line jquery-no-bind-unbind\n        events.on('stage:viewportChangeAfter', $element.slick.bind($element, 'setPosition'));\n    };\n});\n","Magento_PageBuilder/js/content-type/tabs/appearance/default/widget.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'Magento_PageBuilder/js/events',\n    'jquery-ui-modules/tabs'\n], function ($, events) {\n    'use strict';\n\n    return function (config, element) {\n        var $element = $(element);\n\n        // Ignore stage builder preview tabs\n        if ($element.is('.pagebuilder-tabs')) {\n            return;\n        }\n\n        // Disambiguate between the mage/tabs component which is loaded randomly depending on requirejs order.\n        $.ui.tabs({\n            active: $element.data('activeTab') || 0,\n            create:\n\n                /**\n                 * Adjust the margin bottom of the navigation to correctly display the active tab\n                 */\n                function () {\n                    var borderWidth = parseInt($element.find('.tabs-content').css('borderWidth').toString(), 10);\n\n                    $element.find('.tabs-navigation').css('marginBottom', -borderWidth);\n                    $element.find('.tabs-navigation li:not(:first-child)').css('marginLeft', -borderWidth);\n                },\n            activate:\n\n                /**\n                 * Trigger redraw event since new content is being displayed\n                 */\n                function () {\n                    events.trigger('contentType:redrawAfter', {\n                        element: element\n                    });\n                }\n        }, element);\n    };\n});\n","Magento_PageBuilder/js/resource/jarallax/jarallax-video.js":"/*!\n * Video Extension for Jarallax v2.0.3 (https://github.com/nk-o/jarallax)\n * Copyright 2022 nK <https://nkdev.info>\n * Licensed under MIT (https://github.com/nk-o/jarallax/blob/master/LICENSE)\n */\n\n(function (global, factory) {\n  typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :\n      typeof define === 'function' && define.amd ? define(factory) :\n          (global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.jarallaxVideo = factory());\n})(this, (function () { 'use strict';\n\n  /*!\n   * Name    : Video Worker\n   * Version : 2.0.0\n   * Author  : nK <https://nkdev.info>\n   * GitHub  : https://github.com/nk-o/video-worker\n   */\n\n  /* eslint-disable import/no-mutable-exports */\n\n  /* eslint-disable no-restricted-globals */\n  let win$1;\n\n  if (typeof window !== 'undefined') {\n    win$1 = window;\n  } else if (typeof global !== 'undefined') {\n    win$1 = global;\n  } else if (typeof self !== 'undefined') {\n    win$1 = self;\n  } else {\n    win$1 = {};\n  }\n\n  var global$1$1 = win$1; // Deferred\n  // thanks http://stackoverflow.com/questions/18096715/implement-deferred-object-without-using-jquery\n\n  function Deferred() {\n    this.doneCallbacks = [];\n    this.failCallbacks = [];\n  }\n\n  Deferred.prototype = {\n    execute(list, args) {\n      let i = list.length; // eslint-disable-next-line no-param-reassign\n\n      args = Array.prototype.slice.call(args);\n\n      while (i) {\n        i -= 1;\n        list[i].apply(null, args);\n      }\n    },\n\n    resolve(...args) {\n      this.execute(this.doneCallbacks, args);\n    },\n\n    reject(...args) {\n      this.execute(this.failCallbacks, args);\n    },\n\n    done(callback) {\n      this.doneCallbacks.push(callback);\n    },\n\n    fail(callback) {\n      this.failCallbacks.push(callback);\n    }\n\n  };\n  let ID = 0;\n  let YoutubeAPIadded = 0;\n  let VimeoAPIadded = 0;\n  let loadingYoutubePlayer = 0;\n  let loadingVimeoPlayer = 0;\n  const loadingYoutubeDefer = /*#__PURE__*/new Deferred();\n  const loadingVimeoDefer = /*#__PURE__*/new Deferred();\n\n  class VideoWorker {\n    constructor(url, options) {\n      const self = this;\n      self.url = url;\n      self.options_default = {\n        autoplay: false,\n        loop: false,\n        mute: false,\n        volume: 100,\n        showControls: true,\n        accessibilityHidden: false,\n        // start / end video time in seconds\n        startTime: 0,\n        endTime: 0\n      };\n      self.options = self.extend({}, self.options_default, options); // Fix wrong option name.\n      // Thanks to https://github.com/nk-o/video-worker/issues/13.\n\n      if (typeof self.options.showContols !== 'undefined') {\n        self.options.showControls = self.options.showContols;\n        delete self.options.showContols;\n      } // check URL\n\n\n      self.videoID = self.parseURL(url); // init\n\n      if (self.videoID) {\n        self.ID = ID;\n        ID += 1;\n        self.loadAPI();\n        self.init();\n      }\n    } // Extend like jQuery.extend\n    // eslint-disable-next-line class-methods-use-this\n\n\n    extend(...args) {\n      const out = args[0] || {};\n      Object.keys(args).forEach(i => {\n        if (!args[i]) {\n          return;\n        }\n\n        Object.keys(args[i]).forEach(key => {\n          out[key] = args[i][key];\n        });\n      });\n      return out;\n    }\n\n    parseURL(url) {\n      // parse youtube ID\n      function getYoutubeID(ytUrl) {\n        // eslint-disable-next-line no-useless-escape\n        const regExp = /.*(?:youtu.be\\/|v\\/|u\\/\\w\\/|embed\\/|watch\\?v=)([^#\\&\\?]*).*/;\n        const match = ytUrl.match(regExp);\n        return match && match[1].length === 11 ? match[1] : false;\n      } // parse vimeo ID\n\n\n      function getVimeoID(vmUrl) {\n        // eslint-disable-next-line no-useless-escape\n        const regExp = /https?:\\/\\/(?:www\\.|player\\.)?vimeo.com\\/(?:channels\\/(?:\\w+\\/)?|groups\\/([^/]*)\\/videos\\/|album\\/(\\d+)\\/video\\/|video\\/|)(\\d+)(?:$|\\/|\\?)/;\n        const match = vmUrl.match(regExp);\n        return match && match[3] ? match[3] : false;\n      } // parse local string\n\n\n      function getLocalVideos(locUrl) {\n        // eslint-disable-next-line no-useless-escape\n        const videoFormats = locUrl.split(/,(?=mp4\\:|webm\\:|ogv\\:|ogg\\:)/);\n        const result = {};\n        let ready = 0;\n        videoFormats.forEach(val => {\n          // eslint-disable-next-line no-useless-escape\n          const match = val.match(/^(mp4|webm|ogv|ogg)\\:(.*)/);\n\n          if (match && match[1] && match[2]) {\n            // eslint-disable-next-line prefer-destructuring\n            result[match[1] === 'ogv' ? 'ogg' : match[1]] = match[2];\n            ready = 1;\n          }\n        });\n        return ready ? result : false;\n      }\n\n      const Youtube = getYoutubeID(url);\n      const Vimeo = getVimeoID(url);\n      const Local = getLocalVideos(url);\n\n      if (Youtube) {\n        this.type = 'youtube';\n        return Youtube;\n      }\n\n      if (Vimeo) {\n        this.type = 'vimeo';\n        return Vimeo;\n      }\n\n      if (Local) {\n        this.type = 'local';\n        return Local;\n      }\n\n      return false;\n    }\n\n    isValid() {\n      return !!this.videoID;\n    } // events\n\n\n    on(name, callback) {\n      this.userEventsList = this.userEventsList || []; // add new callback in events list\n\n      (this.userEventsList[name] || (this.userEventsList[name] = [])).push(callback);\n    }\n\n    off(name, callback) {\n      if (!this.userEventsList || !this.userEventsList[name]) {\n        return;\n      }\n\n      if (!callback) {\n        delete this.userEventsList[name];\n      } else {\n        this.userEventsList[name].forEach((val, key) => {\n          if (val === callback) {\n            this.userEventsList[name][key] = false;\n          }\n        });\n      }\n    }\n\n    fire(name, ...args) {\n      if (this.userEventsList && typeof this.userEventsList[name] !== 'undefined') {\n        this.userEventsList[name].forEach(val => {\n          // call with all arguments\n          if (val) {\n            val.apply(this, args);\n          }\n        });\n      }\n    }\n\n    play(start) {\n      const self = this;\n\n      if (!self.player) {\n        return;\n      }\n\n      if (self.type === 'youtube' && self.player.playVideo) {\n        if (typeof start !== 'undefined') {\n          self.player.seekTo(start || 0);\n        }\n\n        if (global$1$1.YT.PlayerState.PLAYING !== self.player.getPlayerState()) {\n          self.player.playVideo();\n        }\n      }\n\n      if (self.type === 'vimeo') {\n        if (typeof start !== 'undefined') {\n          self.player.setCurrentTime(start);\n        }\n\n        self.player.getPaused().then(paused => {\n          if (paused) {\n            self.player.play();\n          }\n        });\n      }\n\n      if (self.type === 'local') {\n        if (typeof start !== 'undefined') {\n          self.player.currentTime = start;\n        }\n\n        if (self.player.paused) {\n          self.player.play();\n        }\n      }\n    }\n\n    pause() {\n      const self = this;\n\n      if (!self.player) {\n        return;\n      }\n\n      if (self.type === 'youtube' && self.player.pauseVideo) {\n        if (global$1$1.YT.PlayerState.PLAYING === self.player.getPlayerState()) {\n          self.player.pauseVideo();\n        }\n      }\n\n      if (self.type === 'vimeo') {\n        self.player.getPaused().then(paused => {\n          if (!paused) {\n            self.player.pause();\n          }\n        });\n      }\n\n      if (self.type === 'local') {\n        if (!self.player.paused) {\n          self.player.pause();\n        }\n      }\n    }\n\n    mute() {\n      const self = this;\n\n      if (!self.player) {\n        return;\n      }\n\n      if (self.type === 'youtube' && self.player.mute) {\n        self.player.mute();\n      }\n\n      if (self.type === 'vimeo' && self.player.setVolume) {\n        self.player.setVolume(0);\n      }\n\n      if (self.type === 'local') {\n        self.$video.muted = true;\n      }\n    }\n\n    unmute() {\n      const self = this;\n\n      if (!self.player) {\n        return;\n      }\n\n      if (self.type === 'youtube' && self.player.mute) {\n        self.player.unMute();\n      }\n\n      if (self.type === 'vimeo' && self.player.setVolume) {\n        self.player.setVolume(self.options.volume);\n      }\n\n      if (self.type === 'local') {\n        self.$video.muted = false;\n      }\n    }\n\n    setVolume(volume = false) {\n      const self = this;\n\n      if (!self.player || !volume) {\n        return;\n      }\n\n      if (self.type === 'youtube' && self.player.setVolume) {\n        self.player.setVolume(volume);\n      }\n\n      if (self.type === 'vimeo' && self.player.setVolume) {\n        self.player.setVolume(volume);\n      }\n\n      if (self.type === 'local') {\n        self.$video.volume = volume / 100;\n      }\n    }\n\n    getVolume(callback) {\n      const self = this;\n\n      if (!self.player) {\n        callback(false);\n        return;\n      }\n\n      if (self.type === 'youtube' && self.player.getVolume) {\n        callback(self.player.getVolume());\n      }\n\n      if (self.type === 'vimeo' && self.player.getVolume) {\n        self.player.getVolume().then(volume => {\n          callback(volume);\n        });\n      }\n\n      if (self.type === 'local') {\n        callback(self.$video.volume * 100);\n      }\n    }\n\n    getMuted(callback) {\n      const self = this;\n\n      if (!self.player) {\n        callback(null);\n        return;\n      }\n\n      if (self.type === 'youtube' && self.player.isMuted) {\n        callback(self.player.isMuted());\n      }\n\n      if (self.type === 'vimeo' && self.player.getVolume) {\n        self.player.getVolume().then(volume => {\n          callback(!!volume);\n        });\n      }\n\n      if (self.type === 'local') {\n        callback(self.$video.muted);\n      }\n    }\n\n    getImageURL(callback) {\n      const self = this;\n\n      if (self.videoImage) {\n        callback(self.videoImage);\n        return;\n      }\n\n      if (self.type === 'youtube') {\n        const availableSizes = ['maxresdefault', 'sddefault', 'hqdefault', '0'];\n        let step = 0;\n        const tempImg = new Image();\n\n        tempImg.onload = function () {\n          // if no thumbnail, youtube add their own image with width = 120px\n          if ((this.naturalWidth || this.width) !== 120 || step === availableSizes.length - 1) {\n            // ok\n            self.videoImage = `https://img.youtube.com/vi/${self.videoID}/${availableSizes[step]}.jpg`;\n            callback(self.videoImage);\n          } else {\n            // try another size\n            step += 1;\n            this.src = `https://img.youtube.com/vi/${self.videoID}/${availableSizes[step]}.jpg`;\n          }\n        };\n\n        tempImg.src = `https://img.youtube.com/vi/${self.videoID}/${availableSizes[step]}.jpg`;\n      }\n\n      if (self.type === 'vimeo') {\n        let request = new XMLHttpRequest(); // https://vimeo.com/api/oembed.json?url=https://vimeo.com/235212527\n\n        request.open('GET', `https://vimeo.com/api/oembed.json?url=${self.url}`, true);\n\n        request.onreadystatechange = function () {\n          if (this.readyState === 4) {\n            if (this.status >= 200 && this.status < 400) {\n              // Success!\n              const response = JSON.parse(this.responseText);\n\n              if (response.thumbnail_url) {\n                self.videoImage = response.thumbnail_url;\n                callback(self.videoImage);\n              }\n            }\n          }\n        };\n\n        request.send();\n        request = null;\n      }\n    } // fallback to the old version.\n\n\n    getIframe(callback) {\n      this.getVideo(callback);\n    }\n\n    getVideo(callback) {\n      const self = this; // return generated video block\n\n      if (self.$video) {\n        callback(self.$video);\n        return;\n      } // generate new video block\n\n\n      self.onAPIready(() => {\n        let hiddenDiv;\n\n        if (!self.$video) {\n          hiddenDiv = document.createElement('div');\n          hiddenDiv.style.display = 'none';\n        } // Youtube\n\n\n        if (self.type === 'youtube') {\n          self.playerOptions = {\n            // GDPR Compliance.\n            host: 'https://www.youtube-nocookie.com',\n            videoId: self.videoID,\n            playerVars: {\n              autohide: 1,\n              rel: 0,\n              autoplay: 0,\n              // autoplay enable on mobile devices\n              playsinline: 1\n            }\n          }; // hide controls\n\n          if (!self.options.showControls) {\n            self.playerOptions.playerVars.iv_load_policy = 3;\n            self.playerOptions.playerVars.modestbranding = 1;\n            self.playerOptions.playerVars.controls = 0;\n            self.playerOptions.playerVars.showinfo = 0;\n            self.playerOptions.playerVars.disablekb = 1;\n          } // events\n\n\n          let ytStarted;\n          let ytProgressInterval;\n          self.playerOptions.events = {\n            onReady(e) {\n              // mute\n              if (self.options.mute) {\n                e.target.mute();\n              } else if (self.options.volume) {\n                e.target.setVolume(self.options.volume);\n              } // autoplay\n\n\n              if (self.options.autoplay) {\n                self.play(self.options.startTime);\n              }\n\n              self.fire('ready', e); // For seamless loops, set the endTime to 0.1 seconds less than the video's duration\n              // https://github.com/nk-o/video-worker/issues/2\n\n              if (self.options.loop && !self.options.endTime) {\n                const secondsOffset = 0.1;\n                self.options.endTime = self.player.getDuration() - secondsOffset;\n              } // volumechange\n\n\n              setInterval(() => {\n                self.getVolume(volume => {\n                  if (self.options.volume !== volume) {\n                    self.options.volume = volume;\n                    self.fire('volumechange', e);\n                  }\n                });\n              }, 150);\n            },\n\n            onStateChange(e) {\n              // loop\n              if (self.options.loop && e.data === global$1$1.YT.PlayerState.ENDED) {\n                self.play(self.options.startTime);\n              }\n\n              if (!ytStarted && e.data === global$1$1.YT.PlayerState.PLAYING) {\n                ytStarted = 1;\n                self.fire('started', e);\n              }\n\n              if (e.data === global$1$1.YT.PlayerState.PLAYING) {\n                self.fire('play', e);\n              }\n\n              if (e.data === global$1$1.YT.PlayerState.PAUSED) {\n                self.fire('pause', e);\n              }\n\n              if (e.data === global$1$1.YT.PlayerState.ENDED) {\n                self.fire('ended', e);\n              } // progress check\n\n\n              if (e.data === global$1$1.YT.PlayerState.PLAYING) {\n                ytProgressInterval = setInterval(() => {\n                  self.fire('timeupdate', e); // check for end of video and play again or stop\n\n                  if (self.options.endTime && self.player.getCurrentTime() >= self.options.endTime) {\n                    if (self.options.loop) {\n                      self.play(self.options.startTime);\n                    } else {\n                      self.pause();\n                    }\n                  }\n                }, 150);\n              } else {\n                clearInterval(ytProgressInterval);\n              }\n            },\n\n            onError(e) {\n              self.fire('error', e);\n            }\n\n          };\n          const firstInit = !self.$video;\n\n          if (firstInit) {\n            const div = document.createElement('div');\n            div.setAttribute('id', self.playerID);\n            hiddenDiv.appendChild(div);\n            document.body.appendChild(hiddenDiv);\n          }\n\n          self.player = self.player || new global$1$1.YT.Player(self.playerID, self.playerOptions);\n\n          if (firstInit) {\n            self.$video = document.getElementById(self.playerID); // add accessibility attributes\n\n            if (self.options.accessibilityHidden) {\n              self.$video.setAttribute('tabindex', '-1');\n              self.$video.setAttribute('aria-hidden', 'true');\n            } // get video width and height\n\n\n            self.videoWidth = parseInt(self.$video.getAttribute('width'), 10) || 1280;\n            self.videoHeight = parseInt(self.$video.getAttribute('height'), 10) || 720;\n          }\n        } // Vimeo\n\n\n        if (self.type === 'vimeo') {\n          self.playerOptions = {\n            // GDPR Compliance.\n            dnt: 1,\n            id: self.videoID,\n            autopause: 0,\n            transparent: 0,\n            autoplay: self.options.autoplay ? 1 : 0,\n            loop: self.options.loop ? 1 : 0,\n            muted: self.options.mute ? 1 : 0\n          };\n\n          if (self.options.volume) {\n            self.playerOptions.volume = self.options.volume;\n          } // hide controls\n\n\n          if (!self.options.showControls) {\n            self.playerOptions.badge = 0;\n            self.playerOptions.byline = 0;\n            self.playerOptions.portrait = 0;\n            self.playerOptions.title = 0;\n            self.playerOptions.background = 1;\n          }\n\n          if (!self.$video) {\n            let playerOptionsString = '';\n            Object.keys(self.playerOptions).forEach(key => {\n              if (playerOptionsString !== '') {\n                playerOptionsString += '&';\n              }\n\n              playerOptionsString += `${key}=${encodeURIComponent(self.playerOptions[key])}`;\n            }); // we need to create iframe manually because when we create it using API\n            // js events won't triggers after iframe moved to another place\n\n            self.$video = document.createElement('iframe');\n            self.$video.setAttribute('id', self.playerID);\n            self.$video.setAttribute('src', `https://player.vimeo.com/video/${self.videoID}?${playerOptionsString}`);\n            self.$video.setAttribute('frameborder', '0');\n            self.$video.setAttribute('mozallowfullscreen', '');\n            self.$video.setAttribute('allowfullscreen', '');\n            self.$video.setAttribute('title', 'Vimeo video player'); // add accessibility attributes\n\n            if (self.options.accessibilityHidden) {\n              self.$video.setAttribute('tabindex', '-1');\n              self.$video.setAttribute('aria-hidden', 'true');\n            }\n\n            hiddenDiv.appendChild(self.$video);\n            document.body.appendChild(hiddenDiv);\n          }\n\n          self.player = self.player || new global$1$1.Vimeo.Player(self.$video, self.playerOptions); // set current time for autoplay\n\n          if (self.options.startTime && self.options.autoplay) {\n            self.player.setCurrentTime(self.options.startTime);\n          } // get video width and height\n\n\n          self.player.getVideoWidth().then(width => {\n            self.videoWidth = width || 1280;\n          });\n          self.player.getVideoHeight().then(height => {\n            self.videoHeight = height || 720;\n          }); // events\n\n          let vmStarted;\n          self.player.on('timeupdate', e => {\n            if (!vmStarted) {\n              self.fire('started', e);\n              vmStarted = 1;\n            }\n\n            self.fire('timeupdate', e); // check for end of video and play again or stop\n\n            if (self.options.endTime) {\n              if (self.options.endTime && e.seconds >= self.options.endTime) {\n                if (self.options.loop) {\n                  self.play(self.options.startTime);\n                } else {\n                  self.pause();\n                }\n              }\n            }\n          });\n          self.player.on('play', e => {\n            self.fire('play', e); // check for the start time and start with it\n\n            if (self.options.startTime && e.seconds === 0) {\n              self.play(self.options.startTime);\n            }\n          });\n          self.player.on('pause', e => {\n            self.fire('pause', e);\n          });\n          self.player.on('ended', e => {\n            self.fire('ended', e);\n          });\n          self.player.on('loaded', e => {\n            self.fire('ready', e);\n          });\n          self.player.on('volumechange', e => {\n            self.fire('volumechange', e);\n          });\n          self.player.on('error', e => {\n            self.fire('error', e);\n          });\n        } // Local\n\n\n        function addSourceToLocal(element, src, type) {\n          const source = document.createElement('source');\n          source.src = src;\n          source.type = type;\n          element.appendChild(source);\n        }\n\n        if (self.type === 'local') {\n          if (!self.$video) {\n            self.$video = document.createElement('video'); // show controls\n\n            if (self.options.showControls) {\n              self.$video.controls = true;\n            } // mute\n\n\n            if (self.options.mute) {\n              self.$video.muted = true;\n            } else if (self.$video.volume) {\n              self.$video.volume = self.options.volume / 100;\n            } // loop\n\n\n            if (self.options.loop) {\n              self.$video.loop = true;\n            } // autoplay enable on mobile devices\n\n\n            self.$video.setAttribute('playsinline', '');\n            self.$video.setAttribute('webkit-playsinline', ''); // add accessibility attributes\n\n            if (self.options.accessibilityHidden) {\n              self.$video.setAttribute('tabindex', '-1');\n              self.$video.setAttribute('aria-hidden', 'true');\n            }\n\n            self.$video.setAttribute('id', self.playerID);\n            hiddenDiv.appendChild(self.$video);\n            document.body.appendChild(hiddenDiv);\n            Object.keys(self.videoID).forEach(key => {\n              addSourceToLocal(self.$video, self.videoID[key], `video/${key}`);\n            });\n          }\n\n          self.player = self.player || self.$video;\n          let locStarted;\n          self.player.addEventListener('playing', e => {\n            if (!locStarted) {\n              self.fire('started', e);\n            }\n\n            locStarted = 1;\n          });\n          self.player.addEventListener('timeupdate', function (e) {\n            self.fire('timeupdate', e); // check for end of video and play again or stop\n\n            if (self.options.endTime) {\n              if (self.options.endTime && this.currentTime >= self.options.endTime) {\n                if (self.options.loop) {\n                  self.play(self.options.startTime);\n                } else {\n                  self.pause();\n                }\n              }\n            }\n          });\n          self.player.addEventListener('play', e => {\n            self.fire('play', e);\n          });\n          self.player.addEventListener('pause', e => {\n            self.fire('pause', e);\n          });\n          self.player.addEventListener('ended', e => {\n            self.fire('ended', e);\n          });\n          self.player.addEventListener('loadedmetadata', function () {\n            // get video width and height\n            self.videoWidth = this.videoWidth || 1280;\n            self.videoHeight = this.videoHeight || 720;\n            self.fire('ready'); // autoplay\n\n            if (self.options.autoplay) {\n              self.play(self.options.startTime);\n            }\n          });\n          self.player.addEventListener('volumechange', e => {\n            self.getVolume(volume => {\n              self.options.volume = volume;\n            });\n            self.fire('volumechange', e);\n          });\n          self.player.addEventListener('error', e => {\n            self.fire('error', e);\n          });\n        }\n\n        callback(self.$video);\n      });\n    }\n\n    init() {\n      const self = this;\n      self.playerID = `VideoWorker-${self.ID}`;\n    }\n\n    loadAPI() {\n      const self = this;\n\n      if (YoutubeAPIadded && VimeoAPIadded) {\n        return;\n      }\n\n      let src = ''; // load Youtube API\n\n      if (self.type === 'youtube' && !YoutubeAPIadded) {\n        YoutubeAPIadded = 1;\n        src = 'https://www.youtube.com/iframe_api';\n      } // load Vimeo API\n\n\n      if (self.type === 'vimeo' && !VimeoAPIadded) {\n        VimeoAPIadded = 1; // Useful when Vimeo API added using RequireJS https://github.com/nk-o/video-worker/pull/7\n\n        if (typeof global$1$1.Vimeo !== 'undefined') {\n          return;\n        }\n\n        src = 'https://player.vimeo.com/api/player.js';\n      }\n\n      if (!src) {\n        return;\n      } // add script in head section\n\n\n      let tag = document.createElement('script');\n      let head = document.getElementsByTagName('head')[0];\n      tag.src = src;\n      head.appendChild(tag);\n      head = null;\n      tag = null;\n    }\n\n    onAPIready(callback) {\n      const self = this; // Youtube\n\n      if (self.type === 'youtube') {\n        // Listen for global YT player callback\n        if ((typeof global$1$1.YT === 'undefined' || global$1$1.YT.loaded === 0) && !loadingYoutubePlayer) {\n          // Prevents Ready event from being called twice\n          loadingYoutubePlayer = 1; // Creates deferred so, other players know when to wait.\n\n          global$1$1.onYouTubeIframeAPIReady = function () {\n            global$1$1.onYouTubeIframeAPIReady = null;\n            loadingYoutubeDefer.resolve('done');\n            callback();\n          };\n        } else if (typeof global$1$1.YT === 'object' && global$1$1.YT.loaded === 1) {\n          callback();\n        } else {\n          loadingYoutubeDefer.done(() => {\n            callback();\n          });\n        }\n      } // Vimeo\n\n\n      if (self.type === 'vimeo') {\n        if (typeof global$1$1.Vimeo === 'undefined' && !loadingVimeoPlayer) {\n          loadingVimeoPlayer = 1;\n          const vimeoInterval = setInterval(() => {\n            if (typeof global$1$1.Vimeo !== 'undefined') {\n              clearInterval(vimeoInterval);\n              loadingVimeoDefer.resolve('done');\n              callback();\n            }\n          }, 20);\n        } else if (typeof global$1$1.Vimeo !== 'undefined') {\n          callback();\n        } else {\n          loadingVimeoDefer.done(() => {\n            callback();\n          });\n        }\n      } // Local\n\n\n      if (self.type === 'local') {\n        callback();\n      }\n    }\n\n  }\n\n  function ready(callback) {\n    if ('complete' === document.readyState || 'interactive' === document.readyState) {\n      // Already ready or interactive, execute callback\n      callback();\n    } else {\n      document.addEventListener('DOMContentLoaded', callback, {\n        capture: true,\n        once: true,\n        passive: true\n      });\n    }\n  }\n\n  /* eslint-disable import/no-mutable-exports */\n\n  /* eslint-disable no-restricted-globals */\n  let win;\n\n  if ('undefined' !== typeof window) {\n    win = window;\n  } else if ('undefined' !== typeof global) {\n    win = global;\n  } else if ('undefined' !== typeof self) {\n    win = self;\n  } else {\n    win = {};\n  }\n\n  var global$1 = win;\n\n  function jarallaxVideo(jarallax = global$1.jarallax) {\n    if ('undefined' === typeof jarallax) {\n      return;\n    }\n\n    const Jarallax = jarallax.constructor; // append video after when block will be visible.\n\n    const defOnScroll = Jarallax.prototype.onScroll;\n\n    Jarallax.prototype.onScroll = function () {\n      const self = this;\n      defOnScroll.apply(self);\n      const isReady = !self.isVideoInserted && self.video && (!self.options.videoLazyLoading || self.isElementInViewport) && !self.options.disableVideo();\n\n      if (isReady) {\n        self.isVideoInserted = true;\n        self.video.getVideo(video => {\n          const $parent = video.parentNode;\n          self.css(video, {\n            position: self.image.position,\n            top: '0px',\n            left: '0px',\n            right: '0px',\n            bottom: '0px',\n            width: '100%',\n            height: '100%',\n            maxWidth: 'none',\n            maxHeight: 'none',\n            pointerEvents: 'none',\n            transformStyle: 'preserve-3d',\n            backfaceVisibility: 'hidden',\n            willChange: 'transform,opacity',\n            margin: 0,\n            zIndex: -1\n          });\n          self.$video = video; // add Poster attribute to self-hosted video\n\n          if ('local' === self.video.type) {\n            if (self.image.src) {\n              self.$video.setAttribute('poster', self.image.src);\n            } else if (self.image.$item && 'IMG' === self.image.$item.tagName && self.image.$item.src) {\n              self.$video.setAttribute('poster', self.image.$item.src);\n            }\n          } // insert video tag\n\n\n          self.image.$container.appendChild(video); // remove parent video element (created by VideoWorker)\n\n          $parent.parentNode.removeChild($parent); // call onVideoInsert event\n\n          if (self.options.onVideoInsert) {\n            self.options.onVideoInsert.call(self);\n          }\n        });\n      }\n    }; // cover video\n\n\n    const defCoverImage = Jarallax.prototype.coverImage;\n\n    Jarallax.prototype.coverImage = function () {\n      const self = this;\n      const imageData = defCoverImage.apply(self);\n      const node = self.image.$item ? self.image.$item.nodeName : false;\n\n      if (imageData && self.video && node && ('IFRAME' === node || 'VIDEO' === node)) {\n        let h = imageData.image.height;\n        let w = h * self.image.width / self.image.height;\n        let ml = (imageData.container.width - w) / 2;\n        let mt = imageData.image.marginTop;\n\n        if (imageData.container.width > w) {\n          w = imageData.container.width;\n          h = w * self.image.height / self.image.width;\n          ml = 0;\n          mt += (imageData.image.height - h) / 2;\n        } // add video height over than need to hide controls\n\n\n        if ('IFRAME' === node) {\n          h += 400;\n          mt -= 200;\n        }\n\n        self.css(self.$video, {\n          width: `${w}px`,\n          marginLeft: `${ml}px`,\n          height: `${h}px`,\n          marginTop: `${mt}px`\n        });\n      }\n\n      return imageData;\n    }; // init video\n\n\n    const defInitImg = Jarallax.prototype.initImg;\n\n    Jarallax.prototype.initImg = function () {\n      const self = this;\n      const defaultResult = defInitImg.apply(self);\n\n      if (!self.options.videoSrc) {\n        self.options.videoSrc = self.$item.getAttribute('data-jarallax-video') || null;\n      }\n\n      if (self.options.videoSrc) {\n        self.defaultInitImgResult = defaultResult;\n        return true;\n      }\n\n      return defaultResult;\n    };\n\n    const defCanInitParallax = Jarallax.prototype.canInitParallax;\n\n    Jarallax.prototype.canInitParallax = function () {\n      const self = this;\n      let defaultResult = defCanInitParallax.apply(self);\n\n      if (!self.options.videoSrc) {\n        return defaultResult;\n      } // Init video api\n\n\n      const video = new VideoWorker(self.options.videoSrc, {\n        autoplay: true,\n        loop: self.options.videoLoop,\n        showControls: false,\n        accessibilityHidden: true,\n        startTime: self.options.videoStartTime || 0,\n        endTime: self.options.videoEndTime || 0,\n        mute: self.options.videoVolume ? 0 : 1,\n        volume: self.options.videoVolume || 0\n      }); // call onVideoWorkerInit event\n\n      if (self.options.onVideoWorkerInit) {\n        self.options.onVideoWorkerInit.call(self, video);\n      }\n\n      function resetDefaultImage() {\n        if (self.image.$default_item) {\n          self.image.$item = self.image.$default_item;\n          self.image.$item.style.display = 'block'; // set image width and height\n\n          self.coverImage();\n          self.onScroll();\n        }\n      }\n\n      if (video.isValid()) {\n        // Force enable parallax.\n        // When the parallax disabled on mobile devices, we still need to display videos.\n        // https://github.com/nk-o/jarallax/issues/159\n        if (this.options.disableParallax()) {\n          defaultResult = true;\n          self.image.position = 'absolute';\n          self.options.type = 'scroll';\n          self.options.speed = 1;\n        } // if parallax will not be inited, we can add thumbnail on background.\n\n\n        if (!defaultResult) {\n          if (!self.defaultInitImgResult) {\n            video.getImageURL(url => {\n              // save default user styles\n              const curStyle = self.$item.getAttribute('style');\n\n              if (curStyle) {\n                self.$item.setAttribute('data-jarallax-original-styles', curStyle);\n              } // set new background\n\n\n              self.css(self.$item, {\n                'background-image': `url(\"${url}\")`,\n                'background-position': 'center',\n                'background-size': 'cover'\n              });\n            });\n          } // init video\n\n        } else {\n          video.on('ready', () => {\n            if (self.options.videoPlayOnlyVisible) {\n              const oldOnScroll = self.onScroll;\n\n              self.onScroll = function () {\n                oldOnScroll.apply(self);\n\n                if (!self.videoError && (self.options.videoLoop || !self.options.videoLoop && !self.videoEnded)) {\n                  if (self.isVisible()) {\n                    video.play();\n                  } else {\n                    video.pause();\n                  }\n                }\n              };\n            } else {\n              video.play();\n            }\n          });\n          video.on('started', () => {\n            self.image.$default_item = self.image.$item;\n            self.image.$item = self.$video; // set video width and height\n\n            self.image.width = self.video.videoWidth || 1280;\n            self.image.height = self.video.videoHeight || 720;\n            self.coverImage();\n            self.onScroll(); // hide image\n\n            if (self.image.$default_item) {\n              self.image.$default_item.style.display = 'none';\n            }\n          });\n          video.on('ended', () => {\n            self.videoEnded = true;\n\n            if (!self.options.videoLoop) {\n              // show default image if Loop disabled.\n              resetDefaultImage();\n            }\n          });\n          video.on('error', () => {\n            self.videoError = true; // show default image if video loading error.\n\n            resetDefaultImage();\n          });\n          self.video = video; // set image if not exists\n\n          if (!self.defaultInitImgResult) {\n            // set empty image on self-hosted video if not defined\n            self.image.src = 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';\n\n            if ('local' !== video.type) {\n              video.getImageURL(url => {\n                self.image.bgImage = `url(\"${url}\")`;\n                self.init();\n              });\n              return false;\n            }\n          }\n        }\n      }\n\n      return defaultResult;\n    }; // Destroy video parallax\n\n\n    const defDestroy = Jarallax.prototype.destroy;\n\n    Jarallax.prototype.destroy = function () {\n      const self = this;\n\n      if (self.image.$default_item) {\n        self.image.$item = self.image.$default_item;\n        delete self.image.$default_item;\n      }\n\n      defDestroy.apply(self);\n    };\n  }\n\n  jarallaxVideo(); // data-jarallax-video initialization\n\n  ready(() => {\n    if ('undefined' !== typeof global$1.jarallax) {\n      global$1.jarallax(document.querySelectorAll('[data-jarallax-video]'));\n    }\n  }); // We should add VideoWorker globally, since some project uses it.\n\n  if (!global$1.VideoWorker) {\n    global$1.VideoWorker = VideoWorker;\n  }\n\n  return jarallaxVideo;\n\n}));\n//# sourceMappingURL=jarallax-video.js.map\n","Magento_PageBuilder/js/resource/jarallax/jarallax-wrapper.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_PageBuilder/js/resource/jarallax/jarallax'\n], function(jarallax){\n    'use strict';\n\n    window.jarallax = window.jarallax || jarallax;\n});\n","Magento_PageBuilder/js/resource/jarallax/jarallax.js":"/*!\n * Jarallax v2.0.3 (https://github.com/nk-o/jarallax)\n * Copyright 2022 nK <https://nkdev.info>\n * Licensed under MIT (https://github.com/nk-o/jarallax/blob/master/LICENSE)\n */\n(function (global, factory) {\n  typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :\n      typeof define === 'function' && define.amd ? define(factory) :\n          (global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.jarallax = factory());\n})(this, (function () { 'use strict';\n\n  function ready(callback) {\n    if ('complete' === document.readyState || 'interactive' === document.readyState) {\n      // Already ready or interactive, execute callback\n      callback();\n    } else {\n      document.addEventListener('DOMContentLoaded', callback, {\n        capture: true,\n        once: true,\n        passive: true\n      });\n    }\n  }\n\n  /* eslint-disable import/no-mutable-exports */\n\n  /* eslint-disable no-restricted-globals */\n  let win;\n\n  if ('undefined' !== typeof window) {\n    win = window;\n  } else if ('undefined' !== typeof global) {\n    win = global;\n  } else if ('undefined' !== typeof self) {\n    win = self;\n  } else {\n    win = {};\n  }\n\n  var global$1 = win;\n\n  const {\n    navigator\n  } = global$1;\n  const isMobile = /*#__PURE__*/ /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);\n  let $deviceHelper;\n  /**\n   * The most popular mobile browsers changes height after page scroll and this generates image jumping.\n   * We can fix it using this workaround with vh units.\n   */\n\n  function getDeviceHeight() {\n    if (!$deviceHelper && document.body) {\n      $deviceHelper = document.createElement('div');\n      $deviceHelper.style.cssText = 'position: fixed; top: -9999px; left: 0; height: 100vh; width: 0;';\n      document.body.appendChild($deviceHelper);\n    }\n\n    return ($deviceHelper ? $deviceHelper.clientHeight : 0) || global$1.innerHeight || document.documentElement.clientHeight;\n  } // Window height data\n\n\n  let wndH;\n\n  function updateWndVars() {\n    if (isMobile) {\n      wndH = getDeviceHeight();\n    } else {\n      wndH = global$1.innerHeight || document.documentElement.clientHeight;\n    }\n  }\n\n  updateWndVars();\n  global$1.addEventListener('resize', updateWndVars);\n  global$1.addEventListener('orientationchange', updateWndVars);\n  global$1.addEventListener('load', updateWndVars);\n  ready(() => {\n    updateWndVars();\n  }); // list with all jarallax instances\n  // need to render all in one scroll/resize event\n\n  const jarallaxList = []; // get all parents of the element.\n\n  function getParents(elem) {\n    const parents = [];\n\n    while (null !== elem.parentElement) {\n      elem = elem.parentElement;\n\n      if (1 === elem.nodeType) {\n        parents.push(elem);\n      }\n    }\n\n    return parents;\n  }\n\n  function updateParallax() {\n    if (!jarallaxList.length) {\n      return;\n    }\n\n    jarallaxList.forEach((data, k) => {\n      const {\n        instance,\n        oldData\n      } = data;\n      const clientRect = instance.$item.getBoundingClientRect();\n      const newData = {\n        width: clientRect.width,\n        height: clientRect.height,\n        top: clientRect.top,\n        bottom: clientRect.bottom,\n        wndW: global$1.innerWidth,\n        wndH\n      };\n      const isResized = !oldData || oldData.wndW !== newData.wndW || oldData.wndH !== newData.wndH || oldData.width !== newData.width || oldData.height !== newData.height;\n      const isScrolled = isResized || !oldData || oldData.top !== newData.top || oldData.bottom !== newData.bottom;\n      jarallaxList[k].oldData = newData;\n\n      if (isResized) {\n        instance.onResize();\n      }\n\n      if (isScrolled) {\n        instance.onScroll();\n      }\n    });\n    global$1.requestAnimationFrame(updateParallax);\n  }\n\n  let instanceID = 0; // Jarallax class\n\n  class Jarallax {\n    constructor(item, userOptions) {\n      const self = this;\n      self.instanceID = instanceID;\n      instanceID += 1;\n      self.$item = item;\n      self.defaults = {\n        type: 'scroll',\n        // type of parallax: scroll, scale, opacity, scale-opacity, scroll-opacity\n        speed: 0.5,\n        // supported value from -1 to 2\n        imgSrc: null,\n        imgElement: '.jarallax-img',\n        imgSize: 'cover',\n        imgPosition: '50% 50%',\n        imgRepeat: 'no-repeat',\n        // supported only for background, not for <img> tag\n        keepImg: false,\n        // keep <img> tag in it's default place\n        elementInViewport: null,\n        zIndex: -100,\n        disableParallax: false,\n        disableVideo: false,\n        // video\n        videoSrc: null,\n        videoStartTime: 0,\n        videoEndTime: 0,\n        videoVolume: 0,\n        videoLoop: true,\n        videoPlayOnlyVisible: true,\n        videoLazyLoading: true,\n        // events\n        onScroll: null,\n        // function(calculations) {}\n        onInit: null,\n        // function() {}\n        onDestroy: null,\n        // function() {}\n        onCoverImage: null // function() {}\n\n      }; // prepare data-options\n\n      const dataOptions = self.$item.dataset || {};\n      const pureDataOptions = {};\n      Object.keys(dataOptions).forEach(key => {\n        const loweCaseOption = key.substr(0, 1).toLowerCase() + key.substr(1);\n\n        if (loweCaseOption && 'undefined' !== typeof self.defaults[loweCaseOption]) {\n          pureDataOptions[loweCaseOption] = dataOptions[key];\n        }\n      });\n      self.options = self.extend({}, self.defaults, pureDataOptions, userOptions);\n      self.pureOptions = self.extend({}, self.options); // prepare 'true' and 'false' strings to boolean\n\n      Object.keys(self.options).forEach(key => {\n        if ('true' === self.options[key]) {\n          self.options[key] = true;\n        } else if ('false' === self.options[key]) {\n          self.options[key] = false;\n        }\n      }); // fix speed option [-1.0, 2.0]\n\n      self.options.speed = Math.min(2, Math.max(-1, parseFloat(self.options.speed))); // prepare disableParallax callback\n\n      if ('string' === typeof self.options.disableParallax) {\n        self.options.disableParallax = new RegExp(self.options.disableParallax);\n      }\n\n      if (self.options.disableParallax instanceof RegExp) {\n        const disableParallaxRegexp = self.options.disableParallax;\n\n        self.options.disableParallax = () => disableParallaxRegexp.test(navigator.userAgent);\n      }\n\n      if ('function' !== typeof self.options.disableParallax) {\n        self.options.disableParallax = () => false;\n      } // prepare disableVideo callback\n\n\n      if ('string' === typeof self.options.disableVideo) {\n        self.options.disableVideo = new RegExp(self.options.disableVideo);\n      }\n\n      if (self.options.disableVideo instanceof RegExp) {\n        const disableVideoRegexp = self.options.disableVideo;\n\n        self.options.disableVideo = () => disableVideoRegexp.test(navigator.userAgent);\n      }\n\n      if ('function' !== typeof self.options.disableVideo) {\n        self.options.disableVideo = () => false;\n      } // custom element to check if parallax in viewport\n\n\n      let elementInVP = self.options.elementInViewport; // get first item from array\n\n      if (elementInVP && 'object' === typeof elementInVP && 'undefined' !== typeof elementInVP.length) {\n        [elementInVP] = elementInVP;\n      } // check if dom element\n\n\n      if (!(elementInVP instanceof Element)) {\n        elementInVP = null;\n      }\n\n      self.options.elementInViewport = elementInVP;\n      self.image = {\n        src: self.options.imgSrc || null,\n        $container: null,\n        useImgTag: false,\n        // 1. Position fixed is needed for the most of browsers because absolute position have glitches\n        // 2. On MacOS with smooth scroll there is a huge lags with absolute position - https://github.com/nk-o/jarallax/issues/75\n        // 3. Previously used 'absolute' for mobile devices. But we re-tested on iPhone 12 and 'fixed' position is working better, then 'absolute', so for now position is always 'fixed'\n        position: 'fixed'\n      };\n\n      if (self.initImg() && self.canInitParallax()) {\n        self.init();\n      }\n    } // add styles to element\n    // eslint-disable-next-line class-methods-use-this\n\n\n    css(el, styles) {\n      if ('string' === typeof styles) {\n        return global$1.getComputedStyle(el).getPropertyValue(styles);\n      }\n\n      Object.keys(styles).forEach(key => {\n        el.style[key] = styles[key];\n      });\n      return el;\n    } // Extend like jQuery.extend\n    // eslint-disable-next-line class-methods-use-this\n\n\n    extend(out, ...args) {\n      out = out || {};\n      Object.keys(args).forEach(i => {\n        if (!args[i]) {\n          return;\n        }\n\n        Object.keys(args[i]).forEach(key => {\n          out[key] = args[i][key];\n        });\n      });\n      return out;\n    } // get window size and scroll position. Useful for extensions\n    // eslint-disable-next-line class-methods-use-this\n\n\n    getWindowData() {\n      return {\n        width: global$1.innerWidth || document.documentElement.clientWidth,\n        height: wndH,\n        y: document.documentElement.scrollTop\n      };\n    } // Jarallax functions\n\n\n    initImg() {\n      const self = this; // find image element\n\n      let $imgElement = self.options.imgElement;\n\n      if ($imgElement && 'string' === typeof $imgElement) {\n        $imgElement = self.$item.querySelector($imgElement);\n      } // check if dom element\n\n\n      if (!($imgElement instanceof Element)) {\n        if (self.options.imgSrc) {\n          $imgElement = new Image();\n          $imgElement.src = self.options.imgSrc;\n        } else {\n          $imgElement = null;\n        }\n      }\n\n      if ($imgElement) {\n        if (self.options.keepImg) {\n          self.image.$item = $imgElement.cloneNode(true);\n        } else {\n          self.image.$item = $imgElement;\n          self.image.$itemParent = $imgElement.parentNode;\n        }\n\n        self.image.useImgTag = true;\n      } // true if there is img tag\n\n\n      if (self.image.$item) {\n        return true;\n      } // get image src\n\n\n      if (null === self.image.src) {\n        self.image.src = 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///yH5BAEAAAAALAAAAAABAAEAAAIBRAA7';\n        self.image.bgImage = self.css(self.$item, 'background-image');\n      }\n\n      return !(!self.image.bgImage || 'none' === self.image.bgImage);\n    }\n\n    canInitParallax() {\n      return !this.options.disableParallax();\n    }\n\n    init() {\n      const self = this;\n      const containerStyles = {\n        position: 'absolute',\n        top: 0,\n        left: 0,\n        width: '100%',\n        height: '100%',\n        overflow: 'hidden'\n      };\n      let imageStyles = {\n        pointerEvents: 'none',\n        transformStyle: 'preserve-3d',\n        backfaceVisibility: 'hidden',\n        willChange: 'transform,opacity'\n      };\n\n      if (!self.options.keepImg) {\n        // save default user styles\n        const curStyle = self.$item.getAttribute('style');\n\n        if (curStyle) {\n          self.$item.setAttribute('data-jarallax-original-styles', curStyle);\n        }\n\n        if (self.image.useImgTag) {\n          const curImgStyle = self.image.$item.getAttribute('style');\n\n          if (curImgStyle) {\n            self.image.$item.setAttribute('data-jarallax-original-styles', curImgStyle);\n          }\n        }\n      } // set relative position and z-index to the parent\n\n\n      if ('static' === self.css(self.$item, 'position')) {\n        self.css(self.$item, {\n          position: 'relative'\n        });\n      }\n\n      if ('auto' === self.css(self.$item, 'z-index')) {\n        self.css(self.$item, {\n          zIndex: 0\n        });\n      } // container for parallax image\n\n\n      self.image.$container = document.createElement('div');\n      self.css(self.image.$container, containerStyles);\n      self.css(self.image.$container, {\n        'z-index': self.options.zIndex\n      }); // it will remove some image overlapping\n      // overlapping occur due to an image position fixed inside absolute position element\n      // needed only when background in fixed position\n\n      if ('fixed' === this.image.position) {\n        self.css(self.image.$container, {\n          '-webkit-clip-path': 'polygon(0 0, 100% 0, 100% 100%, 0 100%)',\n          'clip-path': 'polygon(0 0, 100% 0, 100% 100%, 0 100%)'\n        });\n      }\n\n      self.image.$container.setAttribute('id', `jarallax-container-${self.instanceID}`);\n      self.$item.appendChild(self.image.$container); // use img tag\n\n      if (self.image.useImgTag) {\n        imageStyles = self.extend({\n          'object-fit': self.options.imgSize,\n          'object-position': self.options.imgPosition,\n          'max-width': 'none'\n        }, containerStyles, imageStyles); // use div with background image\n      } else {\n        self.image.$item = document.createElement('div');\n\n        if (self.image.src) {\n          imageStyles = self.extend({\n            'background-position': self.options.imgPosition,\n            'background-size': self.options.imgSize,\n            'background-repeat': self.options.imgRepeat,\n            'background-image': self.image.bgImage || `url(\"${self.image.src}\")`\n          }, containerStyles, imageStyles);\n        }\n      }\n\n      if ('opacity' === self.options.type || 'scale' === self.options.type || 'scale-opacity' === self.options.type || 1 === self.options.speed) {\n        self.image.position = 'absolute';\n      } // 1. Check if one of parents have transform style (without this check, scroll transform will be inverted if used parallax with position fixed)\n      //    discussion - https://github.com/nk-o/jarallax/issues/9\n      // 2. Check if parents have overflow scroll\n\n\n      if ('fixed' === self.image.position) {\n        const $parents = getParents(self.$item).filter(el => {\n          const styles = global$1.getComputedStyle(el);\n          const parentTransform = styles['-webkit-transform'] || styles['-moz-transform'] || styles.transform;\n          const overflowRegex = /(auto|scroll)/;\n          return parentTransform && 'none' !== parentTransform || overflowRegex.test(styles.overflow + styles['overflow-y'] + styles['overflow-x']);\n        });\n        self.image.position = $parents.length ? 'absolute' : 'fixed';\n      } // add position to parallax block\n\n\n      imageStyles.position = self.image.position; // insert parallax image\n\n      self.css(self.image.$item, imageStyles);\n      self.image.$container.appendChild(self.image.$item); // set initial position and size\n\n      self.onResize();\n      self.onScroll(true); // call onInit event\n\n      if (self.options.onInit) {\n        self.options.onInit.call(self);\n      } // remove default user background\n\n\n      if ('none' !== self.css(self.$item, 'background-image')) {\n        self.css(self.$item, {\n          'background-image': 'none'\n        });\n      }\n\n      self.addToParallaxList();\n    } // add to parallax instances list\n\n\n    addToParallaxList() {\n      jarallaxList.push({\n        instance: this\n      });\n\n      if (1 === jarallaxList.length) {\n        global$1.requestAnimationFrame(updateParallax);\n      }\n    } // remove from parallax instances list\n\n\n    removeFromParallaxList() {\n      const self = this;\n      jarallaxList.forEach((data, key) => {\n        if (data.instance.instanceID === self.instanceID) {\n          jarallaxList.splice(key, 1);\n        }\n      });\n    }\n\n    destroy() {\n      const self = this;\n      self.removeFromParallaxList(); // return styles on container as before jarallax init\n\n      const originalStylesTag = self.$item.getAttribute('data-jarallax-original-styles');\n      self.$item.removeAttribute('data-jarallax-original-styles'); // null occurs if there is no style tag before jarallax init\n\n      if (!originalStylesTag) {\n        self.$item.removeAttribute('style');\n      } else {\n        self.$item.setAttribute('style', originalStylesTag);\n      }\n\n      if (self.image.useImgTag) {\n        // return styles on img tag as before jarallax init\n        const originalStylesImgTag = self.image.$item.getAttribute('data-jarallax-original-styles');\n        self.image.$item.removeAttribute('data-jarallax-original-styles'); // null occurs if there is no style tag before jarallax init\n\n        if (!originalStylesImgTag) {\n          self.image.$item.removeAttribute('style');\n        } else {\n          self.image.$item.setAttribute('style', originalStylesTag);\n        } // move img tag to its default position\n\n\n        if (self.image.$itemParent) {\n          self.image.$itemParent.appendChild(self.image.$item);\n        }\n      } // remove additional dom elements\n\n\n      if (self.image.$container) {\n        self.image.$container.parentNode.removeChild(self.image.$container);\n      } // call onDestroy event\n\n\n      if (self.options.onDestroy) {\n        self.options.onDestroy.call(self);\n      } // delete jarallax from item\n\n\n      delete self.$item.jarallax;\n    } // Fallback for removed function.\n    // Does nothing now.\n    // eslint-disable-next-line class-methods-use-this\n\n\n    clipContainer() {}\n\n    coverImage() {\n      const self = this;\n      const rect = self.image.$container.getBoundingClientRect();\n      const contH = rect.height;\n      const {\n        speed\n      } = self.options;\n      const isScroll = 'scroll' === self.options.type || 'scroll-opacity' === self.options.type;\n      let scrollDist = 0;\n      let resultH = contH;\n      let resultMT = 0; // scroll parallax\n\n      if (isScroll) {\n        // scroll distance and height for image\n        if (0 > speed) {\n          scrollDist = speed * Math.max(contH, wndH);\n\n          if (wndH < contH) {\n            scrollDist -= speed * (contH - wndH);\n          }\n        } else {\n          scrollDist = speed * (contH + wndH);\n        } // size for scroll parallax\n\n\n        if (1 < speed) {\n          resultH = Math.abs(scrollDist - wndH);\n        } else if (0 > speed) {\n          resultH = scrollDist / speed + Math.abs(scrollDist);\n        } else {\n          resultH += (wndH - contH) * (1 - speed);\n        }\n\n        scrollDist /= 2;\n      } // store scroll distance\n\n\n      self.parallaxScrollDistance = scrollDist; // vertical center\n\n      if (isScroll) {\n        resultMT = (wndH - resultH) / 2;\n      } else {\n        resultMT = (contH - resultH) / 2;\n      } // apply result to item\n\n\n      self.css(self.image.$item, {\n        height: `${resultH}px`,\n        marginTop: `${resultMT}px`,\n        left: 'fixed' === self.image.position ? `${rect.left}px` : '0',\n        width: `${rect.width}px`\n      }); // call onCoverImage event\n\n      if (self.options.onCoverImage) {\n        self.options.onCoverImage.call(self);\n      } // return some useful data. Used in the video cover function\n\n\n      return {\n        image: {\n          height: resultH,\n          marginTop: resultMT\n        },\n        container: rect\n      };\n    }\n\n    isVisible() {\n      return this.isElementInViewport || false;\n    }\n\n    onScroll(force) {\n      const self = this;\n      const rect = self.$item.getBoundingClientRect();\n      const contT = rect.top;\n      const contH = rect.height;\n      const styles = {}; // check if in viewport\n\n      let viewportRect = rect;\n\n      if (self.options.elementInViewport) {\n        viewportRect = self.options.elementInViewport.getBoundingClientRect();\n      }\n\n      self.isElementInViewport = 0 <= viewportRect.bottom && 0 <= viewportRect.right && viewportRect.top <= wndH && viewportRect.left <= global$1.innerWidth; // stop calculations if item is not in viewport\n\n      if (force ? false : !self.isElementInViewport) {\n        return;\n      } // calculate parallax helping variables\n\n\n      const beforeTop = Math.max(0, contT);\n      const beforeTopEnd = Math.max(0, contH + contT);\n      const afterTop = Math.max(0, -contT);\n      const beforeBottom = Math.max(0, contT + contH - wndH);\n      const beforeBottomEnd = Math.max(0, contH - (contT + contH - wndH));\n      const afterBottom = Math.max(0, -contT + wndH - contH);\n      const fromViewportCenter = 1 - 2 * ((wndH - contT) / (wndH + contH)); // calculate on how percent of section is visible\n\n      let visiblePercent = 1;\n\n      if (contH < wndH) {\n        visiblePercent = 1 - (afterTop || beforeBottom) / contH;\n      } else if (beforeTopEnd <= wndH) {\n        visiblePercent = beforeTopEnd / wndH;\n      } else if (beforeBottomEnd <= wndH) {\n        visiblePercent = beforeBottomEnd / wndH;\n      } // opacity\n\n\n      if ('opacity' === self.options.type || 'scale-opacity' === self.options.type || 'scroll-opacity' === self.options.type) {\n        styles.transform = 'translate3d(0,0,0)';\n        styles.opacity = visiblePercent;\n      } // scale\n\n\n      if ('scale' === self.options.type || 'scale-opacity' === self.options.type) {\n        let scale = 1;\n\n        if (0 > self.options.speed) {\n          scale -= self.options.speed * visiblePercent;\n        } else {\n          scale += self.options.speed * (1 - visiblePercent);\n        }\n\n        styles.transform = `scale(${scale}) translate3d(0,0,0)`;\n      } // scroll\n\n\n      if ('scroll' === self.options.type || 'scroll-opacity' === self.options.type) {\n        let positionY = self.parallaxScrollDistance * fromViewportCenter; // fix if parallax block in absolute position\n\n        if ('absolute' === self.image.position) {\n          positionY -= contT;\n        }\n\n        styles.transform = `translate3d(0,${positionY}px,0)`;\n      }\n\n      self.css(self.image.$item, styles); // call onScroll event\n\n      if (self.options.onScroll) {\n        self.options.onScroll.call(self, {\n          section: rect,\n          beforeTop,\n          beforeTopEnd,\n          afterTop,\n          beforeBottom,\n          beforeBottomEnd,\n          afterBottom,\n          visiblePercent,\n          fromViewportCenter\n        });\n      }\n    }\n\n    onResize() {\n      this.coverImage();\n    }\n\n  } // global definition\n\n\n  const jarallax = function (items, options, ...args) {\n    // check for dom element\n    // thanks: http://stackoverflow.com/questions/384286/javascript-isdom-how-do-you-check-if-a-javascript-object-is-a-dom-object\n    if ('object' === typeof HTMLElement ? items instanceof HTMLElement : items && 'object' === typeof items && null !== items && 1 === items.nodeType && 'string' === typeof items.nodeName) {\n      items = [items];\n    }\n\n    const len = items.length;\n    let k = 0;\n    let ret;\n\n    for (k; k < len; k += 1) {\n      if ('object' === typeof options || 'undefined' === typeof options) {\n        if (!items[k].jarallax) {\n          items[k].jarallax = new Jarallax(items[k], options);\n        }\n      } else if (items[k].jarallax) {\n        // eslint-disable-next-line prefer-spread\n        ret = items[k].jarallax[options].apply(items[k].jarallax, args);\n      }\n\n      if ('undefined' !== typeof ret) {\n        return ret;\n      }\n    }\n\n    return items;\n  };\n\n  jarallax.constructor = Jarallax;\n\n  const $ = global$1.jQuery; // jQuery support\n\n  if ('undefined' !== typeof $) {\n    const $Plugin = function (...args) {\n      Array.prototype.unshift.call(args, this);\n      const res = jarallax.apply(global$1, args);\n      return 'object' !== typeof res ? res : this;\n    };\n\n    $Plugin.constructor = jarallax.constructor; // no conflict\n\n    const old$Plugin = $.fn.jarallax;\n    $.fn.jarallax = $Plugin;\n\n    $.fn.jarallax.noConflict = function () {\n      $.fn.jarallax = old$Plugin;\n      return this;\n    };\n  } // data-jarallax initialization\n\n\n  ready(() => {\n    jarallax(document.querySelectorAll('[data-jarallax]'));\n  });\n\n  return jarallax;\n\n}));\n//# sourceMappingURL=jarallax.js.map\n","Magento_PageBuilder/js/resource/slick/slick.min.js":"/*\n     _ _      _       _\n ___| (_) ___| | __  (_)___\n/ __| | |/ __| |/ /  | / __|\n\\__ \\ | | (__|   < _ | \\__ \\\n|___/_|_|\\___|_|\\_(_)/ |___/\n                   |__/\n\n Version: 1.8.1\n  Author: Ken Wheeler\n Website: http://kenwheeler.github.io\n    Docs: http://kenwheeler.github.io/slick\n    Repo: http://github.com/kenwheeler/slick\n  Issues: http://github.com/kenwheeler/slick/issues\n\n */\n!function(i){\"use strict\";\"function\"==typeof define&&define.amd?define([\"jquery\"],i):\"undefined\"!=typeof exports?module.exports=i(require(\"jquery\")):i(jQuery)}(function(i){\"use strict\";var e=window.Slick||{};(e=function(){var e=0;return function(t,o){var s,n=this;n.defaults={accessibility:!0,adaptiveHeight:!1,appendArrows:i(t),appendDots:i(t),arrows:!0,asNavFor:null,prevArrow:'<button class=\"slick-prev\" aria-label=\"Previous\" type=\"button\">Previous</button>',nextArrow:'<button class=\"slick-next\" aria-label=\"Next\" type=\"button\">Next</button>',autoplay:!1,autoplaySpeed:3e3,centerMode:!1,centerPadding:\"50px\",cssEase:\"ease\",customPaging:function(e,t){return i('<button type=\"button\" />').text(t+1)},dots:!1,dotsClass:\"slick-dots\",draggable:!0,easing:\"linear\",edgeFriction:.35,fade:!1,focusOnSelect:!1,focusOnChange:!1,infinite:!0,initialSlide:0,lazyLoad:\"ondemand\",mobileFirst:!1,pauseOnHover:!0,pauseOnFocus:!0,pauseOnDotsHover:!1,respondTo:\"window\",responsive:null,rows:1,rtl:!1,slide:\"\",slidesPerRow:1,slidesToShow:1,slidesToScroll:1,speed:500,swipe:!0,swipeToSlide:!1,touchMove:!0,touchThreshold:5,useCSS:!0,useTransform:!0,variableWidth:!1,vertical:!1,verticalSwiping:!1,waitForAnimate:!0,zIndex:1e3},n.initials={animating:!1,dragging:!1,autoPlayTimer:null,currentDirection:0,currentLeft:null,currentSlide:0,direction:1,$dots:null,listWidth:null,listHeight:null,loadIndex:0,$nextArrow:null,$prevArrow:null,scrolling:!1,slideCount:null,slideWidth:null,$slideTrack:null,$slides:null,sliding:!1,slideOffset:0,swipeLeft:null,swiping:!1,$list:null,touchObject:{},transformsEnabled:!1,unslicked:!1},i.extend(n,n.initials),n.activeBreakpoint=null,n.animType=null,n.animProp=null,n.breakpoints=[],n.breakpointSettings=[],n.cssTransitions=!1,n.focussed=!1,n.interrupted=!1,n.hidden=\"hidden\",n.paused=!0,n.positionProp=null,n.respondTo=null,n.rowCount=1,n.shouldClick=!0,n.$slider=i(t),n.$slidesCache=null,n.transformType=null,n.transitionType=null,n.visibilityChange=\"visibilitychange\",n.windowWidth=0,n.windowTimer=null,s=i(t).data(\"slick\")||{},n.options=i.extend({},n.defaults,o,s),n.currentSlide=n.options.initialSlide,n.originalSettings=n.options,void 0!==document.mozHidden?(n.hidden=\"mozHidden\",n.visibilityChange=\"mozvisibilitychange\"):void 0!==document.webkitHidden&&(n.hidden=\"webkitHidden\",n.visibilityChange=\"webkitvisibilitychange\"),n.autoPlay=i.proxy(n.autoPlay,n),n.autoPlayClear=i.proxy(n.autoPlayClear,n),n.autoPlayIterator=i.proxy(n.autoPlayIterator,n),n.changeSlide=i.proxy(n.changeSlide,n),n.clickHandler=i.proxy(n.clickHandler,n),n.selectHandler=i.proxy(n.selectHandler,n),n.setPosition=i.proxy(n.setPosition,n),n.swipeHandler=i.proxy(n.swipeHandler,n),n.dragHandler=i.proxy(n.dragHandler,n),n.keyHandler=i.proxy(n.keyHandler,n),n.instanceUid=e++,n.htmlExpr=/^(?:\\s*(<[\\w\\W]+>)[^>]*)$/,n.registerBreakpoints(),n.init(!0)}}()).prototype.activateADA=function(){this.$slideTrack.find(\".slick-active\").attr({\"aria-hidden\":\"false\"}).find(\"a, input, button, select\").attr({tabindex:\"0\"})},e.prototype.addSlide=e.prototype.slickAdd=function(e,t,o){var s=this;if(\"boolean\"==typeof t)o=t,t=null;else if(t<0||t>=s.slideCount)return!1;s.unload(),\"number\"==typeof t?0===t&&0===s.$slides.length?i(e).appendTo(s.$slideTrack):o?i(e).insertBefore(s.$slides.eq(t)):i(e).insertAfter(s.$slides.eq(t)):!0===o?i(e).prependTo(s.$slideTrack):i(e).appendTo(s.$slideTrack),s.$slides=s.$slideTrack.children(this.options.slide),s.$slideTrack.children(this.options.slide).detach(),s.$slideTrack.append(s.$slides),s.$slides.each(function(e,t){i(t).attr(\"data-slick-index\",e)}),s.$slidesCache=s.$slides,s.reinit()},e.prototype.animateHeight=function(){var i=this;if(1===i.options.slidesToShow&&!0===i.options.adaptiveHeight&&!1===i.options.vertical){var e=i.$slides.eq(i.currentSlide).outerHeight(!0);i.$list.animate({height:e},i.options.speed)}},e.prototype.animateSlide=function(e,t){var o={},s=this;s.animateHeight(),!0===s.options.rtl&&!1===s.options.vertical&&(e=-e),!1===s.transformsEnabled?!1===s.options.vertical?s.$slideTrack.animate({left:e},s.options.speed,s.options.easing,t):s.$slideTrack.animate({top:e},s.options.speed,s.options.easing,t):!1===s.cssTransitions?(!0===s.options.rtl&&(s.currentLeft=-s.currentLeft),i({animStart:s.currentLeft}).animate({animStart:e},{duration:s.options.speed,easing:s.options.easing,step:function(i){i=Math.ceil(i),!1===s.options.vertical?(o[s.animType]=\"translate(\"+i+\"px, 0px)\",s.$slideTrack.css(o)):(o[s.animType]=\"translate(0px,\"+i+\"px)\",s.$slideTrack.css(o))},complete:function(){t&&t.call()}})):(s.applyTransition(),e=Math.ceil(e),!1===s.options.vertical?o[s.animType]=\"translate3d(\"+e+\"px, 0px, 0px)\":o[s.animType]=\"translate3d(0px,\"+e+\"px, 0px)\",s.$slideTrack.css(o),t&&setTimeout(function(){s.disableTransition(),t.call()},s.options.speed))},e.prototype.getNavTarget=function(){var e=this.options.asNavFor;return e&&null!==e&&(e=i(e).not(this.$slider)),e},e.prototype.asNavFor=function(e){var t=this.getNavTarget();null!==t&&\"object\"==typeof t&&t.each(function(){var t=i(this).slick(\"getSlick\");t.unslicked||t.slideHandler(e,!0)})},e.prototype.applyTransition=function(i){var e=this,t={};!1===e.options.fade?t[e.transitionType]=e.transformType+\" \"+e.options.speed+\"ms \"+e.options.cssEase:t[e.transitionType]=\"opacity \"+e.options.speed+\"ms \"+e.options.cssEase,!1===e.options.fade?e.$slideTrack.css(t):e.$slides.eq(i).css(t)},e.prototype.autoPlay=function(){var i=this;i.autoPlayClear(),i.slideCount>i.options.slidesToShow&&(i.autoPlayTimer=setInterval(i.autoPlayIterator,i.options.autoplaySpeed))},e.prototype.autoPlayClear=function(){this.autoPlayTimer&&clearInterval(this.autoPlayTimer)},e.prototype.autoPlayIterator=function(){var i=this,e=i.currentSlide+i.options.slidesToScroll;i.paused||i.interrupted||i.focussed||(!1===i.options.infinite&&(1===i.direction&&i.currentSlide+1===i.slideCount-1?i.direction=0:0===i.direction&&(e=i.currentSlide-i.options.slidesToScroll,i.currentSlide-1==0&&(i.direction=1))),i.slideHandler(e))},e.prototype.buildArrows=function(){var e=this;!0===e.options.arrows&&(e.$prevArrow=i(e.options.prevArrow).addClass(\"slick-arrow\"),e.$nextArrow=i(e.options.nextArrow).addClass(\"slick-arrow\"),e.slideCount>e.options.slidesToShow?(e.$prevArrow.removeClass(\"slick-hidden\").removeAttr(\"aria-hidden tabindex\"),e.$nextArrow.removeClass(\"slick-hidden\").removeAttr(\"aria-hidden tabindex\"),e.htmlExpr.test(e.options.prevArrow)&&e.$prevArrow.prependTo(e.options.appendArrows),e.htmlExpr.test(e.options.nextArrow)&&e.$nextArrow.appendTo(e.options.appendArrows),!0!==e.options.infinite&&e.$prevArrow.addClass(\"slick-disabled\").attr(\"aria-disabled\",\"true\")):e.$prevArrow.add(e.$nextArrow).addClass(\"slick-hidden\").attr({\"aria-disabled\":\"true\",tabindex:\"-1\"}))},e.prototype.buildDots=function(){var e,t,o=this;if(!0===o.options.dots&&o.slideCount>o.options.slidesToShow){for(o.$slider.addClass(\"slick-dotted\"),t=i(\"<ul />\").addClass(o.options.dotsClass),e=0;e<=o.getDotCount();e+=1)t.append(i(\"<li />\").append(o.options.customPaging.call(this,o,e)));o.$dots=t.appendTo(o.options.appendDots),o.$dots.find(\"li\").first().addClass(\"slick-active\")}},e.prototype.buildOut=function(){var e=this;e.$slides=e.$slider.children(e.options.slide+\":not(.slick-cloned)\").addClass(\"slick-slide\"),e.slideCount=e.$slides.length,e.$slides.each(function(e,t){i(t).attr(\"data-slick-index\",e).data(\"originalStyling\",i(t).attr(\"style\")||\"\")}),e.$slider.addClass(\"slick-slider\"),e.$slideTrack=0===e.slideCount?i('<div class=\"slick-track\"/>').appendTo(e.$slider):e.$slides.wrapAll('<div class=\"slick-track\"/>').parent(),e.$list=e.$slideTrack.wrap('<div class=\"slick-list\"/>').parent(),e.$slideTrack.css(\"opacity\",0),!0!==e.options.centerMode&&!0!==e.options.swipeToSlide||(e.options.slidesToScroll=1),i(\"img[data-lazy]\",e.$slider).not(\"[src]\").addClass(\"slick-loading\"),e.setupInfinite(),e.buildArrows(),e.buildDots(),e.updateDots(),e.setSlideClasses(\"number\"==typeof e.currentSlide?e.currentSlide:0),!0===e.options.draggable&&e.$list.addClass(\"draggable\")},e.prototype.buildRows=function(){var i,e,t,o,s,n,r,l=this;if(o=document.createDocumentFragment(),n=l.$slider.children(),l.options.rows>0){for(r=l.options.slidesPerRow*l.options.rows,s=Math.ceil(n.length/r),i=0;i<s;i++){var d=document.createElement(\"div\");for(e=0;e<l.options.rows;e++){var a=document.createElement(\"div\");for(t=0;t<l.options.slidesPerRow;t++){var c=i*r+(e*l.options.slidesPerRow+t);n.get(c)&&a.appendChild(n.get(c))}d.appendChild(a)}o.appendChild(d)}l.$slider.empty().append(o),l.$slider.children().children().children().css({width:100/l.options.slidesPerRow+\"%\",display:\"inline-block\"})}},e.prototype.checkResponsive=function(e,t){var o,s,n,r=this,l=!1,d=r.$slider.width(),a=window.innerWidth||i(window).width();if(\"window\"===r.respondTo?n=a:\"slider\"===r.respondTo?n=d:\"min\"===r.respondTo&&(n=Math.min(a,d)),r.options.responsive&&r.options.responsive.length&&null!==r.options.responsive){s=null;for(o in r.breakpoints)r.breakpoints.hasOwnProperty(o)&&(!1===r.originalSettings.mobileFirst?n<r.breakpoints[o]&&(s=r.breakpoints[o]):n>r.breakpoints[o]&&(s=r.breakpoints[o]));null!==s?null!==r.activeBreakpoint?(s!==r.activeBreakpoint||t)&&(r.activeBreakpoint=s,\"unslick\"===r.breakpointSettings[s]?r.unslick(s):(r.options=i.extend({},r.originalSettings,r.breakpointSettings[s]),!0===e&&(r.currentSlide=r.options.initialSlide),r.refresh(e)),l=s):(r.activeBreakpoint=s,\"unslick\"===r.breakpointSettings[s]?r.unslick(s):(r.options=i.extend({},r.originalSettings,r.breakpointSettings[s]),!0===e&&(r.currentSlide=r.options.initialSlide),r.refresh(e)),l=s):null!==r.activeBreakpoint&&(r.activeBreakpoint=null,r.options=r.originalSettings,!0===e&&(r.currentSlide=r.options.initialSlide),r.refresh(e),l=s),e||!1===l||r.$slider.trigger(\"breakpoint\",[r,l])}},e.prototype.changeSlide=function(e,t){var o,s,n=this,r=i(e.currentTarget);switch(r.is(\"a\")&&e.preventDefault(),r.is(\"li\")||(r=r.closest(\"li\")),o=n.slideCount%n.options.slidesToScroll!=0?0:(n.slideCount-n.currentSlide)%n.options.slidesToScroll,e.data.message){case\"previous\":s=0===o?n.options.slidesToScroll:n.options.slidesToShow-o,n.slideCount>n.options.slidesToShow&&n.slideHandler(n.currentSlide-s,!1,t);break;case\"next\":s=0===o?n.options.slidesToScroll:o,n.slideCount>n.options.slidesToShow&&n.slideHandler(n.currentSlide+s,!1,t);break;case\"index\":var l=0===e.data.index?0:e.data.index||r.index()*n.options.slidesToScroll;n.slideHandler(n.checkNavigable(l),!1,t),r.children().trigger(\"focus\");break;default:return}},e.prototype.checkNavigable=function(i){var e,t;if(t=0,i>(e=this.getNavigableIndexes())[e.length-1])i=e[e.length-1];else for(var o in e){if(i<e[o]){i=t;break}t=e[o]}return i},e.prototype.cleanUpEvents=function(){var e=this;e.options.dots&&null!==e.$dots&&(i(\"li\",e.$dots).off(\"click.slick\",e.changeSlide).off(\"mouseenter.slick\",i.proxy(e.interrupt,e,!0)).off(\"mouseleave.slick\",i.proxy(e.interrupt,e,!1)),!0===e.options.accessibility&&e.$dots.off(\"keydown.slick\",e.keyHandler)),e.$slider.off(\"focus.slick blur.slick\"),!0===e.options.arrows&&e.slideCount>e.options.slidesToShow&&(e.$prevArrow&&e.$prevArrow.off(\"click.slick\",e.changeSlide),e.$nextArrow&&e.$nextArrow.off(\"click.slick\",e.changeSlide),!0===e.options.accessibility&&(e.$prevArrow&&e.$prevArrow.off(\"keydown.slick\",e.keyHandler),e.$nextArrow&&e.$nextArrow.off(\"keydown.slick\",e.keyHandler))),e.$list.off(\"touchstart.slick mousedown.slick\",e.swipeHandler),e.$list.off(\"touchmove.slick mousemove.slick\",e.swipeHandler),e.$list.off(\"touchend.slick mouseup.slick\",e.swipeHandler),e.$list.off(\"touchcancel.slick mouseleave.slick\",e.swipeHandler),e.$list.off(\"click.slick\",e.clickHandler),i(document).off(e.visibilityChange,e.visibility),e.cleanUpSlideEvents(),!0===e.options.accessibility&&e.$list.off(\"keydown.slick\",e.keyHandler),!0===e.options.focusOnSelect&&i(e.$slideTrack).children().off(\"click.slick\",e.selectHandler),i(window).off(\"orientationchange.slick.slick-\"+e.instanceUid,e.orientationChange),i(window).off(\"resize.slick.slick-\"+e.instanceUid,e.resize),i(\"[draggable!=true]\",e.$slideTrack).off(\"dragstart\",e.preventDefault),i(window).off(\"load.slick.slick-\"+e.instanceUid,e.setPosition)},e.prototype.cleanUpSlideEvents=function(){var e=this;e.$list.off(\"mouseenter.slick\",i.proxy(e.interrupt,e,!0)),e.$list.off(\"mouseleave.slick\",i.proxy(e.interrupt,e,!1))},e.prototype.cleanUpRows=function(){var i;this.options.rows>0&&((i=this.$slides.children().children()).removeAttr(\"style\"),this.$slider.empty().append(i))},e.prototype.clickHandler=function(i){!1===this.shouldClick&&(i.stopImmediatePropagation(),i.stopPropagation(),i.preventDefault())},e.prototype.destroy=function(e){var t=this;t.autoPlayClear(),t.touchObject={},t.cleanUpEvents(),i(\".slick-cloned\",t.$slider).detach(),t.$dots&&t.$dots.remove(),t.$prevArrow&&t.$prevArrow.length&&(t.$prevArrow.removeClass(\"slick-disabled slick-arrow slick-hidden\").removeAttr(\"aria-hidden aria-disabled tabindex\").css(\"display\",\"\"),t.htmlExpr.test(t.options.prevArrow)&&t.$prevArrow.remove()),t.$nextArrow&&t.$nextArrow.length&&(t.$nextArrow.removeClass(\"slick-disabled slick-arrow slick-hidden\").removeAttr(\"aria-hidden aria-disabled tabindex\").css(\"display\",\"\"),t.htmlExpr.test(t.options.nextArrow)&&t.$nextArrow.remove()),t.$slides&&(t.$slides.removeClass(\"slick-slide slick-active slick-center slick-visible slick-current\").removeAttr(\"aria-hidden\").removeAttr(\"data-slick-index\").each(function(){i(this).attr(\"style\",i(this).data(\"originalStyling\"))}),t.$slideTrack.children(this.options.slide).detach(),t.$slideTrack.detach(),t.$list.detach(),t.$slider.append(t.$slides)),t.cleanUpRows(),t.$slider.removeClass(\"slick-slider\"),t.$slider.removeClass(\"slick-initialized\"),t.$slider.removeClass(\"slick-dotted\"),t.unslicked=!0,e||t.$slider.trigger(\"destroy\",[t])},e.prototype.disableTransition=function(i){var e={};e[this.transitionType]=\"\",!1===this.options.fade?this.$slideTrack.css(e):this.$slides.eq(i).css(e)},e.prototype.fadeSlide=function(i,e){var t=this;!1===t.cssTransitions?(t.$slides.eq(i).css({zIndex:t.options.zIndex}),t.$slides.eq(i).animate({opacity:1},t.options.speed,t.options.easing,e)):(t.applyTransition(i),t.$slides.eq(i).css({opacity:1,zIndex:t.options.zIndex}),e&&setTimeout(function(){t.disableTransition(i),e.call()},t.options.speed))},e.prototype.fadeSlideOut=function(i){var e=this;!1===e.cssTransitions?e.$slides.eq(i).animate({opacity:0,zIndex:e.options.zIndex-2},e.options.speed,e.options.easing):(e.applyTransition(i),e.$slides.eq(i).css({opacity:0,zIndex:e.options.zIndex-2}))},e.prototype.filterSlides=e.prototype.slickFilter=function(i){var e=this;null!==i&&(e.$slidesCache=e.$slides,e.unload(),e.$slideTrack.children(this.options.slide).detach(),e.$slidesCache.filter(i).appendTo(e.$slideTrack),e.reinit())},e.prototype.focusHandler=function(){var e=this;e.$slider.off(\"focus.slick blur.slick\").on(\"focus.slick\",\"*\",function(t){var o=i(this);setTimeout(function(){e.options.pauseOnFocus&&o.is(\":focus\")&&(e.focussed=!0,e.autoPlay())},0)}).on(\"blur.slick\",\"*\",function(t){i(this);e.options.pauseOnFocus&&(e.focussed=!1,e.autoPlay())})},e.prototype.getCurrent=e.prototype.slickCurrentSlide=function(){return this.currentSlide},e.prototype.getDotCount=function(){var i=this,e=0,t=0,o=0;if(!0===i.options.infinite)if(i.slideCount<=i.options.slidesToShow)++o;else for(;e<i.slideCount;)++o,e=t+i.options.slidesToScroll,t+=i.options.slidesToScroll<=i.options.slidesToShow?i.options.slidesToScroll:i.options.slidesToShow;else if(!0===i.options.centerMode)o=i.slideCount;else if(i.options.asNavFor)for(;e<i.slideCount;)++o,e=t+i.options.slidesToScroll,t+=i.options.slidesToScroll<=i.options.slidesToShow?i.options.slidesToScroll:i.options.slidesToShow;else o=1+Math.ceil((i.slideCount-i.options.slidesToShow)/i.options.slidesToScroll);return o-1},e.prototype.getLeft=function(i){var e,t,o,s,n=this,r=0;return n.slideOffset=0,t=n.$slides.first().outerHeight(!0),!0===n.options.infinite?(n.slideCount>n.options.slidesToShow&&(n.slideOffset=n.slideWidth*n.options.slidesToShow*-1,s=-1,!0===n.options.vertical&&!0===n.options.centerMode&&(2===n.options.slidesToShow?s=-1.5:1===n.options.slidesToShow&&(s=-2)),r=t*n.options.slidesToShow*s),n.slideCount%n.options.slidesToScroll!=0&&i+n.options.slidesToScroll>n.slideCount&&n.slideCount>n.options.slidesToShow&&(i>n.slideCount?(n.slideOffset=(n.options.slidesToShow-(i-n.slideCount))*n.slideWidth*-1,r=(n.options.slidesToShow-(i-n.slideCount))*t*-1):(n.slideOffset=n.slideCount%n.options.slidesToScroll*n.slideWidth*-1,r=n.slideCount%n.options.slidesToScroll*t*-1))):i+n.options.slidesToShow>n.slideCount&&(n.slideOffset=(i+n.options.slidesToShow-n.slideCount)*n.slideWidth,r=(i+n.options.slidesToShow-n.slideCount)*t),n.slideCount<=n.options.slidesToShow&&(n.slideOffset=0,r=0),!0===n.options.centerMode&&n.slideCount<=n.options.slidesToShow?n.slideOffset=n.slideWidth*Math.floor(n.options.slidesToShow)/2-n.slideWidth*n.slideCount/2:!0===n.options.centerMode&&!0===n.options.infinite?n.slideOffset+=n.slideWidth*Math.floor(n.options.slidesToShow/2)-n.slideWidth:!0===n.options.centerMode&&(n.slideOffset=0,n.slideOffset+=n.slideWidth*Math.floor(n.options.slidesToShow/2)),e=!1===n.options.vertical?i*n.slideWidth*-1+n.slideOffset:i*t*-1+r,!0===n.options.variableWidth&&(o=n.slideCount<=n.options.slidesToShow||!1===n.options.infinite?n.$slideTrack.children(\".slick-slide\").eq(i):n.$slideTrack.children(\".slick-slide\").eq(i+n.options.slidesToShow),e=!0===n.options.rtl?o[0]?-1*(n.$slideTrack.width()-o[0].offsetLeft-o.width()):0:o[0]?-1*o[0].offsetLeft:0,!0===n.options.centerMode&&(o=n.slideCount<=n.options.slidesToShow||!1===n.options.infinite?n.$slideTrack.children(\".slick-slide\").eq(i):n.$slideTrack.children(\".slick-slide\").eq(i+n.options.slidesToShow+1),e=!0===n.options.rtl?o[0]?-1*(n.$slideTrack.width()-o[0].offsetLeft-o.width()):0:o[0]?-1*o[0].offsetLeft:0,e+=(n.$list.width()-o.outerWidth())/2)),e},e.prototype.getOption=e.prototype.slickGetOption=function(i){return this.options[i]},e.prototype.getNavigableIndexes=function(){var i,e=this,t=0,o=0,s=[];for(!1===e.options.infinite?i=e.slideCount:(t=-1*e.options.slidesToScroll,o=-1*e.options.slidesToScroll,i=2*e.slideCount);t<i;)s.push(t),t=o+e.options.slidesToScroll,o+=e.options.slidesToScroll<=e.options.slidesToShow?e.options.slidesToScroll:e.options.slidesToShow;return s},e.prototype.getSlick=function(){return this},e.prototype.getSlideCount=function(){var e,t,o=this;return t=!0===o.options.centerMode?o.slideWidth*Math.floor(o.options.slidesToShow/2):0,!0===o.options.swipeToSlide?(o.$slideTrack.find(\".slick-slide\").each(function(s,n){if(n.offsetLeft-t+i(n).outerWidth()/2>-1*o.swipeLeft)return e=n,!1}),Math.abs(i(e).attr(\"data-slick-index\")-o.currentSlide)||1):o.options.slidesToScroll},e.prototype.goTo=e.prototype.slickGoTo=function(i,e){this.changeSlide({data:{message:\"index\",index:parseInt(i)}},e)},e.prototype.init=function(e){var t=this;i(t.$slider).hasClass(\"slick-initialized\")||(i(t.$slider).addClass(\"slick-initialized\"),t.buildRows(),t.buildOut(),t.setProps(),t.startLoad(),t.loadSlider(),t.initializeEvents(),t.updateArrows(),t.updateDots(),t.checkResponsive(!0),t.focusHandler()),e&&t.$slider.trigger(\"init\",[t]),!0===t.options.accessibility&&t.initADA(),t.options.autoplay&&(t.paused=!1,t.autoPlay())},e.prototype.initADA=function(){var e=this,t=Math.ceil(e.slideCount/e.options.slidesToShow),o=e.getNavigableIndexes().filter(function(i){return i>=0&&i<e.slideCount});e.$slides.add(e.$slideTrack.find(\".slick-cloned\")).attr({\"aria-hidden\":\"true\",tabindex:\"-1\"}).find(\"a, input, button, select\").attr({tabindex:\"-1\"}),null!==e.$dots&&(e.$slides.not(e.$slideTrack.find(\".slick-cloned\")).each(function(t){var s=o.indexOf(t);if(i(this).attr({role:\"tabpanel\",id:\"slick-slide\"+e.instanceUid+t,tabindex:-1}),-1!==s){var n=\"slick-slide-control\"+e.instanceUid+s;i(\"#\"+n).length&&i(this).attr({\"aria-describedby\":n})}}),e.$dots.attr(\"role\",\"tablist\").find(\"li\").each(function(s){var n=o[s];i(this).attr({role:\"presentation\"}),i(this).find(\"button\").first().attr({role:\"tab\",id:\"slick-slide-control\"+e.instanceUid+s,\"aria-controls\":\"slick-slide\"+e.instanceUid+n,\"aria-label\":s+1+\" of \"+t,\"aria-selected\":null,tabindex:\"-1\"})}).eq(e.currentSlide).find(\"button\").attr({\"aria-selected\":\"true\",tabindex:\"0\"}).end());for(var s=e.currentSlide,n=s+e.options.slidesToShow;s<n;s++)e.options.focusOnChange?e.$slides.eq(s).attr({tabindex:\"0\"}):e.$slides.eq(s).removeAttr(\"tabindex\");e.activateADA()},e.prototype.initArrowEvents=function(){var i=this;!0===i.options.arrows&&i.slideCount>i.options.slidesToShow&&(i.$prevArrow.off(\"click.slick\").on(\"click.slick\",{message:\"previous\"},i.changeSlide),i.$nextArrow.off(\"click.slick\").on(\"click.slick\",{message:\"next\"},i.changeSlide),!0===i.options.accessibility&&(i.$prevArrow.on(\"keydown.slick\",i.keyHandler),i.$nextArrow.on(\"keydown.slick\",i.keyHandler)))},e.prototype.initDotEvents=function(){var e=this;!0===e.options.dots&&e.slideCount>e.options.slidesToShow&&(i(\"li\",e.$dots).on(\"click.slick\",{message:\"index\"},e.changeSlide),!0===e.options.accessibility&&e.$dots.on(\"keydown.slick\",e.keyHandler)),!0===e.options.dots&&!0===e.options.pauseOnDotsHover&&e.slideCount>e.options.slidesToShow&&i(\"li\",e.$dots).on(\"mouseenter.slick\",i.proxy(e.interrupt,e,!0)).on(\"mouseleave.slick\",i.proxy(e.interrupt,e,!1))},e.prototype.initSlideEvents=function(){var e=this;e.options.pauseOnHover&&(e.$list.on(\"mouseenter.slick\",i.proxy(e.interrupt,e,!0)),e.$list.on(\"mouseleave.slick\",i.proxy(e.interrupt,e,!1)))},e.prototype.initializeEvents=function(){var e=this;e.initArrowEvents(),e.initDotEvents(),e.initSlideEvents(),e.$list.on(\"touchstart.slick mousedown.slick\",{action:\"start\"},e.swipeHandler),e.$list.on(\"touchmove.slick mousemove.slick\",{action:\"move\"},e.swipeHandler),e.$list.on(\"touchend.slick mouseup.slick\",{action:\"end\"},e.swipeHandler),e.$list.on(\"touchcancel.slick mouseleave.slick\",{action:\"end\"},e.swipeHandler),e.$list.on(\"click.slick\",e.clickHandler),i(document).on(e.visibilityChange,i.proxy(e.visibility,e)),!0===e.options.accessibility&&e.$list.on(\"keydown.slick\",e.keyHandler),!0===e.options.focusOnSelect&&i(e.$slideTrack).children().on(\"click.slick\",e.selectHandler),i(window).on(\"orientationchange.slick.slick-\"+e.instanceUid,i.proxy(e.orientationChange,e)),i(window).on(\"resize.slick.slick-\"+e.instanceUid,i.proxy(e.resize,e)),i(\"[draggable!=true]\",e.$slideTrack).on(\"dragstart\",e.preventDefault),i(window).on(\"load.slick.slick-\"+e.instanceUid,e.setPosition),i(e.setPosition)},e.prototype.initUI=function(){var i=this;!0===i.options.arrows&&i.slideCount>i.options.slidesToShow&&(i.$prevArrow.show(),i.$nextArrow.show()),!0===i.options.dots&&i.slideCount>i.options.slidesToShow&&i.$dots.show()},e.prototype.keyHandler=function(i){var e=this;i.target.tagName.match(\"TEXTAREA|INPUT|SELECT\")||(37===i.keyCode&&!0===e.options.accessibility?e.changeSlide({data:{message:!0===e.options.rtl?\"next\":\"previous\"}}):39===i.keyCode&&!0===e.options.accessibility&&e.changeSlide({data:{message:!0===e.options.rtl?\"previous\":\"next\"}}))},e.prototype.lazyLoad=function(){var e,t,o,s=this;function n(e){i(\"img[data-lazy]\",e).each(function(){var e=i(this),t=i(this).attr(\"data-lazy\"),o=i(this).attr(\"data-srcset\"),n=i(this).attr(\"data-sizes\")||s.$slider.attr(\"data-sizes\"),r=document.createElement(\"img\");r.onload=function(){e.animate({opacity:0},100,function(){o&&(e.attr(\"srcset\",o),n&&e.attr(\"sizes\",n)),e.attr(\"src\",t).animate({opacity:1},200,function(){e.removeAttr(\"data-lazy data-srcset data-sizes\").removeClass(\"slick-loading\")}),s.$slider.trigger(\"lazyLoaded\",[s,e,t])})},r.onerror=function(){e.removeAttr(\"data-lazy\").removeClass(\"slick-loading\").addClass(\"slick-lazyload-error\"),s.$slider.trigger(\"lazyLoadError\",[s,e,t])},r.src=t})}if(!0===s.options.centerMode?!0===s.options.infinite?o=(t=s.currentSlide+(s.options.slidesToShow/2+1))+s.options.slidesToShow+2:(t=Math.max(0,s.currentSlide-(s.options.slidesToShow/2+1)),o=s.options.slidesToShow/2+1+2+s.currentSlide):(t=s.options.infinite?s.options.slidesToShow+s.currentSlide:s.currentSlide,o=Math.ceil(t+s.options.slidesToShow),!0===s.options.fade&&(t>0&&t--,o<=s.slideCount&&o++)),e=s.$slider.find(\".slick-slide\").slice(t,o),\"anticipated\"===s.options.lazyLoad)for(var r=t-1,l=o,d=s.$slider.find(\".slick-slide\"),a=0;a<s.options.slidesToScroll;a++)r<0&&(r=s.slideCount-1),e=(e=e.add(d.eq(r))).add(d.eq(l)),r--,l++;n(e),s.slideCount<=s.options.slidesToShow?n(s.$slider.find(\".slick-slide\")):s.currentSlide>=s.slideCount-s.options.slidesToShow?n(s.$slider.find(\".slick-cloned\").slice(0,s.options.slidesToShow)):0===s.currentSlide&&n(s.$slider.find(\".slick-cloned\").slice(-1*s.options.slidesToShow))},e.prototype.loadSlider=function(){var i=this;i.setPosition(),i.$slideTrack.css({opacity:1}),i.$slider.removeClass(\"slick-loading\"),i.initUI(),\"progressive\"===i.options.lazyLoad&&i.progressiveLazyLoad()},e.prototype.next=e.prototype.slickNext=function(){this.changeSlide({data:{message:\"next\"}})},e.prototype.orientationChange=function(){this.checkResponsive(),this.setPosition()},e.prototype.pause=e.prototype.slickPause=function(){this.autoPlayClear(),this.paused=!0},e.prototype.play=e.prototype.slickPlay=function(){var i=this;i.autoPlay(),i.options.autoplay=!0,i.paused=!1,i.focussed=!1,i.interrupted=!1},e.prototype.postSlide=function(e){var t=this;t.unslicked||(t.$slider.trigger(\"afterChange\",[t,e]),t.animating=!1,t.slideCount>t.options.slidesToShow&&t.setPosition(),t.swipeLeft=null,t.options.autoplay&&t.autoPlay(),!0===t.options.accessibility&&(t.initADA(),t.options.focusOnChange&&i(t.$slides.get(t.currentSlide)).attr(\"tabindex\",0).focus()))},e.prototype.prev=e.prototype.slickPrev=function(){this.changeSlide({data:{message:\"previous\"}})},e.prototype.preventDefault=function(i){i.preventDefault()},e.prototype.progressiveLazyLoad=function(e){e=e||1;var t,o,s,n,r,l=this,d=i(\"img[data-lazy]\",l.$slider);d.length?(t=d.first(),o=t.attr(\"data-lazy\"),s=t.attr(\"data-srcset\"),n=t.attr(\"data-sizes\")||l.$slider.attr(\"data-sizes\"),(r=document.createElement(\"img\")).onload=function(){s&&(t.attr(\"srcset\",s),n&&t.attr(\"sizes\",n)),t.attr(\"src\",o).removeAttr(\"data-lazy data-srcset data-sizes\").removeClass(\"slick-loading\"),!0===l.options.adaptiveHeight&&l.setPosition(),l.$slider.trigger(\"lazyLoaded\",[l,t,o]),l.progressiveLazyLoad()},r.onerror=function(){e<3?setTimeout(function(){l.progressiveLazyLoad(e+1)},500):(t.removeAttr(\"data-lazy\").removeClass(\"slick-loading\").addClass(\"slick-lazyload-error\"),l.$slider.trigger(\"lazyLoadError\",[l,t,o]),l.progressiveLazyLoad())},r.src=o):l.$slider.trigger(\"allImagesLoaded\",[l])},e.prototype.refresh=function(e){var t,o,s=this;o=s.slideCount-s.options.slidesToShow,!s.options.infinite&&s.currentSlide>o&&(s.currentSlide=o),s.slideCount<=s.options.slidesToShow&&(s.currentSlide=0),t=s.currentSlide,s.destroy(!0),i.extend(s,s.initials,{currentSlide:t}),s.init(),e||s.changeSlide({data:{message:\"index\",index:t}},!1)},e.prototype.registerBreakpoints=function(){var e,t,o,s=this,n=s.options.responsive||null;if(\"array\"===i.type(n)&&n.length){s.respondTo=s.options.respondTo||\"window\";for(e in n)if(o=s.breakpoints.length-1,n.hasOwnProperty(e)){for(t=n[e].breakpoint;o>=0;)s.breakpoints[o]&&s.breakpoints[o]===t&&s.breakpoints.splice(o,1),o--;s.breakpoints.push(t),s.breakpointSettings[t]=n[e].settings}s.breakpoints.sort(function(i,e){return s.options.mobileFirst?i-e:e-i})}},e.prototype.reinit=function(){var e=this;e.$slides=e.$slideTrack.children(e.options.slide).addClass(\"slick-slide\"),e.slideCount=e.$slides.length,e.currentSlide>=e.slideCount&&0!==e.currentSlide&&(e.currentSlide=e.currentSlide-e.options.slidesToScroll),e.slideCount<=e.options.slidesToShow&&(e.currentSlide=0),e.registerBreakpoints(),e.setProps(),e.setupInfinite(),e.buildArrows(),e.updateArrows(),e.initArrowEvents(),e.buildDots(),e.updateDots(),e.initDotEvents(),e.cleanUpSlideEvents(),e.initSlideEvents(),e.checkResponsive(!1,!0),!0===e.options.focusOnSelect&&i(e.$slideTrack).children().on(\"click.slick\",e.selectHandler),e.setSlideClasses(\"number\"==typeof e.currentSlide?e.currentSlide:0),e.setPosition(),e.focusHandler(),e.paused=!e.options.autoplay,e.autoPlay(),e.$slider.trigger(\"reInit\",[e])},e.prototype.resize=function(){var e=this;i(window).width()!==e.windowWidth&&(clearTimeout(e.windowDelay),e.windowDelay=window.setTimeout(function(){e.windowWidth=i(window).width(),e.checkResponsive(),e.unslicked||e.setPosition()},50))},e.prototype.removeSlide=e.prototype.slickRemove=function(i,e,t){var o=this;if(i=\"boolean\"==typeof i?!0===(e=i)?0:o.slideCount-1:!0===e?--i:i,o.slideCount<1||i<0||i>o.slideCount-1)return!1;o.unload(),!0===t?o.$slideTrack.children().remove():o.$slideTrack.children(this.options.slide).eq(i).remove(),o.$slides=o.$slideTrack.children(this.options.slide),o.$slideTrack.children(this.options.slide).detach(),o.$slideTrack.append(o.$slides),o.$slidesCache=o.$slides,o.reinit()},e.prototype.setCSS=function(i){var e,t,o=this,s={};!0===o.options.rtl&&(i=-i),e=\"left\"==o.positionProp?Math.ceil(i)+\"px\":\"0px\",t=\"top\"==o.positionProp?Math.ceil(i)+\"px\":\"0px\",s[o.positionProp]=i,!1===o.transformsEnabled?o.$slideTrack.css(s):(s={},!1===o.cssTransitions?(s[o.animType]=\"translate(\"+e+\", \"+t+\")\",o.$slideTrack.css(s)):(s[o.animType]=\"translate3d(\"+e+\", \"+t+\", 0px)\",o.$slideTrack.css(s)))},e.prototype.setDimensions=function(){var i=this;!1===i.options.vertical?!0===i.options.centerMode&&i.$list.css({padding:\"0px \"+i.options.centerPadding}):(i.$list.height(i.$slides.first().outerHeight(!0)*i.options.slidesToShow),!0===i.options.centerMode&&i.$list.css({padding:i.options.centerPadding+\" 0px\"})),i.listWidth=i.$list.width(),i.listHeight=i.$list.height(),!1===i.options.vertical&&!1===i.options.variableWidth?(i.slideWidth=Math.ceil(i.listWidth/i.options.slidesToShow),i.$slideTrack.width(Math.ceil(i.slideWidth*i.$slideTrack.children(\".slick-slide\").length))):!0===i.options.variableWidth?i.$slideTrack.width(5e3*i.slideCount):(i.slideWidth=Math.ceil(i.listWidth),i.$slideTrack.height(Math.ceil(i.$slides.first().outerHeight(!0)*i.$slideTrack.children(\".slick-slide\").length)));var e=i.$slides.first().outerWidth(!0)-i.$slides.first().width();!1===i.options.variableWidth&&i.$slideTrack.children(\".slick-slide\").width(i.slideWidth-e)},e.prototype.setFade=function(){var e,t=this;t.$slides.each(function(o,s){e=t.slideWidth*o*-1,!0===t.options.rtl?i(s).css({position:\"relative\",right:e,top:0,zIndex:t.options.zIndex-2,opacity:0}):i(s).css({position:\"relative\",left:e,top:0,zIndex:t.options.zIndex-2,opacity:0})}),t.$slides.eq(t.currentSlide).css({zIndex:t.options.zIndex-1,opacity:1})},e.prototype.setHeight=function(){var i=this;if(1===i.options.slidesToShow&&!0===i.options.adaptiveHeight&&!1===i.options.vertical){var e=i.$slides.eq(i.currentSlide).outerHeight(!0);i.$list.css(\"height\",e)}},e.prototype.setOption=e.prototype.slickSetOption=function(){var e,t,o,s,n,r=this,l=!1;if(\"object\"===i.type(arguments[0])?(o=arguments[0],l=arguments[1],n=\"multiple\"):\"string\"===i.type(arguments[0])&&(o=arguments[0],s=arguments[1],l=arguments[2],\"responsive\"===arguments[0]&&\"array\"===i.type(arguments[1])?n=\"responsive\":void 0!==arguments[1]&&(n=\"single\")),\"single\"===n)r.options[o]=s;else if(\"multiple\"===n)i.each(o,function(i,e){r.options[i]=e});else if(\"responsive\"===n)for(t in s)if(\"array\"!==i.type(r.options.responsive))r.options.responsive=[s[t]];else{for(e=r.options.responsive.length-1;e>=0;)r.options.responsive[e].breakpoint===s[t].breakpoint&&r.options.responsive.splice(e,1),e--;r.options.responsive.push(s[t])}l&&(r.unload(),r.reinit())},e.prototype.setPosition=function(){var i=this;i.setDimensions(),i.setHeight(),!1===i.options.fade?i.setCSS(i.getLeft(i.currentSlide)):i.setFade(),i.$slider.trigger(\"setPosition\",[i])},e.prototype.setProps=function(){var i=this,e=document.body.style;i.positionProp=!0===i.options.vertical?\"top\":\"left\",\"top\"===i.positionProp?i.$slider.addClass(\"slick-vertical\"):i.$slider.removeClass(\"slick-vertical\"),void 0===e.WebkitTransition&&void 0===e.MozTransition&&void 0===e.msTransition||!0===i.options.useCSS&&(i.cssTransitions=!0),i.options.fade&&(\"number\"==typeof i.options.zIndex?i.options.zIndex<3&&(i.options.zIndex=3):i.options.zIndex=i.defaults.zIndex),void 0!==e.OTransform&&(i.animType=\"OTransform\",i.transformType=\"-o-transform\",i.transitionType=\"OTransition\",void 0===e.perspectiveProperty&&void 0===e.webkitPerspective&&(i.animType=!1)),void 0!==e.MozTransform&&(i.animType=\"MozTransform\",i.transformType=\"-moz-transform\",i.transitionType=\"MozTransition\",void 0===e.perspectiveProperty&&void 0===e.MozPerspective&&(i.animType=!1)),void 0!==e.webkitTransform&&(i.animType=\"webkitTransform\",i.transformType=\"-webkit-transform\",i.transitionType=\"webkitTransition\",void 0===e.perspectiveProperty&&void 0===e.webkitPerspective&&(i.animType=!1)),void 0!==e.msTransform&&(i.animType=\"msTransform\",i.transformType=\"-ms-transform\",i.transitionType=\"msTransition\",void 0===e.msTransform&&(i.animType=!1)),void 0!==e.transform&&!1!==i.animType&&(i.animType=\"transform\",i.transformType=\"transform\",i.transitionType=\"transition\"),i.transformsEnabled=i.options.useTransform&&null!==i.animType&&!1!==i.animType},e.prototype.setSlideClasses=function(i){var e,t,o,s,n=this;if(t=n.$slider.find(\".slick-slide\").removeClass(\"slick-active slick-center slick-current\").attr(\"aria-hidden\",\"true\"),n.$slides.eq(i).addClass(\"slick-current\"),!0===n.options.centerMode){var r=n.options.slidesToShow%2==0?1:0;e=Math.floor(n.options.slidesToShow/2),!0===n.options.infinite&&(i>=e&&i<=n.slideCount-1-e?n.$slides.slice(i-e+r,i+e+1).addClass(\"slick-active\").attr(\"aria-hidden\",\"false\"):(o=n.options.slidesToShow+i,t.slice(o-e+1+r,o+e+2).addClass(\"slick-active\").attr(\"aria-hidden\",\"false\")),0===i?t.eq(t.length-1-n.options.slidesToShow).addClass(\"slick-center\"):i===n.slideCount-1&&t.eq(n.options.slidesToShow).addClass(\"slick-center\")),n.$slides.eq(i).addClass(\"slick-center\")}else i>=0&&i<=n.slideCount-n.options.slidesToShow?n.$slides.slice(i,i+n.options.slidesToShow).addClass(\"slick-active\").attr(\"aria-hidden\",\"false\"):t.length<=n.options.slidesToShow?t.addClass(\"slick-active\").attr(\"aria-hidden\",\"false\"):(s=n.slideCount%n.options.slidesToShow,o=!0===n.options.infinite?n.options.slidesToShow+i:i,n.options.slidesToShow==n.options.slidesToScroll&&n.slideCount-i<n.options.slidesToShow?t.slice(o-(n.options.slidesToShow-s),o+s).addClass(\"slick-active\").attr(\"aria-hidden\",\"false\"):t.slice(o,o+n.options.slidesToShow).addClass(\"slick-active\").attr(\"aria-hidden\",\"false\"));\"ondemand\"!==n.options.lazyLoad&&\"anticipated\"!==n.options.lazyLoad||n.lazyLoad()},e.prototype.setupInfinite=function(){var e,t,o,s=this;if(!0===s.options.fade&&(s.options.centerMode=!1),!0===s.options.infinite&&!1===s.options.fade&&(t=null,s.slideCount>s.options.slidesToShow)){for(o=!0===s.options.centerMode?s.options.slidesToShow+1:s.options.slidesToShow,e=s.slideCount;e>s.slideCount-o;e-=1)t=e-1,i(s.$slides[t]).clone(!0).attr(\"id\",\"\").attr(\"data-slick-index\",t-s.slideCount).prependTo(s.$slideTrack).addClass(\"slick-cloned\");for(e=0;e<o+s.slideCount;e+=1)t=e,i(s.$slides[t]).clone(!0).attr(\"id\",\"\").attr(\"data-slick-index\",t+s.slideCount).appendTo(s.$slideTrack).addClass(\"slick-cloned\");s.$slideTrack.find(\".slick-cloned\").find(\"[id]\").each(function(){i(this).attr(\"id\",\"\")})}},e.prototype.interrupt=function(i){i||this.autoPlay(),this.interrupted=i},e.prototype.selectHandler=function(e){var t=i(e.target).is(\".slick-slide\")?i(e.target):i(e.target).parents(\".slick-slide\"),o=parseInt(t.attr(\"data-slick-index\"));o||(o=0),this.slideCount<=this.options.slidesToShow?this.slideHandler(o,!1,!0):this.slideHandler(o)},e.prototype.slideHandler=function(i,e,t){var o,s,n,r,l,d,a=this;if(e=e||!1,!(!0===a.animating&&!0===a.options.waitForAnimate||!0===a.options.fade&&a.currentSlide===i))if(!1===e&&a.asNavFor(i),o=i,l=a.getLeft(o),r=a.getLeft(a.currentSlide),a.currentLeft=null===a.swipeLeft?r:a.swipeLeft,!1===a.options.infinite&&!1===a.options.centerMode&&(i<0||i>a.getDotCount()*a.options.slidesToScroll))!1===a.options.fade&&(o=a.currentSlide,!0!==t&&a.slideCount>a.options.slidesToShow?a.animateSlide(r,function(){a.postSlide(o)}):a.postSlide(o));else if(!1===a.options.infinite&&!0===a.options.centerMode&&(i<0||i>a.slideCount-a.options.slidesToScroll))!1===a.options.fade&&(o=a.currentSlide,!0!==t&&a.slideCount>a.options.slidesToShow?a.animateSlide(r,function(){a.postSlide(o)}):a.postSlide(o));else{if(a.options.autoplay&&clearInterval(a.autoPlayTimer),s=o<0?a.slideCount%a.options.slidesToScroll!=0?a.slideCount-a.slideCount%a.options.slidesToScroll:a.slideCount+o:o>=a.slideCount?a.slideCount%a.options.slidesToScroll!=0?0:o-a.slideCount:o,a.animating=!0,a.$slider.trigger(\"beforeChange\",[a,a.currentSlide,s]),n=a.currentSlide,a.currentSlide=s,a.setSlideClasses(a.currentSlide),a.options.asNavFor&&(d=(d=a.getNavTarget()).slick(\"getSlick\")).slideCount<=d.options.slidesToShow&&d.setSlideClasses(a.currentSlide),a.updateDots(),a.updateArrows(),!0===a.options.fade)return!0!==t?(a.fadeSlideOut(n),a.fadeSlide(s,function(){a.postSlide(s)})):a.postSlide(s),void a.animateHeight();!0!==t&&a.slideCount>a.options.slidesToShow?a.animateSlide(l,function(){a.postSlide(s)}):a.postSlide(s)}},e.prototype.startLoad=function(){var i=this;!0===i.options.arrows&&i.slideCount>i.options.slidesToShow&&(i.$prevArrow.hide(),i.$nextArrow.hide()),!0===i.options.dots&&i.slideCount>i.options.slidesToShow&&i.$dots.hide(),i.$slider.addClass(\"slick-loading\")},e.prototype.swipeDirection=function(){var i,e,t,o,s=this;return i=s.touchObject.startX-s.touchObject.curX,e=s.touchObject.startY-s.touchObject.curY,t=Math.atan2(e,i),(o=Math.round(180*t/Math.PI))<0&&(o=360-Math.abs(o)),o<=45&&o>=0?!1===s.options.rtl?\"left\":\"right\":o<=360&&o>=315?!1===s.options.rtl?\"left\":\"right\":o>=135&&o<=225?!1===s.options.rtl?\"right\":\"left\":!0===s.options.verticalSwiping?o>=35&&o<=135?\"down\":\"up\":\"vertical\"},e.prototype.swipeEnd=function(i){var e,t,o=this;if(o.dragging=!1,o.swiping=!1,o.scrolling)return o.scrolling=!1,!1;if(o.interrupted=!1,o.shouldClick=!(o.touchObject.swipeLength>10),void 0===o.touchObject.curX)return!1;if(!0===o.touchObject.edgeHit&&o.$slider.trigger(\"edge\",[o,o.swipeDirection()]),o.touchObject.swipeLength>=o.touchObject.minSwipe){switch(t=o.swipeDirection()){case\"left\":case\"down\":e=o.options.swipeToSlide?o.checkNavigable(o.currentSlide+o.getSlideCount()):o.currentSlide+o.getSlideCount(),o.currentDirection=0;break;case\"right\":case\"up\":e=o.options.swipeToSlide?o.checkNavigable(o.currentSlide-o.getSlideCount()):o.currentSlide-o.getSlideCount(),o.currentDirection=1}\"vertical\"!=t&&(o.slideHandler(e),o.touchObject={},o.$slider.trigger(\"swipe\",[o,t]))}else o.touchObject.startX!==o.touchObject.curX&&(o.slideHandler(o.currentSlide),o.touchObject={})},e.prototype.swipeHandler=function(i){var e=this;if(!(!1===e.options.swipe||\"ontouchend\"in document&&!1===e.options.swipe||!1===e.options.draggable&&-1!==i.type.indexOf(\"mouse\")))switch(e.touchObject.fingerCount=i.originalEvent&&void 0!==i.originalEvent.touches?i.originalEvent.touches.length:1,e.touchObject.minSwipe=e.listWidth/e.options.touchThreshold,!0===e.options.verticalSwiping&&(e.touchObject.minSwipe=e.listHeight/e.options.touchThreshold),i.data.action){case\"start\":e.swipeStart(i);break;case\"move\":e.swipeMove(i);break;case\"end\":e.swipeEnd(i)}},e.prototype.swipeMove=function(i){var e,t,o,s,n,r,l=this;return n=void 0!==i.originalEvent?i.originalEvent.touches:null,!(!l.dragging||l.scrolling||n&&1!==n.length)&&(e=l.getLeft(l.currentSlide),l.touchObject.curX=void 0!==n?n[0].pageX:i.clientX,l.touchObject.curY=void 0!==n?n[0].pageY:i.clientY,l.touchObject.swipeLength=Math.round(Math.sqrt(Math.pow(l.touchObject.curX-l.touchObject.startX,2))),r=Math.round(Math.sqrt(Math.pow(l.touchObject.curY-l.touchObject.startY,2))),!l.options.verticalSwiping&&!l.swiping&&r>4?(l.scrolling=!0,!1):(!0===l.options.verticalSwiping&&(l.touchObject.swipeLength=r),t=l.swipeDirection(),void 0!==i.originalEvent&&l.touchObject.swipeLength>4&&(l.swiping=!0,i.preventDefault()),s=(!1===l.options.rtl?1:-1)*(l.touchObject.curX>l.touchObject.startX?1:-1),!0===l.options.verticalSwiping&&(s=l.touchObject.curY>l.touchObject.startY?1:-1),o=l.touchObject.swipeLength,l.touchObject.edgeHit=!1,!1===l.options.infinite&&(0===l.currentSlide&&\"right\"===t||l.currentSlide>=l.getDotCount()&&\"left\"===t)&&(o=l.touchObject.swipeLength*l.options.edgeFriction,l.touchObject.edgeHit=!0),!1===l.options.vertical?l.swipeLeft=e+o*s:l.swipeLeft=e+o*(l.$list.height()/l.listWidth)*s,!0===l.options.verticalSwiping&&(l.swipeLeft=e+o*s),!0!==l.options.fade&&!1!==l.options.touchMove&&(!0===l.animating?(l.swipeLeft=null,!1):void l.setCSS(l.swipeLeft))))},e.prototype.swipeStart=function(i){var e,t=this;if(t.interrupted=!0,1!==t.touchObject.fingerCount||t.slideCount<=t.options.slidesToShow)return t.touchObject={},!1;void 0!==i.originalEvent&&void 0!==i.originalEvent.touches&&(e=i.originalEvent.touches[0]),t.touchObject.startX=t.touchObject.curX=void 0!==e?e.pageX:i.clientX,t.touchObject.startY=t.touchObject.curY=void 0!==e?e.pageY:i.clientY,t.dragging=!0},e.prototype.unfilterSlides=e.prototype.slickUnfilter=function(){var i=this;null!==i.$slidesCache&&(i.unload(),i.$slideTrack.children(this.options.slide).detach(),i.$slidesCache.appendTo(i.$slideTrack),i.reinit())},e.prototype.unload=function(){var e=this;i(\".slick-cloned\",e.$slider).remove(),e.$dots&&e.$dots.remove(),e.$prevArrow&&e.htmlExpr.test(e.options.prevArrow)&&e.$prevArrow.remove(),e.$nextArrow&&e.htmlExpr.test(e.options.nextArrow)&&e.$nextArrow.remove(),e.$slides.removeClass(\"slick-slide slick-active slick-visible slick-current\").attr(\"aria-hidden\",\"true\").css(\"width\",\"\")},e.prototype.unslick=function(i){this.$slider.trigger(\"unslick\",[this,i]),this.destroy()},e.prototype.updateArrows=function(){var i=this;Math.floor(i.options.slidesToShow/2),!0===i.options.arrows&&i.slideCount>i.options.slidesToShow&&!i.options.infinite&&(i.$prevArrow.removeClass(\"slick-disabled\").attr(\"aria-disabled\",\"false\"),i.$nextArrow.removeClass(\"slick-disabled\").attr(\"aria-disabled\",\"false\"),0===i.currentSlide?(i.$prevArrow.addClass(\"slick-disabled\").attr(\"aria-disabled\",\"true\"),i.$nextArrow.removeClass(\"slick-disabled\").attr(\"aria-disabled\",\"false\")):i.currentSlide>=i.slideCount-i.options.slidesToShow&&!1===i.options.centerMode?(i.$nextArrow.addClass(\"slick-disabled\").attr(\"aria-disabled\",\"true\"),i.$prevArrow.removeClass(\"slick-disabled\").attr(\"aria-disabled\",\"false\")):i.currentSlide>=i.slideCount-1&&!0===i.options.centerMode&&(i.$nextArrow.addClass(\"slick-disabled\").attr(\"aria-disabled\",\"true\"),i.$prevArrow.removeClass(\"slick-disabled\").attr(\"aria-disabled\",\"false\")))},e.prototype.updateDots=function(){var i=this;null!==i.$dots&&(i.$dots.find(\"li\").removeClass(\"slick-active\").end(),i.$dots.find(\"li\").eq(Math.floor(i.currentSlide/i.options.slidesToScroll)).addClass(\"slick-active\"))},e.prototype.visibility=function(){this.options.autoplay&&(document[this.hidden]?this.interrupted=!0:this.interrupted=!1)},i.fn.slick=function(){var i,t,o=this,s=arguments[0],n=Array.prototype.slice.call(arguments,1),r=o.length;for(i=0;i<r;i++)if(\"object\"==typeof s||void 0===s?o[i].slick=new e(o[i],s):t=o[i].slick[s].apply(o[i].slick,n),void 0!==t)return t;return o}});\n","Magento_PageBuilder/js/utils/breakpoints.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore'\n], function (_) {\n    'use strict';\n\n    return {\n        /**\n         * Build media query.\n         *\n         * @param {Object} conditions\n         * @returns {String}\n         */\n        buildMedia: function (conditions) {\n            var result = _.map(_.pairs(conditions), function (condition) {\n                return '(' + condition.join(': ') + ')';\n            });\n\n            return result.join(' and ');\n        }\n    };\n});\n","Magento_PageBuilder/js/utils/map.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * googleMaps dependency is added within googlemaps.phtml through shim based on API key being set\n *\n * @api\n */\ndefine([\n    'underscore',\n    'module',\n    'Magento_PageBuilder/js/events'\n], function (_, module, events) {\n    'use strict';\n\n    var google = window.google || {},\n\n        /**\n         * Generates a google map usable latitude and longitude object\n         *\n         * @param {Object} position\n         * @return {google.maps.LatLng}\n         */\n        getGoogleLatitudeLongitude = function (position) {\n            return new google.maps.LatLng(position.latitude, position.longitude);\n        },\n        gmAuthFailure = false;\n\n    // jscs:disable requireCamelCaseOrUpperCaseIdentifiers\n    /**\n     * Google's error listener for map loader failures\n     */\n    window.gm_authFailure = function () {\n        events.trigger('googleMaps:authFailure');\n        gmAuthFailure = true;\n    };\n    // jscs:enable requireCamelCaseOrUpperCaseIdentifiers\n\n    return function (element, markers, additionalOptions) {\n        var options,\n            style;\n\n        // If we've previously had an API key error, throw the error even again\n        if (gmAuthFailure) {\n            events.trigger('googleMaps:authFailure');\n\n            return;\n        }\n\n        // If Google Maps isn't loaded don't try init the map, it won't work\n        if (typeof google.maps === 'undefined') {\n            return;\n        }\n\n        /**\n         * Just in case of a bad JSON that bypassed validation\n         */\n        try {\n            style = module.config().style ? JSON.parse(module.config().style) : [];\n        }\n        catch (error) {\n            style = [];\n        }\n        options = _.extend({\n            zoom: 8,\n            center: getGoogleLatitudeLongitude({\n                latitude: 30.2672,\n                longitude: -97.7431\n            }),\n            scrollwheel: false,\n            disableDoubleClickZoom: false,\n            disableDefaultUI: false,\n            mapTypeControl: true,\n            mapTypeControlOptions: {\n                style: google.maps.MapTypeControlStyle.DEFAULT\n            },\n            styles: style\n        }, additionalOptions);\n\n        /* Create the map */\n        this.map = new google.maps.Map(element, options);\n        this.markers = [];\n\n        /**\n         * Callback function on map config update\n         * @param {Array} newMarkers\n         * @param {Object} updateOptions\n         */\n        this.onUpdate = function (newMarkers, updateOptions) {\n            this.map.setOptions(updateOptions);\n            this.setMarkers(newMarkers);\n        };\n\n        /**\n         * Sets the markers to selected map\n         * @param {Object} newMarkers\n         */\n        this.setMarkers = function (newMarkers) {\n            var activeInfoWindow,\n                latitudeLongitudeBounds = new google.maps.LatLngBounds();\n\n            this.markers.forEach(function (marker) {\n                marker.setMap(null);\n            }, this);\n\n            this.markers = [];\n            this.bounds = [];\n\n            /**\n             * Creates and set listener for markers\n             */\n            if (newMarkers && newMarkers.length) {\n                newMarkers.forEach(function (newMarker) {\n                    var location = _.escape(newMarker['location_name']) || '',\n                    comment = newMarker.comment ?\n                        '<p>' + _.escape(newMarker.comment).replace(/(?:\\r\\n|\\r|\\n)/g, '<br/>') + '</p>'\n                        : '',\n                    phone = newMarker.phone ? '<p>Phone: ' + _.escape(newMarker.phone) + '</p>' : '',\n                    address = newMarker.address ? _.escape(newMarker.address) + '<br/>' : '',\n                    city = _.escape(newMarker.city) || '',\n                    country = newMarker.country ? _.escape(newMarker.country) : '',\n                    state = newMarker.state ? _.escape(newMarker.state) + ' ' : '',\n                    zipCode = newMarker.zipcode ? _.escape(newMarker.zipcode) : '',\n                    cityComma = city !== '' && (zipCode !== '' || state !== '') ? ', ' : '',\n                    lineBreak = city !== '' || zipCode !== '' ? '<br/>' : '',\n                    contentString =\n                        '<div>' +\n                        '<h3><b>' + location + '</b></h3>' +\n                        comment +\n                        phone +\n                        '<p><span>' + address +\n                        city + cityComma + state + zipCode + lineBreak +\n                        country + '</span></p>' +\n                        '</div>',\n                    infowindow = new google.maps.InfoWindow({\n                        content: contentString,\n                        maxWidth: 350\n                    }),\n                    newCreatedMarker = new google.maps.Marker({\n                        map: this.map,\n                        position: getGoogleLatitudeLongitude(newMarker.position),\n                        title: location\n                    });\n\n                    if (location) {\n                        newCreatedMarker.addListener('click', function () {\n                            if (activeInfoWindow) {\n                                activeInfoWindow.close();\n                            }\n\n                            infowindow.open(this.map, newCreatedMarker);\n                            activeInfoWindow = infowindow;\n                        }, this);\n                    }\n\n                    this.markers.push(newCreatedMarker);\n                    this.bounds.push(getGoogleLatitudeLongitude(newMarker.position));\n                }, this);\n            }\n\n            /**\n             * This sets the bounds of the map for multiple locations\n             */\n            if (this.bounds.length > 1) {\n                this.bounds.forEach(function (bound) {\n                    latitudeLongitudeBounds.extend(bound);\n                });\n                this.map.fitBounds(latitudeLongitudeBounds);\n            }\n\n            /**\n             * Zoom to 8 if there is only a single location\n             */\n            if (this.bounds.length === 1) {\n                this.map.setCenter(this.bounds[0]);\n                this.map.setZoom(8);\n            }\n        };\n\n        this.setMarkers(markers);\n    };\n});\n","Magento_PageBuilder/js/widget/show-on-hover.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['jquery'], function ($) {\n    'use strict';\n\n    /**\n     * Show the overlay on hover of specific elements\n     *\n     * @param {JQuery<Element>[]} $elements\n     */\n    function showOverlayOnHover($elements) {\n        $elements.each(function (index, element) {\n            var overlayEl = $(element).find('.pagebuilder-overlay'),\n                overlayColor = overlayEl.attr('data-overlay-color');\n\n            $(element).on('mouseenter', function () {\n                overlayEl.css('background-color', overlayColor);\n            });\n\n            $(element).on('mouseleave', function () {\n                overlayEl.css('background-color', 'transparent');\n            });\n        });\n    }\n\n    /**\n     * Show button on hover of specific elements\n     *\n     * @param {JQuery<Element>[]} $elements\n     * @param {String} buttonClass\n     */\n    function showButtonOnHover($elements, buttonClass) {\n        $elements.each(function (index, element) {\n            var buttonEl = $(element).find(buttonClass);\n\n            $(element).on('mouseenter', function () {\n                buttonEl.css({\n                    'opacity': '1',\n                    'visibility': 'visible'\n                });\n            });\n\n            $(element).on('mouseleave', function () {\n                buttonEl.css({\n                    'opacity': '0',\n                    'visibility': 'hidden'\n                });\n            });\n        });\n    }\n\n    return function (config) {\n\n        var buttonSelector = config.buttonSelector,\n            overlayHoverSelector = 'div[data-content-type=\"%s\"][data-show-overlay=\"%s\"]'\n                .replace('%s', config.dataRole)\n                .replace('%s', config.showOverlay),\n            overlayButtonSelector = 'div[data-content-type=\"%s\"][data-show-button=\"%s\"]'\n                .replace('%s', config.dataRole)\n                .replace('%s', config.showOverlay);\n\n        showOverlayOnHover($(overlayHoverSelector));\n        showButtonOnHover($(overlayButtonSelector), buttonSelector);\n    };\n});\n","Magento_PageBuilder/js/widget/video-background.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'jarallax',\n    'jarallaxVideo',\n    'vimeoWrapper'\n], function ($) {\n    'use strict';\n\n    return function (config, element) {\n        var $element = $(element),\n            parallaxSpeed = $element.data('enableParallax') !== 1 ? 1 : parseFloat($element.data('parallaxSpeed'));\n\n        if ($element.data('background-type') !== 'video') {\n            return;\n        }\n\n        $element.addClass('jarallax');\n        $element.attr('data-jarallax', '');\n\n        window.jarallax($element[0], {\n            imgSrc: $element.data('videoFallbackSrc'),\n            speed: !isNaN(parallaxSpeed) ? parallaxSpeed : 0.5,\n            videoLoop: $element.data('videoLoop'),\n            videoPlayOnlyVisible: $element.data('videoPlayOnlyVisible'),\n            videoLazyLoading: $element.data('videoLazyLoad'),\n            disableVideo: false,\n            elementInViewport: $element.data('elementInViewport') &&\n                $element[0].querySelector($element.data('elementInViewport'))\n        });\n        $element[0].jarallax.video && $element[0].jarallax.video.on('started', function () {\n            if ($element[0].jarallax.$video) {\n                $element[0].jarallax.$video.style.visibility = 'visible';\n            }\n        });\n    };\n});\n","Magento_PageCache/js/form-key-provider.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine(function () {\n    'use strict';\n\n    return function (settings) {\n        var formKey,\n            inputElements,\n            inputSelector = 'input[name=\"form_key\"]';\n\n        /**\n         * Set form_key cookie\n         * @private\n         */\n        function setFormKeyCookie(value) {\n            var expires,\n                secure,\n                date = new Date(),\n                cookiesConfig = window.cookiesConfig || {},\n                isSecure = !!cookiesConfig.secure,\n                samesite = cookiesConfig.samesite || 'lax';\n\n            date.setTime(date.getTime() + 86400000);\n            expires = '; expires=' + date.toUTCString();\n            secure = isSecure ? '; secure' : '';\n            samesite = '; samesite=' + samesite;\n\n            document.cookie = 'form_key=' + (value || '') + expires + secure + '; path=/' + samesite;\n        }\n\n        /**\n         * Retrieves form key from cookie\n         * @private\n         */\n        function getFormKeyCookie() {\n            var cookie,\n                i,\n                nameEQ = 'form_key=',\n                cookieArr = document.cookie.split(';');\n\n            for (i = 0; i < cookieArr.length; i++) {\n                cookie = cookieArr[i];\n\n                while (cookie.charAt(0) === ' ') {\n                    cookie = cookie.substring(1, cookie.length);\n                }\n\n                if (cookie.indexOf(nameEQ) === 0) {\n                    return cookie.substring(nameEQ.length, cookie.length);\n                }\n            }\n\n            return null;\n        }\n\n        /**\n         * Get form key from UI input hidden\n         * @private\n         */\n        function getFormKeyFromUI() {\n            return document.querySelector(inputSelector).value;\n        }\n\n        /**\n         * Generate form key string\n         * @private\n         */\n        function generateFormKeyString() {\n            var result = '',\n                length = 16,\n                chars = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';\n\n            while (length--) {\n                result += chars[Math.round(Math.random() * (chars.length - 1))];\n            }\n\n            return result;\n        }\n\n        /**\n         * Init form_key inputs with value\n         * @private\n         */\n        function initFormKey() {\n            formKey = getFormKeyCookie();\n\n            if (settings && settings.isPaginationCacheEnabled && !formKey) {\n                formKey = getFormKeyFromUI();\n                setFormKeyCookie(formKey);\n            }\n\n            if (!formKey) {\n                formKey = generateFormKeyString();\n                setFormKeyCookie(formKey);\n            }\n            inputElements = document.querySelectorAll(inputSelector);\n\n            if (inputElements.length) {\n                Array.prototype.forEach.call(inputElements, function (element) {\n                    element.setAttribute('value', formKey);\n                });\n            }\n        }\n\n        initFormKey();\n    };\n});\n","Magento_PageCache/js/page-cache.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'domReady',\n    'consoleLogger',\n    'Magento_PageCache/js/form-key-provider',\n    'jquery-ui-modules/widget',\n    'mage/cookies'\n], function ($, domReady, consoleLogger, formKeyInit) {\n    'use strict';\n\n    /**\n     * Helper. Generate random string\n     * TODO: Merge with mage/utils\n     * @param {String} chars - list of symbols\n     * @param {Number} length - length for need string\n     * @returns {String}\n     */\n    function generateRandomString(chars, length) {\n        var result = '';\n\n        length = length > 0 ? length : 1;\n\n        while (length--) {\n            result += chars[Math.round(Math.random() * (chars.length - 1))];\n        }\n\n        return result;\n    }\n\n    /**\n     * Nodes tree to flat list converter\n     * @returns {Array}\n     */\n    $.fn.comments = function () {\n        var elements = [],\n            contents,\n            elementContents;\n\n        /**\n         * @param {jQuery} element - Comment holder\n         */\n        (function lookup(element) {\n            var iframeHostName;\n\n            // prevent cross origin iframe content reading\n            if ($(element).prop('tagName') === 'IFRAME') {\n                iframeHostName = $('<a>').prop('href', $(element).prop('src'))\n                    .prop('hostname');\n\n                if (window.location.hostname !== iframeHostName) {\n                    return [];\n                }\n            }\n\n            /**\n             * Rewrite jQuery contents().\n             *\n             * @param {jQuery} elem\n             */\n            contents = function (elem) {\n                return $.map(elem, function (el) {\n                    try {\n                        return el.nodeName.toLowerCase() === 'iframe' ?\n                            el.contentDocument || (el.contentWindow ? el.contentWindow.document : []) :\n                            $.merge([], el.childNodes);\n                    } catch (e) {\n                        consoleLogger.error(e);\n\n                        return [];\n                    }\n                });\n            };\n\n            elementContents = contents($(element));\n\n            $.each(elementContents, function (index, el) {\n                switch (el.nodeType) {\n                    case 1: // ELEMENT_NODE\n                        lookup(el);\n                        break;\n\n                    case 8: // COMMENT_NODE\n                        elements.push(el);\n                        break;\n\n                    case 9: // DOCUMENT_NODE\n                        lookup($(el).find('body'));\n                        break;\n                }\n            });\n        })(this);\n\n        return elements;\n    };\n\n    /**\n     * FormKey Widget - this widget is generating from key, saves it to cookie and\n     * @deprecated see Magento/PageCache/view/frontend/web/js/form-key-provider.js\n     */\n    $.widget('mage.formKey', {\n        options: {\n            inputSelector: 'input[name=\"form_key\"]',\n            allowedCharacters: '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ',\n            length: 16\n        },\n\n        /**\n         * Creates widget 'mage.formKey'\n         * @private\n         */\n        _create: function () {\n            var formKey = $.mage.cookies.get('form_key'),\n                options = {\n                    secure: window.cookiesConfig ? window.cookiesConfig.secure : false\n                };\n\n            if (!formKey) {\n                formKey = generateRandomString(this.options.allowedCharacters, this.options.length);\n                $.mage.cookies.set('form_key', formKey, options);\n            }\n            $(this.options.inputSelector).val(formKey);\n        }\n    });\n\n    /**\n     * PageCache Widget\n     * Handles additional ajax request for rendering user private content.\n     */\n    $.widget('mage.pageCache', {\n        options: {\n            url: '/',\n            patternPlaceholderOpen: /^ BLOCK (.+) $/,\n            patternPlaceholderClose: /^ \\/BLOCK (.+) $/,\n            versionCookieName: 'private_content_version',\n            handles: []\n        },\n\n        /**\n         * Creates widget 'mage.pageCache'\n         * @private\n         */\n        _create: function () {\n            var placeholders,\n                version = $.mage.cookies.get(this.options.versionCookieName);\n\n            if (!version) {\n                return;\n            }\n            placeholders = this._searchPlaceholders(this.element.comments());\n\n            if (placeholders && placeholders.length) {\n                this._ajax(placeholders, version);\n            }\n        },\n\n        /**\n         * Parse page for placeholders.\n         * @param {Array} elements\n         * @returns {Array}\n         * @private\n         */\n        _searchPlaceholders: function (elements) {\n            var placeholders = [],\n                tmp = {},\n                ii,\n                len,\n                el, matches, name;\n\n            if (!(elements && elements.length)) {\n                return placeholders;\n            }\n\n            for (ii = 0, len = elements.length; ii < len; ii++) {\n                el = elements[ii];\n                matches = this.options.patternPlaceholderOpen.exec(el.nodeValue);\n                name = null;\n\n                if (matches) {\n                    name = matches[1];\n                    tmp[name] = {\n                        name: name,\n                        openElement: el\n                    };\n                } else {\n                    matches = this.options.patternPlaceholderClose.exec(el.nodeValue);\n\n                    if (matches) { //eslint-disable-line max-depth\n                        name = matches[1];\n\n                        if (tmp[name]) { //eslint-disable-line max-depth\n                            tmp[name].closeElement = el;\n                            placeholders.push(tmp[name]);\n                            delete tmp[name];\n                        }\n                    }\n                }\n            }\n\n            return placeholders;\n        },\n\n        /**\n         * Parse for page and replace placeholders\n         * @param {Object} placeholder\n         * @param {Object} html\n         * @protected\n         */\n        _replacePlaceholder: function (placeholder, html) {\n            var startReplacing = false,\n                prevSibling = null,\n                parent, contents, yy, len, element;\n\n            if (!placeholder || !html) {\n                return;\n            }\n\n            parent = $(placeholder.openElement).parent();\n            contents = parent.contents();\n\n            for (yy = 0, len = contents.length; yy < len; yy++) {\n                element = contents[yy];\n\n                if (element == placeholder.openElement) { //eslint-disable-line eqeqeq\n                    startReplacing = true;\n                }\n\n                if (startReplacing) {\n                    $(element).remove();\n                } else if (element.nodeType != 8) { //eslint-disable-line eqeqeq\n                    //due to comment tag doesn't have siblings we try to find it manually\n                    prevSibling = element;\n                }\n\n                if (element == placeholder.closeElement) { //eslint-disable-line eqeqeq\n                    break;\n                }\n            }\n\n            if (prevSibling) {\n                $(prevSibling).after(html);\n            } else {\n                $(parent).prepend(html);\n            }\n\n            // trigger event to use mage-data-init attribute\n            $(parent).trigger('contentUpdated');\n        },\n\n        /**\n         * AJAX helper\n         * @param {Object} placeholders\n         * @param {String} version\n         * @private\n         */\n        _ajax: function (placeholders, version) {\n            var ii,\n                data = {\n                    blocks: [],\n                    handles: this.options.handles,\n                    originalRequest: this.options.originalRequest,\n                    version: version\n                };\n\n            for (ii = 0; ii < placeholders.length; ii++) {\n                data.blocks.push(placeholders[ii].name);\n            }\n            data.blocks = JSON.stringify(data.blocks.sort());\n            data.handles = JSON.stringify(data.handles);\n            data.originalRequest = JSON.stringify(data.originalRequest);\n            $.ajax({\n                url: this.options.url,\n                data: data,\n                type: 'GET',\n                cache: true,\n                dataType: 'json',\n                context: this,\n\n                /**\n                 * Response handler\n                 * @param {Object} response\n                 */\n                success: function (response) {\n                    var placeholder, i;\n\n                    for (i = 0; i < placeholders.length; i++) {\n                        placeholder = placeholders[i];\n\n                        if (response.hasOwnProperty(placeholder.name)) {\n                            this._replacePlaceholder(placeholder, response[placeholder.name]);\n                        }\n                    }\n                }\n            });\n        }\n    });\n\n    domReady(function () {\n        formKeyInit();\n    });\n\n    return {\n        'pageCache': $.mage.pageCache,\n        'formKey': $.mage.formKey\n    };\n});\n","Magento_Payment/js/cc-type.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.creditCardType', {\n        options: {\n            typeCodes: ['SS', 'SM', 'SO'] // Type codes for Switch/Maestro/Solo credit cards.\n        },\n\n        /**\n         * Bind change handler to select element and trigger the event to show/hide\n         * the Switch/Maestro or Solo credit card type container for those credit card types.\n         * @private\n         */\n        _create: function () {\n            this.element.on('change', $.proxy(this._toggleCardType, this)).trigger('change');\n        },\n\n        /**\n         * Toggle the Switch/Maestro and Solo credit card type container depending on which\n         * credit card type is selected.\n         * @private\n         */\n        _toggleCardType: function () {\n            $(this.options.creditCardTypeContainer)\n                .toggle($.inArray(this.element.val(), this.options.typeCodes) !== -1);\n        }\n    });\n\n    return $.mage.creditCardType;\n});\n","Magento_Payment/js/transparent.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'jquery',\n    'mage/template',\n    'Magento_Ui/js/modal/alert',\n    'jquery-ui-modules/widget',\n    'Magento_Payment/js/model/credit-card-validation/validator',\n    'Magento_Checkout/js/model/full-screen-loader'\n], function ($, mageTemplate, alert, ui, validator, fullScreenLoader) {\n    'use strict';\n\n    $.widget('mage.transparent', {\n        options: {\n            context: null,\n            placeOrderSelector: '[data-role=\"review-save\"]',\n            paymentFormSelector: '#co-payment-form',\n            updateSelectorPrefix: '#checkout-',\n            updateSelectorSuffix: '-load',\n            hiddenFormTmpl:\n                '<form target=\"<%= data.target %>\" action=\"<%= data.action %>\" method=\"POST\" ' +\n                'hidden enctype=\"application/x-www-form-urlencoded\" class=\"no-display\">' +\n                    '<% _.each(data.inputs, function(val, key){ %>' +\n                    '<input value=\"<%= val %>\" name=\"<%= key %>\" type=\"hidden\">' +\n                    '<% }); %>' +\n                '</form>',\n            reviewAgreementForm: '#checkout-agreements',\n            cgiUrl: null,\n            orderSaveUrl: null,\n            controller: null,\n            gateway: null,\n            dateDelim: null,\n            cardFieldsMap: null,\n            expireYearLength: 2\n        },\n\n        /**\n         * {Function}\n         * @private\n         */\n        _create: function () {\n            this.hiddenFormTmpl = mageTemplate(this.options.hiddenFormTmpl);\n\n            if (this.options.context) {\n                this.options.context.setPlaceOrderHandler($.proxy(this._orderSave, this));\n                this.options.context.setValidateHandler($.proxy(this._validateHandler, this));\n            } else {\n                $(this.options.placeOrderSelector)\n                    .off('click')\n                    .on('click', $.proxy(this._placeOrderHandler, this));\n            }\n\n            this.element.validation();\n            $('[data-container=\"' + this.options.gateway + '-cc-number\"]').on('focusout', function () {\n                $(this).valid();\n            });\n        },\n\n        /**\n         * handler for credit card validation\n         * @return {Boolean}\n         * @private\n         */\n        _validateHandler: function () {\n            return this.element.validation && this.element.validation('isValid');\n        },\n\n        /**\n         * handler for Place Order button to call gateway for credit card validation\n         * @return {Boolean}\n         * @private\n         */\n        _placeOrderHandler: function () {\n            if (this._validateHandler()) {\n                this._orderSave();\n            }\n\n            return false;\n        },\n\n        /**\n         * Save order and generate post data for gateway call\n         * @private\n         */\n        _orderSave: function () {\n            var postData = $(this.options.paymentFormSelector).serialize();\n\n            if ($(this.options.reviewAgreementForm).length) {\n                postData += '&' + $(this.options.reviewAgreementForm).serialize();\n            }\n            postData += '&controller=' + this.options.controller;\n            postData += '&cc_type=' + this.element.find(\n                '[data-container=\"' + this.options.gateway + '-cc-type\"]'\n            ).val();\n\n            return $.ajax({\n                url: this.options.orderSaveUrl,\n                type: 'post',\n                context: this,\n                data: postData,\n                dataType: 'json',\n\n                /**\n                 * {Function}\n                 */\n                beforeSend: function () {\n                    fullScreenLoader.startLoader();\n                },\n\n                /**\n                 * {Function}\n                 */\n                success: function (response) {\n                    var preparedData,\n                        msg,\n\n                        /**\n                         * {Function}\n                         */\n                        alertActionHandler = function () {\n                            // default action\n                        };\n\n                    if (response.success && response[this.options.gateway]) {\n                        preparedData = this._preparePaymentData(\n                            response[this.options.gateway].fields,\n                            this.options.cardFieldsMap\n                        );\n                        this._postPaymentToGateway(preparedData);\n                    } else {\n                        fullScreenLoader.stopLoader(true);\n\n                        msg = response['error_messages'];\n\n                        if (this.options.context) {\n                            this.options.context.clearTimeout().fail();\n                            alertActionHandler = this.options.context.alertActionHandler;\n                        }\n\n                        if (typeof msg === 'object') {\n                            msg = msg.join('\\n');\n                        }\n\n                        if (msg) {\n                            alert(\n                                {\n                                    content: msg,\n                                    actions: {\n\n                                        /**\n                                         * {Function}\n                                         */\n                                        always: alertActionHandler\n                                    }\n                                }\n                            );\n                        }\n                    }\n                }.bind(this)\n            });\n        },\n\n        /**\n         * Post data to gateway for credit card validation\n         * @param {Object} data\n         * @private\n         */\n        _postPaymentToGateway: function (data) {\n            var tmpl,\n                iframeSelector = '[data-container=\"' + this.options.gateway + '-transparent-iframe\"]';\n\n            tmpl = this.hiddenFormTmpl({\n                data: {\n                    target: $(iframeSelector).attr('name'),\n                    action: this.options.cgiUrl,\n                    inputs: data\n                }\n            });\n            $(tmpl).appendTo($(iframeSelector)).trigger('submit');\n        },\n\n        /**\n         * Add credit card fields to post data for gateway\n         * @param {Object} data\n         * @param {Object} ccfields\n         * @private\n         */\n        _preparePaymentData: function (data, ccfields) {\n            var preparedata;\n\n            if (this.element.find('[data-container=\"' + this.options.gateway + '-cc-cvv\"]').length) {\n                data[ccfields.cccvv] = this.element.find(\n                    '[data-container=\"' + this.options.gateway + '-cc-cvv\"]'\n                ).val();\n            }\n            preparedata = this._prepareExpDate();\n            data[ccfields.ccexpdate] = preparedata.month + this.options.dateDelim + preparedata.year;\n            data[ccfields.ccnum] = this.element.find(\n                '[data-container=\"' + this.options.gateway + '-cc-number\"]'\n            ).val();\n\n            return data;\n        },\n\n        /**\n         * Grab Month and Year into one\n         * @returns {Object}\n         * @private\n         */\n        _prepareExpDate: function () {\n            var year = this.element.find('[data-container=\"' + this.options.gateway + '-cc-year\"]').val(),\n                month = parseInt(\n                    this.element.find('[data-container=\"' + this.options.gateway + '-cc-month\"]').val(),\n                    10\n                );\n\n            if (year.length > this.options.expireYearLength) {\n                year = year.substring(year.length - this.options.expireYearLength);\n            }\n\n            if (month < 10) {\n                month = '0' + month;\n            }\n\n            return {\n                month: month, year: year\n            };\n        }\n    });\n\n    return $.mage.transparent;\n});\n","Magento_Payment/js/model/credit-card-validation/credit-card-data.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([], function () {\n    'use strict';\n\n    return {\n        creditCard: null,\n        creditCardNumber: null,\n        expirationMonth: null,\n        expirationYear: null,\n        cvvCode: null\n    };\n});\n","Magento_Payment/js/model/credit-card-validation/credit-card-number-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'mageUtils',\n    'Magento_Payment/js/model/credit-card-validation/credit-card-number-validator/luhn10-validator',\n    'Magento_Payment/js/model/credit-card-validation/credit-card-number-validator/credit-card-type'\n], function (utils, luhn10, creditCardTypes) {\n    'use strict';\n\n    /**\n     * @param {*} card\n     * @param {*} isPotentiallyValid\n     * @param {*} isValid\n     * @return {Object}\n     */\n    function resultWrapper(card, isPotentiallyValid, isValid) {\n        return {\n            card: card,\n            isValid: isValid,\n            isPotentiallyValid: isPotentiallyValid\n        };\n    }\n\n    return function (value) {\n        var potentialTypes,\n            cardType,\n            valid,\n            i,\n            maxLength;\n\n        if (utils.isEmpty(value)) {\n            return resultWrapper(null, false, false);\n        }\n\n        value = value.replace(/\\s+/g, '');\n\n        if (!/^\\d*$/.test(value)) {\n            return resultWrapper(null, false, false);\n        }\n\n        potentialTypes = creditCardTypes.getCardTypes(value);\n\n        if (potentialTypes.length === 0) {\n            return resultWrapper(null, false, false);\n        } else if (potentialTypes.length !== 1) {\n            return resultWrapper(null, true, false);\n        }\n\n        cardType = potentialTypes[0];\n\n        if (cardType.type === 'unionpay') {  // UnionPay is not Luhn 10 compliant\n            valid = true;\n        } else {\n            valid = luhn10(value);\n        }\n\n        for (i = 0; i < cardType.lengths.length; i++) {\n            if (cardType.lengths[i] === value.length) {\n                return resultWrapper(cardType, valid, valid);\n            }\n        }\n\n        maxLength = Math.max.apply(null, cardType.lengths);\n\n        if (value.length < maxLength) {\n            return resultWrapper(cardType, true, false);\n        }\n\n        return resultWrapper(cardType, false, false);\n    };\n});\n","Magento_Payment/js/model/credit-card-validation/cvv-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([], function () {\n    'use strict';\n\n    /**\n     * @param {*} isValid\n     * @param {*} isPotentiallyValid\n     * @return {Object}\n     */\n    function resultWrapper(isValid, isPotentiallyValid) {\n        return {\n            isValid: isValid,\n            isPotentiallyValid: isPotentiallyValid\n        };\n    }\n\n    /**\n     * CVV number validation.\n     * Validate digit count for CVV code.\n     *\n     * @param {*} value\n     * @param {Number} maxLength\n     * @return {Object}\n     */\n    return function (value, maxLength) {\n        var DEFAULT_LENGTH = 3;\n\n        maxLength = maxLength || DEFAULT_LENGTH;\n\n        if (!/^\\d*$/.test(value)) {\n            return resultWrapper(false, false);\n        }\n\n        if (value.length === maxLength) {\n            return resultWrapper(true, true);\n        }\n\n        if (value.length < maxLength) {\n            return resultWrapper(false, true);\n        }\n\n        if (value.length > maxLength) {\n            return resultWrapper(false, false);\n        }\n    };\n});\n","Magento_Payment/js/model/credit-card-validation/expiration-date-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'mageUtils',\n    'Magento_Payment/js/model/credit-card-validation/expiration-date-validator/parse-date',\n    'Magento_Payment/js/model/credit-card-validation/expiration-date-validator/expiration-month-validator',\n    'Magento_Payment/js/model/credit-card-validation/expiration-date-validator/expiration-year-validator'\n], function (utils, parseDate, expirationMonth, expirationYear) {\n    'use strict';\n\n    /**\n     * @param {*} isValid\n     * @param {*} isPotentiallyValid\n     * @param {*} month\n     * @param {*} year\n     * @return {Object}\n     */\n    function resultWrapper(isValid, isPotentiallyValid, month, year) {\n        return {\n            isValid: isValid,\n            isPotentiallyValid: isPotentiallyValid,\n            month: month,\n            year: year\n        };\n    }\n\n    return function (value) {\n        var date,\n            monthValid,\n            yearValid;\n\n        if (utils.isEmpty(value)) {\n            return resultWrapper(false, false, null, null);\n        }\n\n        value = value.replace(/^(\\d\\d) (\\d\\d(\\d\\d)?)$/, '$1/$2');\n        date = parseDate(value);\n        monthValid = expirationMonth(date.month);\n        yearValid = expirationYear(date.year);\n\n        if (monthValid.isValid && yearValid.isValid) {\n            return resultWrapper(true, true, date.month, date.year);\n        }\n\n        if (monthValid.isPotentiallyValid && yearValid.isPotentiallyValid) {\n            return resultWrapper(false, true, null, null);\n        }\n\n        return resultWrapper(false, false, null, null);\n    };\n});\n","Magento_Payment/js/model/credit-card-validation/validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'jquery',\n    'Magento_Payment/js/model/credit-card-validation/cvv-validator',\n    'Magento_Payment/js/model/credit-card-validation/credit-card-number-validator',\n    'Magento_Payment/js/model/credit-card-validation/expiration-date-validator/expiration-year-validator',\n    'Magento_Payment/js/model/credit-card-validation/expiration-date-validator/expiration-month-validator',\n    'Magento_Payment/js/model/credit-card-validation/credit-card-data',\n    'mage/translate'\n], function ($, cvvValidator, creditCardNumberValidator, yearValidator, monthValidator, creditCardData) {\n    'use strict';\n\n    $('.payment-method-content input[type=\"number\"]').on('keyup', function () {\n        if ($(this).val() < 0) {\n            $(this).val($(this).val().replace(/^-/, ''));\n        }\n    });\n\n    $.each({\n        'validate-card-type': [\n            function (number, item, allowedTypes) {\n                var cardInfo,\n                    i,\n                    l;\n\n                if (!creditCardNumberValidator(number).isValid) {\n                    return false;\n                }\n\n                cardInfo = creditCardNumberValidator(number).card;\n\n                for (i = 0, l = allowedTypes.length; i < l; i++) {\n                    if (cardInfo.title == allowedTypes[i].type) { //eslint-disable-line eqeqeq\n                        return true;\n                    }\n                }\n\n                return false;\n            },\n            $.mage.__('Please enter a valid credit card type number.')\n        ],\n        'validate-card-number': [\n\n            /**\n             * Validate credit card number based on mod 10\n             *\n             * @param {*} number - credit card number\n             * @return {Boolean}\n             */\n            function (number) {\n                return creditCardNumberValidator(number).isValid;\n            },\n            $.mage.__('Please enter a valid credit card number.')\n        ],\n        'validate-card-date': [\n\n            /**\n             * Validate credit card expiration month\n             *\n             * @param {String} date - month\n             * @return {Boolean}\n             */\n            function (date) {\n                return monthValidator(date).isValid;\n            },\n            $.mage.__('Incorrect credit card expiration month.')\n        ],\n        'validate-card-cvv': [\n\n            /**\n             * Validate cvv\n             *\n             * @param {String} cvv - card verification value\n             * @return {Boolean}\n             */\n            function (cvv) {\n                var maxLength = creditCardData.creditCard ? creditCardData.creditCard.code.size : 3;\n\n                return cvvValidator(cvv, maxLength).isValid;\n            },\n            $.mage.__('Please enter a valid credit card verification number.')\n        ],\n        'validate-card-year': [\n\n            /**\n             * Validate credit card expiration year\n             *\n             * @param {String} date - year\n             * @return {Boolean}\n             */\n            function (date) {\n                return yearValidator(date).isValid;\n            },\n            $.mage.__('Incorrect credit card expiration year.')\n        ]\n\n    }, function (i, rule) {\n        rule.unshift(i);\n        $.validator.addMethod.apply($.validator, rule);\n    });\n});\n","Magento_Payment/js/model/credit-card-validation/credit-card-number-validator/credit-card-type.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'jquery',\n    'mageUtils'\n], function ($, utils) {\n    'use strict';\n\n    var types = [\n        {\n            title: 'Visa',\n            type: 'VI',\n            pattern: '^4\\\\d*$',\n            gaps: [4, 8, 12],\n            lengths: [16],\n            code: {\n                name: 'CVV',\n                size: 3\n            }\n        },\n        {\n            title: 'MasterCard',\n            type: 'MC',\n            pattern: '^(?:5[1-5][0-9]{2}|222[1-9]|22[3-9][0-9]|2[3-6][0-9]{2}|27[01][0-9]|2720)[0-9]{12}$',\n            gaps: [4, 8, 12],\n            lengths: [16],\n            code: {\n                name: 'CVC',\n                size: 3\n            }\n        },\n        {\n            title: 'American Express',\n            type: 'AE',\n            pattern: '^3([47]\\\\d*)?$',\n            isAmex: true,\n            gaps: [4, 10],\n            lengths: [15],\n            code: {\n                name: 'CID',\n                size: 4\n            }\n        },\n        {\n            title: 'Diners',\n            type: 'DN',\n            pattern: '^(3(0[0-5]|095|6|[8-9]))\\\\d*$',\n            gaps: [4, 10],\n            lengths: [14, 16, 17, 18, 19],\n            code: {\n                name: 'CVV',\n                size: 3\n            }\n        },\n        {\n            title: 'Discover',\n            type: 'DI',\n            pattern: '^(6011(0|[2-4]|74|7[7-9]|8[6-9]|9)|6(4[4-9]|5))\\\\d*$',\n            gaps: [4, 8, 12],\n            lengths: [16, 17, 18, 19],\n            code: {\n                name: 'CID',\n                size: 3\n            }\n        },\n        {\n            title: 'JCB',\n            type: 'JCB',\n            pattern: '^35(2[8-9]|[3-8])\\\\d*$',\n            gaps: [4, 8, 12],\n            lengths: [16, 17, 18, 19],\n            code: {\n                name: 'CVV',\n                size: 3\n            }\n        },\n        {\n            title: 'UnionPay',\n            type: 'UN',\n            pattern: '^(622(1(2[6-9]|[3-9])|[3-8]|9([[0-1]|2[0-5]))|62[4-6]|628([2-8]))\\\\d*?$',\n            gaps: [4, 8, 12],\n            lengths: [16, 17, 18, 19],\n            code: {\n                name: 'CVN',\n                size: 3\n            }\n        },\n        {\n            title: 'Maestro International',\n            type: 'MI',\n            pattern: '^(5(0|[6-9])|63|67(?!59|6770|6774))\\\\d*$',\n            gaps: [4, 8, 12],\n            lengths: [12, 13, 14, 15, 16, 17, 18, 19],\n            code: {\n                name: 'CVC',\n                size: 3\n            }\n        },\n        {\n            title: 'Maestro Domestic',\n            type: 'MD',\n            pattern: '^6759(?!24|38|40|6[3-9]|70|76)|676770|676774\\\\d*$',\n            gaps: [4, 8, 12],\n            lengths: [12, 13, 14, 15, 16, 17, 18, 19],\n            code: {\n                name: 'CVC',\n                size: 3\n            }\n        },\n        {\n            title: 'Hipercard',\n            type: 'HC',\n            pattern: '^((606282)|(637095)|(637568)|(637599)|(637609)|(637612))\\\\d*$',\n            gaps: [4, 8, 12],\n            lengths: [13, 16],\n            code: {\n                name: 'CVC',\n                size: 3\n            }\n        },\n        {\n            title: 'Elo',\n            type: 'ELO',\n            pattern: '^((509091)|(636368)|(636297)|(504175)|(438935)|(40117[8-9])|(45763[1-2])|' +\n                '(457393)|(431274)|(50990[0-2])|(5099[7-9][0-9])|(50996[4-9])|(509[1-8][0-9][0-9])|' +\n                '(5090(0[0-2]|0[4-9]|1[2-9]|[24589][0-9]|3[1-9]|6[0-46-9]|7[0-24-9]))|' +\n                '(5067(0[0-24-8]|1[0-24-9]|2[014-9]|3[0-379]|4[0-9]|5[0-3]|6[0-5]|7[0-8]))|' +\n                '(6504(0[5-9]|1[0-9]|2[0-9]|3[0-9]))|' +\n                '(6504(8[5-9]|9[0-9])|6505(0[0-9]|1[0-9]|2[0-9]|3[0-8]))|' +\n                '(6505(4[1-9]|5[0-9]|6[0-9]|7[0-9]|8[0-9]|9[0-8]))|' +\n                '(6507(0[0-9]|1[0-8]))|(65072[0-7])|(6509(0[1-9]|1[0-9]|20))|' +\n                '(6516(5[2-9]|6[0-9]|7[0-9]))|(6550(0[0-9]|1[0-9]))|' +\n                '(6550(2[1-9]|3[0-9]|4[0-9]|5[0-8])))\\\\d*$',\n            gaps: [4, 8, 12],\n            lengths: [16],\n            code: {\n                name: 'CVC',\n                size: 3\n            }\n        },\n        {\n            title: 'Aura',\n            type: 'AU',\n            pattern: '^5078\\\\d*$',\n            gaps: [4, 8, 12],\n            lengths: [19],\n            code: {\n                name: 'CVC',\n                size: 3\n            }\n        }\n    ];\n\n    return {\n        /**\n         * @param {*} cardNumber\n         * @return {Array}\n         */\n        getCardTypes: function (cardNumber) {\n            var i, value,\n                result = [];\n\n            if (utils.isEmpty(cardNumber)) {\n                return result;\n            }\n\n            if (cardNumber === '') {\n                return $.extend(true, {}, types);\n            }\n\n            for (i = 0; i < types.length; i++) {\n                value = types[i];\n\n                if (new RegExp(value.pattern).test(cardNumber)) {\n                    result.push($.extend(true, {}, value));\n                }\n            }\n\n            return result;\n        }\n    };\n});\n","Magento_Payment/js/model/credit-card-validation/credit-card-number-validator/luhn10-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    /**\n     * Luhn algorithm verification\n     */\n    return function (a, b, c, d, e) {\n        for (d = +a[b = a.length - 1], e = 0; b--;) {\n            c = +a[b];\n            d += ++e % 2 ? 2 * c % 10 + (c > 4) : c;\n        }\n\n        return !(d % 10);\n    };\n});\n","Magento_Payment/js/model/credit-card-validation/expiration-date-validator/expiration-month-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    /**\n     * @param {*} isValid\n     * @param {*} isPotentiallyValid\n     * @return {Object}\n     */\n    function resultWrapper(isValid, isPotentiallyValid) {\n        return {\n            isValid: isValid,\n            isPotentiallyValid: isPotentiallyValid\n        };\n    }\n\n    return function (value) {\n        var month,\n            monthValid;\n\n        if (value.replace(/\\s/g, '') === '' || value === '0') {\n            return resultWrapper(false, true);\n        }\n\n        if (!/^\\d*$/.test(value)) {\n            return resultWrapper(false, false);\n        }\n\n        if (isNaN(value)) {\n            return resultWrapper(false, false);\n        }\n\n        month = parseInt(value, 10);\n        monthValid = month > 0 && month < 13;\n\n        return resultWrapper(monthValid, monthValid);\n    };\n});\n","Magento_Payment/js/model/credit-card-validation/expiration-date-validator/expiration-year-validator.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    /**\n     * @param {*} isValid\n     * @param {*} isPotentiallyValid\n     * @return {Object}\n     */\n    function resultWrapper(isValid, isPotentiallyValid) {\n        return {\n            isValid: isValid,\n            isPotentiallyValid: isPotentiallyValid\n        };\n    }\n\n    return function (value) {\n        var currentYear = new Date().getFullYear(),\n            len = value.length,\n            valid,\n            expMaxLifetime = 19;\n\n        if (value.replace(/\\s/g, '') === '') {\n            return resultWrapper(false, true);\n        }\n\n        if (!/^\\d*$/.test(value)) {\n            return resultWrapper(false, false);\n        }\n\n        if (len !== 4) {\n            return resultWrapper(false, true);\n        }\n\n        value = parseInt(value, 10);\n        valid = value >= currentYear && value <= currentYear + expMaxLifetime;\n\n        return resultWrapper(valid, valid);\n    };\n});\n","Magento_Payment/js/model/credit-card-validation/expiration-date-validator/parse-date.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return function (value) {\n        var month, len;\n\n        if (value.match('/')) {\n            value = value.split(/\\s*\\/\\s*/g);\n\n            return {\n                month: value[0],\n                year: value.slice(1).join()\n            };\n        }\n\n        len = value[0] === '0' || value.length > 5 || value.length === 4 || value.length === 3 ? 2 : 1;\n        month = value.substr(0, len);\n\n        return {\n            month: month,\n            year: value.substr(month.length, 4)\n        };\n    };\n});\n","Magento_Payment/js/view/payment/cc-form.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'underscore',\n    'Magento_Checkout/js/view/payment/default',\n    'Magento_Payment/js/model/credit-card-validation/credit-card-data',\n    'Magento_Payment/js/model/credit-card-validation/credit-card-number-validator',\n    'mage/translate'\n], function (_, Component, creditCardData, cardNumberValidator, $t) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            creditCardType: '',\n            creditCardExpYear: '',\n            creditCardExpMonth: '',\n            creditCardNumber: '',\n            creditCardSsStartMonth: '',\n            creditCardSsStartYear: '',\n            creditCardSsIssue: '',\n            creditCardVerificationNumber: '',\n            selectedCardType: null\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super()\n                .observe([\n                    'creditCardType',\n                    'creditCardExpYear',\n                    'creditCardExpMonth',\n                    'creditCardNumber',\n                    'creditCardVerificationNumber',\n                    'creditCardSsStartMonth',\n                    'creditCardSsStartYear',\n                    'creditCardSsIssue',\n                    'selectedCardType'\n                ]);\n\n            return this;\n        },\n\n        /**\n         * Init component\n         */\n        initialize: function () {\n            var self = this;\n\n            this._super();\n\n            //Set credit card number to credit card data object\n            this.creditCardNumber.subscribe(function (value) {\n                var result;\n\n                self.selectedCardType(null);\n\n                if (value === '' || value === null) {\n                    return false;\n                }\n                result = cardNumberValidator(value);\n\n                if (!result.isPotentiallyValid && !result.isValid) {\n                    return false;\n                }\n\n                if (result.card !== null) {\n                    self.selectedCardType(result.card.type);\n                    creditCardData.creditCard = result.card;\n                }\n\n                if (result.isValid) {\n                    creditCardData.creditCardNumber = value;\n                    self.creditCardType(result.card.type);\n                }\n            });\n\n            //Set expiration year to credit card data object\n            this.creditCardExpYear.subscribe(function (value) {\n                creditCardData.expirationYear = value;\n            });\n\n            //Set expiration month to credit card data object\n            this.creditCardExpMonth.subscribe(function (value) {\n                creditCardData.expirationMonth = value;\n            });\n\n            //Set cvv code to credit card data object\n            this.creditCardVerificationNumber.subscribe(function (value) {\n                creditCardData.cvvCode = value;\n            });\n        },\n\n        /**\n         * Get code\n         * @returns {String}\n         */\n        getCode: function () {\n            return 'cc';\n        },\n\n        /**\n         * Get data\n         * @returns {Object}\n         */\n        getData: function () {\n            return {\n                'method': this.item.method,\n                'additional_data': {\n                    'cc_cid': this.creditCardVerificationNumber(),\n                    'cc_ss_start_month': this.creditCardSsStartMonth(),\n                    'cc_ss_start_year': this.creditCardSsStartYear(),\n                    'cc_ss_issue': this.creditCardSsIssue(),\n                    'cc_type': this.creditCardType(),\n                    'cc_exp_year': this.creditCardExpYear(),\n                    'cc_exp_month': this.creditCardExpMonth(),\n                    'cc_number': this.creditCardNumber()\n                }\n            };\n        },\n\n        /**\n         * Get list of available credit card types\n         * @returns {Object}\n         */\n        getCcAvailableTypes: function () {\n            return window.checkoutConfig.payment.ccform.availableTypes[this.getCode()];\n        },\n\n        /**\n         * Get payment icons\n         * @param {String} type\n         * @returns {Boolean}\n         */\n        getIcons: function (type) {\n            return window.checkoutConfig.payment.ccform.icons.hasOwnProperty(type) ?\n                window.checkoutConfig.payment.ccform.icons[type]\n                : false;\n        },\n\n        /**\n         * Get list of months\n         * @returns {Object}\n         */\n        getCcMonths: function () {\n            return window.checkoutConfig.payment.ccform.months[this.getCode()];\n        },\n\n        /**\n         * Get list of years\n         * @returns {Object}\n         */\n        getCcYears: function () {\n            return window.checkoutConfig.payment.ccform.years[this.getCode()];\n        },\n\n        /**\n         * Check if current payment has verification\n         * @returns {Boolean}\n         */\n        hasVerification: function () {\n            return window.checkoutConfig.payment.ccform.hasVerification[this.getCode()];\n        },\n\n        /**\n         * @deprecated\n         * @returns {Boolean}\n         */\n        hasSsCardType: function () {\n            return window.checkoutConfig.payment.ccform.hasSsCardType[this.getCode()];\n        },\n\n        /**\n         * Get image url for CVV\n         * @returns {String}\n         */\n        getCvvImageUrl: function () {\n            return window.checkoutConfig.payment.ccform.cvvImageUrl[this.getCode()];\n        },\n\n        /**\n         * Get image for CVV\n         * @returns {String}\n         */\n        getCvvImageHtml: function () {\n            return '<img src=\"' + this.getCvvImageUrl() +\n                '\" alt=\"' + $t('Card Verification Number Visual Reference') +\n                '\" title=\"' + $t('Card Verification Number Visual Reference') +\n                '\" />';\n        },\n\n        /**\n         * Get unsanitized html for image for CVV\n         * @returns {String}\n         */\n        getCvvImageUnsanitizedHtml: function () {\n            return this.getCvvImageHtml();\n        },\n\n        /**\n         * @deprecated\n         * @returns {Object}\n         */\n        getSsStartYears: function () {\n            return window.checkoutConfig.payment.ccform.ssStartYears[this.getCode()];\n        },\n\n        /**\n         * Get list of available credit card types values\n         * @returns {Object}\n         */\n        getCcAvailableTypesValues: function () {\n            return _.map(this.getCcAvailableTypes(), function (value, key) {\n                return {\n                    'value': key,\n                    'type': value\n                };\n            });\n        },\n\n        /**\n         * Get list of available month values\n         * @returns {Object}\n         */\n        getCcMonthsValues: function () {\n            return _.map(this.getCcMonths(), function (value, key) {\n                return {\n                    'value': key,\n                    'month': value\n                };\n            });\n        },\n\n        /**\n         * Get list of available year values\n         * @returns {Object}\n         */\n        getCcYearsValues: function () {\n            return _.map(this.getCcYears(), function (value, key) {\n                return {\n                    'value': key,\n                    'year': value\n                };\n            });\n        },\n\n        /**\n         * @deprecated\n         * @returns {Object}\n         */\n        getSsStartYearsValues: function () {\n            return _.map(this.getSsStartYears(), function (value, key) {\n                return {\n                    'value': key,\n                    'year': value\n                };\n            });\n        },\n\n        /**\n         * Is legend available to display\n         * @returns {Boolean}\n         */\n        isShowLegend: function () {\n            return false;\n        },\n\n        /**\n         * Get available credit card type by code\n         * @param {String} code\n         * @returns {String}\n         */\n        getCcTypeTitleByCode: function (code) {\n            var title = '',\n                keyValue = 'value',\n                keyType = 'type';\n\n            _.each(this.getCcAvailableTypesValues(), function (value) {\n                if (value[keyValue] === code) {\n                    title = value[keyType];\n                }\n            });\n\n            return title;\n        },\n\n        /**\n         * Prepare credit card number to output\n         * @param {String} number\n         * @returns {String}\n         */\n        formatDisplayCcNumber: function (number) {\n            return 'xxxx-' + number.substr(-4);\n        },\n\n        /**\n         * Get credit card details\n         * @returns {Array}\n         */\n        getInfo: function () {\n            return [\n                {\n                    'name': 'Credit Card Type', value: this.getCcTypeTitleByCode(this.creditCardType())\n                },\n                {\n                    'name': 'Credit Card Number', value: this.formatDisplayCcNumber(this.creditCardNumber())\n                }\n            ];\n        }\n    });\n});\n","Magento_Payment/js/view/payment/iframe.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'jquery',\n    'Magento_Payment/js/view/payment/cc-form',\n    'Magento_Ui/js/model/messageList',\n    'mage/translate',\n    'Magento_Checkout/js/model/full-screen-loader',\n    'Magento_Checkout/js/action/set-payment-information',\n    'Magento_Checkout/js/model/payment/additional-validators',\n    'Magento_Ui/js/modal/alert'\n], function (\n    $,\n    Component,\n    messageList,\n    $t,\n    fullScreenLoader,\n    setPaymentInformationAction,\n    additionalValidators,\n    alert\n) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Payment/payment/iframe',\n            timeoutId: null,\n            timeoutMessage: 'Sorry, but something went wrong.'\n        },\n\n        /**\n         * @returns {String}\n         */\n        getSource: function () {\n            return window.checkoutConfig.payment.iframe.source[this.getCode()];\n        },\n\n        /**\n         * @returns {String}\n         */\n        getControllerName: function () {\n            return window.checkoutConfig.payment.iframe.controllerName[this.getCode()];\n        },\n\n        /**\n         * @returns {String}\n         */\n        getPlaceOrderUrl: function () {\n            return window.checkoutConfig.payment.iframe.placeOrderUrl[this.getCode()];\n        },\n\n        /**\n         * @returns {String}\n         */\n        getCgiUrl: function () {\n            return window.checkoutConfig.payment.iframe.cgiUrl[this.getCode()];\n        },\n\n        /**\n         * @returns {String}\n         */\n        getSaveOrderUrl: function () {\n            return window.checkoutConfig.payment.iframe.saveOrderUrl[this.getCode()];\n        },\n\n        /**\n         * @returns {String}\n         */\n        getDateDelim: function () {\n            return window.checkoutConfig.payment.iframe.dateDelim[this.getCode()];\n        },\n\n        /**\n         * @returns {String}\n         */\n        getCardFieldsMap: function () {\n            return window.checkoutConfig.payment.iframe.cardFieldsMap[this.getCode()];\n        },\n\n        /**\n         * @returns {String}\n         */\n        getExpireYearLength: function () {\n            return window.checkoutConfig.payment.iframe.expireYearLength[this.getCode()];\n        },\n\n        /**\n         * @param {Object} parent\n         * @returns {Function}\n         */\n        originalPlaceOrder: function (parent) {\n            return parent.placeOrder.bind(parent);\n        },\n\n        /**\n         * @returns {Number}\n         */\n        getTimeoutTime: function () {\n            return window.checkoutConfig.payment.iframe.timeoutTime[this.getCode()];\n        },\n\n        /**\n         * @returns {String}\n         */\n        getTimeoutMessage: function () {\n            return $t(this.timeoutMessage);\n        },\n\n        /**\n         * @override\n         */\n        placeOrder: function () {\n            var self = this;\n\n            if (this.validateHandler() &&\n                additionalValidators.validate() &&\n                this.isPlaceOrderActionAllowed() === true\n            ) {\n                fullScreenLoader.startLoader();\n\n                this.isPlaceOrderActionAllowed(false);\n\n                $.when(\n                    this.setPaymentInformation()\n                ).done(\n                    this.done.bind(this)\n                ).fail(\n                    this.fail.bind(this)\n                ).always(\n                    function () {\n                        self.isPlaceOrderActionAllowed(true);\n                    }\n                );\n\n                this.initTimeoutHandler();\n            }\n        },\n\n        /**\n         * {Function}\n         */\n        setPaymentInformation: function () {\n            return setPaymentInformationAction(\n                this.messageContainer,\n                {\n                    method: this.getCode()\n                }\n            );\n        },\n\n        /**\n         * {Function}\n         */\n        initTimeoutHandler: function () {\n            this.timeoutId = setTimeout(\n                this.timeoutHandler.bind(this),\n                this.getTimeoutTime()\n            );\n\n            $(window).off('clearTimeout')\n                .on('clearTimeout', this.clearTimeout.bind(this));\n        },\n\n        /**\n         * {Function}\n         */\n        clearTimeout: function () {\n            clearTimeout(this.timeoutId);\n            this.fail();\n\n            return this;\n        },\n\n        /**\n         * {Function}\n         */\n        timeoutHandler: function () {\n            this.clearTimeout();\n\n            alert(\n                {\n                    content: this.getTimeoutMessage(),\n                    actions: {\n\n                        /**\n                         * {Function}\n                         */\n                        always: this.alertActionHandler.bind(this)\n                    }\n                }\n            );\n\n            this.fail();\n        },\n\n        /**\n         * {Function}\n         */\n        alertActionHandler: function () {\n            fullScreenLoader.startLoader();\n            window.location.reload();\n        },\n\n        /**\n         * {Function}\n         */\n        fail: function () {\n            fullScreenLoader.stopLoader();\n\n            return this;\n        },\n\n        /**\n         * {Function}\n         */\n        done: function () {\n            this.placeOrderHandler().fail(function () {\n                fullScreenLoader.stopLoader();\n            });\n\n            return this;\n        }\n    });\n});\n","Magento_Payment/js/view/payment/payments.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/payment/renderer-list'\n], function (Component, rendererList) {\n    'use strict';\n\n    rendererList.push(\n        {\n            type: 'free',\n            component: 'Magento_Payment/js/view/payment/method-renderer/free-method'\n        }\n    );\n\n    /** Add view logic here if needed */\n    return Component.extend({});\n});\n","Magento_Payment/js/view/payment/method-renderer/free-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* @api */\ndefine([\n    'Magento_Checkout/js/view/payment/default',\n    'Magento_Checkout/js/model/quote'\n], function (Component, quote) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Payment/payment/free'\n        },\n\n        /** Returns is method available */\n        isAvailable: function () {\n            return quote.totals()['grand_total'] <= 0;\n        }\n    });\n});\n","Magento_Paypal/js/order-review.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/alert',\n    'jquery-ui-modules/widget',\n    'mage/translate',\n    'mage/mage',\n    'mage/validation'\n], function ($, alert) {\n    'use strict';\n\n    $.widget('mage.orderReview', {\n        options: {\n            orderReviewSubmitSelector: '#review-button',\n            shippingSelector: '#shipping_method',\n            shippingSubmitFormSelector: null,\n            updateOrderSelector: '#update-order',\n            billingAsShippingSelector: '#billing\\\\:as_shipping',\n            updateContainerSelector: '#details-reload',\n            waitLoadingContainer: '#review-please-wait',\n            shippingMethodContainer: '#shipping-method-container',\n            agreementSelector: 'div.checkout-agreements input',\n            isAjax: false,\n            shippingMethodUpdateUrl: null,\n            updateOrderSubmitUrl: null,\n            canEditShippingMethod: false\n        },\n\n        /**\n         * Widget instance properties\n         */\n        triggerPropertyChange: true,\n        isShippingSubmitForm: false,\n\n        /** @inheritdoc */\n        _create: function () {\n            var isDisable;\n\n            //change handler for ajaxEnabled\n            if (this.options.isAjax) {\n                this._submitOrder = this._ajaxSubmitOrder;\n            }\n\n            this.element.on('click', this.options.orderReviewSubmitSelector, $.proxy(this._submitOrder, this))\n                .on('click', this.options.billingAsShippingSelector, $.proxy(this._shippingTobilling, this))\n                .on('change',\n                    this.options.shippingSelector,\n                    $.proxy(this._submitUpdateOrder,\n                        this,\n                        this.options.updateOrderSubmitUrl,\n                        this.options.updateContainerSelector\n                    )\n                ).find(this.options.updateOrderSelector).on('click', $.proxy(this._updateOrderHandler, this)).end();\n            this._shippingTobilling();\n\n            if ($(this.options.shippingSubmitFormSelector).length && this.options.canEditShippingMethod) {\n                this.isShippingSubmitForm = true;\n                $(this.options.shippingSubmitFormSelector)\n                    .on('change',\n                        this.options.shippingSelector,\n                        $.proxy(\n                            this._submitUpdateOrder,\n                            this,\n                            $(this.options.shippingSubmitFormSelector).prop('action'),\n                            this.options.updateContainerSelector\n                        )\n                    );\n                this._updateOrderSubmit(!$(this.options.shippingSubmitFormSelector)\n                    .find(this.options.shippingSelector).val());\n            } else {\n                isDisable = this.isShippingSubmitForm && this.element.find(this.options.shippingSelector).val();\n                this.element\n                    .on('input propertychange', ':input[name]',\n                        $.proxy(this._updateOrderSubmit, this, isDisable, this._onShippingChange)\n                    ).find('select').not(this.options.shippingSelector).on('change', this._propertyChange);\n                this._updateOrderSubmit(isDisable);\n            }\n        },\n\n        /**\n         * show ajax loader\n         */\n        _ajaxBeforeSend: function () {\n            this.element.find(this.options.waitLoadingContainer).show();\n        },\n\n        /**\n         * hide ajax loader\n         */\n        _ajaxComplete: function () {\n            this.element.find(this.options.waitLoadingContainer).hide();\n        },\n\n        /**\n         * trigger propertychange for input type select\n         */\n        _propertyChange: function () {\n            $(this).trigger('propertychange');\n        },\n\n        /**\n         * trigger change for the update of shipping methods from server\n         */\n        _updateOrderHandler: function () {\n            $(this.options.shippingSelector).trigger('change');\n        },\n\n        /**\n         * Attempt to submit order\n         */\n        _submitOrder: function () {\n            if (this._validateForm()) {\n                this.element.find(this.options.updateOrderSelector).fadeTo(0, 0.5)\n                    .end().find(this.options.waitLoadingContainer).show()\n                    .end().trigger('submit');\n                this._updateOrderSubmit(true);\n            }\n        },\n\n        /**\n         * Attempt to ajax submit order\n         */\n        _ajaxSubmitOrder: function () {\n            if (this.element.find(this.options.waitLoadingContainer).is(':visible')) {\n                return false;\n            }\n            $.ajax({\n                url: this.element.prop('action'),\n                type: 'post',\n                context: this,\n                data: {\n                    isAjax: 1\n                },\n                dataType: 'json',\n                beforeSend: this._ajaxBeforeSend,\n                complete: this._ajaxComplete,\n\n                /** @inheritdoc */\n                success: function (response) {\n                    var msg;\n\n                    if (typeof response === 'object' && !$.isEmptyObject(response)) {\n                        if (response['error_messages']) {\n                            this._ajaxComplete();\n                            msg = response['error_messages'];\n\n                            /* eslint-disable max-depth */\n                            if (msg) {\n                                if (Array.isArray(msg)) {\n                                    msg = msg.join('\\n');\n                                }\n                            }\n\n                            /* eslint-enablemax-depth */\n                            alert({\n                                content: msg\n                            });\n\n                            return false;\n                        }\n\n                        if (response.redirect) {\n                            $.mage.redirect(response.redirect);\n\n                            return false;\n                        } else if (response.success) {\n                            $.mage.redirect(this.options.successUrl);\n\n                            return false;\n                        }\n                        this._ajaxComplete();\n                        alert({\n                            content: $.mage.__('Sorry, something went wrong.')\n                        });\n                    }\n                },\n\n                /** @inheritdoc */\n                error: function () {\n                    alert({\n                        content: $.mage.__('Sorry, something went wrong. Please try again later.')\n                    });\n                    this._ajaxComplete();\n                }\n            });\n        },\n\n        /**\n         * Validate Order form\n         */\n        _validateForm: function () {\n            this.element.find(this.options.agreementSelector).off('change').on('change', $.proxy(function () {\n                var isValid = this._validateForm();\n\n                this._updateOrderSubmit(!isValid);\n            }, this));\n\n            if (this.element.data('mageValidation')) {\n                return this.element.validation().valid();\n            }\n\n            return true;\n        },\n\n        /**\n         * Check/Set whether order can be submitted\n         * Also disables form submission element, if any\n         * @param {*} shouldDisable - whether should prevent order submission explicitly\n         * @param {Function} [fn] - function for shipping change handler\n         * @param {*} [*] - if true the property change will be set to true\n         */\n        _updateOrderSubmit: function (shouldDisable, fn) {\n            this._toggleButton(this.options.orderReviewSubmitSelector, shouldDisable);\n\n            if (typeof fn === 'function') {\n                fn.call(this);\n            }\n        },\n\n        /**\n         * Enable/Disable button\n         * @param {jQuery} button - button selector to be toggled\n         * @param {*} disable - boolean for toggling\n         */\n        _toggleButton: function (button, disable) {\n            $(button).prop({\n                'disabled': disable\n            }).toggleClass('no-checkout', disable).fadeTo(0, disable ? 0.5 : 1);\n        },\n\n        /**\n         * Copy element value from shipping to billing address\n         * @param {jQuery.Event} e - optional\n         */\n        _shippingTobilling: function (e) {\n            var isChecked, opacity;\n\n            if (this.options.shippingSubmitFormSelector) {\n                return false;\n            }\n            isChecked = $(this.options.billingAsShippingSelector).is(':checked');\n            opacity = isChecked ? 0.5 : 1;\n\n            if (isChecked) {\n                this.element.validation('clearError', ':input[name^=\"billing\"]');\n            }\n            $(':input[name^=\"shipping\"]', this.element).each($.proxy(function (key, value) {\n                var fieldObj = $(value.id.replace('shipping:', '#billing\\\\:'));\n\n                if (isChecked) {\n                    fieldObj = fieldObj.val($(value).val());\n                }\n                fieldObj.prop({\n                    'readonly': isChecked,\n                    'disabled': isChecked\n                }).fadeTo(0, opacity);\n\n                if (fieldObj.is('select')) {\n                    this.triggerPropertyChange = false;\n                    fieldObj.trigger('change');\n                }\n            }, this));\n\n            if (isChecked || e) {\n                this._updateOrderSubmit(true);\n            }\n            this.triggerPropertyChange = true;\n        },\n\n        /**\n         * Dispatch an ajax request of Update Order submission\n         * @param {*} url - url where to submit shipping method\n         * @param {*} resultId - id of element to be updated\n         */\n        _submitUpdateOrder: function (url, resultId) {\n            var isChecked, formData, callBackResponseHandler, shippingMethod;\n\n            if (this.element.find(this.options.waitLoadingContainer).is(':visible')) {\n                return false;\n            }\n            isChecked = $(this.options.billingAsShippingSelector).is(':checked');\n            formData = null;\n            callBackResponseHandler = null;\n            let val = $(this.options.shippingSelector).val();\n\n            shippingMethod = val.trim();\n            this._shippingTobilling();\n\n            if (url && resultId && shippingMethod) {\n                this._updateOrderSubmit(true);\n                this._toggleButton(this.options.updateOrderSelector, true);\n\n                // form data and callBack updated based on the shipping Form element\n                if (this.isShippingSubmitForm) {\n                    formData = $(this.options.shippingSubmitFormSelector).serialize() + '&isAjax=true';\n\n                    /**\n                     * @param {Object} response\n                     */\n                    callBackResponseHandler = function (response) {\n                        $(resultId).html(response);\n                        this._updateOrderSubmit(false);\n                        this._ajaxComplete();\n                    };\n                } else {\n                    formData = this.element.serialize() + '&isAjax=true';\n\n                    /**\n                     * @param {Object} response\n                     */\n                    callBackResponseHandler = function (response) {\n                        $(resultId).html(response);\n                        this._ajaxShippingUpdate(shippingMethod);\n                    };\n                }\n\n                if (isChecked) {\n                    $(this.options.shippingSelect).prop('disabled', true);\n                }\n                $.ajax({\n                    url: url,\n                    type: 'post',\n                    context: this,\n                    beforeSend: this._ajaxBeforeSend,\n                    data: formData,\n                    success: callBackResponseHandler\n                });\n            }\n        },\n\n        /**\n         * Update Shipping Methods Element from server\n         * @param {*} shippingMethod\n         */\n        _ajaxShippingUpdate: function (shippingMethod) {\n            $.ajax({\n                url: this.options.shippingMethodUpdateUrl,\n                data: {\n                    isAjax: true,\n                    'shipping_method': shippingMethod\n                },\n                type: 'post',\n                context: this,\n\n                /** @inheritdoc */\n                success: function (response) {\n                    $(this.options.shippingMethodContainer).parent().html(response);\n                    this._toggleButton(this.options.updateOrderSelector, false);\n                    this._updateOrderSubmit(false);\n                },\n                complete: this._ajaxComplete\n            });\n        },\n\n        /**\n         * Actions on change Shipping Address data\n         */\n        _onShippingChange: function () {\n            let val = $(this.options.shippingSelector).val();\n\n            if (this.triggerPropertyChange && val.trim()) {\n                this.element.find(this.options.shippingSelector).hide().end()\n                    .find(this.options.shippingSelector + '_update').show();\n            }\n        }\n    });\n\n    return $.mage.orderReview;\n});\n","Magento_Paypal/js/paypal-checkout.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/confirm',\n    'Magento_Customer/js/customer-data',\n    'jquery-ui-modules/widget',\n    'mage/mage'\n], function ($, confirm, customerData) {\n    'use strict';\n\n    $.widget('mage.paypalCheckout', {\n        options: {\n            originalForm:\n                'form:not(#product_addtocart_form_from_popup):has(input[name=\"product\"][value=%1])',\n            productId: 'input[type=\"hidden\"][name=\"product\"]',\n            ppCheckoutSelector: '[data-role=pp-checkout-url]',\n            ppCheckoutInput: '<input type=\"hidden\" data-role=\"pp-checkout-url\" name=\"return_url\" value=\"\"/>'\n        },\n\n        /**\n         * Initialize store credit events\n         * @private\n         */\n        _create: function () {\n            this.element.on('click', '[data-action=\"checkout-form-submit\"]', $.proxy(function (e) {\n                var $target = $(e.target),\n                    returnUrl = $target.data('checkout-url'),\n                    productId = $target.closest('form').find(this.options.productId).val(),\n                    originalForm = this.options.originalForm.replace('%1', productId),\n                    self = this,\n                    billingAgreement = customerData.get('paypal-billing-agreement');\n\n                e.preventDefault();\n\n                if (billingAgreement().askToCreate) {\n                    confirm({\n                        content: billingAgreement().confirmMessage,\n                        actions: {\n\n                            /**\n                             * Confirmation handler\n                             *\n                             */\n                            confirm: function () {\n                                returnUrl = billingAgreement().confirmUrl;\n                                self._redirect(returnUrl, originalForm);\n                            },\n\n                            /**\n                             * Cancel confirmation handler\n                             *\n                             */\n                            cancel: function (event) {\n                                if (event && !$(event.target).hasClass('action-close')) {\n                                    self._redirect(returnUrl);\n                                }\n                            }\n                        }\n                    });\n                } else {\n                    this._redirect(returnUrl, originalForm);\n                }\n            }, this));\n        },\n\n        /**\n         * Redirect to certain url, with optional form\n         * @param {String} returnUrl\n         * @param {HTMLElement} originalForm\n         *\n         */\n        _redirect: function (returnUrl, originalForm) {\n            var $form,\n                ppCheckoutInput;\n\n            if (this.options.isCatalogProduct) {\n                // find the form from which the button was clicked\n                $form = originalForm ? $(originalForm) : $($(this.options.shortcutContainerClass).closest('form'));\n\n                ppCheckoutInput = $form.find(this.options.ppCheckoutSelector)[0];\n\n                if (!ppCheckoutInput) {\n                    ppCheckoutInput = $(this.options.ppCheckoutInput);\n                    ppCheckoutInput.appendTo($form);\n                }\n                $(ppCheckoutInput).val(returnUrl);\n\n                $form.trigger('submit');\n            } else {\n                $.mage.redirect(returnUrl);\n            }\n        }\n    });\n\n    return $.mage.paypalCheckout;\n});\n","Magento_Paypal/js/action/set-payment-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/action/set-payment-information'\n], function (quote, setPaymentInformation) {\n    'use strict';\n\n    return function (messageContainer) {\n        return setPaymentInformation(messageContainer, quote.paymentMethod());\n    };\n});\n","Magento_Paypal/js/in-context/billing-agreement.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'Magento_Ui/js/modal/confirm',\n    'Magento_Customer/js/customer-data'\n], function ($, confirm, customerData) {\n    'use strict';\n\n    $.widget('mage.billingAgreement', {\n        options: {\n            invalidateOnLoad: false,\n            cancelButtonSelector: '.block-billing-agreements-view button.cancel',\n            cancelMessage: '',\n            cancelUrl: ''\n        },\n\n        /**\n         * Initialize billing agreements events\n         * @private\n         */\n        _create: function () {\n            var self = this;\n\n            if (this.options.invalidateOnLoad) {\n                this.invalidate();\n            }\n            $(this.options.cancelButtonSelector).on('click', function () {\n                confirm({\n                    content: self.options.cancelMessage,\n                    actions: {\n                        /**\n                         * 'Confirm' action handler.\n                         */\n                        confirm: function () {\n                            self.invalidate();\n                            window.location.href = self.options.cancelUrl;\n                        }\n                    }\n                });\n\n                return false;\n            });\n        },\n\n        /**\n         * clear paypal billing agreement customer data\n         * @returns void\n         */\n        invalidate: function () {\n            customerData.invalidate(['paypal-billing-agreement']);\n        }\n    });\n\n    return $.mage.billingAgreement;\n});\n","Magento_Paypal/js/in-context/button.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'uiComponent',\n    'jquery',\n    'Magento_Paypal/js/in-context/express-checkout-wrapper',\n    'Magento_Customer/js/customer-data'\n], function (Component, $, Wrapper, customerData) {\n    'use strict';\n\n    return Component.extend(Wrapper).extend({\n        defaults: {\n            declinePayment: false\n        },\n\n        /** @inheritdoc */\n        initialize: function (config, element) {\n            var cart = customerData.get('cart'),\n                customer = customerData.get('customer');\n\n            this._super();\n            this.renderPayPalButtons(element);\n\n            if (cart().isGuestCheckoutAllowed === undefined) {\n                cart.subscribe(function (updatedCart) {\n                    this.declinePayment = !customer().firstname && !cart().isGuestCheckoutAllowed;\n\n                    return updatedCart;\n                }.bind(this));\n            }\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        beforePayment: function (resolve, reject) {\n            var promise = $.Deferred();\n\n            if (this.declinePayment) {\n                this.addError(this.signInMessage, 'warning');\n\n                reject();\n            } else {\n                promise.resolve();\n            }\n\n            return promise;\n        },\n\n        /** @inheritdoc */\n        prepareClientConfig: function () {\n            this._super();\n\n            return this.clientConfig;\n        }\n    });\n});\n","Magento_Paypal/js/in-context/express-checkout-smart-buttons.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/* eslint-disable max-nested-callbacks */\ndefine([\n    'underscore',\n    'jquery',\n    'Magento_Paypal/js/in-context/paypal-sdk',\n    'Magento_Customer/js/customer-data',\n    'domReady!'\n], function (_, $, paypalSdk, customerData) {\n    'use strict';\n\n    /**\n     * Triggers beforePayment action on PayPal buttons\n     *\n     * @param {Object} clientConfig\n     * @returns {Object} jQuery promise\n     */\n    function performCreateOrder(clientConfig) {\n        var params = {\n            'quote_id': clientConfig.quoteId,\n            'customer_id': clientConfig.customerId || '',\n            'form_key': clientConfig.formKey,\n            button: clientConfig.button\n        };\n\n        return $.Deferred(function (deferred) {\n            clientConfig.rendererComponent.beforePayment(deferred.resolve, deferred.reject).then(function () {\n                $.post(clientConfig.getTokenUrl, params).done(function (res) {\n                    clientConfig.rendererComponent.afterPayment(res, deferred.resolve, deferred.reject);\n                }).fail(function (jqXHR, textStatus, err) {\n                    clientConfig.rendererComponent.catchPayment(err, deferred.resolve, deferred.reject);\n                });\n            });\n        }).promise();\n    }\n\n    /**\n     * Triggers beforeOnAuthorize action on PayPal buttons\n     * @param {Object} clientConfig\n     * @param {Object} data\n     * @param {Object} actions\n     * @returns {Object} jQuery promise\n     */\n    function performOnApprove(clientConfig, data, actions) {\n        var params = {\n            paymentToken: data.orderID,\n            payerId: data.payerID,\n            paypalFundingSource: customerData.get('paypal-funding-source'),\n            'form_key': clientConfig.formKey\n        };\n\n        return $.Deferred(function (deferred) {\n            clientConfig.rendererComponent.beforeOnAuthorize(deferred.resolve, deferred.reject, actions)\n                .then(function () {\n                    $.post(clientConfig.onAuthorizeUrl, params).done(function (res) {\n                        clientConfig.rendererComponent\n                            .afterOnAuthorize(res, deferred.resolve, deferred.reject, actions);\n                        customerData.set('paypal-funding-source', '');\n                    }).fail(function (jqXHR, textStatus, err) {\n                        clientConfig.rendererComponent.catchOnAuthorize(err, deferred.resolve, deferred.reject);\n                        customerData.set('paypal-funding-source', '');\n                    });\n                });\n        }).promise();\n    }\n\n    return function (clientConfig, element) {\n        paypalSdk(clientConfig.sdkUrl, clientConfig.dataAttributes).done(function (paypal) {\n            paypal.Buttons({\n                style: clientConfig.styles,\n\n                /**\n                 * onInit is called when the button first renders\n                 * @param {Object} data\n                 * @param {Object} actions\n                 */\n                onInit: function (data, actions) {\n                    clientConfig.rendererComponent.validate(actions);\n                },\n\n                /**\n                 * Triggers beforePayment action on PayPal buttons\n                 * @returns {Object} jQuery promise\n                 */\n                createOrder: function () {\n                    return performCreateOrder(clientConfig);\n                },\n\n                /**\n                 * Triggers beforeOnAuthorize action on PayPal buttons\n                 * @param {Object} data\n                 * @param {Object} actions\n                 */\n                onApprove: function (data, actions) {\n                    performOnApprove(clientConfig, data, actions);\n                },\n\n                /**\n                 * Execute logic on Paypal button click\n                 */\n                onClick: function (data) {\n                    customerData.set('paypal-funding-source', data.fundingSource);\n                    clientConfig.rendererComponent.validate();\n                    clientConfig.rendererComponent.onClick();\n                },\n\n                /**\n                 * Process cancel action\n                 * @param {Object} data\n                 * @param {Object} actions\n                 */\n                onCancel: function (data, actions) {\n                    clientConfig.rendererComponent.onCancel(data, actions);\n                },\n\n                /**\n                 * Process errors\n                 *\n                 * @param {Error} err\n                 */\n                onError: function (err) {\n                    clientConfig.rendererComponent.onError(err);\n                }\n            }).render(element);\n        });\n    };\n});\n","Magento_Paypal/js/in-context/express-checkout-wrapper.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'mage/translate',\n    'Magento_Customer/js/customer-data',\n    'Magento_Paypal/js/in-context/express-checkout-smart-buttons',\n    'Magento_Ui/js/modal/alert',\n    'mage/cookies'\n], function ($, $t, customerData, checkoutSmartButtons, alert) {\n    'use strict';\n\n    return {\n        defaults: {\n            paymentActionError: $t('Something went wrong with your request. Please try again later.'),\n            signInMessage: $t('To check out, please sign in with your email address.')\n        },\n\n        /**\n         * Render PayPal buttons using checkout.js\n         */\n        renderPayPalButtons: function (element) {\n            checkoutSmartButtons(this.prepareClientConfig(), element);\n        },\n\n        /**\n         * Validate payment method\n         *\n         * @param {Object} actions\n         */\n        validate: function (actions) {\n            this.actions = actions || this.actions;\n        },\n\n        /**\n         * Execute logic on Paypal button click\n         */\n        onClick: function () {},\n\n        /**\n         * Before payment execute\n         *\n         * @param {Function} resolve\n         * @param {Function} reject\n         * @return {*}\n         */\n        beforePayment: function (resolve, reject) { //eslint-disable-line no-unused-vars\n            return $.Deferred().resolve();\n        },\n\n        /**\n         * After payment execute\n         *\n         * @param {Object} res\n         * @param {Function} resolve\n         * @param {Function} reject\n         *\n         * @return {*}\n         */\n        afterPayment: function (res, resolve, reject) {\n\n            if (res.success) {\n                return resolve(res.token);\n            }\n\n            return reject(new Error(res['error_message']));\n        },\n\n        /**\n         * Catch payment\n         *\n         * @param {Error} err\n         * @param {Function} resolve\n         * @param {Function} reject\n         */\n        catchPayment: function (err, resolve, reject) {\n            this.addAlert(this.paymentActionError);\n            reject(err);\n        },\n\n        /**\n         * Before onAuthorize execute\n         *\n         * @param {Function} resolve\n         * @param {Function} reject\n         * @param {Object} actions\n         *\n         * @return {jQuery.Deferred}\n         */\n        beforeOnAuthorize: function (resolve, reject, actions) { //eslint-disable-line no-unused-vars\n            //display loading widget.\n            $('body').trigger('processStart');\n\n            return $.Deferred().resolve();\n        },\n\n        /**\n         * After onAuthorize execute\n         *\n         * @param {Object} res\n         * @param {Function} resolve\n         * @param {Function} reject\n         * @param {Object} actions\n         *\n         * @return {*}\n         */\n        afterOnAuthorize: function (res, resolve, reject, actions) {\n            $('body').trigger('processStop');\n\n            if (res.success) {\n                resolve();\n\n                return actions.redirect(res.redirectUrl);\n            }\n\n            return reject(new Error(res['error_message']));\n        },\n\n        /**\n         * Catch payment\n         *\n         * @param {Error} err\n         * @param {Function} resolve\n         * @param {Function} reject\n         */\n        catchOnAuthorize: function (err, resolve, reject) {\n            $('body').trigger('processStop');\n            this.addAlert(this.paymentActionError);\n            reject(err);\n        },\n\n        /**\n         * Process cancel action\n         *\n         * @param {Object} data\n         * @param {Object} actions\n         */\n        onCancel: function (data, actions) {\n            $('body').trigger('processStop');\n            actions.redirect(this.clientConfig.onCancelUrl);\n        },\n\n        /**\n         * Process errors\n         *\n         * @param {Error} err\n         */\n        onError: function (err) { //eslint-disable-line no-unused-vars\n            // Uncaught error isn't displayed in the console\n        },\n\n        /**\n         * Adds error message\n         *\n         * @param {String} message\n         * @param {String} [type]\n         */\n        addError: function (message, type) {\n            type = type || 'error';\n            customerData.set('messages', {\n                messages: [{\n                    type: type,\n                    text: message\n                }],\n                'data_id': Math.floor(Date.now() / 1000)\n            });\n        },\n\n        /**\n         * Add alert message\n         *\n         * @param {String} message\n         */\n        addAlert: function (message) {\n            alert({\n                content: message\n            });\n        },\n\n        /**\n         * @returns {String}\n         */\n        getButtonId: function () {\n            return this.inContextId;\n        },\n\n        /**\n         * Populate client config with all required data\n         *\n         * @return {Object}\n         */\n        prepareClientConfig: function () {\n            this.clientConfig.rendererComponent = this;\n            this.clientConfig.formKey = $.mage.cookies.get('form_key');\n\n            return this.clientConfig;\n        }\n    };\n});\n","Magento_Paypal/js/in-context/paypal-sdk.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    var dfd = $.Deferred();\n\n    /**\n     * Loads the PayPal SDK object\n     * @param {String} paypalUrl - the url of the PayPal SDK\n     * @param {Array} dataAttributes - Array of the Attributes for PayPal SDK Script tag\n     */\n    return function loadPaypalScript(paypalUrl, dataAttributes) {\n        //configuration for loaded PayPal script\n        require.config({\n            paths: {\n                paypalSdk: paypalUrl\n            },\n            shim: {\n                paypalSdk: {\n                    exports: 'paypal'\n                }\n            },\n            attributes: {\n                'paypalSdk': dataAttributes\n            },\n\n            /**\n             * Add attributes under Paypal SDK Script tag\n             */\n            onNodeCreated: function (node, config, name) {\n                if (config.attributes && config.attributes[name]) {\n                    $.each(dataAttributes, function (index, elem) {\n                        node.setAttribute(index, elem);\n                    });\n                }\n            }\n        });\n\n        if (dfd.state() !== 'resolved') {\n            require(['paypalSdk'], function (paypalObject) {\n                dfd.resolve(paypalObject);\n            });\n        }\n\n        return dfd.promise();\n    };\n});\n","Magento_Paypal/js/in-context/product-express-checkout.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'underscore',\n    'jquery',\n    'uiComponent',\n    'Magento_Paypal/js/in-context/express-checkout-wrapper',\n    'Magento_Customer/js/customer-data'\n], function (_, $, Component, Wrapper, customerData) {\n    'use strict';\n\n    return Component.extend(Wrapper).extend({\n        defaults: {\n            productFormSelector: '#product_addtocart_form',\n            declinePayment: false,\n            formInvalid: false,\n            productAddedToCart: false\n        },\n\n        /** @inheritdoc */\n        initialize: function (config, element) {\n            var cart = customerData.get('cart'),\n                customer = customerData.get('customer'),\n                isGuestCheckoutAllowed;\n\n            this._super();\n\n            isGuestCheckoutAllowed = cart().isGuestCheckoutAllowed;\n\n            if (typeof isGuestCheckoutAllowed === 'undefined') {\n                isGuestCheckoutAllowed = config.clientConfig.isGuestCheckoutAllowed;\n            }\n\n            if (config.clientConfig.isVisibleOnProductPage) {\n                this.renderPayPalButtons(element);\n            }\n\n            this.declinePayment = !customer().firstname && !isGuestCheckoutAllowed;\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        onClick: function () {\n            var $form = $(this.productFormSelector);\n\n            if (!this.declinePayment && !this.productAddedToCart) {\n                $form.trigger('submit');\n                this.formInvalid = !$form.validation('isValid');\n                this.productAddedToCart = true;\n            }\n        },\n\n        /** @inheritdoc */\n        beforePayment: function (resolve, reject) {\n            var promise = $.Deferred();\n\n            if (this.declinePayment) {\n                this.addError(this.signInMessage, 'warning');\n                reject();\n            } else if (this.formInvalid) {\n                reject();\n            } else {\n                $(document).on('ajax:addToCart', function (e, data) {\n                    if (_.isEmpty(data.response)) {\n                        return promise.resolve();\n                    }\n\n                    return reject();\n                });\n                $(document).on('ajax:addToCart:error', reject);\n            }\n\n            return promise;\n        },\n\n        /**\n         * After payment execute\n         *\n         * @param {Object} res\n         * @param {Function} resolve\n         * @param {Function} reject\n         *\n         * @return {*}\n         */\n        afterPayment: function (res, resolve, reject) {\n            if (res.success) {\n                return resolve(res.token);\n            }\n\n            this.addAlert(res['error_message']);\n\n            return reject(new Error(res['error_message']));\n        },\n\n        /** @inheritdoc */\n        prepareClientConfig: function () {\n            this._super();\n            this.clientConfig.quoteId = '';\n            this.clientConfig.customerId = '';\n\n            return this.clientConfig;\n        },\n\n        /** @inheritdoc */\n        onError: function (err) {\n            this.productAddedToCart = false;\n            this._super(err);\n        },\n\n        /** @inheritdoc */\n        onCancel: function (data, actions) {\n            this.productAddedToCart = false;\n            this._super(data, actions);\n        },\n\n        /** @inheritdoc */\n        afterOnAuthorize: function (res, resolve, reject, actions) {\n            this.productAddedToCart = false;\n\n            return this._super(res, resolve, reject, actions);\n        }\n    });\n});\n","Magento_Paypal/js/model/iframe-redirect.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'ko',\n    'Magento_Paypal/js/model/iframe',\n    'Magento_Ui/js/model/messageList'\n],\nfunction (ko, iframe, messageList) {\n    'use strict';\n\n    return function (cartUrl, errorMessage, goToSuccessPage, successUrl) {\n        if (this === window.self) {\n            window.location = cartUrl;\n        }\n\n        if (!!errorMessage.message) { //eslint-disable-line no-extra-boolean-cast\n            document.removeEventListener('click', iframe.stopEventPropagation, true);\n            iframe.isInAction(false);\n            messageList.addErrorMessage(errorMessage);\n        } else if (!!goToSuccessPage) { //eslint-disable-line no-extra-boolean-cast\n            window.location = successUrl;\n        } else {\n            window.location = cartUrl;\n        }\n    };\n});\n","Magento_Paypal/js/model/iframe.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['ko'], function (ko) {\n    'use strict';\n\n    var isInAction = ko.observable(false);\n\n    return {\n        isInAction: isInAction,\n\n        /**\n         * @param {jQuery.Event} event\n         */\n        stopEventPropagation: function (event) {\n            event.stopImmediatePropagation();\n            event.preventDefault();\n        }\n    };\n});\n","Magento_Paypal/js/view/paylater.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'ko',\n    'uiElement',\n    'uiLayout',\n    'Magento_Paypal/js/in-context/paypal-sdk',\n    'domReady!'\n], function (\n    $,\n    ko,\n    Component,\n    layout,\n    paypalSdk\n) {\n    'use strict';\n\n    return Component.extend({\n\n        defaults: {\n            template: 'Magento_Paypal/paylater',\n            sdkUrl: '',\n            attributes: {\n                class: 'pay-later-message'\n            },\n            dataAttributes: {},\n            refreshSelector: '',\n            displayAmount: false,\n            amountComponentConfig: {\n                name: '${ $.name }.amountProvider',\n                component: ''\n            }\n        },\n        paypal: null,\n        amount: null,\n\n        /**\n         * Initialize\n         *\n         * @returns {*}\n         */\n        initialize: function () {\n            this._super()\n                .observe(['amount']);\n\n            if (this.displayAmount) {\n                layout([this.amountComponentConfig]);\n            }\n\n            if (this.sdkUrl !== '') {\n                this.loadPayPalSdk(this.sdkUrl, this.dataAttributes)\n                    .then(this._setPayPalObject.bind(this));\n            }\n\n            if (this.refreshSelector) {\n                $(this.refreshSelector).on('click', this._refreshMessages.bind(this));\n            }\n\n            return this;\n        },\n\n        /**\n         * Get attribute value from configuration\n         *\n         * @param {String} attributeName\n         * @returns {*|null}\n         */\n        getAttribute: function (attributeName) {\n            return typeof this.attributes[attributeName] !== 'undefined' ?\n                this.attributes[attributeName] : null;\n        },\n\n        /**\n         * Load PP SDK with preconfigured options\n         *\n         * @param {String} sdkUrl - the url of the PayPal SDK\n         * @param {Array} dataAttributes - Array of the Attributes for PayPal SDK Script tag\n         */\n        loadPayPalSdk: function (sdkUrl, dataAttributes) {\n            return paypalSdk(sdkUrl, dataAttributes);\n        },\n\n        /**\n         * Set reference to paypal Sdk object\n         *\n         * @param {Object} paypal\n         * @private\n         */\n        _setPayPalObject: function (paypal) {\n            this.paypal = paypal;\n        },\n\n        /**\n         * Render messages\n         *\n         * @private\n         */\n        _refreshMessages: function () {\n            if (this.paypal) {\n                this.paypal.Messages.render();\n            }\n        }\n    });\n});\n","Magento_Paypal/js/view/amountProviders/checkout.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'ko',\n    'uiElement',\n    'uiRegistry',\n    'Magento_Checkout/js/model/quote',\n    'domReady!'\n], function (\n    $,\n    ko,\n    Component,\n    registry,\n    quote\n) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            amount: null\n        },\n\n        /**\n         * Initialize\n         *\n         * @returns {*}\n         */\n        initialize: function () {\n            this._super();\n\n            this.updateAmount();\n\n            return this;\n        },\n\n        /**\n         * Update amount\n         */\n        updateAmount: function () {\n            var payLater = registry.get(this.parentName);\n\n            quote.totals.subscribe(function (newValue) {\n                payLater.amount(newValue['base_grand_total']);\n            });\n        }\n    });\n});\n","Magento_Paypal/js/view/amountProviders/product-grouped.js":"/**\n* Copyright \u00a9 Magento, Inc. All rights reserved.\n* See COPYING.txt for license details.\n*/\n\ndefine([\n    'jquery',\n    'uiElement',\n    'uiRegistry',\n    'domReady!'\n], function (\n    $,\n    Component,\n    registry\n) {\n    'use strict';\n\n    return Component.extend({\n\n        defaults: {\n            tableWrapperSelector: '.table-wrapper.grouped',\n            priceBoxSelector: '[data-role=\"priceBox\"]',\n            qtyFieldSelector: '.input-text.qty',\n            amount: null\n        },\n        priceInfo: {},\n\n        /**\n         * Initialize\n         *\n         * @returns {*}\n         */\n        initialize: function () {\n            var self = this;\n\n            this._super();\n\n            $('tbody tr', this.tableWrapperSelector).each(function (index, element) {\n                var priceBox = $(self.priceBoxSelector, element),\n                    qtyElement = $(self.qtyFieldSelector, element),\n                    productId = priceBox.data('productId'),\n                    priceElement = $('#product-price-' + productId);\n\n                self.priceInfo[productId] = {\n                    qty: self._getQty(qtyElement),\n                    price: priceElement.data('priceAmount')\n                };\n            });\n\n            $(this.qtyFieldSelector).on('change', this._onQtyChange.bind(this));\n\n            this._updateAmount();\n\n            return this;\n        },\n\n        /**\n         * Get product quantity\n         *\n         * @param {jQuery.Element} element\n         * @private\n         */\n        _getQty: function (element) {\n            var qty = parseFloat(element.val());\n\n            return !isNaN(qty) && qty ? qty : 0;\n        },\n\n        /**\n         * Handle changed product quantity\n         *\n         * @param {jQuery.Event} event\n         * @private\n         */\n        _onQtyChange: function (event) {\n            var qtyElement = $(event.target),\n                parent = qtyElement.parents('tr'),\n                priceBox = $(this.priceBoxSelector, parent),\n                productId = priceBox.data('productId');\n\n            if (this.priceInfo[productId]) {\n                this.priceInfo[productId].qty = this._getQty(qtyElement);\n            }\n\n            this._updateAmount();\n        },\n\n        /**\n         * Calculate and update amount\n         *\n         * @private\n         */\n        _updateAmount: function () {\n            var productId,\n                amount = 0,\n                payLater = registry.get(this.parentName);\n\n            for (productId in this.priceInfo) {\n                if (this.priceInfo.hasOwnProperty(productId)) {\n                    amount += this.priceInfo[productId].price * this.priceInfo[productId].qty;\n                }\n            }\n\n            payLater.amount(amount);\n        }\n    });\n});\n","Magento_Paypal/js/view/amountProviders/product.js":"/**\n* Copyright \u00a9 Magento, Inc. All rights reserved.\n* See COPYING.txt for license details.\n*/\n\ndefine([\n    'jquery',\n    'uiElement',\n    'uiRegistry',\n    'priceBox',\n    'domReady!'\n], function (\n    $,\n    Component,\n    registry\n) {\n    'use strict';\n\n    return Component.extend({\n\n        defaults: {\n            priceBoxSelector: '.price-box',\n            qtyFieldSelector: '#product_addtocart_form [name=\"qty\"]',\n            amount: null\n        },\n        qty: 1,\n        price: 0,\n        priceType: '',\n\n        /**\n         * Initialize\n         *\n         * @returns {*}\n         */\n        initialize: function () {\n            var priceBox;\n\n            this._super();\n\n            priceBox = $(this.priceBoxSelector);\n            priceBox.on('priceUpdated', this._onPriceChange.bind(this));\n\n            if (priceBox.priceBox('option') &&\n                priceBox.priceBox('option').prices &&\n                (priceBox.priceBox('option').prices.finalPrice || priceBox.priceBox('option').prices.basePrice)\n            ) {\n                this.priceType = priceBox.priceBox('option').prices.finalPrice ? 'finalPrice' : 'basePrice';\n                this.price = priceBox.priceBox('option').prices[this.priceType].amount;\n            }\n\n            $(this.qtyFieldSelector).on('change', this._onQtyChange.bind(this));\n\n            priceBox.trigger('updatePrice');\n\n            return this;\n        },\n\n        /**\n         * Handle changed product qty\n         *\n         * @param {jQuery.Event} event\n         * @private\n         */\n        _onQtyChange: function (event) {\n            var qty = parseFloat($(event.target).val());\n\n            this.qty = !isNaN(qty) && qty ? qty : 1;\n            this._updateAmount();\n        },\n\n        /**\n         * Handle product price change\n         *\n         * @param {jQuery.Event} event\n         * @param {Object} data\n         * @private\n         */\n        _onPriceChange: function (event, data) {\n            this.price = data[this.priceType].amount;\n            this._updateAmount();\n        },\n\n        /**\n         * Calculate and update amount\n         *\n         * @private\n         */\n        _updateAmount: function () {\n            var amount = this.price * this.qty,\n                payLater = registry.get(this.parentName);\n\n            if (amount !== 0) {\n                payLater.amount(amount);\n            }\n        }\n    });\n});\n","Magento_Paypal/js/view/payment/paypal-payments.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Checkout/js/model/payment/renderer-list'\n], function (Component, rendererList) {\n    'use strict';\n\n    var isContextCheckout = window.checkoutConfig.payment.paypalExpress.isContextCheckout,\n        paypalExpress = 'Magento_Paypal/js/view/payment/method-renderer' +\n            (isContextCheckout ? '/in-context/checkout-express' : '/paypal-express');\n\n    rendererList.push(\n        {\n            type: 'paypal_express',\n            component: paypalExpress,\n            config: window.checkoutConfig.payment.paypalExpress.inContextConfig\n        },\n        {\n            type: 'payflow_express',\n            component: 'Magento_Paypal/js/view/payment/method-renderer/payflow-express'\n        },\n        {\n            type: 'payflow_express_bml',\n            component: 'Magento_Paypal/js/view/payment/method-renderer/payflow-express-bml'\n        },\n        {\n            type: 'payflowpro',\n            component: 'Magento_Paypal/js/view/payment/method-renderer/payflowpro-method'\n        },\n        {\n            type: 'payflow_link',\n            component: 'Magento_Paypal/js/view/payment/method-renderer/iframe-methods'\n        },\n        {\n            type: 'payflow_advanced',\n            component: 'Magento_Paypal/js/view/payment/method-renderer/iframe-methods'\n        },\n        {\n            type: 'hosted_pro',\n            component: 'Magento_Paypal/js/view/payment/method-renderer/iframe-methods'\n        },\n        {\n            type: 'paypal_billing_agreement',\n            component: 'Magento_Paypal/js/view/payment/method-renderer/paypal-billing-agreement'\n        }\n    );\n\n    /**\n     * Add view logic here if needed\n     **/\n    return Component.extend({});\n});\n","Magento_Paypal/js/view/payment/method-renderer/iframe-methods.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'Magento_Checkout/js/view/payment/default',\n    'Magento_Paypal/js/model/iframe',\n    'Magento_Checkout/js/model/full-screen-loader'\n], function (Component, iframe, fullScreenLoader) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Paypal/payment/iframe-methods',\n            paymentReady: false\n        },\n        redirectAfterPlaceOrder: false,\n        isInAction: iframe.isInAction,\n\n        /**\n         * @return {exports}\n         */\n        initObservable: function () {\n            this._super()\n                .observe('paymentReady');\n\n            return this;\n        },\n\n        /**\n         * @return {*}\n         */\n        isPaymentReady: function () {\n            return this.paymentReady();\n        },\n\n        /**\n         * Get action url for payment method iframe.\n         * @returns {String}\n         */\n        getActionUrl: function () {\n            return this.isInAction() ? window.checkoutConfig.payment.paypalIframe.actionUrl[this.getCode()] : '';\n        },\n\n        /**\n         * Places order in pending payment status.\n         */\n        placePendingPaymentOrder: function () {\n            if (this.placeOrder()) {\n                fullScreenLoader.startLoader();\n                this.isInAction(true);\n                // capture all click events\n                document.addEventListener('click', iframe.stopEventPropagation, true);\n            }\n        },\n\n        /**\n         * @return {*}\n         */\n        getPlaceOrderDeferredObject: function () {\n            var self = this;\n\n            return this._super().fail(function () {\n                fullScreenLoader.stopLoader();\n                self.isInAction(false);\n                document.removeEventListener('click', iframe.stopEventPropagation, true);\n            });\n        },\n\n        /**\n         * After place order callback\n         */\n        afterPlaceOrder: function () {\n            if (this.iframeIsLoaded) {\n                document.getElementById(this.getCode() + '-iframe')\n                    .contentWindow.location.reload();\n                this.paymentReady(false);\n            }\n\n            this.paymentReady(true);\n            this.iframeIsLoaded = true;\n            this.isPlaceOrderActionAllowed(true);\n            fullScreenLoader.stopLoader();\n        },\n\n        /**\n         * Hide loader when iframe is fully loaded.\n         */\n        iframeLoaded: function () {\n            fullScreenLoader.stopLoader();\n        }\n    });\n});\n","Magento_Paypal/js/view/payment/method-renderer/payflow-express-bml.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Paypal/js/view/payment/method-renderer/paypal-express-abstract'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Paypal/payment/payflow-express-bml'\n        }\n    });\n});\n","Magento_Paypal/js/view/payment/method-renderer/payflow-express.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Paypal/js/view/payment/method-renderer/paypal-express-abstract'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Paypal/payment/payflow-express'\n        }\n    });\n});\n","Magento_Paypal/js/view/payment/method-renderer/payflowpro-method.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Payment/js/view/payment/iframe',\n    'Magento_Checkout/js/model/payment/additional-validators',\n    'Magento_Checkout/js/action/set-payment-information',\n    'Magento_Checkout/js/model/full-screen-loader',\n    'Magento_Vault/js/view/payment/vault-enabler'\n], function ($, Component, additionalValidators, setPaymentInformationAction, fullScreenLoader, VaultEnabler) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Paypal/payment/payflowpro-form'\n        },\n        placeOrderHandler: null,\n        validateHandler: null,\n\n        /**\n         * @returns {exports.initialize}\n         */\n        initialize: function () {\n            this._super();\n            this.vaultEnabler = new VaultEnabler();\n            this.vaultEnabler.setPaymentCode(this.getVaultCode());\n\n            return this;\n        },\n\n        /**\n         * @param {Function} handler\n         */\n        setPlaceOrderHandler: function (handler) {\n            this.placeOrderHandler = handler;\n        },\n\n        /**\n         * @param {Function} handler\n         */\n        setValidateHandler: function (handler) {\n            this.validateHandler = handler;\n        },\n\n        /**\n         * @returns {Object}\n         */\n        context: function () {\n            return this;\n        },\n\n        /**\n         * @returns {Boolean}\n         */\n        isShowLegend: function () {\n            return true;\n        },\n\n        /**\n         * @returns {String}\n         */\n        getCode: function () {\n            return 'payflowpro';\n        },\n\n        /**\n         * @returns {Boolean}\n         */\n        isActive: function () {\n            return true;\n        },\n\n        /**\n         * @override\n         */\n        placeOrder: function () {\n            var self = this;\n\n            if (this.validateHandler() &&\n                additionalValidators.validate() &&\n                this.isPlaceOrderActionAllowed() === true\n            ) {\n                this.isPlaceOrderActionAllowed(false);\n                fullScreenLoader.startLoader();\n                $.when(\n                    setPaymentInformationAction(this.messageContainer, self.getData())\n                ).done(\n                    function () {\n                        self.placeOrderHandler().fail(\n                            function () {\n                                fullScreenLoader.stopLoader();\n                            }\n                        );\n                    }\n                ).always(\n                    function () {\n                        self.isPlaceOrderActionAllowed(true);\n                        fullScreenLoader.stopLoader();\n                    }\n                );\n            }\n        },\n\n        /**\n         * @returns {Object}\n         */\n        getData: function () {\n            var data = {\n                'method': this.getCode(),\n                'additional_data': {\n                    'cc_type': this.creditCardType(),\n                    'cc_exp_year': this.creditCardExpYear(),\n                    'cc_exp_month': this.creditCardExpMonth(),\n                    'cc_last_4': this.creditCardNumber().substr(-4)\n                }\n            };\n\n            this.vaultEnabler.visitAdditionalData(data);\n\n            return data;\n        },\n\n        /**\n         * @returns {Bool}\n         */\n        isVaultEnabled: function () {\n            return this.vaultEnabler.isVaultEnabled();\n        },\n\n        /**\n         * @returns {String}\n         */\n        getVaultCode: function () {\n            return 'payflowpro_cc_vault';\n        }\n    });\n});\n","Magento_Paypal/js/view/payment/method-renderer/paypal-billing-agreement.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Checkout/js/view/payment/default',\n    'mage/validation'\n], function ($, Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Paypal/payment/paypal_billing_agreement-form',\n            selectedBillingAgreement: ''\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super()\n                .observe('selectedBillingAgreement');\n\n            return this;\n        },\n\n        /**\n         * @return {*}\n         */\n        getTransportName: function () {\n            return window.checkoutConfig.payment.paypalBillingAgreement.transportName;\n        },\n\n        /**\n         * @return {*}\n         */\n        getBillingAgreements: function () {\n            return window.checkoutConfig.payment.paypalBillingAgreement.agreements;\n        },\n\n        /**\n         * @return {Object}\n         */\n        getData: function () {\n            var additionalData = null;\n\n            if (this.getTransportName()) {\n                additionalData = {};\n                additionalData[this.getTransportName()] = this.selectedBillingAgreement();\n            }\n\n            return {\n                'method': this.item.method,\n                'additional_data': additionalData\n            };\n        },\n\n        /**\n         * @return {jQuery}\n         */\n        validate: function () {\n            var form = '#billing-agreement-form';\n\n            return $(form).validation() && $(form).validation('isValid');\n        }\n    });\n});\n","Magento_Paypal/js/view/payment/method-renderer/paypal-express-abstract.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Checkout/js/view/payment/default',\n    'Magento_Paypal/js/action/set-payment-method',\n    'Magento_Checkout/js/model/payment/additional-validators',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Customer/js/customer-data'\n], function ($, Component, setPaymentMethodAction, additionalValidators, quote, customerData) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Paypal/payment/payflow-express-bml',\n            billingAgreement: ''\n        },\n\n        /** Init observable variables */\n        initObservable: function () {\n            this._super()\n                .observe('billingAgreement');\n\n            return this;\n        },\n\n        /** Open window with  */\n        showAcceptanceWindow: function (data, event) {\n            window.open(\n                $(event.currentTarget).attr('href'),\n                'olcwhatispaypal',\n                'toolbar=no, location=no,' +\n                ' directories=no, status=no,' +\n                ' menubar=no, scrollbars=yes,' +\n                ' resizable=yes, ,left=0,' +\n                ' top=0, width=400, height=350'\n            );\n\n            return false;\n        },\n\n        /** Returns payment acceptance mark link path */\n        getPaymentAcceptanceMarkHref: function () {\n            return window.checkoutConfig.payment.paypalExpress.paymentAcceptanceMarkHref;\n        },\n\n        /** Returns payment acceptance mark image path */\n        getPaymentAcceptanceMarkSrc: function () {\n            return window.checkoutConfig.payment.paypalExpress.paymentAcceptanceMarkSrc;\n        },\n\n        /** Returns billing agreement data */\n        getBillingAgreementCode: function () {\n            return window.checkoutConfig.payment.paypalExpress.billingAgreementCode[this.item.method];\n        },\n\n        /** Returns payment information data */\n        getData: function () {\n            var parent = this._super(),\n                additionalData = null;\n\n            if (this.getBillingAgreementCode()) {\n                additionalData = {};\n                additionalData[this.getBillingAgreementCode()] = this.billingAgreement();\n            }\n\n            return $.extend(true, parent, {\n                'additional_data': additionalData\n            });\n        },\n\n        /** Redirect to paypal */\n        continueToPayPal: function () {\n            if (additionalValidators.validate()) {\n                //update payment method information if additional data was changed\n                setPaymentMethodAction(this.messageContainer).done(\n                    function () {\n                        customerData.invalidate(['cart']);\n                        $.mage.redirect(\n                            window.checkoutConfig.payment.paypalExpress.redirectUrl[quote.paymentMethod().method]\n                        );\n                    }\n                );\n\n                return false;\n            }\n        }\n    });\n});\n","Magento_Paypal/js/view/payment/method-renderer/paypal-express.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Paypal/js/view/payment/method-renderer/paypal-express-abstract'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Paypal/payment/paypal-express'\n        }\n    });\n});\n","Magento_Paypal/js/view/payment/method-renderer/in-context/checkout-express.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'Magento_Paypal/js/view/payment/method-renderer/paypal-express-abstract',\n    'Magento_Paypal/js/in-context/express-checkout-wrapper',\n    'Magento_Paypal/js/action/set-payment-method',\n    'Magento_Checkout/js/model/payment/additional-validators',\n    'Magento_Ui/js/model/messageList',\n    'Magento_Ui/js/lib/view/utils/async'\n], function ($, Component, Wrapper, setPaymentMethod, additionalValidators, messageList) {\n    'use strict';\n\n    return Component.extend(Wrapper).extend({\n        defaults: {\n            template: 'Magento_Paypal/payment/paypal-express-in-context',\n            validationElements: 'input'\n        },\n\n        /**\n         * Listens element on change and validate it.\n         *\n         * @param {HTMLElement} context\n         */\n        initListeners: function (context) {\n            $.async(this.validationElements, context, function (element) {\n                $(element).on('change', function () {\n                    this.validate();\n                }.bind(this));\n            }.bind(this));\n        },\n\n        /**\n         *  Validates Smart Buttons\n         */\n        validate: function () {\n            this._super();\n\n            if (this.actions) {\n                additionalValidators.validate(true) ? this.actions.enable() : this.actions.disable();\n            }\n        },\n\n        /** @inheritdoc */\n        beforePayment: function (resolve, reject) {\n            var promise = $.Deferred();\n\n            setPaymentMethod(this.messageContainer).done(function () {\n                return promise.resolve();\n            }).fail(function (response) {\n                var error;\n\n                try {\n                    error = JSON.parse(response.responseText);\n                } catch (exception) {\n                    error = this.paymentActionError;\n                }\n\n                this.addError(error);\n\n                return reject(new Error(error));\n            }.bind(this));\n\n            return promise;\n        },\n\n        /**\n         * Populate client config with all required data\n         *\n         * @return {Object}\n         */\n        prepareClientConfig: function () {\n            this._super();\n            this.clientConfig.quoteId = window.checkoutConfig.quoteData['entity_id'];\n            this.clientConfig.customerId = window.customerData.id;\n            this.clientConfig.button = 0;\n\n            return this.clientConfig;\n        },\n\n        /**\n         * Adding logic to be triggered onClick action for smart buttons component\n         */\n        onClick: function () {\n            additionalValidators.validate();\n        },\n\n        /**\n         * Adds error message\n         *\n         * @param {String} message\n         */\n        addError: function (message) {\n            messageList.addErrorMessage({\n                message: message\n            });\n        },\n\n        /**\n         * After payment execute\n         *\n         * @param {Object} res\n         * @param {Function} resolve\n         * @param {Function} reject\n         *\n         * @return {*}\n         */\n        afterPayment: function (res, resolve, reject) {\n            if (res.success) {\n                return resolve(res.token);\n            }\n\n            this.addError(res['error_message']);\n\n            return reject(new Error(res['error_message']));\n        },\n\n        /**\n         * After onAuthorize execute\n         *\n         * @param {Object} res\n         * @param {Function} resolve\n         * @param {Function} reject\n         * @param {Object} actions\n         *\n         * @return {*}\n         */\n        afterOnAuthorize: function (res, resolve, reject, actions) {\n            if (res.success) {\n                resolve();\n\n                return actions.redirect(res.redirectUrl);\n            }\n\n            this.addError(res['error_message']);\n\n            return reject(new Error(res['error_message']));\n        }\n    });\n});\n","Magento_Paypal/js/view/payment/method-renderer/payflowpro/vault.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Vault/js/view/payment/method-renderer/vault'\n], function (VaultComponent) {\n    'use strict';\n\n    return VaultComponent.extend({\n        defaults: {\n            template: 'Magento_Vault/payment/form'\n        },\n\n        /**\n         * @returns {String}\n         */\n        getToken: function () {\n            return this.publicHash;\n        },\n\n        /**\n         * Get last 4 digits of card\n         * @returns {String}\n         */\n        getMaskedCard: function () {\n            return this.details['cc_last_4'];\n        },\n\n        /**\n         * Get expiration date\n         * @returns {String}\n         */\n        getExpirationDate: function () {\n            return this.details['cc_exp_month'] + '/' + this.details['cc_exp_year'];\n        },\n\n        /**\n         * Get card type\n         * @returns {String}\n         */\n        getCardType: function () {\n            return this.details['cc_type'];\n        }\n    });\n});\n","Magento_PaypalCaptcha/js/model/skipRefreshCaptcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['ko'], function (ko) {\n    'use strict';\n\n    return {\n        skip: ko.observable(false)\n    };\n});\n","Magento_PaypalCaptcha/js/view/checkout/defaultCaptcha-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_PaypalCaptcha/js/model/skipRefreshCaptcha'\n], function (skipRefreshCaptcha) {\n    'use strict';\n\n    var defaultCaptchaMixin = {\n        /**\n         * @override\n         */\n        refresh: function () {\n            if (!skipRefreshCaptcha.skip()) {\n                this._super();\n            } else {\n                skipRefreshCaptcha.skip(false);\n            }\n        }\n    };\n\n    return function (defaultCaptcha) {\n        return defaultCaptcha.extend(defaultCaptchaMixin);\n    };\n});\n","Magento_PaypalCaptcha/js/view/checkout/paymentCaptcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Captcha/js/view/checkout/defaultCaptcha',\n    'Magento_Captcha/js/model/captchaList',\n    'Magento_Captcha/js/model/captcha'\n],\nfunction ($, defaultCaptcha, captchaList, Captcha) {\n    'use strict';\n\n    return defaultCaptcha.extend({\n\n        /** @inheritdoc */\n        initialize: function () {\n            var captchaConfigPayment,\n                currentCaptcha;\n\n            this._super();\n\n            if (window[this.configSource] && window[this.configSource].captchaPayments) {\n                captchaConfigPayment = window[this.configSource].captchaPayments;\n\n                $.each(captchaConfigPayment, function (formId, captchaData) {\n                    var captcha;\n\n                    captchaData.formId = formId;\n                    captcha = Captcha(captchaData);\n                    captchaList.add(captcha);\n                });\n            }\n\n            currentCaptcha = captchaList.getCaptchaByFormId(this.formId);\n\n            if (currentCaptcha != null) {\n                currentCaptcha.setIsVisible(true);\n                this.setCurrentCaptcha(currentCaptcha);\n            }\n        }\n    });\n});\n","Magento_PaypalCaptcha/js/view/payment/list-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'Magento_Captcha/js/model/captchaList'\n], function ($, captchaList) {\n    'use strict';\n\n    var mixin = {\n\n        formId: 'co-payment-form',\n\n        /**\n         * Sets custom template for Payflow Pro\n         *\n         * @param {Object} payment\n         * @returns {Object}\n         */\n        createComponent: function (payment) {\n\n            var component = this._super(payment);\n\n            if (component.component === 'Magento_Paypal/js/view/payment/method-renderer/payflowpro-method') {\n                component.template = 'Magento_PaypalCaptcha/payment/payflowpro-form';\n                $(window).off('clearTimeout')\n                    .on('clearTimeout', this.clearTimeout.bind(this));\n            }\n\n            return component;\n        },\n\n        /**\n         * Overrides default window.clearTimeout() to catch errors from iframe and reload Captcha.\n         *\n         * @param {Number} timeoutID\n         */\n        clearTimeout: function (timeoutID) {\n            var captcha = captchaList.getCaptchaByFormId(this.formId);\n\n            if (captcha !== null) {\n                captcha.refresh();\n            }\n            clearTimeout(timeoutID);\n        }\n    };\n\n    /**\n     * Overrides `Magento_Checkout/js/view/payment/list::createComponent`\n     */\n    return function (target) {\n        return target.extend(mixin);\n    };\n});\n","Magento_PaypalCaptcha/js/view/payment/method-renderer/payflowpro-method-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_PaypalCaptcha/js/model/skipRefreshCaptcha'\n], function (skipRefreshCaptcha) {\n    'use strict';\n\n    var payflowProMethodMixin = {\n        /**\n         * @override\n         */\n        placeOrder: function () {\n            skipRefreshCaptcha.skip(true);\n            this._super();\n        }\n    };\n\n    return function (payflowProMethod) {\n        return payflowProMethod.extend(payflowProMethodMixin);\n    };\n});\n","Magento_Persistent/js/view/additional-welcome.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/translate',\n    'Magento_Customer/js/customer-data'\n], function ($, $t, customerData) {\n    'use strict';\n\n    return {\n        /**\n         * Init.\n         */\n        init: function () {\n            var persistent = customerData.get('persistent');\n\n            if (persistent().fullname === undefined) {\n                customerData.get('persistent').subscribe(this.replacePersistentWelcome);\n            } else {\n                this.replacePersistentWelcome();\n            }\n        },\n\n        /**\n         * Replace welcome message for customer with persistent cookie.\n         */\n        replacePersistentWelcome: function () {\n            var persistent = customerData.get('persistent'),\n                welcomeElems;\n\n            if (persistent().fullname !== undefined) {\n                welcomeElems = $('li.greet.welcome > span.not-logged-in');\n\n                if (welcomeElems.length) {\n                    $(welcomeElems).each(function () {\n                        var html = $t('Welcome, %1!').replace('%1', persistent().fullname);\n\n                        $(this).attr('data-bind', html);\n                        $(this).html(html);\n                    });\n                    $(welcomeElems).append(' <span><a ' + window.notYouLink + '>' + $t('Not you?') + '</a>');\n                }\n            }\n        },\n\n        /**\n         * @constructor\n         */\n        'Magento_Persistent/js/view/additional-welcome': function () {\n            this.init();\n        }\n    };\n});\n","Magento_Persistent/js/view/customer-data-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'mage/utils/wrapper'\n], function ($, wrapper) {\n    'use strict';\n\n    var mixin = {\n\n        /**\n         * Check if persistent section is expired due to lifetime.\n         *\n         * @param {Function} originFn - Original method.\n         * @return {Array}\n         */\n        getExpiredSectionNames: function (originFn) {\n            var expiredSections = originFn(),\n                storage = $.initNamespaceStorage('mage-cache-storage').localStorage,\n                currentTimestamp = Math.floor(Date.now() / 1000),\n                persistentIndex = expiredSections.indexOf('persistent'),\n                persistentLifeTime = 0,\n                sectionData;\n\n            if (window.persistent !== undefined && window.persistent.expirationLifetime !== undefined) {\n                persistentLifeTime = window.persistent.expirationLifetime;\n            }\n\n            if (persistentIndex !== -1) {\n                sectionData = storage.get('persistent');\n\n                if (typeof sectionData === 'object' &&\n                    sectionData['data_id'] + persistentLifeTime >= currentTimestamp\n                ) {\n                    expiredSections.splice(persistentIndex, 1);\n                }\n            }\n\n            return expiredSections;\n        },\n\n        /**\n         * @param {Object} settings\n         * @constructor\n         */\n        'Magento_Customer/js/customer-data': function (originFn) {\n            let mageCacheTimeout = new Date($.localStorage.get('mage-cache-timeout')),\n                mageCacheSessId = $.cookieStorage.isSet('mage-cache-sessid');\n\n            originFn();\n            if (window.persistent !== undefined && (mageCacheTimeout < new Date() || !mageCacheSessId)) {\n                this.reload(['persistent','cart'],true);\n            }\n        }\n    };\n\n    /**\n     * Override default customer-data.getExpiredSectionNames().\n     */\n    return function (target) {\n        return wrapper.extend(target, mixin);\n    };\n});\n","Magento_ProductAlert/js/form-submitter.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    return function (data, element) {\n\n        $(element).trigger('submit');\n    };\n});\n","Magento_ProductVideo/js/fotorama-add-video-events.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget',\n    'catalogGallery',\n    'loadPlayer'\n], function ($) {\n    'use strict';\n\n    /**\n     * @private\n     */\n    var allowBase = true; //global var is needed because fotorama always fully reloads events in case of fullscreen\n\n    /**\n     * @private\n     */\n    function parseHref(href) {\n        var a = document.createElement('a');\n\n        a.href = href;\n\n        return a;\n    }\n\n    /**\n     * @private\n     */\n    function parseURL(href, forceVideo) {\n        var id,\n            type,\n            ampersandPosition,\n            vimeoRegex,\n            useYoutubeNocookie = false;\n\n        /**\n         * Get youtube ID\n         * @param {String} srcid\n         * @returns {{}}\n         */\n        function _getYoutubeId(srcid) {\n            if (srcid) {\n                ampersandPosition = srcid.indexOf('&');\n\n                if (ampersandPosition === -1) {\n                    return srcid;\n                }\n\n                srcid = srcid.substring(0, ampersandPosition);\n            }\n\n            return srcid;\n        }\n\n        if (typeof href !== 'string') {\n            return href;\n        }\n\n        href = parseHref(href);\n\n        if (href.host.match(/youtube\\.com/) && href.search) {\n            id = href.search.split('v=')[1];\n\n            if (id) {\n                id = _getYoutubeId(id);\n                type = 'youtube';\n            }\n        } else if (href.host.match(/youtube\\.com|youtu\\.be|youtube-nocookie.com/)) {\n            id = href.pathname.replace(/^\\/(embed\\/|v\\/)?/, '').replace(/\\/.*/, '');\n            type = 'youtube';\n\n            if (href.host.match(/youtube-nocookie.com/)) {\n                useYoutubeNocookie = true;\n            }\n        } else if (href.host.match(/vimeo\\.com/)) {\n            type = 'vimeo';\n            vimeoRegex = new RegExp(['https?:\\\\/\\\\/(?:www\\\\.|player\\\\.)?vimeo.com\\\\/(?:channels\\\\/(?:\\\\w+\\\\/)',\n                '?|groups\\\\/([^\\\\/]*)\\\\/videos\\\\/|album\\\\/(\\\\d+)\\\\/video\\\\/|video\\\\/|)(\\\\d+)(?:$|\\\\/|\\\\?)'\n            ].join(''));\n            id = href.href.match(vimeoRegex)[3];\n        }\n\n        if ((!id || !type) && forceVideo) {\n            id = href.href;\n            type = 'custom';\n        }\n\n        return id ? {\n            id: id, type: type, s: href.search.replace(/^\\?/, ''), useYoutubeNocookie: useYoutubeNocookie\n        } : false;\n    }\n\n    //create AddFotoramaVideoEvents widget\n    $.widget('mage.AddFotoramaVideoEvents', {\n        options: {\n            videoData: '',\n            videoSettings: '',\n            optionsVideoData: '',\n            dataMergeStrategy: 'replace'\n        },\n\n        /**\n         * @private\n         */\n        onVimeoJSFramework: function () {},\n        defaultVideoData: [],\n        PV: 'product-video', // [CONST]\n        VU: 'video-unplayed',\n        PVLOADED: 'fotorama__product-video--loaded', // [CONST]\n        PVLOADING: 'fotorama__product-video--loading', // [CONST]\n        VID: 'video', // [CONST]\n        VI: 'vimeo', // [CONST]\n        FTVC: 'fotorama__video-close',\n        FTAR: 'fotorama__arr',\n        fotoramaSpinner: 'fotorama__spinner',\n        fotoramaSpinnerShow: 'fotorama__spinner--show',\n        TI: 'video-thumb-icon',\n        isFullscreen: false,\n        FTCF: '[data-gallery-role=\"fotorama__fullscreen-icon\"]',\n        Base: 0, //on check for video is base this setting become true if there is any video with base role\n        MobileMaxWidth: 768,\n        GP: 'gallery-placeholder', //gallery placeholder class is needed to find and erase <script> tag\n        videoData: null,\n        videoDataPlaceholder: [{\n            id: '',\n            isBase: true,\n            mediaType: 'image',\n            provider: ''\n        }],\n\n        /**\n         * Creates widget\n         * @private\n         */\n        _create: function () {\n            $(this.element).data('gallery') ?\n                this._onGalleryLoaded() :\n                $(this.element).on('gallery:loaded', this._onGalleryLoaded.bind(this));\n        },\n\n        /**\n         *\n         * @private\n         */\n        _initialize: function () {\n            if (!this.defaultVideoData.length) {\n                this.defaultVideoData = this.options.videoData;\n            }\n\n            // If product does not have images, no video data generated,\n            // but for configurable product we still need a video data, in case of 'prepend' gallery strategy.\n            if (!this.defaultVideoData.length && !this.options.videoData.length) {\n                this.defaultVideoData = this.options.videoData = this.videoDataPlaceholder;\n            }\n\n            this.clearEvents();\n\n            if (this._checkForVideoExist()) {\n                this._checkFullscreen();\n                this._listenForFullscreen();\n                this._isVideoBase();\n                this._initFotoramaVideo();\n                this._attachFotoramaEvents();\n            }\n        },\n\n        /**\n         * Callback which fired after gallery gets initialized.\n         */\n        _onGalleryLoaded: function () {\n            this.fotoramaItem = $(this.element).find('.fotorama-item');\n            this._initialize();\n        },\n\n        /**\n         * Clear gallery events to prevent duplicated calls.\n         *\n         * @private\n         */\n        clearEvents: function () {\n            if (this.fotoramaItem !== undefined) {\n                this.fotoramaItem.off(\n                    'fotorama:show.' + this.PV +\n                    ' fotorama:showend.' + this.PV +\n                    ' fotorama:fullscreenenter.' + this.PV +\n                    ' fotorama:fullscreenexit.' + this.PV\n                );\n            }\n        },\n\n        /**\n         *\n         * @param {Object} options\n         * @private\n         */\n        _setOptions: function (options) {\n            if (options.videoData && options.videoData.length) {\n                this.options.videoData = options.videoData;\n            }\n\n            this._loadVideoData(options);\n            this._initialize();\n        },\n\n        /**\n         * Set video data for configurable product.\n         *\n         * @param {Object} options\n         * @private\n         */\n        _loadVideoData: function (options) {\n            if (options.selectedOption) {\n                if (options.dataMergeStrategy === 'prepend') {\n                    this.options.videoData = [].concat(\n                        this.options.optionsVideoData[options.selectedOption],\n                        this.defaultVideoData\n                    );\n                } else {\n                    this.options.videoData = this.options.optionsVideoData[options.selectedOption];\n                }\n            } else {\n                this.options.videoData = this.defaultVideoData;\n            }\n        },\n\n        /**\n         *\n         * @private\n         */\n        _checkFullscreen: function () {\n            if (this.fotoramaItem.data('fotorama').fullScreen || false) {\n                this.isFullscreen = true;\n            }\n        },\n\n        /**\n         *\n         * @private\n         */\n        _listenForFullscreen: function () {\n            this.fotoramaItem.on('fotorama:fullscreenenter.' + this.PV, $.proxy(function () {\n                this.isFullscreen = true;\n            }, this));\n\n            this.fotoramaItem.on('fotorama:fullscreenexit.' + this.PV, $.proxy(function () {\n                this.isFullscreen = false;\n                this._hideVideoArrows();\n            }, this));\n        },\n\n        /**\n         *\n         * @param {Object} inputData\n         * @param {bool} isJSON\n         * @returns {{}}\n         * @private\n         */\n        _createVideoData: function (inputData, isJSON) {\n            var videoData = [],\n                dataUrl,\n                tmpVideoData,\n                tmpInputData,\n                i;\n\n            if (isJSON) {\n                inputData = JSON.parse(inputData);\n            }\n\n            for (i = 0; i < inputData.length; i++) {\n                tmpInputData = inputData[i];\n                dataUrl = '';\n                tmpVideoData = {\n                    mediaType: '',\n                    isBase: '',\n                    id: '',\n                    provider: ''\n                };\n                tmpVideoData.mediaType = this.VID;\n\n                if (tmpInputData.mediaType !== 'external-video') {\n                    tmpVideoData.mediaType = tmpInputData.mediaType;\n                }\n\n                tmpVideoData.isBase = tmpInputData.isBase;\n\n                if (tmpInputData.videoUrl && tmpInputData.videoUrl !== null) {\n                    dataUrl = tmpInputData.videoUrl;\n                    dataUrl = parseURL(dataUrl);\n                    tmpVideoData.id = dataUrl.id;\n                    tmpVideoData.provider = dataUrl.type;\n                    tmpVideoData.videoUrl = tmpInputData.videoUrl;\n                    tmpVideoData.useYoutubeNocookie = dataUrl.useYoutubeNocookie;\n                }\n\n                videoData.push(tmpVideoData);\n            }\n\n            return videoData;\n        },\n\n        /**\n         *\n         * @param {Object} fotorama\n         * @param {bool} isBase\n         * @private\n         */\n        _createCloseVideo: function (fotorama, isBase) {\n            var closeVideo;\n\n            this.fotoramaItem.find('.' + this.FTVC).remove();\n            this.fotoramaItem.append('<div class=\"' + this.FTVC + '\"></div>');\n            this.fotoramaItem.css('position', 'relative');\n            closeVideo = this.fotoramaItem.find('.' + this.FTVC);\n            this._closeVideoSetEvents(closeVideo, fotorama);\n\n            if (\n                isBase &&\n                this.options.videoData[fotorama.activeIndex].isBase &&\n                $(window).width() > this.MobileMaxWidth) {\n                this._showCloseVideo();\n            }\n        },\n\n        /**\n         *\n         * @private\n         */\n        _hideCloseVideo: function () {\n            this.fotoramaItem\n                .find('.' + this.FTVC)\n                .removeClass('fotorama-show-control');\n        },\n\n        /**\n         *\n         * @private\n         */\n        _showCloseVideo: function () {\n            this.fotoramaItem\n                .find('.' + this.FTVC)\n                .addClass('fotorama-show-control');\n        },\n\n        /**\n         *\n         * @param {jQuery} $closeVideo\n         * @param {jQuery} fotorama\n         * @private\n         */\n        _closeVideoSetEvents: function ($closeVideo, fotorama) {\n            $closeVideo.on('click', $.proxy(function () {\n                this._unloadVideoPlayer(fotorama.activeFrame.$stageFrame.parent(), fotorama, true);\n                this._hideCloseVideo();\n            }, this));\n        },\n\n        /**\n         *\n         * @returns {Boolean}\n         * @private\n         */\n        _checkForVideoExist: function () {\n            var key, result, checker, videoSettings;\n\n            if (!this.options.videoData) {\n                return false;\n            }\n\n            if (!this.options.videoSettings) {\n                return false;\n            }\n\n            result = this._createVideoData(this.options.videoData, false);\n            checker = false;\n            videoSettings = this.options.videoSettings[0];\n            videoSettings.playIfBase = parseInt(videoSettings.playIfBase, 10);\n            videoSettings.showRelated = parseInt(videoSettings.showRelated, 10);\n            videoSettings.videoAutoRestart = parseInt(videoSettings.videoAutoRestart, 10);\n\n            for (key in result) {\n                if (result[key].mediaType === this.VID) {\n                    checker = true;\n                }\n            }\n\n            if (checker) {\n                this.options.videoData = result;\n            }\n\n            return checker;\n        },\n\n        /**\n         *\n         * @private\n         */\n        _isVideoBase: function () {\n            var allVideoData = this.options.videoData,\n                videoItem,\n                allVideoDataKeys,\n                key,\n                i;\n\n            allVideoDataKeys = Object.keys(allVideoData);\n\n            for (i = 0; i < allVideoDataKeys.length; i++) {\n                key = allVideoDataKeys[i];\n                videoItem = allVideoData[key];\n\n                if (\n                    videoItem.mediaType === this.VID && videoItem.isBase &&\n                    this.options.videoSettings[0].playIfBase && allowBase\n                ) {\n                    this.Base = true;\n                    allowBase = false;\n                }\n            }\n\n            if (!this.isFullscreen) {\n                this._createCloseVideo(this.fotoramaItem.data('fotorama'), this.Base);\n            }\n        },\n\n        /**\n         *\n         * @param {Event} e\n         * @private\n         */\n        _initFotoramaVideo: function (e) {\n            var fotorama = this.fotoramaItem.data('fotorama'),\n                thumbsParent,\n                thumbs,\n                t;\n\n            if (!fotorama.activeFrame.$navThumbFrame) {\n                this.fotoramaItem.on('fotorama:showend.' + this.PV, $.proxy(function (evt, fotoramaData) {\n                    $(fotoramaData.activeFrame.$stageFrame).removeAttr('href');\n                }, this));\n\n                this._startPrepareForPlayer(e, fotorama);\n\n                return null;\n            }\n\n            fotorama.data.map($.proxy(this._setItemType, this));\n            thumbsParent = fotorama.activeFrame.$navThumbFrame.parent();\n            thumbs = thumbsParent.find('.fotorama__nav__frame:visible');\n\n            for (t = 0; t < thumbs.length; t++) {\n                this._setThumbsIcon(thumbs.eq(t), t);\n                this._checkForVideo(e, fotorama, t + 1);\n            }\n\n            this.fotoramaItem.on('fotorama:showend.' + this.PV, $.proxy(function (evt, fotoramaData) {\n                $(fotoramaData.activeFrame.$stageFrame).removeAttr('href');\n            }, this));\n        },\n\n        /**\n         *\n         * @param {Object} elem\n         * @param {Number} i\n         * @private\n         */\n        _setThumbsIcon: function (elem, i) {\n            var fotorama = this.fotoramaItem.data('fotorama');\n\n            if (fotorama.options.nav === 'dots' && elem.hasClass(this.TI)) {\n                elem.removeClass(this.TI);\n            }\n\n            if (this.options.videoData[i].mediaType === this.VID &&\n                fotorama.data[i].type ===  this.VID &&\n                fotorama.options.nav === 'thumbs') {\n                elem.addClass(this.TI);\n            }\n        },\n\n        /**\n         * Temporary solution with adding types for configurable product items\n         *\n         * @param {Object} item\n         * @param {Number} i\n         * @private\n         */\n        _setItemType: function (item, i) {\n            !item.type && (item.type = this.options.videoData[i].mediaType);\n        },\n\n        /**\n         * Attach\n         *\n         * @private\n         */\n        _attachFotoramaEvents: function () {\n            this.fotoramaItem.on('fotorama:showend.' + this.PV, $.proxy(function (e, fotorama) {\n                this._startPrepareForPlayer(e, fotorama);\n            }, this));\n\n            this.fotoramaItem.on('fotorama:show.' + this.PV, $.proxy(function (e, fotorama) {\n                this._unloadVideoPlayer(fotorama.activeFrame.$stageFrame.parent(), fotorama, true);\n            }, this));\n\n            this.fotoramaItem.on('fotorama:fullscreenexit.' + this.PV, $.proxy(function (e, fotorama) {\n                fotorama.activeFrame.$stageFrame.find('.' + this.PV).remove();\n                this._startPrepareForPlayer(e, fotorama);\n            }, this));\n        },\n\n        /**\n         * Start prepare for player\n         *\n         * @param {Event} e\n         * @param {jQuery} fotorama\n         * @private\n         */\n        _startPrepareForPlayer: function (e, fotorama) {\n            this._unloadVideoPlayer(fotorama.activeFrame.$stageFrame.parent(), fotorama, false);\n            this._checkForVideo(e, fotorama, fotorama.activeFrame.i);\n            this._checkForVideo(e, fotorama, fotorama.activeFrame.i - 1);\n            this._checkForVideo(e, fotorama, fotorama.activeFrame.i + 1);\n        },\n\n        /**\n         * Check for video\n         *\n         * @param {Event} e\n         * @param {jQuery} fotorama\n         * @param {Number} number\n         * @private\n         */\n        _checkForVideo: function (e, fotorama, number) {\n            var videoData = this.options.videoData[number - 1],\n                $image = fotorama.data[number - 1];\n\n            if ($image) {\n                !$image.type && this._setItemType($image, number - 1);\n\n                if ($image.type === 'image') {\n                    $image.$navThumbFrame && $image.$navThumbFrame.removeClass(this.TI);\n                    this._hideCloseVideo();\n\n                    return;\n                } else if ($image.$navThumbFrame && $image.type === 'video') {\n                    !$image.$navThumbFrame.hasClass(this.TI) && $image.$navThumbFrame.addClass(this.TI);\n                }\n\n                $image = $image.$stageFrame;\n            }\n\n            if ($image && videoData && videoData.mediaType === this.VID) {\n                $(fotorama.activeFrame.$stageFrame).removeAttr('href');\n                this._prepareForVideoContainer($image, videoData, fotorama, number);\n            }\n\n            if (this.isFullscreen && this.fotoramaItem.data('fotorama').activeFrame.i === number) {\n                this.fotoramaItem.data('fotorama').activeFrame.$stageFrame[0].trigger('click');\n            }\n        },\n\n        /**\n         * Prepare for video container\n         *\n         * @param {jQuery} $image\n         * @param {Object} videoData\n         * @param {Object} fotorama\n         * @param {Number} number\n         * @private\n         */\n        _prepareForVideoContainer: function ($image, videoData, fotorama, number) {\n            $image.addClass('fotorama-video-container').addClass(this.VU);\n            this._createVideoContainer(videoData, $image);\n            this._setVideoEvent($image, this.PV, fotorama, number);\n        },\n\n        /**\n         * Create video container\n         *\n         * @param {Object} videoData\n         * @param {jQuery} $image\n         * @private\n         */\n        _createVideoContainer: function (videoData, $image) {\n            var videoSettings;\n\n            videoSettings = this.options.videoSettings[0];\n            $image.find('.' + this.PV).remove();\n            $image.append(\n                '<div class=\"' +\n                this.PV +\n                '\" data-related=\"' +\n                videoSettings.showRelated +\n                '\" data-loop=\"' +\n                videoSettings.videoAutoRestart +\n                '\" data-type=\"' +\n                videoData.provider +\n                '\" data-code=\"' +\n                videoData.id +\n                '\"  data-youtubenocookie=\"' +\n                videoData.useYoutubeNocookie +\n                '\" data-width=\"100%\" data-height=\"100%\"></div>'\n            );\n        },\n\n        /**\n         *\n         * @param {Object} $image\n         * @param {Object} PV\n         * @param {Object} fotorama\n         * @param {Number} number\n         * @private\n         */\n        _setVideoEvent: function ($image, PV, fotorama, number) {\n            $image.find('.magnify-lens').remove();\n            $image\n                .off('click tap', $.proxy(this._clickHandler, this))\n                .on('click tap', $.proxy(this._clickHandler, this));\n            this._handleBaseVideo(fotorama, number); //check for video is it base and handle it if it's base\n        },\n\n        /**\n         * Hides preview arrows above video player.\n         * @private\n         */\n        _hideVideoArrows: function () {\n            var arrows = $('.' + this.FTAR);\n\n            arrows.removeClass('fotorama__arr--shown');\n            arrows.removeClass('fotorama__arr--hidden');\n        },\n\n        /**\n         * @private\n         */\n        _showLoader: function () {\n            var spinner = this.fotoramaItem.find('.' + this.fotoramaSpinner);\n\n            spinner.addClass(this.fotoramaSpinnerShow);\n            this.fotoramaItem.data('fotorama').activeFrame.$stageFrame.addClass(this.PVLOADING);\n        },\n\n        /**\n         * @private\n         */\n        _hideLoader: function () {\n            var spinner = this.fotoramaItem.find('.' + this.fotoramaSpinner);\n\n            spinner.removeClass(this.fotoramaSpinnerShow);\n            this.fotoramaItem.data('fotorama').activeFrame.$stageFrame.removeClass(this.PVLOADING);\n        },\n\n        /**\n         * @param {Event} event\n         * @private\n         */\n        _clickHandler: function (event) {\n            var type;\n\n            if ($(event.target).hasClass(this.VU) && $(event.target).find('iframe').length === 0) {\n                $(event.target).removeClass(this.VU);\n                type = $(event.target).find('.' + this.PV).data('type');\n\n                if (type === this.VI) {\n                    $(event.target).find('.' + this.PV).productVideoLoader();\n                } else if (type === this.VI) {\n                    this._showLoader();\n                    this.onVimeoJSFramework = function () {\n                        $(event.target).find('.' + this.PV).productVideoLoader();\n                        this._hideLoader();\n                    }.bind(this);\n                } else {\n                    $(event.target).find('.' + this.PV).productVideoLoader();\n                }\n\n                $('.' + this.FTAR).addClass(this.isFullscreen ? 'fotorama__arr--shown' : 'fotorama__arr--hidden');\n                $('.' + this.FTVC).addClass('fotorama-show-control');\n            }\n        },\n\n        /**\n         * Handle base video\n         * @param {Object} fotorama\n         * @param {Number} srcNumber\n         * @private\n         */\n        _handleBaseVideo: function (fotorama, srcNumber) {\n            var videoData = this.options.videoData,\n                activeIndex = fotorama.activeIndex,\n                number = parseInt(srcNumber, 10),\n                activeIndexIsBase = videoData[activeIndex];\n\n            if (!this.Base) {\n                return;\n            }\n\n            if (activeIndexIsBase && number === 1 && $(window).width() > this.MobileMaxWidth) {\n                setTimeout($.proxy(function () {\n                    fotorama.requestFullScreen();\n                    this.fotoramaItem.data('fotorama').activeFrame.$stageFrame[0].trigger('click');\n                    this.Base = false;\n                }, this), 50);\n            }\n        },\n\n        /**\n         * Destroy video player\n         * @param {jQuery} $wrapper\n         * @param {jQuery} current\n         * @param {bool} close\n         * @private\n         */\n        _unloadVideoPlayer: function ($wrapper, current, close) {\n            var self = this;\n\n            if (!$wrapper) {\n                return;\n            }\n\n            $wrapper.find('.' + this.PVLOADED).removeClass(this.PVLOADED);\n            this._hideLoader();\n\n            $wrapper.find('.' + this.PV).each(function () {\n                var $item = $(this).parent(),\n                    cloneVideoDiv,\n                    iframeElement = $(this).find('iframe'),\n                    currentIndex,\n                    itemIndex;\n\n                if (iframeElement.length === 0) {\n                    return;\n                }\n\n                currentIndex = current.activeFrame.$stageFrame.index();\n                itemIndex = $item.index();\n\n                if (currentIndex === itemIndex && !close) {\n                    return;\n                }\n\n                if (currentIndex !== itemIndex && close) {\n                    return;\n                }\n\n                iframeElement.remove();\n                cloneVideoDiv = $(this).clone();\n                $(this).remove();\n                $item.append(cloneVideoDiv);\n                $item.addClass(self.VU);\n\n                self._hideCloseVideo();\n                self._hideVideoArrows();\n\n                if (self.isFullscreen && !self.fotoramaItem.data('fotorama').options.fullscreen.arrows) {\n                    if ($('.' + self.FTAR + '--prev').is(':focus') || $('.' + self.FTAR + '--next').is(':focus')) {\n                        $(self.FTCF).trigger('focus');\n                    }\n                }\n            });\n        }\n    });\n\n    return $.mage.AddFotoramaVideoEvents;\n});\n","Magento_ProductVideo/js/load-player.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n @version 0.0.1\n @requires jQuery & jQuery UI\n */\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget',\n    'vimeoWrapper'\n], function ($) {\n    'use strict';\n\n    var videoRegister = {\n        _register: {},\n\n        /**\n         * Checks, if api is already registered\n         *\n         * @param {String} api\n         * @returns {bool}\n         */\n        isRegistered: function (api) {\n            return this._register[api] !== undefined;\n        },\n\n        /**\n         * Checks, if api is loaded\n         *\n         * @param {String} api\n         * @returns {bool}\n         */\n        isLoaded: function (api) {\n            return this._register[api] !== undefined && this._register[api] === true;\n        },\n\n        /**\n         * Register new video api\n         * @param {String} api\n         * @param {bool} loaded\n         */\n        register: function (api, loaded) {\n            loaded = loaded || false;\n            this._register[api] = loaded;\n        }\n    };\n\n    $.widget('mage.productVideoLoader', {\n\n        /**\n         * @private\n         */\n        _create: function () {\n            switch (this.element.data('type')) {\n                case 'youtube':\n                    this.element.videoYoutube();\n                    this._player = this.element.data('mageVideoYoutube');\n                    break;\n\n                case 'vimeo':\n                    this.element.videoVimeo();\n                    this._player = this.element.data('mageVideoVimeo');\n                    break;\n                default:\n                    throw {\n                        name: 'Video Error',\n                        message: 'Unknown video type',\n\n                        /**\n                         * join name with message\n                         */\n                        toString: function () {\n                            return this.name + ': ' + this.message;\n                        }\n                    };\n            }\n        },\n\n        /**\n         * Initializes variables\n         * @private\n         */\n        _initialize: function () {\n            this._params = this.element.data('params') || {};\n            this._code = this.element.data('code');\n            this._width = this.element.data('width');\n            this._height = this.element.data('height');\n            this._autoplay = !!this.element.data('autoplay');\n            this._playing = this._autoplay || false;\n            this._loop = this.element.data('loop');\n            this._rel = this.element.data('related');\n            this.useYoutubeNocookie = this.element.data('youtubenocookie') || false;\n\n            this._responsive = this.element.data('responsive') !== false;\n\n            if (this._responsive === true) {\n                this.element.addClass('responsive');\n            }\n\n            this._calculateRatio();\n        },\n\n        /**\n         * Abstract play command\n         */\n        play: function () {\n            this._player.play();\n        },\n\n        /**\n         * Abstract pause command\n         */\n        pause: function () {\n            this._player.pause();\n        },\n\n        /**\n         * Abstract stop command\n         */\n        stop: function () {\n            this._player.stop();\n        },\n\n        /**\n         * Abstract playing command\n         */\n        playing: function () {\n            return this._player.playing();\n        },\n\n        /**\n         * Calculates ratio for responsive videos\n         * @private\n         */\n        _calculateRatio: function () {\n            if (!this._responsive) {\n                return;\n            }\n            this.element.css('paddingBottom', this._height / this._width * 100 + '%');\n        }\n    });\n\n    $.widget('mage.videoYoutube', $.mage.productVideoLoader, {\n\n        /**\n         * Initialization of the Youtube widget\n         * @private\n         */\n        _create: function () {\n            var self = this;\n\n            this._initialize();\n\n            this.element.append('<div></div>');\n\n            this._on(window, {\n\n                /**\n                 * Handle event\n                 */\n                'youtubeapiready': function () {\n                    var host = 'https://www.youtube.com';\n\n                    if (self.useYoutubeNocookie) {\n                        host = 'https://www.youtube-nocookie.com';\n                    }\n\n                    if (self._player !== undefined) {\n                        return;\n                    }\n                    self._autoplay = true;\n\n                    if (self._autoplay) {\n                        self._params.autoplay = 1;\n                    }\n\n                    if (!self._rel) {\n                        self._params.rel = 0;\n                    }\n\n                    self._player = new window.YT.Player(self.element.children(':first')[0], {\n                        height: self._height,\n                        width: self._width,\n                        videoId: self._code,\n                        playerVars: self._params,\n                        host: host,\n                        events: {\n\n                            /**\n                             * Get duration\n                             */\n                            'onReady': function onPlayerReady() {\n                                self._player.getDuration();\n                                self.element.closest('.fotorama__stage__frame')\n                                    .addClass('fotorama__product-video--loaded');\n                            },\n\n                            /**\n                             * Event observer\n                             */\n                            onStateChange: function (data) {\n                                switch (window.parseInt(data.data, 10)) {\n                                    case 1:\n                                        self._playing = true;\n                                        break;\n                                    default:\n                                        self._playing = false;\n                                        break;\n                                }\n\n                                self._trigger('statechange', {}, data);\n\n                                if (data.data === window.YT.PlayerState.ENDED && self._loop) {\n                                    self._player.playVideo();\n                                }\n                            }\n                        }\n\n                    });\n                }\n            });\n\n            this._loadApi();\n        },\n\n        /**\n         * Loads Youtube API and triggers event, when loaded\n         * @private\n         */\n        _loadApi: function () {\n            var element,\n                scriptTag;\n\n            if (videoRegister.isRegistered('youtube')) {\n                if (videoRegister.isLoaded('youtube')) {\n                    $(window).trigger('youtubeapiready');\n                }\n\n                return;\n            }\n\n            // if script already loaded by other library\n            if (window.YT) {\n                videoRegister.register('youtube', true);\n                $(window).trigger('youtubeapiready');\n\n                return;\n            }\n            videoRegister.register('youtube');\n\n            element = document.createElement('script');\n            scriptTag = document.getElementsByTagName('script')[0];\n\n            element.async = true;\n            element.src = 'https://www.youtube.com/iframe_api';\n            scriptTag.parentNode.insertBefore(element, scriptTag);\n\n            /**\n             * Event observe and handle\n             */\n            window.onYouTubeIframeAPIReady = function () {\n                $(window).trigger('youtubeapiready');\n                videoRegister.register('youtube', true);\n            };\n        },\n\n        /**\n         * Play command for Youtube\n         */\n        play: function () {\n            this._player.playVideo();\n            this._playing = true;\n        },\n\n        /**\n         * Pause command for Youtube\n         */\n        pause: function () {\n            this._player.pauseVideo();\n            this._playing = false;\n        },\n\n        /**\n         * Stop command for Youtube\n         */\n        stop: function () {\n            this._player.stopVideo();\n            this._playing = false;\n        },\n\n        /**\n         * Playing command for Youtube\n         */\n        playing: function () {\n            return this._playing;\n        },\n\n        /**\n         * stops and unloads player\n         * @private\n         */\n        _destroy: function () {\n            this.stop();\n        }\n    });\n\n    $.widget('mage.videoVimeo', $.mage.productVideoLoader, {\n\n        /**\n         * Initialize the Vimeo widget\n         * @private\n         */\n        _create: function () {\n            var timestamp,\n                additionalParams = '',\n                src,\n                id;\n\n            this._initialize();\n            timestamp = new Date().getTime();\n            this._autoplay = true;\n\n            if (this._autoplay) {\n                additionalParams += '&autoplay=1';\n            }\n\n            if (this._loop) {\n                additionalParams += '&loop=1';\n            }\n            src = 'https://player.vimeo.com/video/' +\n                this._code + '?api=1&player_id=vimeo' +\n                this._code +\n                timestamp +\n                additionalParams;\n            id = 'vimeo' + this._code + timestamp;\n            this.element.append(\n                $('<iframe></iframe>')\n                    .attr('frameborder', 0)\n                    .attr('id', id)\n                    .attr('width', this._width)\n                    .attr('height', this._height)\n                    .attr('src', src)\n                    .attr('webkitallowfullscreen', '')\n                    .attr('mozallowfullscreen', '')\n                    .attr('allowfullscreen', '')\n                    .attr('referrerPolicy', 'origin')\n                    .attr('allow', 'autoplay')\n            );\n\n            /* eslint-disable no-undef */\n            this._player = new Vimeo.Player(this.element.children(':first')[0]);\n\n            this._player.ready().then(function () {\n                $('#' + id).closest('.fotorama__stage__frame').addClass('fotorama__product-video--loaded');\n            });\n        },\n\n        /**\n         * Play command for Vimeo\n         */\n        play: function () {\n            this._player.play();\n            this._playing = true;\n        },\n\n        /**\n         * Pause command for Vimeo\n         */\n        pause: function () {\n            this._player.pause();\n            this._playing = false;\n        },\n\n        /**\n         * Stop command for Vimeo\n         */\n        stop: function () {\n            this._player.unload();\n            this._playing = false;\n        },\n\n        /**\n         * Playing command for Vimeo\n         */\n        playing: function () {\n            return this._playing;\n        }\n    });\n});\n","Magento_ReCaptchaCheckout/js/model/place-order-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* eslint-disable max-nested-callbacks */\n\ndefine([\n    'jquery',\n    'mage/utils/wrapper',\n    'Magento_ReCaptchaWebapiUi/js/webapiReCaptchaRegistry'\n], function ($, wrapper, recaptchaRegistry) {\n    'use strict';\n\n    return function (placeOrder) {\n        return wrapper.wrap(placeOrder, function (originalAction, serviceUrl, payload, messageContainer) {\n            var recaptchaDeferred;\n\n            if (recaptchaRegistry.triggers.hasOwnProperty('recaptcha-checkout-place-order')) {\n                //ReCaptcha is present for checkout\n                recaptchaDeferred = $.Deferred();\n                recaptchaRegistry.addListener('recaptcha-checkout-place-order', function (token) {\n                    //Add reCaptcha value to place-order request and resolve deferred with the API call results\n                    payload.xReCaptchaValue = token;\n                    originalAction(serviceUrl, payload, messageContainer).done(function () {\n                        recaptchaDeferred.resolve.apply(recaptchaDeferred, arguments);\n                    }).fail(function () {\n                        recaptchaDeferred.reject.apply(recaptchaDeferred, arguments);\n                    });\n                });\n                //Trigger ReCaptcha validation\n                recaptchaRegistry.triggers['recaptcha-checkout-place-order']();\n\n                return recaptchaDeferred;\n            }\n\n            //No ReCaptcha, just sending the request\n            return originalAction(serviceUrl, payload, messageContainer);\n        });\n    };\n});\n","Magento_ReCaptchaCheckoutSalesRule/js/checkout-sales-rule.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/* global grecaptcha */\ndefine(\n    [\n        'Magento_ReCaptchaWebapiUi/js/webapiReCaptcha',\n        'Magento_ReCaptchaWebapiUi/js/webapiReCaptchaRegistry',\n        'jquery',\n        'Magento_SalesRule/js/action/set-coupon-code',\n        'Magento_SalesRule/js/action/cancel-coupon',\n        'Magento_Checkout/js/model/quote',\n        'ko'\n    ], function (Component, recaptchaRegistry, $, setCouponCodeAction, cancelCouponAction, quote, ko) {\n        'use strict';\n\n        var totals = quote.getTotals(),\n            couponCode = ko.observable(null),\n            isApplied;\n\n        if (totals()) {\n            couponCode(totals()['coupon_code']);\n        }\n        //Captcha can only be required for adding a coupon so we need to know if one was added already.\n        isApplied = ko.observable(couponCode() != null);\n\n        return Component.extend({\n\n            /**\n             * Initialize parent form.\n             *\n             * @param {Object} parentForm\n             * @param {String} widgetId\n             */\n            initParentForm: function (parentForm, widgetId) {\n                var self = this,\n                    xRecaptchaValue,\n                    captchaId = this.getReCaptchaId();\n\n                this._super();\n\n                if (couponCode() != null) {\n                    if (isApplied) {\n                        self.validateReCaptcha(true);\n                        $('#' + captchaId).hide();\n                    }\n                }\n\n                if (recaptchaRegistry.triggers.hasOwnProperty('recaptcha-checkout-coupon-apply')) {\n                    recaptchaRegistry.addListener('recaptcha-checkout-coupon-apply', function (token) {\n                        //Add reCaptcha value to coupon request\n                        xRecaptchaValue = token;\n                    });\n                }\n\n                setCouponCodeAction.registerDataModifier(function (headers) {\n                    headers['X-ReCaptcha'] = xRecaptchaValue;\n                });\n\n                if (self.getIsInvisibleRecaptcha()) {\n                    grecaptcha.execute(widgetId);\n                    self.validateReCaptcha(true);\n                }\n                //Refresh reCaptcha after failed request.\n                setCouponCodeAction.registerFailCallback(function () {\n                    if (self.getIsInvisibleRecaptcha()) {\n                        grecaptcha.execute(widgetId);\n                        self.validateReCaptcha(true);\n                    } else {\n                        self.validateReCaptcha(false);\n                        grecaptcha.reset(widgetId);\n                        $('#' + captchaId).show();\n                    }\n                });\n                //Hide captcha when a coupon has been applied.\n                setCouponCodeAction.registerSuccessCallback(function () {\n                    self.validateReCaptcha(true);\n                    $('#' + captchaId).hide();\n                });\n                //Show captcha again if it was canceled.\n                cancelCouponAction.registerSuccessCallback(function () {\n                    self.validateReCaptcha(false);\n                    grecaptcha.reset(widgetId);\n                    $('#' + captchaId).show();\n                });\n            }\n        });\n    });\n","Magento_ReCaptchaFrontendUi/js/nonInlineReCaptchaRenderer.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global grecaptcha */\ndefine([\n    'jquery',\n    'jquery/z-index'\n], function ($) {\n    'use strict';\n\n    var reCaptchaEntities = [],\n        initialized = false,\n        rendererRecaptchaId = 'recaptcha-invisible',\n        rendererReCaptcha = null;\n\n    return {\n        /**\n         * Add reCaptcha entity to checklist.\n         *\n         * @param {jQuery} reCaptchaEntity\n         * @param {Object} parameters\n         */\n        add: function (reCaptchaEntity, parameters) {\n            if (!initialized) {\n                this.init();\n                grecaptcha.render(rendererRecaptchaId, parameters);\n                setInterval(this.resolveVisibility, 100);\n                initialized = true;\n            }\n\n            reCaptchaEntities.push(reCaptchaEntity);\n        },\n\n        /**\n         * Show additional reCaptcha instance if any other should be visible, otherwise hide it.\n         */\n        resolveVisibility: function () {\n            reCaptchaEntities.some(function (entity) {\n                return entity.is(':visible') &&\n                    // 900 is some magic z-index value of modal popups.\n                    (entity.closest('[data-role=\\'modal\\']').length === 0 || entity.zIndex() > 900);\n            }) ? rendererReCaptcha.show() : rendererReCaptcha.hide();\n        },\n\n        /**\n         * Initialize additional reCaptcha instance.\n         */\n        init: function () {\n            rendererReCaptcha = $('<div/>', {\n                'id': rendererRecaptchaId\n            });\n            rendererReCaptcha.hide();\n            $('body').append(rendererReCaptcha);\n        }\n    };\n});\n","Magento_ReCaptchaFrontendUi/js/reCaptcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/* global grecaptcha */\ndefine(\n    [\n        'uiComponent',\n        'jquery',\n        'ko',\n        'underscore',\n        'Magento_ReCaptchaFrontendUi/js/registry',\n        'Magento_ReCaptchaFrontendUi/js/reCaptchaScriptLoader',\n        'Magento_ReCaptchaFrontendUi/js/nonInlineReCaptchaRenderer'\n    ], function (Component, $, ko, _, registry, reCaptchaLoader, nonInlineReCaptchaRenderer) {\n        'use strict';\n\n        return Component.extend({\n\n            defaults: {\n                template: 'Magento_ReCaptchaFrontendUi/reCaptcha',\n                reCaptchaId: 'recaptcha'\n            },\n\n            /**\n             * @inheritdoc\n             */\n            initialize: function () {\n                this._super();\n                this._loadApi();\n            },\n\n            /**\n             * Loads recaptchaapi API and triggers event, when loaded\n             * @private\n             */\n            _loadApi: function () {\n                if (this._isApiRegistered !== undefined) {\n                    if (this._isApiRegistered === true) {\n                        $(window).trigger('recaptchaapiready');\n                    }\n\n                    return;\n                }\n                this._isApiRegistered = false;\n\n                // global function\n                window.globalOnRecaptchaOnLoadCallback = function () {\n                    this._isApiRegistered = true;\n                    $(window).trigger('recaptchaapiready');\n                }.bind(this);\n\n                reCaptchaLoader.addReCaptchaScriptTag();\n            },\n\n            /**\n             * Checking that reCAPTCHA is invisible type\n             * @returns {Boolean}\n             */\n            getIsInvisibleRecaptcha: function () {\n                if (this.settings ===\n\n                    void 0) {\n                    return false;\n                }\n\n                return this.settings.invisible;\n            },\n\n            /**\n             * reCAPTCHA callback\n             * @param {String} token\n             */\n            reCaptchaCallback: function (token) {\n                if (this.getIsInvisibleRecaptcha()) {\n                    this.tokenField.value = token;\n                    this.$parentForm.submit();\n                }\n            },\n\n            /**\n             * Initialize reCAPTCHA after first rendering\n             */\n            initCaptcha: function () {\n                var $parentForm,\n                    $wrapper,\n                    $reCaptcha,\n                    widgetId,\n                    parameters;\n\n                if (this.captchaInitialized || this.settings ===\n\n                    void 0) {\n                    return;\n                }\n\n                this.captchaInitialized = true;\n\n                /*\n                 * Workaround for data-bind issue:\n                 * We cannot use data-bind to link a dynamic id to our component\n                 * See:\n                 * https://stackoverflow.com/questions/46657573/recaptcha-the-bind-parameter-must-be-an-element-or-id\n                 *\n                 * We create a wrapper element with a wrapping id and we inject the real ID with jQuery.\n                 * In this way we have no data-bind attribute at all in our reCAPTCHA div\n                 */\n                $wrapper = $('#' + this.getReCaptchaId() + '-wrapper');\n                $reCaptcha = $wrapper.find('.g-recaptcha');\n                $reCaptcha.attr('id', this.getReCaptchaId());\n\n                $parentForm = $wrapper.parents('form');\n\n                if (this.settings === undefined) {\n\n                    return;\n                }\n\n                parameters = _.extend(\n                    {\n                        'callback': function (token) { // jscs:ignore jsDoc\n                            this.reCaptchaCallback(token);\n                            this.validateReCaptcha(true);\n                        }.bind(this),\n                        'expired-callback': function () {\n                            this.validateReCaptcha(false);\n                        }.bind(this)\n                    },\n                    this.settings.rendering\n                );\n\n                if (parameters.size === 'invisible' && parameters.badge !== 'inline') {\n                    nonInlineReCaptchaRenderer.add($reCaptcha, parameters);\n                }\n\n                // eslint-disable-next-line no-undef\n                widgetId = grecaptcha.render(this.getReCaptchaId(), parameters);\n                this.initParentForm($parentForm, widgetId);\n\n                registry.ids.push(this.getReCaptchaId());\n                registry.captchaList.push(widgetId);\n                registry.tokenFields.push(this.tokenField);\n\n            },\n\n            /**\n             * Initialize parent form.\n             *\n             * @param {Object} parentForm\n             * @param {String} widgetId\n             */\n            initParentForm: function (parentForm, widgetId) {\n                var listeners;\n\n                if (this.getIsInvisibleRecaptcha() && parentForm.length > 0) {\n                    parentForm.submit(function (event) {\n                        if (!this.tokenField.value) {\n                            // eslint-disable-next-line no-undef\n                            grecaptcha.execute(widgetId);\n                            event.preventDefault(event);\n                            event.stopImmediatePropagation();\n                        }\n                    }.bind(this));\n\n                    // Move our (last) handler topmost. We need this to avoid submit bindings with ko.\n                    listeners = $._data(parentForm[0], 'events').submit;\n                    listeners.unshift(listeners.pop());\n\n                    // Create a virtual token field\n                    this.tokenField = $('<input type=\"text\" name=\"token\" style=\"display: none\" />')[0];\n                    this.$parentForm = parentForm;\n                    parentForm.append(this.tokenField);\n                } else {\n                    this.tokenField = null;\n                }\n                if ($('#send2').length > 0) {$('#send2').prop('disabled', false);}\n            },\n\n            /**\n             * Validates reCAPTCHA\n             * @param {*} state\n             * @returns {jQuery}\n             */\n            validateReCaptcha: function (state) {\n                if (!this.getIsInvisibleRecaptcha()) {\n                    return $(document).find('input[type=checkbox].required-captcha').prop('checked', state);\n                }\n            },\n\n            /**\n             * Render reCAPTCHA\n             */\n            renderReCaptcha: function () {\n                if (window.grecaptcha && window.grecaptcha.render) { // Check if reCAPTCHA is already loaded\n                    this.initCaptcha();\n                } else { // Wait for reCAPTCHA to be loaded\n                    $(window).on('recaptchaapiready', function () {\n                        this.initCaptcha();\n                    }.bind(this));\n                }\n            },\n\n            /**\n             * Get reCAPTCHA ID\n             * @returns {String}\n             */\n            getReCaptchaId: function () {\n                return this.reCaptchaId;\n            }\n        });\n    });\n","Magento_ReCaptchaFrontendUi/js/reCaptchaScriptLoader.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    var scriptTagAdded = false;\n\n    return {\n        /**\n         * Add script tag. Script tag should be added once\n         */\n        addReCaptchaScriptTag: function () {\n            var element, scriptTag;\n\n            if (!scriptTagAdded) {\n                element = document.createElement('script');\n                scriptTag = document.getElementsByTagName('script')[0];\n\n                element.async = true;\n                element.src = 'https://www.google.com/recaptcha/api.js' +\n                    '?onload=globalOnRecaptchaOnLoadCallback&render=explicit';\n\n                scriptTag.parentNode.insertBefore(element, scriptTag);\n                scriptTagAdded = true;\n            }\n        }\n    };\n});\n","Magento_ReCaptchaFrontendUi/js/registry.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['ko'], function (ko) {\n    'use strict';\n\n    return {\n        ids: ko.observableArray([]),\n        captchaList: ko.observableArray([]),\n        tokenFields: ko.observableArray([])\n    };\n});\n","Magento_ReCaptchaFrontendUi/js/ui-messages-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['Magento_ReCaptchaFrontendUi/js/registry'], function (registry) {\n    'use strict';\n\n    return function (originalComponent) {\n        return originalComponent.extend({\n            /**\n             * Initialize reset on messages\n             * @returns {initialize}\n             */\n            initialize: function () {\n                this._super();\n\n                this.messageContainer.errorMessages.subscribe(function () {\n                    var\n                        i,\n                        captchaList = registry.captchaList(),\n                        tokenFieldsList = registry.tokenFields();\n\n                    for (i = 0; i < captchaList.length; i++) {\n                        // eslint-disable-next-line no-undef\n                        grecaptcha.reset(captchaList[i]);\n\n                        if (tokenFieldsList[i]) {\n                            tokenFieldsList[i].value = '';\n                        }\n                    }\n                }, null, 'arrayChange');\n\n                return this;\n            }\n        });\n    };\n});\n","Magento_ReCaptchaPaypal/js/payflowpro-method-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'Magento_Checkout/js/model/payment/additional-validators'\n], function ($, additionalValidators) {\n    'use strict';\n\n    return function (originalComponent) {\n        return originalComponent.extend({\n            /**\n             * Initializes reCaptcha\n             */\n            placeOrder: function () {\n                var original = this._super.bind(this),\n                    // jscs:disable requireCamelCaseOrUpperCaseIdentifiers\n                    isEnabledForPaypal = window.checkoutConfig.recaptcha_paypal,\n                    // jscs:enable requireCamelCaseOrUpperCaseIdentifiers\n                    paymentFormSelector = $('#co-payment-form'),\n                    startEvent = 'captcha:startExecute',\n                    endEvent = 'captcha:endExecute';\n\n                if (!this.validateHandler() || !additionalValidators.validate() || !isEnabledForPaypal) {\n                    return original();\n                }\n\n                paymentFormSelector.off(endEvent).on(endEvent, function () {\n                    original();\n                    paymentFormSelector.off(endEvent);\n                });\n\n                paymentFormSelector.trigger(startEvent);\n            }\n        });\n    };\n});\n","Magento_ReCaptchaPaypal/js/reCaptchaPaypal.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(\n    [\n        'Magento_ReCaptchaFrontendUi/js/reCaptcha',\n        'jquery'\n    ],\n    function (Component, $) {\n        'use strict';\n\n        return Component.extend({\n\n            /**\n             * Recaptcha callback\n             * @param {String} token\n             */\n            reCaptchaCallback: function (token) {\n                this.tokenField.value = token;\n                this.$parentForm.trigger('captcha:endExecute');\n            },\n\n            /**\n             * Initialize parent form.\n             *\n             * @param {Object} parentForm\n             * @param {String} widgetId\n             */\n            initParentForm: function (parentForm, widgetId) {\n                var me = this;\n\n                parentForm.on('captcha:startExecute', function (event) {\n                    if (!me.tokenField.value && me.getIsInvisibleRecaptcha()) {\n                        // eslint-disable-next-line no-undef\n                        grecaptcha.execute(widgetId);\n                        event.preventDefault(event);\n                        event.stopImmediatePropagation();\n                    } else {\n                        me.$parentForm.trigger('captcha:endExecute');\n                    }\n                });\n\n                // Create a virtual token field\n                this.tokenField = $('<input type=\"text\" name=\"token\" style=\"display: none\" />')[0];\n                this.$parentForm = parentForm;\n                parentForm.append(this.tokenField);\n            }\n        });\n    }\n);\n","Magento_ReCaptchaStorePickup/js/reCaptchaStorePickup.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine(['Magento_ReCaptchaFrontendUi/js/reCaptcha'], function (reCaptcha) {\n    'use strict';\n\n    return reCaptcha.extend({\n\n        /**\n         * @inheritdoc\n         */\n        renderReCaptcha: function () {\n            this.captchaInitialized = false;\n            this._super();\n        }\n    });\n});\n","Magento_ReCaptchaWebapiUi/js/jquery-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n// jscs:disable requireDotNotation\n\ndefine([\n    'mage/utils/wrapper'\n], function (wrapper) {\n    'use strict';\n\n    return function (jQuery) {\n        jQuery.ajax = wrapper.wrapSuper(jQuery.ajax, function () {\n            //Moving ReCaptcha value from payload to the header for requests to web API\n            var settings,\n                payload;\n\n            if (arguments.length !== 0) {\n                settings = arguments.length === 1 ? arguments[0] : arguments[1];\n            }\n\n            if (settings && settings.hasOwnProperty('data')) {\n                //The request has a body, trying to parse JSON data\n                try {\n                    payload = JSON.parse(settings.data);\n                } catch (e) {\n                    //Not JSON\n                }\n            }\n\n            if (payload && payload.hasOwnProperty('xReCaptchaValue')) {\n                if (!settings.hasOwnProperty('headers')) {\n                    settings.headers = {};\n                }\n                settings.headers['X-ReCaptcha'] = payload.xReCaptchaValue;\n                delete payload['xReCaptchaValue'];\n                settings.data = JSON.stringify(payload);\n            }\n\n            return this._super.apply(this, arguments);\n        });\n\n        return jQuery;\n    };\n});\n","Magento_ReCaptchaWebapiUi/js/webapiReCaptcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n// jscs:disable jsDoc\n\n/* global grecaptcha */\ndefine(\n    [\n        'Magento_ReCaptchaFrontendUi/js/reCaptcha',\n        'Magento_ReCaptchaWebapiUi/js/webapiReCaptchaRegistry'\n    ],\n    function (Component, registry) {\n        'use strict';\n\n        return Component.extend({\n            defaults: {\n                autoTrigger: false\n            },\n\n            /**\n             * Provide the token to the registry.\n             *\n             * @param {String} token\n             */\n            reCaptchaCallback: function (token) {\n                //Make the token retrievable in other UI components.\n                registry.tokens[this.getReCaptchaId()] = token;\n\n                if (typeof registry._listeners[this.getReCaptchaId()] !== 'undefined') {\n                    registry._listeners[this.getReCaptchaId()](token);\n                }\n            },\n\n            /**\n             * Register this ReCaptcha.\n             *\n             * @param {Object} parentForm\n             * @param {String} widgetId\n             */\n            initParentForm: function (parentForm, widgetId) {\n                var self = this,\n                    trigger;\n\n                if (this.getIsInvisibleRecaptcha()) {\n                    trigger = function () {\n                        grecaptcha.execute(widgetId);\n                    };\n                } else {\n                    trigger = function () {\n                        self.reCaptchaCallback(grecaptcha.getResponse(widgetId));\n                    };\n                }\n\n                if (this.autoTrigger) {\n                    //Validate ReCaptcha when initiated\n                    trigger();\n                    registry.triggers[this.getReCaptchaId()] = new Function();\n                } else {\n                    registry.triggers[this.getReCaptchaId()] = trigger;\n                }\n                this.tokenField = null;\n            }\n        });\n    }\n);\n","Magento_ReCaptchaWebapiUi/js/webapiReCaptchaRegistry.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return {\n        /**\n         * recaptchaId: token map.\n         *\n         * Tokens for already verified recaptcha.\n         */\n        tokens: {},\n\n        /**\n         * recaptchaId: triggerFn map.\n         *\n         * Call a trigger to initiate a recaptcha verification.\n         */\n        triggers: {},\n\n        /**\n         * recaptchaId: callback map\n         */\n        _listeners: {},\n\n        /**\n         * Add a listener to when the ReCaptcha finishes verification\n         * @param {String} id - ReCaptchaId\n         * @param {Function} func - Will be called back with the token\n         */\n        addListener: function (id, func) {\n            if (this.tokens.hasOwnProperty(id)) {\n                func(this.tokens[id]);\n            } else {\n                this._listeners[id] = func;\n            }\n        }\n    };\n});\n","Magento_Reports/js/recently-viewed.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.recentlyViewedProducts', {\n        options: {\n            localStorageKey: 'recently-viewed-products',\n            productBlock: '#widget_viewed_item',\n            viewedContainer: 'ol'\n        },\n\n        /**\n         * Bind events to the appropriate handlers.\n         * @private\n         */\n        _create: function () {\n            var productHtml = $(this.options.productBlock).html(),\n                productSku = $(this.options.productBlock).data('sku'),\n                products = JSON.parse(window.localStorage.getItem(this.options.localStorageKey)),\n                productsLength, maximum, showed, index;\n\n            if (products) {\n                productsLength = products.sku.length;\n                maximum = $(this.element).data('count');\n                showed = 0;\n\n                for (index = 0; index <= productsLength; index++) {\n                    if (products.sku[index] == productSku || showed >= maximum) { //eslint-disable-line\n                        products.sku.splice(index, 1);\n                        products.html.splice(index, 1);\n                    } else {\n                        $(this.element).find(this.options.viewedContainer).append(products.html[index]);\n                        $(this.element).show();\n                        showed++;\n                    }\n                }\n                $(this.element).find(this.options.productBlock).show();\n            } else {\n                products = {};\n                products.sku = [];\n                products.html = [];\n            }\n            products.sku.unshift(productSku);\n            products.html.unshift(productHtml);\n            window.localStorage.setItem(this.options.localStorageKey, JSON.stringify(products));\n        }\n    });\n\n    return $.mage.recentlyViewedProducts;\n});\n","Magento_Review/js/error-placement.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'mage/mage'\n], function ($) {\n    'use strict';\n\n    return function (config, element) {\n        $(element).mage('validation', {\n            /** @inheritdoc */\n            errorPlacement: function (error, el) {\n\n                if (el.parents('#product-review-table').length) {\n                    $('#product-review-table').siblings(this.errorElement + '.' + this.errorClass).remove();\n                    $('#product-review-table').after(error);\n                } else {\n                    el.after(error);\n                }\n            }\n        });\n    };\n});\n","Magento_Review/js/process-reviews.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'tabs',\n    'collapsible'\n], function ($) {\n    'use strict';\n\n    /**\n     * @param {String} url\n     * @param {*} fromPages\n     */\n    function processReviews(url, fromPages) {\n        $.ajax({\n            url: url,\n            cache: true,\n            dataType: 'html',\n            showLoader: false,\n            loaderContext: $('.product.data.items')\n        }).done(function (data) {\n            $('#product-review-container').html(data).trigger('contentUpdated');\n            $('[data-role=\"product-review\"] .pages a').each(function (index, element) {\n                $(element).on('click', function (event) { //eslint-disable-line max-nested-callbacks\n                    processReviews($(element).attr('href'), true);\n                    event.preventDefault();\n                });\n            });\n        }).always(function () {\n            if (fromPages == true) { //eslint-disable-line eqeqeq\n                $('html, body').animate({\n                    scrollTop: $('#reviews').offset().top - 50\n                }, 300);\n            }\n        });\n    }\n\n    return function (config) {\n        var reviewTab = $(config.reviewsTabSelector),\n            requiredReviewTabRole = 'tab';\n\n        if (reviewTab.attr('role') === requiredReviewTabRole && reviewTab.hasClass('active')) {\n            processReviews(config.productReviewUrl, location.hash === '#reviews');\n        } else {\n            reviewTab.one('beforeOpen', function () {\n                processReviews(config.productReviewUrl);\n            });\n        }\n\n        $(function () {\n            $('.product-info-main .reviews-actions a').on('click', function (event) {\n                var anchor, addReviewBlock;\n\n                event.preventDefault();\n                anchor = $(this).attr('href').replace(/^.*?(#|$)/, '');\n                addReviewBlock = $('#' + anchor);\n\n                if (addReviewBlock.length) {\n                    $('.product.data.items [data-role=\"content\"]').each(function (index) { //eslint-disable-line\n                        if (this.id == 'reviews') { //eslint-disable-line eqeqeq\n                            $('.product.data.items').tabs('activate', index);\n                        }\n                    });\n                    $('html, body').animate({\n                        scrollTop: addReviewBlock.offset().top - 50\n                    }, 300);\n                }\n\n            });\n        });\n    };\n});\n","Magento_Review/js/submit-review.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    return function (config, element) {\n        $(element).on('submit', function () {\n            if ($(this).valid()) {\n                $(this).find('.submit').attr('disabled', true);\n            }\n        });\n    };\n});\n","Magento_Review/js/validate-review.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery/validate',\n    'mage/translate'\n], function ($) {\n    'use strict';\n\n    $.validator.addMethod(\n        'rating-required', function (value) {\n            return value !== undefined;\n        }, $.mage.__('Please select one of each of the ratings above.'));\n});\n","Magento_Review/js/view/review.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Customer/js/customer-data',\n    'Magento_Customer/js/view/customer'\n], function (Component, customerData) {\n    'use strict';\n\n    return Component.extend({\n        /** @inheritdoc */\n        initialize: function () {\n            this._super();\n\n            this.review = customerData.get('review').extend({\n                disposableCustomerData: 'review'\n            });\n        },\n\n        /**\n         * @return {*}\n         */\n        nickname: function () {\n            return this.review().nickname || customerData.get('customer')().firstname;\n        }\n    });\n});\n","Magento_Sales/js/gift-message.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.giftMessage', {\n        options: {\n            rowPrefix: '#order-item-row-', // Selector prefix for item's row in the table.\n            linkPrefix: '#order-item-gift-message-link-', // Selector prefix for the 'Gift Message' link.\n            duration: 100, // Toggle duration.\n            expandedClass: 'expanded', // Class added/removed to/from the 'Gift Message' link.\n            expandedContentClass: 'expanded-content', // Class added/removed to/from the 'Gift Message' content.\n            lastClass: 'last' // Class added/removed to/from the last item's row in the products table.\n        },\n\n        /**\n         * Bind a click handler on the widget's element to toggle the gift message.\n         * @private\n         */\n        _create: function () {\n            this.element.on('click', $.proxy(this._toggleGiftMessage, this));\n        },\n\n        /**\n         * Toggle the display of the item's corresponding gift message.\n         * @private\n         * @param {jQuery.Event} event - Click event.\n         */\n        _toggleGiftMessage: function (event) {\n            var element = $(event.target), // Click target. The 'Gift Message' link or 'Close' button.\n                options = this.options, // Cached widget options.\n                itemId = element.data('item-id'), // The individual item's numeric id.\n                link = $(options.linkPrefix + itemId), // The 'Gift Message' expandable link.\n                row = $(options.rowPrefix + itemId), // The item's row in the products table.\n                region = $('#' + element.attr('aria-controls')); // The gift message container region.\n\n            region.toggleClass(options.expandedContentClass, options.duration, function () {\n                if (region.attr('aria-expanded') === 'true') {\n                    region.attr('aria-expanded', 'false');\n\n                    if (region.hasClass(options.lastClass)) {\n                        row.addClass(options.lastClass);\n                    }\n                } else {\n                    region.attr('aria-expanded', 'true');\n\n                    if (region.hasClass(options.lastClass)) {\n                        row.removeClass(options.lastClass);\n                    }\n                }\n                link.toggleClass(options.expandedClass);\n            });\n            event.preventDefault(); // Prevent event propagation and avoid going to the link's href.\n        }\n    });\n\n    return $.mage.giftMessage;\n});\n","Magento_Sales/js/orders-returns.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'jquery-ui-modules/widget'\n], function ($) {\n    'use strict';\n\n    $.widget('mage.ordersReturns', {\n        options: {\n            zipCode: '#oar-zip', // Search by zip code.\n            emailAddress: '#oar-email', // Search by email address.\n            searchType: '#quick-search-type-id' // Search element used for choosing between the two.\n        },\n\n        /** @inheritdoc */\n        _create: function () {\n            $(this.options.searchType).on('change', $.proxy(this._showIdentifyBlock, this)).trigger('change');\n        },\n\n        /**\n         * Show either the search by zip code option or the search by email address option.\n         * @private\n         * @param {jQuery.Event} e - Change event. Event target value is either 'zip' or 'email'.\n         */\n        _showIdentifyBlock: function (e) {\n            var value = $(e.target).val();\n\n            $(this.options.zipCode).toggle(value === 'zip');\n            $(this.options.emailAddress).toggle(value === 'email');\n        }\n    });\n\n    return $.mage.ordersReturns;\n});\n","Magento_Sales/js/view/last-ordered-items.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Customer/js/customer-data',\n    'underscore'\n], function (Component, customerData, _) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            isShowAddToCart: false\n        },\n\n        /** @inheritdoc */\n        initialize: function () {\n            this._super();\n            this.lastOrderedItems = customerData.get('last-ordered-items');\n            this.lastOrderedItems.subscribe(this.checkSalableItems.bind(this));\n            this.checkSalableItems();\n\n            return this;\n        },\n\n        /** @inheritdoc */\n        initObservable: function () {\n            this._super()\n                .observe('isShowAddToCart');\n\n            return this;\n        },\n\n        /**\n         * Check if items is_saleable and change add to cart button visibility.\n         */\n        checkSalableItems: function () {\n            var isShowAddToCart = _.some(this.lastOrderedItems().items, {\n                'is_saleable': true\n            });\n\n            this.isShowAddToCart(isShowAddToCart);\n        }\n    });\n});\n","Magento_SalesRule/js/action/cancel-coupon.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * Customer store credit(balance) application\n */\ndefine([\n    'jquery',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/resource-url-manager',\n    'Magento_Checkout/js/model/error-processor',\n    'Magento_SalesRule/js/model/payment/discount-messages',\n    'mage/storage',\n    'Magento_Checkout/js/action/get-payment-information',\n    'Magento_Checkout/js/model/totals',\n    'mage/translate',\n    'Magento_Checkout/js/model/full-screen-loader',\n    'Magento_Checkout/js/action/recollect-shipping-rates'\n], function ($, quote, urlManager, errorProcessor, messageContainer, storage, getPaymentInformationAction, totals, $t,\n  fullScreenLoader, recollectShippingRates\n) {\n    'use strict';\n\n    var successCallbacks = [],\n        action,\n        callSuccessCallbacks;\n\n    /**\n     * Execute callbacks when a coupon is successfully canceled.\n     */\n    callSuccessCallbacks = function () {\n        successCallbacks.forEach(function (callback) {\n            callback();\n        });\n    };\n\n    /**\n     * Cancel applied coupon.\n     *\n     * @param {Boolean} isApplied\n     * @returns {Deferred}\n     */\n    action =  function (isApplied) {\n        var quoteId = quote.getQuoteId(),\n            url = urlManager.getCancelCouponUrl(quoteId),\n            message = $t('Your coupon was successfully removed.');\n\n        messageContainer.clear();\n        fullScreenLoader.startLoader();\n\n        return storage.delete(\n            url,\n            false\n        ).done(function () {\n            var deferred = $.Deferred();\n\n            totals.isLoading(true);\n            recollectShippingRates();\n            getPaymentInformationAction(deferred);\n            $.when(deferred).done(function () {\n                isApplied(false);\n                totals.isLoading(false);\n                fullScreenLoader.stopLoader();\n                //Allowing to tap into coupon-cancel process.\n                callSuccessCallbacks();\n            });\n            messageContainer.addSuccessMessage({\n                'message': message\n            });\n        }).fail(function (response) {\n            totals.isLoading(false);\n            fullScreenLoader.stopLoader();\n            errorProcessor.process(response, messageContainer);\n        });\n    };\n\n    /**\n     * Callback for when the cancel-coupon process is finished.\n     *\n     * @param {Function} callback\n     */\n    action.registerSuccessCallback = function (callback) {\n        successCallbacks.push(callback);\n    };\n\n    return action;\n});\n","Magento_SalesRule/js/action/select-payment-method-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'mage/utils/wrapper',\n    'Magento_Checkout/js/model/quote',\n    'Magento_SalesRule/js/model/payment/discount-messages',\n    'Magento_Checkout/js/action/set-payment-information-extended',\n    'Magento_Checkout/js/action/get-totals',\n    'Magento_SalesRule/js/model/coupon'\n], function ($, wrapper, quote, messageContainer, setPaymentInformationExtended, getTotalsAction, coupon) {\n    'use strict';\n\n    return function (selectPaymentMethodAction) {\n\n        return wrapper.wrap(selectPaymentMethodAction, function (originalSelectPaymentMethodAction, paymentMethod) {\n\n            originalSelectPaymentMethodAction(paymentMethod);\n\n            if (paymentMethod === null) {\n                return;\n            }\n\n            $.when(\n                setPaymentInformationExtended(\n                    messageContainer,\n                    {\n                        method: paymentMethod.method\n                    },\n                    true\n                )\n            ).done(\n                function () {\n                    var deferred = $.Deferred(),\n\n                        /**\n                         * Update coupon form.\n                         */\n                        updateCouponCallback = function () {\n                            if (quote.totals() && !quote.totals()['coupon_code']) {\n                                coupon.setCouponCode('');\n                                coupon.setIsApplied(false);\n                            }\n                        };\n\n                    getTotalsAction([], deferred);\n                    $.when(deferred).done(updateCouponCallback);\n                }\n            );\n        });\n    };\n\n});\n","Magento_SalesRule/js/action/set-coupon-code.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * Customer store credit(balance) application\n */\ndefine([\n    'ko',\n    'jquery',\n    'Magento_Checkout/js/model/quote',\n    'Magento_Checkout/js/model/resource-url-manager',\n    'Magento_Checkout/js/model/error-processor',\n    'Magento_SalesRule/js/model/payment/discount-messages',\n    'mage/storage',\n    'mage/translate',\n    'Magento_Checkout/js/action/get-payment-information',\n    'Magento_Checkout/js/model/totals',\n    'Magento_Checkout/js/model/full-screen-loader',\n    'Magento_Checkout/js/action/recollect-shipping-rates'\n], function (ko, $, quote, urlManager, errorProcessor, messageContainer, storage, $t, getPaymentInformationAction,\n    totals, fullScreenLoader, recollectShippingRates\n) {\n    'use strict';\n\n    var dataModifiers = [],\n        successCallbacks = [],\n        failCallbacks = [],\n        action;\n\n    /**\n     * Apply provided coupon.\n     *\n     * @param {String} couponCode\n     * @param {Boolean}isApplied\n     * @returns {Deferred}\n     */\n    action = function (couponCode, isApplied) {\n        var quoteId = quote.getQuoteId(),\n            url = urlManager.getApplyCouponUrl(couponCode, quoteId),\n            message = $t('Your coupon was successfully applied.'),\n            data = {},\n            headers = {};\n\n        //Allowing to modify coupon-apply request\n        dataModifiers.forEach(function (modifier) {\n            modifier(headers, data);\n        });\n        fullScreenLoader.startLoader();\n\n        return storage.put(\n            url,\n            data,\n            false,\n            null,\n            headers\n        ).done(function (response) {\n            var deferred;\n\n            if (response) {\n                deferred = $.Deferred();\n\n                isApplied(true);\n                totals.isLoading(true);\n                recollectShippingRates();\n                getPaymentInformationAction(deferred);\n                $.when(deferred).done(function () {\n                    fullScreenLoader.stopLoader();\n                    totals.isLoading(false);\n                });\n                messageContainer.addSuccessMessage({\n                    'message': message\n                });\n                //Allowing to tap into apply-coupon process.\n                successCallbacks.forEach(function (callback) {\n                    callback(response);\n                });\n            }\n        }).fail(function (response) {\n            fullScreenLoader.stopLoader();\n            totals.isLoading(false);\n            errorProcessor.process(response, messageContainer);\n            //Allowing to tap into apply-coupon process.\n            failCallbacks.forEach(function (callback) {\n                callback(response);\n            });\n        });\n    };\n\n    /**\n     * Modifying data to be sent.\n     *\n     * @param {Function} modifier\n     */\n    action.registerDataModifier = function (modifier) {\n        dataModifiers.push(modifier);\n    };\n\n    /**\n     * When successfully added a coupon.\n     *\n     * @param {Function} callback\n     */\n    action.registerSuccessCallback = function (callback) {\n        successCallbacks.push(callback);\n    };\n\n    /**\n     * When failed to add a coupon.\n     *\n     * @param {Function} callback\n     */\n    action.registerFailCallback = function (callback) {\n        failCallbacks.push(callback);\n    };\n\n    return action;\n});\n","Magento_SalesRule/js/form/element/coupon-type.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'uiRegistry',\n    'Magento_Ui/js/form/element/select'\n], function (_, uiRegistry, select) {\n    'use strict';\n\n    return select.extend({\n\n        /**\n         * Hide fields on coupon tab\n         */\n        onUpdate: function () {\n\n            /* eslint-disable eqeqeq */\n            if (this.value() != this.displayOnlyForCouponType) {\n                uiRegistry.get('sales_rule_form.sales_rule_form.rule_information.use_auto_generation').checked(false);\n            }\n\n            this.enableDisableFields();\n        },\n\n        /**\n         * Enable/disable fields on Coupons tab\n         */\n        enableDisableFields: function () {\n            var selector,\n                isUseAutoGenerationChecked,\n                couponType,\n                disableAuto;\n\n            selector = '[id=sales-rule-form-tab-coupons] input, [id=sales-rule-form-tab-coupons] select, ' +\n                    '[id=sales-rule-form-tab-coupons] button';\n            isUseAutoGenerationChecked = uiRegistry\n                    .get('sales_rule_form.sales_rule_form.rule_information.use_auto_generation')\n                    .checked();\n            couponType = uiRegistry\n                .get('sales_rule_form.sales_rule_form.rule_information.coupon_type')\n                .value();\n            disableAuto = couponType === 3 || isUseAutoGenerationChecked;\n            _.each(\n                document.querySelectorAll(selector),\n                function (element) {\n                    element.disabled = !disableAuto;\n                }\n            );\n        }\n    });\n});\n","Magento_SalesRule/js/form/element/manage-coupon-codes.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'underscore',\n    'uiRegistry',\n    'Magento_Ui/js/form/components/fieldset',\n    'Magento_Ui/js/lib/view/utils/async'\n], function (_, uiRegistry, fieldset, async) {\n    'use strict';\n\n    return fieldset.extend({\n\n        /*eslint-disable no-unused-vars*/\n        /**\n         * Initialize element\n         *\n         * @returns {Abstract} Chainable\n         */\n        initialize: function (elems, position) {\n            var obj = this;\n\n            this._super();\n\n            async.async('#sales-rule-form-tab-coupons', document.getElementById('container'), function (node) {\n                var useAutoGeneration = uiRegistry.get(\n                    'sales_rule_form.sales_rule_form.rule_information.use_auto_generation'\n                );\n\n                useAutoGeneration.on('checked', function () {\n                    obj.enableDisableFields();\n                });\n                obj.enableDisableFields();\n            });\n\n            return this;\n        },\n\n        /*eslint-enable no-unused-vars*/\n        /*eslint-disable lines-around-comment*/\n\n        /**\n         * Enable/disable fields on Coupons tab\n         */\n        enableDisableFields: function () {\n            var selector,\n                isUseAutoGenerationChecked,\n                couponType,\n                disableAuto;\n\n            selector = '[id=sales-rule-form-tab-coupons] input, [id=sales-rule-form-tab-coupons] select, ' +\n                    '[id=sales-rule-form-tab-coupons] button';\n            isUseAutoGenerationChecked = uiRegistry\n                    .get('sales_rule_form.sales_rule_form.rule_information.use_auto_generation')\n                    .checked();\n            couponType = uiRegistry\n                .get('sales_rule_form.sales_rule_form.rule_information.coupon_type')\n                .value();\n            /**\n             * \\Magento\\Rule\\Model\\AbstractModel::COUPON_TYPE_AUTO\n             */\n            disableAuto = couponType === 3 || isUseAutoGenerationChecked;\n            _.each(\n                document.querySelectorAll(selector),\n                function (element) {\n                    element.disabled = !disableAuto;\n                }\n            );\n        }\n    });\n});\n","Magento_SalesRule/js/model/coupon.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n/**\n * Coupon model.\n */\ndefine([\n    'ko',\n    'domReady!'\n], function (ko) {\n    'use strict';\n\n    var couponCode = ko.observable(null),\n        isApplied = ko.observable(null);\n\n    return {\n        couponCode: couponCode,\n        isApplied: isApplied,\n\n        /**\n         * @return {*}\n         */\n        getCouponCode: function () {\n            return couponCode;\n        },\n\n        /**\n         * @return {Boolean}\n         */\n        getIsApplied: function () {\n            return isApplied;\n        },\n\n        /**\n         * @param {*} couponCodeValue\n         */\n        setCouponCode: function (couponCodeValue) {\n            couponCode(couponCodeValue);\n        },\n\n        /**\n         * @param {Boolean} isAppliedValue\n         */\n        setIsApplied: function (isAppliedValue) {\n            isApplied(isAppliedValue);\n        }\n    };\n});\n","Magento_SalesRule/js/model/place-order-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'jquery',\n    'mage/utils/wrapper',\n    'Magento_Checkout/js/model/quote',\n    'Magento_SalesRule/js/model/coupon',\n    'Magento_Checkout/js/action/get-totals'\n], function ($, wrapper, quote, coupon, getTotalsAction) {\n    'use strict';\n\n    return function (placeOrderAction) {\n        return wrapper.wrap(placeOrderAction, function (originalAction, paymentData, messageContainer) {\n            var result;\n\n            $.when(\n                result = originalAction(paymentData, messageContainer)\n            ).fail(\n                function () {\n                    var deferred = $.Deferred(),\n\n                        /**\n                         * Update coupon form\n                         */\n                        updateCouponCallback = function () {\n                            if (quote.totals() && !quote.totals()['coupon_code']) {\n                                coupon.setCouponCode('');\n                                coupon.setIsApplied(false);\n                            }\n                        };\n\n                    getTotalsAction([], deferred);\n                    $.when(deferred).done(updateCouponCallback);\n                }\n            );\n\n            return result;\n        });\n    };\n});\n","Magento_SalesRule/js/model/shipping-save-processor-mixin.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\ndefine([\n    'mage/utils/wrapper',\n    'Magento_Checkout/js/model/quote',\n    'Magento_SalesRule/js/model/coupon'\n], function (wrapper, quote, coupon) {\n    'use strict';\n\n    return function (shippingSaveProcessor) {\n        shippingSaveProcessor.saveShippingInformation = wrapper.wrapSuper(\n            shippingSaveProcessor.saveShippingInformation,\n            function (type) {\n                var updateCouponCallback;\n\n                /**\n                 * Update coupon form\n                 */\n                updateCouponCallback = function () {\n                    if (quote.totals() && !quote.totals()['coupon_code']) {\n                        coupon.setCouponCode('');\n                        coupon.setIsApplied(false);\n                    }\n                };\n\n                return this._super(type).done(updateCouponCallback);\n            }\n        );\n\n        return shippingSaveProcessor;\n    };\n});\n","Magento_SalesRule/js/model/payment/discount-messages.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Ui/js/model/messages'\n], function (Messages) {\n    'use strict';\n\n    return new Messages();\n});\n","Magento_SalesRule/js/view/cart/totals/discount.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_SalesRule/js/view/summary/discount'\n], function (Component) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_SalesRule/cart/totals/discount'\n        },\n\n        /**\n         * @override\n         *\n         * @returns {Boolean}\n         */\n        isDisplayed: function () {\n            return this.getPureValue() != 0; //eslint-disable-line eqeqeq\n        }\n    });\n});\n","Magento_SalesRule/js/view/payment/captcha.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n        'Magento_Captcha/js/view/checkout/defaultCaptcha',\n        'Magento_Captcha/js/model/captchaList',\n        'Magento_SalesRule/js/action/set-coupon-code',\n        'Magento_SalesRule/js/action/cancel-coupon',\n        'Magento_Checkout/js/model/quote',\n        'ko'\n    ],\n    function (defaultCaptcha, captchaList, setCouponCodeAction, cancelCouponAction, quote, ko) {\n        'use strict';\n\n        var totals = quote.getTotals(),\n            couponCode = ko.observable(null),\n            isApplied;\n\n        if (totals()) {\n            couponCode(totals()['coupon_code']);\n        }\n        //Captcha can only be required for adding a coupon so we need to know if one was added already.\n        isApplied = ko.observable(couponCode() != null);\n\n        return defaultCaptcha.extend({\n            /** @inheritdoc */\n            initialize: function () {\n                var self = this,\n                    currentCaptcha;\n\n                this._super();\n                //Getting coupon captcha model.\n                currentCaptcha = captchaList.getCaptchaByFormId(this.formId);\n\n                if (currentCaptcha != null) {\n                    if (!isApplied()) {\n                        //Show captcha if we don't have a coupon applied.\n                        currentCaptcha.setIsVisible(true);\n                    }\n                    this.setCurrentCaptcha(currentCaptcha);\n                    //Add captcha code to coupon-apply request.\n                    setCouponCodeAction.registerDataModifier(function (headers) {\n                        if (self.isRequired()) {\n                            headers['X-Captcha'] = self.captchaValue()();\n                        }\n                    });\n                    //Refresh captcha after failed request.\n                    setCouponCodeAction.registerFailCallback(function () {\n                        if (self.isRequired()) {\n                            self.refresh();\n                        }\n                    });\n                    //Hide captcha when a coupon has been applied.\n                    setCouponCodeAction.registerSuccessCallback(function () {\n                        self.setIsVisible(false);\n                    });\n                    //Show captcha again if it was canceled.\n                    cancelCouponAction.registerSuccessCallback(function () {\n                        if (self.isRequired()) {\n                            self.setIsVisible(true);\n                        }\n                    });\n                }\n            }\n        });\n    });\n","Magento_SalesRule/js/view/payment/discount-messages.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Ui/js/view/messages',\n    '../../model/payment/discount-messages'\n], function (Component, messageContainer) {\n    'use strict';\n\n    return Component.extend({\n        /** @inheritdoc */\n        initialize: function (config) {\n            return this._super(config, messageContainer);\n        }\n    });\n});\n","Magento_SalesRule/js/view/payment/discount.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'ko',\n    'uiComponent',\n    'Magento_Checkout/js/model/quote',\n    'Magento_SalesRule/js/action/set-coupon-code',\n    'Magento_SalesRule/js/action/cancel-coupon',\n    'Magento_SalesRule/js/model/coupon'\n], function ($, ko, Component, quote, setCouponCodeAction, cancelCouponAction, coupon) {\n    'use strict';\n\n    var totals = quote.getTotals(),\n        couponCode = coupon.getCouponCode(),\n        isApplied = coupon.getIsApplied();\n\n    if (totals()) {\n        couponCode(totals()['coupon_code']);\n    }\n    isApplied(couponCode() != null);\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_SalesRule/payment/discount'\n        },\n        couponCode: couponCode,\n\n        /**\n         * Applied flag\n         */\n        isApplied: isApplied,\n\n        /**\n         * Coupon code application procedure\n         */\n        apply: function () {\n            if (this.validate()) {\n                setCouponCodeAction(couponCode(), isApplied);\n            }\n        },\n\n        /**\n         * Cancel using coupon\n         */\n        cancel: function () {\n            if (this.validate()) {\n                couponCode('');\n                cancelCouponAction(isApplied);\n            }\n        },\n\n        /**\n         * Coupon form validation\n         *\n         * @returns {Boolean}\n         */\n        validate: function () {\n            var form = '#discount-form';\n\n            return $(form).validation() && $(form).validation('isValid');\n        }\n    });\n});\n","Magento_SalesRule/js/view/summary/discount.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'Magento_Checkout/js/view/summary/abstract-total',\n    'Magento_Checkout/js/model/quote'\n], function (Component, quote) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_SalesRule/summary/discount'\n        },\n        totals: quote.getTotals(),\n\n        /**\n         * @return {*|Boolean}\n         */\n        isDisplayed: function () {\n            return this.isFullMode() && this.getPureValue() != 0; //eslint-disable-line eqeqeq\n        },\n\n        /**\n         * @return {*}\n         */\n        getCouponCode: function () {\n            if (!this.totals()) {\n                return null;\n            }\n\n            return this.totals()['coupon_code'];\n        },\n\n        /**\n         * @return {*}\n         */\n        getCouponLabel: function () {\n            if (!this.totals()) {\n                return null;\n            }\n\n            return this.totals()['coupon_label'];\n        },\n\n        /**\n         * Get discount title\n         *\n         * @returns {null|String}\n         */\n        getTitle: function () {\n            var discountSegments;\n\n            if (!this.totals()) {\n                return null;\n            }\n\n            discountSegments = this.totals()['total_segments'].filter(function (segment) {\n                return segment.code.indexOf('discount') !== -1;\n            });\n\n            return discountSegments.length ? discountSegments[0].title : null;\n        },\n\n        /**\n         * @return {Number}\n         */\n        getPureValue: function () {\n            var price = 0;\n\n            if (this.totals() && this.totals()['discount_amount']) {\n                price = parseFloat(this.totals()['discount_amount']);\n            }\n\n            return price;\n        },\n\n        /**\n         * @return {*|String}\n         */\n        getValue: function () {\n            return this.getFormattedPrice(this.getPureValue());\n        }\n    });\n});\n","Magento_Search/js/form-mini.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * @api\n */\ndefine([\n    'jquery',\n    'underscore',\n    'mage/template',\n    'matchMedia',\n    'jquery-ui-modules/widget',\n    'jquery-ui-modules/core',\n    'mage/translate'\n], function ($, _, mageTemplate, mediaCheck) {\n    'use strict';\n\n    /**\n     * Check whether the incoming string is not empty or if doesn't consist of spaces.\n     *\n     * @param {String} value - Value to check.\n     * @returns {Boolean}\n     */\n    function isEmpty(value) {\n        return value.length === 0 || value == null || /^\\s+$/.test(value);\n    }\n\n    $.widget('mage.quickSearch', {\n        options: {\n            autocomplete: 'off',\n            minSearchLength: 3,\n            responseFieldElements: 'ul li',\n            selectClass: 'selected',\n            template:\n                '<li class=\"<%- data.row_class %>\" id=\"qs-option-<%- data.index %>\" role=\"option\">' +\n                '<span class=\"qs-option-name\">' +\n                ' <%- data.title %>' +\n                '</span>' +\n                '<span aria-hidden=\"true\" class=\"amount\">' +\n                '<%- data.num_results %>' +\n                '</span>' +\n                '</li>',\n            submitBtn: 'button[type=\"submit\"]',\n            searchLabel: '[data-role=minisearch-label]',\n            isExpandable: null,\n            suggestionDelay: 300\n        },\n\n        /** @inheritdoc */\n        _create: function () {\n            this.responseList = {\n                indexList: null,\n                selected: null\n            };\n            this.autoComplete = $(this.options.destinationSelector);\n            this.searchForm = $(this.options.formSelector);\n            this.submitBtn = this.searchForm.find(this.options.submitBtn)[0];\n            this.searchLabel = this.searchForm.find(this.options.searchLabel);\n            this.isExpandable = this.options.isExpandable;\n\n            _.bindAll(this, '_onKeyDown', '_onPropertyChange', '_onSubmit');\n\n            this.submitBtn.disabled = true;\n\n            this.element.attr('autocomplete', this.options.autocomplete);\n\n            mediaCheck({\n                media: '(max-width: 768px)',\n                entry: function () {\n                    this.isExpandable = true;\n                }.bind(this),\n                exit: function () {\n                    this.isExpandable = true;\n                }.bind(this)\n            });\n\n            this.searchLabel.on('click', function (e) {\n                // allow input to lose its' focus when clicking on label\n                if (this.isExpandable && this.isActive()) {\n                    e.preventDefault();\n                }\n            }.bind(this));\n\n            this.element.on('blur', $.proxy(function () {\n                if (!this.searchLabel.hasClass('active')) {\n                    return;\n                }\n\n                setTimeout($.proxy(function () {\n                    if (this.autoComplete.is(':hidden')) {\n                        this.setActiveState(false);\n                    } else {\n                        this.element.trigger('focus');\n                    }\n                    this.autoComplete.hide();\n                    this._updateAriaHasPopup(false);\n                }, this), 250);\n            }, this));\n\n            if (this.element.get(0) === document.activeElement) {\n                this.setActiveState(true);\n            }\n\n            this.element.on('focus', this.setActiveState.bind(this, true));\n            this.element.on('keydown', this._onKeyDown);\n            // Prevent spamming the server with requests by waiting till the user has stopped typing for period of time\n            this.element.on('input propertychange', _.debounce(this._onPropertyChange, this.options.suggestionDelay));\n\n            this.searchForm.on('submit', $.proxy(function (e) {\n                this._onSubmit(e);\n                this._updateAriaHasPopup(false);\n            }, this));\n        },\n\n        /**\n         * Checks if search field is active.\n         *\n         * @returns {Boolean}\n         */\n        isActive: function () {\n            return this.searchLabel.hasClass('active');\n        },\n\n        /**\n         * Sets state of the search field to provided value.\n         *\n         * @param {Boolean} isActive\n         */\n        setActiveState: function (isActive) {\n            var searchValue;\n\n            this.searchForm.toggleClass('active', isActive);\n            this.searchLabel.toggleClass('active', isActive);\n\n            if (this.isExpandable) {\n                this.element.attr('aria-expanded', isActive);\n                searchValue = this.element.val();\n                this.element.val('');\n                this.element.val(searchValue);\n            }\n        },\n\n        /**\n         * @private\n         * @return {Element} The first element in the suggestion list.\n         */\n        _getFirstVisibleElement: function () {\n            return this.responseList.indexList ? this.responseList.indexList.first() : false;\n        },\n\n        /**\n         * @private\n         * @return {Element} The last element in the suggestion list.\n         */\n        _getLastElement: function () {\n            return this.responseList.indexList ? this.responseList.indexList.last() : false;\n        },\n\n        /**\n         * @private\n         * @param {Boolean} show - Set attribute aria-haspopup to \"true/false\" for element.\n         */\n        _updateAriaHasPopup: function (show) {\n            if (show) {\n                this.element.attr('aria-haspopup', 'true');\n            } else {\n                this.element.attr('aria-haspopup', 'false');\n            }\n        },\n\n        /**\n         * Clears the item selected from the suggestion list and resets the suggestion list.\n         * @private\n         * @param {Boolean} all - Controls whether to clear the suggestion list.\n         */\n        _resetResponseList: function (all) {\n            this.responseList.selected = null;\n\n            if (all === true) {\n                this.responseList.indexList = null;\n            }\n        },\n\n        /**\n         * Executes when the search box is submitted. Sets the search input field to the\n         * value of the selected item.\n         * @private\n         * @param {Event} e - The submit event\n         */\n        _onSubmit: function (e) {\n            var value = this.element.val();\n\n            if (isEmpty(value)) {\n                e.preventDefault();\n            }\n\n            if (this.responseList.selected) {\n                this.element.val(this.responseList.selected.find('.qs-option-name').text());\n            }\n        },\n\n        /**\n         * Executes when keys are pressed in the search input field. Performs specific actions\n         * depending on which keys are pressed.\n         * @private\n         * @param {Event} e - The key down event\n         * @return {Boolean} Default return type for any unhandled keys\n         */\n        _onKeyDown: function (e) {\n            var keyCode = e.keyCode || e.which;\n\n            switch (keyCode) {\n                case $.ui.keyCode.HOME:\n                    if (this._getFirstVisibleElement()) {\n                        this._getFirstVisibleElement().addClass(this.options.selectClass);\n                        this.responseList.selected = this._getFirstVisibleElement();\n                    }\n                    break;\n\n                case $.ui.keyCode.END:\n                    if (this._getLastElement()) {\n                        this._getLastElement().addClass(this.options.selectClass);\n                        this.responseList.selected = this._getLastElement();\n                    }\n                    break;\n\n                case $.ui.keyCode.ESCAPE:\n                    this._resetResponseList(true);\n                    this.autoComplete.hide();\n                    break;\n\n                case $.ui.keyCode.ENTER:\n                    if (this.element.val().length >= parseInt(this.options.minSearchLength, 10)) {\n                        this.searchForm.trigger('submit');\n                        e.preventDefault();\n                    }\n                    break;\n\n                case $.ui.keyCode.DOWN:\n                    if (this.responseList.indexList) {\n                        if (!this.responseList.selected) {  //eslint-disable-line max-depth\n                            this._getFirstVisibleElement().addClass(this.options.selectClass);\n                            this.responseList.selected = this._getFirstVisibleElement();\n                        } else if (!this._getLastElement().hasClass(this.options.selectClass)) {\n                            this.responseList.selected = this.responseList.selected\n                                .removeClass(this.options.selectClass).next().addClass(this.options.selectClass);\n                        } else {\n                            this.responseList.selected.removeClass(this.options.selectClass);\n                            this._getFirstVisibleElement().addClass(this.options.selectClass);\n                            this.responseList.selected = this._getFirstVisibleElement();\n                        }\n                        this.element.val(this.responseList.selected.find('.qs-option-name').text());\n                        this.element.attr('aria-activedescendant', this.responseList.selected.attr('id'));\n                        this._updateAriaHasPopup(true);\n                        this.autoComplete.show();\n                    }\n                    break;\n\n                case $.ui.keyCode.UP:\n                    if (this.responseList.indexList !== null) {\n                        if (!this._getFirstVisibleElement().hasClass(this.options.selectClass)) {\n                            this.responseList.selected = this.responseList.selected\n                                .removeClass(this.options.selectClass).prev().addClass(this.options.selectClass);\n\n                        } else {\n                            this.responseList.selected.removeClass(this.options.selectClass);\n                            this._getLastElement().addClass(this.options.selectClass);\n                            this.responseList.selected = this._getLastElement();\n                        }\n                        this.element.val(this.responseList.selected.find('.qs-option-name').text());\n                        this.element.attr('aria-activedescendant', this.responseList.selected.attr('id'));\n                        this._updateAriaHasPopup(true);\n                        this.autoComplete.show();\n                    }\n                    break;\n                default:\n                    return true;\n            }\n        },\n\n        /**\n         * Executes when the value of the search input field changes. Executes a GET request\n         * to populate a suggestion list based on entered text. Handles click (select), hover,\n         * and mouseout events on the populated suggestion list dropdown.\n         * @private\n         */\n        _onPropertyChange: function () {\n            var searchField = this.element,\n                clonePosition = {\n                    position: 'absolute',\n                    // Removed to fix display issues\n                    // left: searchField.offset().left,\n                    // top: searchField.offset().top + searchField.outerHeight(),\n                    width: searchField.outerWidth()\n                },\n                source = this.options.template,\n                template = mageTemplate(source),\n                dropdown = $('<ul role=\"listbox\"></ul>'),\n                value = this.element.val();\n\n            this.submitBtn.disabled = true;\n\n            if (value.length >= parseInt(this.options.minSearchLength, 10)) {\n                this.submitBtn.disabled = false;\n\n                if (this.options.url !== '') { //eslint-disable-line eqeqeq\n                    $.getJSON(this.options.url, {\n                        q: value\n                    }, $.proxy(function (data) {\n                        if (data.length) {\n                            $.each(data, function (index, element) {\n                                var html;\n\n                                element.index = index;\n                                html = template({\n                                    data: element\n                                });\n                                dropdown.append(html);\n                            });\n\n                            this._resetResponseList(true);\n\n                            this.responseList.indexList = this.autoComplete.html(dropdown)\n                                .css(clonePosition)\n                                .show()\n                                .find(this.options.responseFieldElements + ':visible');\n\n                            this.element.removeAttr('aria-activedescendant');\n\n                            if (this.responseList.indexList.length) {\n                                this._updateAriaHasPopup(true);\n                            } else {\n                                this._updateAriaHasPopup(false);\n                            }\n\n                            this.responseList.indexList\n                                .on('click', function (e) {\n                                    this.responseList.selected = $(e.currentTarget);\n                                    this.searchForm.trigger('submit');\n                                }.bind(this))\n                                .on('mouseenter mouseleave', function (e) {\n                                    this.responseList.indexList.removeClass(this.options.selectClass);\n                                    $(e.target).addClass(this.options.selectClass);\n                                    this.responseList.selected = $(e.target);\n                                    this.element.attr('aria-activedescendant', $(e.target).attr('id'));\n                                }.bind(this))\n                                .on('mouseout', function (e) {\n                                    if (!this._getLastElement() &&\n                                        this._getLastElement().hasClass(this.options.selectClass)) {\n                                        $(e.target).removeClass(this.options.selectClass);\n                                        this._resetResponseList(false);\n                                    }\n                                }.bind(this));\n                        } else {\n                            this._resetResponseList(true);\n                            this.autoComplete.hide();\n                            this._updateAriaHasPopup(false);\n                            this.element.removeAttr('aria-activedescendant');\n                        }\n                    }, this));\n                }\n            } else {\n                this._resetResponseList(true);\n                this.autoComplete.hide();\n                this._updateAriaHasPopup(false);\n                this.element.removeAttr('aria-activedescendant');\n            }\n        }\n    });\n\n    return $.mage.quickSearch;\n});\n","Magento_Security/js/escaper.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\n/**\n * A loose JavaScript version of Magento\\Framework\\Escaper\n *\n * Due to differences in how XML/HTML is processed in PHP vs JS there are a couple of minor differences in behavior\n * from the PHP counterpart.\n *\n * The first difference is that the default invocation of escapeHtml without allowedTags will double-escape existing\n * entities as the intention of such an invocation is that the input isn't supposed to contain any HTML.\n *\n * The second difference is that escapeHtml will not escape quotes. Since the input is actually being processed by the\n * DOM there is no chance of quotes being mixed with HTML syntax. And, since escapeHtml is not\n * intended to be used with raw injection into a HTML attribute, this is acceptable.\n *\n * @api\n */\ndefine([], function () {\n    'use strict';\n\n    return {\n        neverAllowedElements: ['script', 'img', 'embed', 'iframe', 'video', 'source', 'object', 'audio'],\n        generallyAllowedAttributes: ['id', 'class', 'href', 'title', 'style'],\n        forbiddenAttributesByElement: {\n            a: ['style']\n        },\n\n        /**\n         * Escape a string for safe injection into HTML\n         *\n         * @param {String} data\n         * @param {Array|null} allowedTags\n         * @returns {String}\n         */\n        escapeHtml: function (data, allowedTags) {\n            var domParser = new DOMParser(),\n                fragment = domParser.parseFromString('<div></div>', 'text/html');\n\n            fragment = fragment.body.childNodes[0];\n            allowedTags = typeof allowedTags === 'object' && allowedTags.length ? allowedTags : null;\n\n            if (allowedTags) {\n                fragment.innerHTML = data || '';\n                allowedTags = this._filterProhibitedTags(allowedTags);\n\n                this._removeComments(fragment);\n                this._removeNotAllowedElements(fragment, allowedTags);\n                this._removeNotAllowedAttributes(fragment);\n\n                return fragment.innerHTML;\n            }\n\n            fragment.textContent = data || '';\n\n            return fragment.innerHTML;\n        },\n\n        /**\n         * Remove the always forbidden tags from a list of provided tags\n         *\n         * @param {Array} tags\n         * @returns {Array}\n         * @private\n         */\n        _filterProhibitedTags: function (tags) {\n            return tags.filter(function (n) {\n                return this.neverAllowedElements.indexOf(n) === -1;\n            }.bind(this));\n        },\n\n        /**\n         * Remove comment nodes from the given node\n         *\n         * @param {Node} node\n         * @private\n         */\n        _removeComments: function (node) {\n            var treeWalker = node.ownerDocument.createTreeWalker(\n                    node,\n                    NodeFilter.SHOW_COMMENT,\n                    function () {\n                        return NodeFilter.FILTER_ACCEPT;\n                    },\n                    false\n                ),\n                nodesToRemove = [];\n\n            while (treeWalker.nextNode()) {\n                nodesToRemove.push(treeWalker.currentNode);\n            }\n\n            nodesToRemove.forEach(function (nodeToRemove) {\n                nodeToRemove.parentNode.removeChild(nodeToRemove);\n            });\n        },\n\n        /**\n         * Strip the given node of all disallowed tags while permitting any nested text nodes\n         *\n         * @param {Node} node\n         * @param {Array|null} allowedTags\n         * @private\n         */\n        _removeNotAllowedElements: function (node, allowedTags) {\n            var treeWalker = node.ownerDocument.createTreeWalker(\n                    node,\n                    NodeFilter.SHOW_ELEMENT,\n                    function (currentNode) {\n                        return allowedTags.indexOf(currentNode.nodeName.toLowerCase()) === -1 ?\n                            NodeFilter.FILTER_ACCEPT\n                            // SKIP instead of REJECT because REJECT also rejects child nodes\n                            : NodeFilter.FILTER_SKIP;\n                    },\n                false\n                ),\n                nodesToRemove = [];\n\n            while (treeWalker.nextNode()) {\n                if (allowedTags.indexOf(treeWalker.currentNode.nodeName.toLowerCase()) === -1) {\n                    nodesToRemove.push(treeWalker.currentNode);\n                }\n            }\n\n            nodesToRemove.forEach(function (nodeToRemove) {\n                nodeToRemove.parentNode.replaceChild(\n                    node.ownerDocument.createTextNode(nodeToRemove.textContent),\n                    nodeToRemove\n                );\n            });\n        },\n\n        /**\n         * Remove any invalid attributes from the given node\n         *\n         * @param {Node} node\n         * @private\n         */\n        _removeNotAllowedAttributes: function (node) {\n            var treeWalker = node.ownerDocument.createTreeWalker(\n                    node,\n                    NodeFilter.SHOW_ELEMENT,\n                    function () {\n                        return NodeFilter.FILTER_ACCEPT;\n                    },\n                false\n                ),\n                i,\n                attribute,\n                nodeName,\n                attributesToRemove = [];\n\n            while (treeWalker.nextNode()) {\n                for (i = 0; i < treeWalker.currentNode.attributes.length; i++) {\n                    attribute = treeWalker.currentNode.attributes[i];\n                    nodeName = treeWalker.currentNode.nodeName.toLowerCase();\n\n                    if (this.generallyAllowedAttributes.indexOf(attribute.name) === -1  || // eslint-disable-line max-depth,max-len\n                        this._checkHrefValue(attribute) ||\n                        this.forbiddenAttributesByElement[nodeName] &&\n                        this.forbiddenAttributesByElement[nodeName].indexOf(attribute.name) !== -1\n                    ) {\n                        attributesToRemove.push(attribute);\n                    }\n                }\n            }\n\n            attributesToRemove.forEach(function (attributeToRemove) {\n                attributeToRemove.ownerElement.removeAttribute(attributeToRemove.name);\n            });\n        },\n\n        /**\n         * Check that attribute contains script content\n         *\n         * @param {Object} attribute\n         * @private\n         */\n        _checkHrefValue: function (attribute) {\n            return attribute.nodeName === 'href' && attribute.nodeValue.startsWith('javascript');\n        }\n    };\n});\n","Magento_SendFriend/requirejs-config.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\nvar config = {\n    map: {\n        '*': {\n            'Magento_SendFriend/back-event': 'Magento_SendFriend/js/back-event'\n        }\n    }\n};\n","Magento_SendFriend/js/back-event.js":"/**\n* Copyright \u00a9 Magento, Inc. All rights reserved.\n* See COPYING.txt for license details.\n*/\n\ndefine([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    return function (config, element) {\n        $(element).on('click', function () {\n            history.back();\n\n            return false;\n        });\n    };\n});\n","Magento_Shipping/js/model/config.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([], function () {\n    'use strict';\n\n    return function () {\n        return window.checkoutConfig.shippingPolicy;\n    };\n});\n","Magento_Shipping/js/view/checkout/shipping/shipping-policy.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'uiComponent',\n    'Magento_Shipping/js/model/config'\n\n], function (Component, config) {\n    'use strict';\n\n    return Component.extend({\n        defaults: {\n            template: 'Magento_Shipping/checkout/shipping/shipping-policy'\n        },\n        config: config()\n    });\n});\n","Magento_Swatches/js/catalog-add-to-cart.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\nrequire([\n    'jquery'\n], function ($) {\n    'use strict';\n\n    /**\n     * Add selected swatch attributes to redirect url\n     *\n     * @see Magento_Catalog/js/catalog-add-to-cart\n     */\n    $('body').on('catalogCategoryAddToCartRedirect', function (event, data) {\n        $(data.form).find('[name*=\"super\"]').each(function (index, item) {\n            var $item = $(item),\n                attr;\n\n            if ($item.attr('data-attr-name')) {\n                attr = $item.attr('data-attr-name');\n            } else {\n                attr = $item.parent().attr('attribute-code');\n            }\n            data.redirectParameters.push(attr + '=' + $item.val());\n\n        });\n    });\n});\n","Magento_Swatches/js/configurable-customer-data.js":"require([\n    'jquery',\n    'Magento_ConfigurableProduct/js/options-updater'\n], function ($, Updater) {\n    'use strict';\n\n    var selectors = {\n            formSelector: '#product_addtocart_form',\n            swatchSelector: '.swatch-opt'\n        },\n        swatchWidgetName = 'mageSwatchRenderer',\n        widgetInitEvent = 'swatch.initialized',\n\n    /**\n    * Sets all configurable swatch attribute's selected values\n    */\n    updateSwatchOptions = function () {\n        var swatchWidget = $(selectors.swatchSelector).data(swatchWidgetName);\n\n        if (!swatchWidget || !swatchWidget._EmulateSelectedByAttributeId) {\n            return;\n        }\n        swatchWidget._EmulateSelectedByAttributeId(this.productOptions);\n    },\n    updater = new Updater(widgetInitEvent, updateSwatchOptions);\n\n    updater.listen();\n});\n","Magento_Swatches/js/swatch-renderer.js":"/**\n * Copyright \u00a9 Magento, Inc. All rights reserved.\n * See COPYING.txt for license details.\n */\n\ndefine([\n    'jquery',\n    'underscore',\n    'mage/template',\n    'mage/smart-keyboard-handler',\n    'mage/translate',\n    'priceUtils',\n    'jquery-ui-modules/widget',\n    'jquery/jquery.parsequery',\n    'mage/validation/validation'\n], function ($, _, mageTemplate, keyboardHandler, $t, priceUtils) {\n    'use strict';\n\n    /**\n     * Extend form validation to support swatch accessibility\n     */\n    $.widget('mage.validation', $.mage.validation, {\n        /**\n         * Handle form with swatches validation. Focus on first invalid swatch block.\n         *\n         * @param {jQuery.Event} event\n         * @param {Object} validation\n         */\n        listenFormValidateHandler: function (event, validation) {\n            var swatchWrapper, firstActive, swatches, swatch, successList, errorList, firstSwatch;\n\n            this._superApply(arguments);\n\n            swatchWrapper = '.swatch-attribute-options';\n            swatches = $(event.target).find(swatchWrapper);\n\n            if (!swatches.length) {\n                return;\n            }\n\n            swatch = '.swatch-attribute';\n            firstActive = $(validation.errorList[0].element || []);\n            successList = validation.successList;\n            errorList = validation.errorList;\n            firstSwatch = $(firstActive).parent(swatch).find(swatchWrapper);\n\n            keyboardHandler.focus(swatches);\n\n            $.each(successList, function (index, item) {\n                $(item).parent(swatch).find(swatchWrapper).attr('aria-invalid', false);\n            });\n\n            $.each(errorList, function (index, item) {\n                $(item.element).parent(swatch).find(swatchWrapper).attr('aria-invalid', true);\n            });\n\n            if (firstSwatch.length) {\n                $(firstSwatch).trigger('focus');\n            }\n        }\n    });\n\n    /**\n     * Render tooltips by attributes (only to up).\n     * Required element attributes:\n     *  - data-option-type (integer, 0-3)\n     *  - data-option-label (string)\n     *  - data-option-tooltip-thumb\n     *  - data-option-tooltip-value\n     *  - data-thumb-width\n     *  - data-thumb-height\n     */\n    $.widget('mage.SwatchRendererTooltip', {\n        options: {\n            delay: 200,                             //how much ms before tooltip to show\n            tooltipClass: 'swatch-option-tooltip'  //configurable, but remember about css\n        },\n\n        /**\n         * @private\n         */\n        _init: function () {\n            var $widget = this,\n                $this = this.element,\n                $element = $('.' + $widget.options.tooltipClass),\n                timer,\n                type = parseInt($this.data('option-type'), 10),\n                label = $this.data('option-label'),\n                thumb = $this.data('option-tooltip-thumb'),\n                value = $this.data('option-tooltip-value'),\n                width = $this.data('thumb-width'),\n                height = $this.data('thumb-height'),\n                $image,\n                $title,\n                $corner;\n\n            if (!$element.length) {\n                $element = $('<div class=\"' +\n                    $widget.options.tooltipClass +\n                    '\"><div class=\"image\"></div><div class=\"title\"></div><div class=\"corner\"></div></div>'\n                );\n                $('body').append($element);\n            }\n\n            $image = $element.find('.image');\n            $title = $element.find('.title');\n            $corner = $element.find('.corner');\n\n            $this.on('mouseenter', function () {\n                if (!$this.hasClass('disabled')) {\n                    timer = setTimeout(\n                        function () {\n                            var leftOpt = null,\n                                leftCorner = 0,\n                                left,\n                                $window;\n\n                            if (type === 2) {\n                                // Image\n                                $image.css({\n                                    'background': 'url(\"' + thumb + '\") no-repeat center', //Background case\n                                    'background-size': 'initial',\n                                    'width': width + 'px',\n                                    'height': height + 'px'\n                                });\n                                $image.show();\n                            } else if (type === 1) {\n                                // Color\n                                $image.css({\n                                    background: value\n                                });\n                                $image.show();\n                            } else if (type === 0 || type === 3) {\n                                // Default\n                                $image.hide();\n                            }\n\n                            $title.text(label);\n\n                            leftOpt = $this.offset().left;\n                            left = leftOpt + $this.width() / 2 - $element.width() / 2;\n                            $window = $(window);\n\n                            // the numbers (5 and 5) is magick constants for offset from left or right page\n                            if (left < 0) {\n                                left = 5;\n                            } else if (left + $element.width() > $window.width()) {\n                                left = $window.width() - $element.width() - 5;\n                            }\n\n                            // the numbers (6,  3 and 18) is magick constants for offset tooltip\n                            leftCorner = 0;\n\n                            if ($element.width() < $this.width()) {\n                                leftCorner = $element.width() / 2 - 3;\n                            } else {\n                                leftCorner = (leftOpt > left ? leftOpt - left : left - leftOpt) + $this.width() / 2 - 6;\n                            }\n\n                            $corner.css({\n                                left: leftCorner\n                            });\n                            $element.css({\n                                left: left,\n                                top: $this.offset().top - $element.height() - $corner.height() - 18\n                            }).show();\n                        },\n                        $widget.options.delay\n                    );\n                }\n            });\n\n            $this.on('mouseleave', function () {\n                $element.hide();\n                clearTimeout(timer);\n            });\n\n            $(document).on('tap', function () {\n                $element.hide();\n                clearTimeout(timer);\n            });\n\n            $this.on('tap', function (event) {\n                event.stopPropagation();\n            });\n        }\n    });\n\n    /**\n     * Render swatch controls with options and use tooltips.\n     * Required two json:\n     *  - jsonConfig (magento's option config)\n     *  - jsonSwatchConfig (swatch's option config)\n     *\n     *  Tuning:\n     *  - numberToShow (show \"more\" button if options are more)\n     *  - onlySwatches (hide selectboxes)\n     *  - moreButtonText (text for \"more\" button)\n     *  - selectorProduct (selector for product container)\n     *  - selectorProductPrice (selector for change price)\n     */\n    $.widget('mage.SwatchRenderer', {\n        options: {\n            classes: {\n                attributeClass: 'swatch-attribute',\n                attributeLabelClass: 'swatch-attribute-label',\n                attributeSelectedOptionLabelClass: 'swatch-attribute-selected-option',\n                attributeOptionsWrapper: 'swatch-attribute-options',\n                attributeInput: 'swatch-input',\n                optionClass: 'swatch-option',\n                selectClass: 'swatch-select',\n                moreButton: 'swatch-more',\n                loader: 'swatch-option-loading'\n            },\n            // option's json config\n            jsonConfig: {},\n\n            // swatch's json config\n            jsonSwatchConfig: {},\n\n            // selector of parental block of prices and swatches (need to know where to seek for price block)\n            selectorProduct: '.product-info-main',\n\n            // selector of price wrapper (need to know where set price)\n            selectorProductPrice: '[data-role=priceBox]',\n\n            //selector of product images gallery wrapper\n            mediaGallerySelector: '[data-gallery-role=gallery-placeholder]',\n\n            // selector of category product tile wrapper\n            selectorProductTile: '.product-item',\n\n            // number of controls to show (false or zero = show all)\n            numberToShow: false,\n\n            // show only swatch controls\n            onlySwatches: false,\n\n            // enable label for control\n            enableControlLabel: true,\n\n            // control label id\n            controlLabelId: '',\n\n            // text for more button\n            moreButtonText: $t('More'),\n\n            // Callback url for media\n            mediaCallback: '',\n\n            // Local media cache\n            mediaCache: {},\n\n            // Cache for BaseProduct images. Needed when option unset\n            mediaGalleryInitial: [{}],\n\n            // Use ajax to get image data\n            useAjax: false,\n\n            /**\n             * Defines the mechanism of how images of a gallery should be\n             * updated when user switches between configurations of a product.\n             *\n             * As for now value of this option can be either 'replace' or 'prepend'.\n             *\n             * @type {String}\n             */\n            gallerySwitchStrategy: 'replace',\n\n            // whether swatches are rendered in product list or on product page\n            inProductList: false,\n\n            // sly-old-price block selector\n            slyOldPriceSelector: '.sly-old-price',\n\n            // tier prise selectors start\n            tierPriceTemplateSelector: '#tier-prices-template',\n            tierPriceBlockSelector: '[data-role=\"tier-price-block\"]',\n            tierPriceTemplate: '',\n            // tier prise selectors end\n\n            // A price label selector\n            normalPriceLabelSelector: '.product-info-main .normal-price .price-label',\n            qtyInfo: '#qty'\n        },\n\n        /**\n         * Get chosen product\n         *\n         * @returns int|null\n         */\n        getProduct: function () {\n            var products = this._CalcProducts();\n\n            return _.isArray(products) ? products[0] : null;\n        },\n\n        /**\n         * Get chosen product id\n         *\n         * @returns int|null\n         */\n        getProductId: function () {\n            var products = this._CalcProducts();\n\n            return _.isArray(products) && products.length === 1 ? products[0] : null;\n        },\n\n        /**\n         * @private\n         */\n        _init: function () {\n            // Don't render the same set of swatches twice\n            if ($(this.element).attr('data-rendered')) {\n                return;\n            }\n\n            $(this.element).attr('data-rendered', true);\n\n            if (_.isEmpty(this.options.jsonConfig.images)) {\n                this.options.useAjax = true;\n                // creates debounced variant of _LoadProductMedia()\n                // to use it in events handlers instead of _LoadProductMedia()\n                this._debouncedLoadProductMedia = _.debounce(this._LoadProductMedia.bind(this), 500);\n            }\n\n            this.options.tierPriceTemplate = $(this.options.tierPriceTemplateSelector).html();\n\n            if (this.options.jsonConfig !== '' && this.options.jsonSwatchConfig !== '') {\n                // store unsorted attributes\n                this.options.jsonConfig.mappedAttributes = _.clone(this.options.jsonConfig.attributes);\n                this._sortAttributes();\n                this._RenderControls();\n                this._setPreSelectedGallery();\n                $(this.element).trigger('swatch.initialized');\n            } else {\n                console.log('SwatchRenderer: No input data received');\n            }\n        },\n\n        /**\n         * @private\n         */\n        _sortAttributes: function () {\n            this.options.jsonConfig.attributes = _.sortBy(this.options.jsonConfig.attributes, function (attribute) {\n                return parseInt(attribute.position, 10);\n            });\n        },\n\n        /**\n         * @private\n         */\n        _create: function () {\n            var options = this.options,\n                gallery = $('[data-gallery-role=gallery-placeholder]', '.column.main'),\n                productData = this._determineProductData(),\n                $main = productData.isInProductView ?\n                    this.element.parents('.column.main') :\n                    this.element.parents('.product-item-info');\n\n            if (productData.isInProductView) {\n                gallery.data('gallery') ?\n                    this._onGalleryLoaded(gallery) :\n                    gallery.on('gallery:loaded', this._onGalleryLoaded.bind(this, gallery));\n            } else {\n                options.mediaGalleryInitial = [{\n                    'img': $main.find('.product-image-photo').attr('src')\n                }];\n            }\n\n            this.productForm = this.element.parents(this.options.selectorProductTile).find('form:first');\n            this.inProductList = this.productForm.length > 0;\n            $(this.options.qtyInfo).on('input', this._onQtyChanged.bind(this));\n        },\n\n        /**\n         * Determine product id and related data\n         *\n         * @returns {{productId: *, isInProductView: bool}}\n         * @private\n         */\n        _determineProductData: function () {\n            // Check if product is in a list of products.\n            var productId,\n                isInProductView = false;\n\n            productId = this.element.parents('.product-item-details')\n                    .find('.price-box.price-final_price').attr('data-product-id');\n\n            if (!productId) {\n                // Check individual product.\n                productId = $('[name=product]').val();\n                isInProductView = productId > 0;\n            }\n\n            return {\n                productId: productId,\n                isInProductView: isInProductView\n            };\n        },\n\n        /**\n         * Render controls\n         *\n         * @private\n         */\n        _RenderControls: function () {\n            var $widget = this,\n                container = this.element,\n                classes = this.options.classes,\n                chooseText = this.options.jsonConfig.chooseText,\n                showTooltip = this.options.showTooltip;\n\n            $widget.optionsMap = {};\n\n            $.each(this.options.jsonConfig.attributes, function () {\n                var item = this,\n                    controlLabelId = 'option-label-' + item.code + '-' + item.id,\n                    options = $widget._RenderSwatchOptions(item, controlLabelId),\n                    select = $widget._RenderSwatchSelect(item, chooseText),\n                    input = $widget._RenderFormInput(item),\n                    listLabel = '',\n                    label = '';\n\n                // Show only swatch controls\n                if ($widget.options.onlySwatches && !$widget.options.jsonSwatchConfig.hasOwnProperty(item.id)) {\n                    return;\n                }\n\n                if ($widget.options.enableControlLabel) {\n                    label +=\n                        '<span id=\"' + controlLabelId + '\" class=\"' + classes.attributeLabelClass + '\">' +\n                        $('<i></i>').text(item.label).html() +\n                        '</span>' +\n                        '<span class=\"' + classes.attributeSelectedOptionLabelClass + '\"></span>';\n                }\n\n                if ($widget.inProductList) {\n                    $widget.productForm.append(input);\n                    input = '';\n                    listLabel = 'aria-label=\"' + $('<i></i>').text(item.label).html() + '\"';\n                } else {\n                    listLabel = 'aria-labelledby=\"' + controlLabelId + '\"';\n                }\n\n                // Create new control\n                container.append(\n                    '<div class=\"' + classes.attributeClass + ' ' + item.code + '\" ' +\n                         'data-attribute-code=\"' + item.code + '\" ' +\n                         'data-attribute-id=\"' + item.id + '\">' +\n                        label +\n                        '<div aria-activedescendant=\"\" ' +\n                             'tabindex=\"0\" ' +\n                             'aria-invalid=\"false\" ' +\n                             'aria-required=\"true\" ' +\n                             'role=\"listbox\" ' + listLabel +\n                             'class=\"' + classes.attributeOptionsWrapper + ' clearfix\">' +\n                            options + select +\n                        '</div>' + input +\n                    '</div>'\n                );\n\n                $widget.optionsMap[item.id] = {};\n\n                // Aggregate options array to hash (key => value)\n                $.each(item.options, function () {\n                    if (this.products.length > 0) {\n                        $widget.optionsMap[item.id][this.id] = {\n                            price: parseInt(\n                                $widget.options.jsonConfig.optionPrices[this.products[0]].finalPrice.amount,\n                                10\n                            ),\n                            products: this.products\n                        };\n                    }\n                });\n            });\n\n            if (showTooltip === 1) {\n                // Connect Tooltip\n                container\n                    .find('[data-option-type=\"1\"], [data-option-type=\"2\"],' +\n                        ' [data-option-type=\"0\"], [data-option-type=\"3\"]')\n                    .SwatchRendererTooltip();\n            }\n\n            // Hide all elements below more button\n            $('.' + classes.moreButton).nextAll().hide();\n\n            // Handle events like click or change\n            $widget._EventListener();\n\n            // Rewind options\n            $widget._Rewind(container);\n\n            //Emulate click on all swatches from Request\n            $widget._EmulateSelected($.parseQuery());\n            $widget._EmulateSelected($widget._getSelectedAttributes());\n        },\n\n        /**\n         * Render swatch options by part of config\n         *\n         * @param {Object} config\n         * @param {String} controlId\n         * @returns {String}\n         * @private\n         */\n        _RenderSwatchOptions: function (config, controlId) {\n            var optionConfig = this.options.jsonSwatchConfig[config.id],\n                optionClass = this.options.classes.optionClass,\n                sizeConfig = this.options.jsonSwatchImageSizeConfig,\n                moreLimit = parseInt(this.options.numberToShow, 10),\n                moreClass = this.options.classes.moreButton,\n                moreText = this.options.moreButtonText,\n                countAttributes = 0,\n                html = '';\n\n            if (!this.options.jsonSwatchConfig.hasOwnProperty(config.id)) {\n                return '';\n            }\n\n            $.each(config.options, function (index) {\n                var id,\n                    type,\n                    value,\n                    thumb,\n                    label,\n                    width,\n                    height,\n                    attr,\n                    swatchImageWidth,\n                    swatchImageHeight;\n\n                if (!optionConfig.hasOwnProperty(this.id)) {\n                    return '';\n                }\n\n                // Add more button\n                if (moreLimit === countAttributes++) {\n                    html += '<a href=\"#\" class=\"' + moreClass + '\"><span>' + moreText + '</span></a>';\n                }\n\n                id = this.id;\n                type = parseInt(optionConfig[id].type, 10);\n                value = optionConfig[id].hasOwnProperty('value') ?\n                    $('<i></i>').text(optionConfig[id].value).html() : '';\n                thumb = optionConfig[id].hasOwnProperty('thumb') ? optionConfig[id].thumb : '';\n                width = _.has(sizeConfig, 'swatchThumb') ? sizeConfig.swatchThumb.width : 110;\n                height = _.has(sizeConfig, 'swatchThumb') ? sizeConfig.swatchThumb.height : 90;\n                label = this.label ? $('<i></i>').text(this.label).html() : '';\n                attr =\n                    ' id=\"' + controlId + '-item-' + id + '\"' +\n                    ' index=\"' + index + '\"' +\n                    ' aria-checked=\"false\"' +\n                    ' aria-describedby=\"' + controlId + '\"' +\n                    ' tabindex=\"0\"' +\n                    ' data-option-type=\"' + type + '\"' +\n                    ' data-option-id=\"' + id + '\"' +\n                    ' data-option-label=\"' + label + '\"' +\n                    ' aria-label=\"' + label + '\"' +\n                    ' role=\"option\"' +\n                    ' data-thumb-width=\"' + width + '\"' +\n                    ' data-thumb-height=\"' + height + '\"';\n\n                attr += thumb !== '' ? ' data-option-tooltip-thumb=\"' + thumb + '\"' : '';\n                attr += value !== '' ? ' data-option-tooltip-value=\"' + value + '\"' : '';\n\n                swatchImageWidth = _.has(sizeConfig, 'swatchImage') ? sizeConfig.swatchImage.width : 30;\n                swatchImageHeight = _.has(sizeConfig, 'swatchImage') ? sizeConfig.swatchImage.height : 20;\n\n                if (!this.hasOwnProperty('products') || this.products.length <= 0) {\n                    attr += ' data-option-empty=\"true\"';\n                }\n\n                if (type === 0) {\n                    // Text\n                    html += '<div class=\"' + optionClass + ' text\" ' + attr + '>' + (value ? value : label) +\n                        '</div>';\n                } else if (type === 1) {\n                    // Color\n                    html += '<div class=\"' + optionClass + ' color\" ' + attr +\n                        ' style=\"background: ' + value +\n                        ' no-repeat center; background-size: initial;\">' + '' +\n                        '</div>';\n                } else if (type === 2) {\n                    // Image\n                    html += '<div class=\"' + optionClass + ' image\" ' + attr +\n                        ' style=\"background: url(' + value + ') no-repeat center; background-size: initial;width:' +\n                        swatchImageWidth + 'px; height:' + swatchImageHeight + 'px\">' + '' +\n                        '</div>';\n                } else if (type === 3) {\n                    // Clear\n                    html += '<div class=\"' + optionClass + '\" ' + attr + '></div>';\n                } else {\n                    // Default\n                    html += '<div class=\"' + optionClass + '\" ' + attr + '>' + label + '</div>';\n                }\n            });\n\n            return html;\n        },\n\n        /**\n         * Render select by part of config\n         *\n         * @param {Object} config\n         * @param {String} chooseText\n         * @returns {String}\n         * @private\n         */\n        _RenderSwatchSelect: function (config, chooseText) {\n            var html;\n\n            if (this.options.jsonSwatchConfig.hasOwnProperty(config.id)) {\n                return '';\n            }\n\n            html =\n                '<select class=\"' + this.options.classes.selectClass + ' ' + config.code + '\">' +\n                '<option value=\"0\" data-option-id=\"0\">' + chooseText + '</option>';\n\n            $.each(config.options, function () {\n                var label = this.label,\n                    attr = ' value=\"' + this.id + '\" data-option-id=\"' + this.id + '\"';\n\n                if (!this.hasOwnProperty('products') || this.products.length <= 0) {\n                    attr += ' data-option-empty=\"true\"';\n                }\n\n                html += '<option ' + attr + '>' + label + '</option>';\n            });\n\n            html += '</select>';\n\n            return html;\n        },\n\n        /**\n         * Input for submit form.\n         * This control shouldn't have \"type=hidden\", \"display: none\" for validation work :(\n         *\n         * @param {Object} config\n         * @private\n         */\n        _RenderFormInput: function (config) {\n            return '<input class=\"' + this.options.classes.attributeInput + ' super-attribute-select\" ' +\n                'name=\"super_attribute[' + config.id + ']\" ' +\n                'type=\"text\" ' +\n                'value=\"\" ' +\n                'data-selector=\"super_attribute[' + config.id + ']\" ' +\n                'data-validate=\"{required: true}\" ' +\n                'aria-required=\"true\" ' +\n                'aria-invalid=\"false\">';\n        },\n\n        /**\n         * Event listener\n         *\n         * @private\n         */\n        _EventListener: function () {\n            var $widget = this,\n                options = this.options.classes,\n                target;\n\n            $widget.element.on('click', '.' + options.optionClass, function () {\n                return $widget._OnClick($(this), $widget);\n            });\n\n            $widget.element.on('change', '.' + options.selectClass, function () {\n                return $widget._OnChange($(this), $widget);\n            });\n\n            $widget.element.on('click', '.' + options.moreButton, function (e) {\n                e.preventDefault();\n\n                return $widget._OnMoreClick($(this));\n            });\n\n            $widget.element.on('keydown', function (e) {\n                if (e.which === 13) {\n                    target = $(e.target);\n\n                    if (target.is('.' + options.optionClass)) {\n                        return $widget._OnClick(target, $widget);\n                    } else if (target.is('.' + options.selectClass)) {\n                        return $widget._OnChange(target, $widget);\n                    } else if (target.is('.' + options.moreButton)) {\n                        e.preventDefault();\n\n                        return $widget._OnMoreClick(target);\n                    }\n                }\n            });\n        },\n\n        /**\n         * Load media gallery using ajax or json config.\n         *\n         * @private\n         */\n        _loadMedia: function () {\n            var $main = this.inProductList ?\n                    this.element.parents('.product-item-info') :\n                    this.element.parents('.column.main'),\n                images;\n\n            if (this.options.useAjax) {\n                this._debouncedLoadProductMedia();\n            }  else {\n                images = this.options.jsonConfig.images[this.getProduct()];\n\n                if (!images) {\n                    images = this.options.mediaGalleryInitial;\n                }\n                this.updateBaseImage(this._sortImages(images), $main, !this.inProductList);\n            }\n        },\n\n        /**\n         * Sorting images array\n         *\n         * @private\n         */\n        _sortImages: function (images) {\n            return _.sortBy(images, function (image) {\n                return parseInt(image.position, 10);\n            });\n        },\n\n        /**\n         * Event for swatch options\n         *\n         * @param {Object} $this\n         * @param {Object} $widget\n         * @private\n         */\n        _OnClick: function ($this, $widget) {\n            var $parent = $this.parents('.' + $widget.options.classes.attributeClass),\n                $wrapper = $this.parents('.' + $widget.options.classes.attributeOptionsWrapper),\n                $label = $parent.find('.' + $widget.options.classes.attributeSelectedOptionLabelClass),\n                attributeId = $parent.data('attribute-id'),\n                $input = $parent.find('.' + $widget.options.classes.attributeInput),\n                checkAdditionalData = JSON.parse(this.options.jsonSwatchConfig[attributeId]['additional_data']),\n                $priceBox = $widget.element.parents($widget.options.selectorProduct)\n                    .find(this.options.selectorProductPrice);\n\n            if ($widget.inProductList) {\n                $input = $widget.productForm.find(\n                    '.' + $widget.options.classes.attributeInput + '[name=\"super_attribute[' + attributeId + ']\"]'\n                );\n            }\n\n            if ($this.hasClass('disabled')) {\n                return;\n            }\n\n            if ($this.hasClass('selected')) {\n                $parent.removeAttr('data-option-selected').find('.selected').removeClass('selected');\n                $input.val('');\n                $label.text('');\n                $this.attr('aria-checked', false);\n            } else {\n                $parent.attr('data-option-selected', $this.data('option-id')).find('.selected').removeClass('selected');\n                $label.text($this.data('option-label'));\n                $input.val($this.data('option-id'));\n                $input.attr('data-attr-name', this._getAttributeCodeById(attributeId));\n                $this.addClass('selected');\n                $widget._toggleCheckedAttributes($this, $wrapper);\n            }\n\n            $widget._Rebuild();\n\n            if ($priceBox.is(':data(mage-priceBox)')) {\n                $widget._UpdatePrice();\n            }\n\n            $(document).trigger('updateMsrpPriceBlock',\n                [\n                    this._getSelectedOptionPriceIndex(),\n                    $widget.options.jsonConfig.optionPrices,\n                    $priceBox\n                ]);\n\n            if (parseInt(checkAdditionalData['update_product_preview_image'], 10) === 1) {\n                $widget._loadMedia();\n            }\n\n            $input.trigger('change');\n        },\n\n        /**\n         * Get selected option price index\n         *\n         * @return {String|undefined}\n         * @private\n         */\n        _getSelectedOptionPriceIndex: function () {\n            var allowedProduct = this._getAllowedProductWithMinPrice(this._CalcProducts());\n\n            if (_.isEmpty(allowedProduct)) {\n                return undefined;\n            }\n\n            return allowedProduct;\n        },\n\n        /**\n         * Get human readable attribute code (eg. size, color) by it ID from configuration\n         *\n         * @param {Number} attributeId\n         * @returns {*}\n         * @private\n         */\n        _getAttributeCodeById: function (attributeId) {\n            var attribute = this.options.jsonConfig.mappedAttributes[attributeId];\n\n            return attribute ? attribute.code : attributeId;\n        },\n\n        /**\n         * Toggle accessibility attributes\n         *\n         * @param {Object} $this\n         * @param {Object} $wrapper\n         * @private\n         */\n        _toggleCheckedAttributes: function ($this, $wrapper) {\n            $wrapper.attr('aria-activedescendant', $this.attr('id'))\n                    .find('.' + this.options.classes.optionClass).attr('aria-checked', false);\n            $this.attr('aria-checked', true);\n        },\n\n        /**\n         * Event for select\n         *\n         * @param {Object} $this\n         * @param {Object} $widget\n         * @private\n         */\n        _OnChange: function ($this, $widget) {\n            var $parent = $this.parents('.' + $widget.options.classes.attributeClass),\n                attributeId = $parent.data('attribute-id'),\n                $input = $parent.find('.' + $widget.options.classes.attributeInput);\n\n            if ($widget.productForm.length > 0) {\n                $input = $widget.productForm.find(\n                    '.' + $widget.options.classes.attributeInput + '[name=\"super_attribute[' + attributeId + ']\"]'\n                );\n            }\n\n            if ($this.val() > 0) {\n                $parent.attr('data-option-selected', $this.val());\n                $input.val($this.val());\n            } else {\n                $parent.removeAttr('data-option-selected');\n                $input.val('');\n            }\n\n            $widget._Rebuild();\n            $widget._UpdatePrice();\n            $widget._loadMedia();\n            $input.trigger('change');\n        },\n\n        /**\n         * Event for more switcher\n         *\n         * @param {Object} $this\n         * @private\n         */\n        _OnMoreClick: function ($this) {\n            $this.nextAll().show();\n            $this.trigger('blur').remove();\n        },\n\n        /**\n         * Rewind options for controls\n         *\n         * @private\n         */\n        _Rewind: function (controls) {\n            controls.find('div[data-option-id], option[data-option-id]')\n                .removeClass('disabled')\n                .prop('disabled', false);\n            controls.find('div[data-option-empty], option[data-option-empty]')\n                .attr('disabled', true)\n                .addClass('disabled')\n                .attr('tabindex', '-1');\n        },\n\n        /**\n         * Rebuild container\n         *\n         * @private\n         */\n        _Rebuild: function () {\n            var $widget = this,\n                controls = $widget.element.find('.' + $widget.options.classes.attributeClass + '[data-attribute-id]'),\n                selected = controls.filter('[data-option-selected]');\n\n            // Enable all options\n            $widget._Rewind(controls);\n\n            // done if nothing selected\n            if (selected.length <= 0) {\n                return;\n            }\n\n            // Disable not available options\n            controls.each(function () {\n                var $this = $(this),\n                    id = $this.data('attribute-id'),\n                    products = $widget._CalcProducts(id);\n\n                if (selected.length === 1 && selected.first().data('attribute-id') === id) {\n                    return;\n                }\n\n                $this.find('[data-option-id]').each(function () {\n                    var $element = $(this),\n                        option = $element.data('option-id');\n\n                    if (!$widget.optionsMap.hasOwnProperty(id) || !$widget.optionsMap[id].hasOwnProperty(option) ||\n                        $element.hasClass('selected') ||\n                        $element.is(':selected')) {\n                        return;\n                    }\n\n                    if (_.intersection(products, $widget.optionsMap[id][option].products).length <= 0) {\n                        $element.attr('disabled', true).addClass('disabled');\n                    }\n                });\n            });\n        },\n\n        /**\n         * Get selected product list\n         *\n         * @returns {Array}\n         * @private\n         */\n        _CalcProducts: function ($skipAttributeId) {\n            var $widget = this,\n                selectedOptions = '.' + $widget.options.classes.attributeClass + '[data-option-selected]',\n                products = [];\n\n            // Generate intersection of products\n            $widget.element.find(selectedOptions).each(function () {\n                var id = $(this).data('attribute-id'),\n                    option = $(this).attr('data-option-selected');\n\n                if ($skipAttributeId !== undefined && $skipAttributeId === id) {\n                    return;\n                }\n\n                if (!$widget.optionsMap.hasOwnProperty(id) || !$widget.optionsMap[id].hasOwnProperty(option)) {\n                    return;\n                }\n\n                if (products.length === 0) {\n                    products = $widget.optionsMap[id][option].products;\n                } else {\n                    products = _.intersection(products, $widget.optionsMap[id][option].products);\n                }\n            });\n\n            return products;\n        },\n\n        /**\n         * Update total price\n         *\n         * @private\n         */\n        _UpdatePrice: function () {\n            var $widget = this,\n                $product = $widget.element.parents($widget.options.selectorProduct),\n                $productPrice = $product.find(this.options.selectorProductPrice),\n                result = $widget._getNewPrices(),\n                tierPriceHtml,\n                isShow;\n\n            $productPrice.trigger(\n                'updatePrice',\n                {\n                    'prices': $widget._getPrices(result, $productPrice.priceBox('option').prices)\n                }\n            );\n\n            isShow = typeof result != 'undefined' && result.oldPrice.amount !== result.finalPrice.amount;\n\n            $productPrice.find('span:first').toggleClass('special-price', isShow);\n\n            $product.find(this.options.slyOldPriceSelector)[isShow ? 'show' : 'hide']();\n\n            if (typeof result != 'undefined' && result.tierPrices && result.tierPrices.length) {\n                if (this.options.tierPriceTemplate) {\n                    tierPriceHtml = mageTemplate(\n                        this.options.tierPriceTemplate,\n                        {\n                            'tierPrices': result.tierPrices,\n                            '$t': $t,\n                            'currencyFormat': this.options.jsonConfig.currencyFormat,\n                            'priceUtils': priceUtils\n                        }\n                    );\n                    $(this.options.tierPriceBlockSelector).html(tierPriceHtml).show();\n                }\n            } else {\n                $(this.options.tierPriceBlockSelector).hide();\n            }\n\n            $(this.options.normalPriceLabelSelector).hide();\n\n            _.each($('.' + this.options.classes.attributeOptionsWrapper), function (attribute) {\n                if ($(attribute).find('.' + this.options.classes.optionClass + '.selected').length === 0) {\n                    if ($(attribute).find('.' + this.options.classes.selectClass).length > 0) {\n                        _.each($(attribute).find('.' + this.options.classes.selectClass), function (dropdown) {\n                            if ($(dropdown).val() === '0') {\n                                $(this.options.normalPriceLabelSelector).show();\n                            }\n                        }.bind(this));\n                    } else {\n                        $(this.options.normalPriceLabelSelector).show();\n                    }\n                }\n            }.bind(this));\n        },\n\n        /**\n         * Get new prices for selected options\n         *\n         * @returns {*}\n         * @private\n         */\n        _getNewPrices: function () {\n            var $widget = this,\n                newPrices = $widget.options.jsonConfig.prices,\n                allowedProduct = this._getAllowedProductWithMinPrice(this._CalcProducts());\n\n            if (!_.isEmpty(allowedProduct)) {\n                newPrices = this.options.jsonConfig.optionPrices[allowedProduct];\n            }\n\n            return newPrices;\n        },\n\n        /**\n         * Get prices\n         *\n         * @param {Object} newPrices\n         * @param {Object} displayPrices\n         * @returns {*}\n         * @private\n         */\n        _getPrices: function (newPrices, displayPrices) {\n            var $widget = this;\n\n            if (_.isEmpty(newPrices)) {\n                newPrices = $widget._getNewPrices();\n            }\n            _.each(displayPrices, function (price, code) {\n\n                if (newPrices[code]) {\n                    displayPrices[code].amount = newPrices[code].amount - displayPrices[code].amount;\n                }\n            });\n\n            return displayPrices;\n        },\n\n        /**\n         * Get product with minimum price from selected options.\n         *\n         * @param {Array} allowedProducts\n         * @returns {String}\n         * @private\n         */\n        _getAllowedProductWithMinPrice: function (allowedProducts) {\n            var optionPrices = this.options.jsonConfig.optionPrices,\n                product = {},\n                optionFinalPrice, optionMinPrice;\n\n            _.each(allowedProducts, function (allowedProduct) {\n                optionFinalPrice = parseFloat(optionPrices[allowedProduct].finalPrice.amount);\n\n                if (_.isEmpty(product) || optionFinalPrice < optionMinPrice) {\n                    optionMinPrice = optionFinalPrice;\n                    product = allowedProduct;\n                }\n            }, this);\n\n            return product;\n        },\n\n        /**\n         * Gets all product media and change current to the needed one\n         *\n         * @private\n         */\n        _LoadProductMedia: function () {\n            var $widget = this,\n                $this = $widget.element,\n                productData = this._determineProductData(),\n                mediaCallData,\n                mediaCacheKey,\n\n                /**\n                 * Processes product media data\n                 *\n                 * @param {Object} data\n                 * @returns void\n                 */\n                mediaSuccessCallback = function (data) {\n                    if (!(mediaCacheKey in $widget.options.mediaCache)) {\n                        $widget.options.mediaCache[mediaCacheKey] = data;\n                    }\n                    $widget._ProductMediaCallback($this, data, productData.isInProductView);\n                    setTimeout(function () {\n                        $widget._DisableProductMediaLoader($this);\n                    }, 300);\n                };\n\n            if (!$widget.options.mediaCallback) {\n                return;\n            }\n\n            mediaCallData = {\n                'product_id': this.getProduct()\n            };\n\n            mediaCacheKey = JSON.stringify(mediaCallData);\n\n            if (mediaCacheKey in $widget.options.mediaCache) {\n                $widget._XhrKiller();\n                $widget._EnableProductMediaLoader($this);\n                mediaSuccessCallback($widget.options.mediaCache[mediaCacheKey]);\n            } else {\n                mediaCallData.isAjax = true;\n                $widget._XhrKiller();\n                $widget._EnableProductMediaLoader($this);\n                $widget.xhr = $.ajax({\n                    url: $widget.options.mediaCallback,\n                    cache: true,\n                    type: 'GET',\n                    dataType: 'json',\n                    data: mediaCallData,\n                    success: mediaSuccessCallback\n                }).done(function () {\n                    $widget._XhrKiller();\n                });\n            }\n        },\n\n        /**\n         * Enable loader\n         *\n         * @param {Object} $this\n         * @private\n         */\n        _EnableProductMediaLoader: function ($this) {\n            var $widget = this;\n\n            if ($('body.catalog-product-view').length > 0) {\n                $this.parents('.column.main').find('.photo.image')\n                    .addClass($widget.options.classes.loader);\n            } else {\n                //Category View\n                $this.parents('.product-item-info').find('.product-image-photo')\n                    .addClass($widget.options.classes.loader);\n            }\n        },\n\n        /**\n         * Disable loader\n         *\n         * @param {Object} $this\n         * @private\n         */\n        _DisableProductMediaLoader: function ($this) {\n            var $widget = this;\n\n            if ($('body.catalog-product-view').length > 0) {\n                $this.parents('.column.main').find('.photo.image')\n                    .removeClass($widget.options.classes.loader);\n            } else {\n                //Category View\n                $this.parents('.product-item-info').find('.product-image-photo')\n                    .removeClass($widget.options.classes.loader);\n            }\n        },\n\n        /**\n         * Callback for product media\n         *\n         * @param {Object} $this\n         * @param {String} response\n         * @param {Boolean} isInProductView\n         * @private\n         */\n        _ProductMediaCallback: function ($this, response, isInProductView) {\n            var $main = isInProduc